/******************************************************************
*
*	CyberNet for C
*
*	Copyright (C) Satoshi Konno 2005
*
*       Copyright (C) 2006 Nokia Corporation. All rights reserved.
*
*       This is licensed under BSD-style license with patent exclusion,
*       see file COPYING.
*
*	File: csubscriber.c
*
*	Revision:
*
*	02/01/05
*		- first revision
*
*	12-Jan-06 Heikki Junnila
*		- Added API comments
******************************************************************/

#include <cybergarage/upnp/event/cevent.h>
#include <cybergarage/upnp/event/cnotify.h>
#include <cybergarage/upnp/cupnp_limit.h>
#include <cybergarage/util/clog.h>

/****************************************
* CG_UPNP_NOUSE_SUBSCRIPTION (Begin)
****************************************/

#if !defined(CG_UPNP_NOUSE_SUBSCRIPTION)

/**
 * Create a new event subscriber
 */
CgUpnpSubscriber *cg_upnp_subscriber_new()
{
	cg_log_debug_l4("Entering...\n");

	CgUpnpSubscriber *sub = (CgUpnpSubscriber *)malloc(sizeof(CgUpnpSubscriber));

	if ( NULL != sub )
	{
		cg_list_node_init((CgList *)sub);
		
		sub->sid = cg_string_new();
		sub->ifAddr = cg_string_new();
		sub->deliveryURL = cg_net_url_new();
		
		cg_upnp_subscriber_settimeout(sub, 0);
		cg_upnp_subscriber_renew(sub);
	}
	
	return sub;

	cg_log_debug_l4("Leaving...\n");
}

/**
 * Destroy an event subscriber
 *
 * @param sub The event subscriber
 */
void cg_upnp_subscriber_delete(CgUpnpSubscriber *sub)
{
	cg_log_debug_l4("Entering...\n");

	cg_upnp_subscriber_clear(sub);
	cg_list_remove((CgList *)sub);

	cg_string_delete(sub->sid);
	cg_string_delete(sub->ifAddr);
	cg_net_url_delete(sub->deliveryURL);
	
	free(sub);

	cg_log_debug_l4("Leaving...\n");
}

/**
 * Clear the contents of an event subscriber
 *
 * @todo Lacks implementation (is it necessary?)
 *
 * @param sub The event subscriber
 */
void cg_upnp_subscriber_clear(CgUpnpSubscriber *sub)
{
	cg_log_debug_l4("Entering...\n");


	cg_log_debug_l4("Leaving...\n");
}

/**
 * Renew a subscription. Essentially sets subscription time (duration) 
 * to zero and resets notify count (== event key).
 *
 * @param sub The event subscriber
 */
void cg_upnp_subscriber_renew(CgUpnpSubscriber *sub)
{
	cg_log_debug_l4("Entering...\n");

	cg_upnp_subscriber_setsubscriptiontime(sub, time(NULL));
	cg_upnp_subscriber_setnotifycount(sub, 0);

	cg_log_debug_l4("Leaving...\n");
}

/**
 * Increment the event notify count by one
 *
 * @param sub The event subscriber
 * @return The new notify count
 */
long cg_upnp_subscriber_incrementnotifycount(CgUpnpSubscriber *sub)
{
	cg_log_debug_l4("Entering...\n");

	if (CG_UPNP_NOTIFY_COUNT_MAX <= sub->notifyCount)
		sub->notifyCount = 0;
	sub->notifyCount++;
	return sub->notifyCount;

	cg_log_debug_l4("Leaving...\n");
}

/**
 * Check, whether a subscriber's event subscription has been expired
 *
 * @param sub The subscriber
 * @return TRUE if the subscription has been expired; otherwise FALSE
 */
BOOL cg_upnp_subscriber_isexpired(CgUpnpSubscriber *sub)
{
	cg_log_debug_l4("Entering...\n");

	long currTime;
	long timeout;
	long expiredTime;
	
	timeout = cg_upnp_subscriber_gettimeout(sub);
	if(timeout == CG_UPNP_SUBSCRIPTION_INFINITE_VALUE) 
		return FALSE; 
			
	currTime = time(NULL);
	expiredTime = cg_upnp_subscriber_getsubscriptiontime(sub) + timeout;
	if (expiredTime < currTime)
		return TRUE;
			
	return FALSE;

	cg_log_debug_l4("Leaving...\n");
}

/**
 * Post a notification to an event subscriber. This is called in a device.
 *
 * @param sub The event subscriber
 * @param statVar The evented state variable
 * @return TRUE if succesful; otherwise FALSE
 */
BOOL cg_upnp_subscriber_notify(CgUpnpSubscriber *sub, CgUpnpStateVariable *statVar)
{
	cg_log_debug_l4("Entering...\n");

	char *varName;
	char *varValue;
	char *host;
	int port;
	CgUpnpNotifyRequest *notifyReq;
	CgUpnpNotifyResponse *notifyRes;
	BOOL notifySuccess;
	
	varName = cg_upnp_statevariable_getname(statVar);
	varValue = cg_upnp_statevariable_getvalue(statVar);

	host = cg_upnp_subscriber_getdeliveryhost(sub);
	port = cg_upnp_subscriber_getdeliveryport(sub);

	notifyReq = cg_upnp_event_notify_request_new();
	cg_upnp_event_notify_request_setpropertysetnode(notifyReq, sub, statVar);
	notifyRes = cg_upnp_event_notify_request_post(notifyReq, host, port);
	notifySuccess = cg_upnp_event_notify_response_issuccessful(notifyRes);
	cg_upnp_event_notify_request_delete(notifyReq);
	
	if (notifySuccess == FALSE)
		return FALSE;
		
	cg_upnp_subscriber_incrementnotifycount(sub);
			
	return TRUE;

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* CG_UPNP_NOUSE_SUBSCRIPTION (End)
****************************************/

#endif
