/*
 * UPnP Browser for Maemo
 *
 * upnp.c
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license with patent exclusion,
 * see file COPYING.
 */

#include <gtk/gtk.h>
#include <cybergarage/util/cstring.h>
#include <cybergarage/contentdirectory/ccontentdirectory.h>
#include <cybergarage/contentdirectory/ccdscontrolpoint.h>
#include <cybergarage/contentdirectory/cdidllite.h>
#include <cybergarage/renderingcontrol/crcscontrolpoint.h>
#include <cybergarage/avtransport/cavtcontrolpoint.h>
#include <cybergarage/avdebug.h>
#include <cybergarage/avcontrol/cavcontrolpoint.h>

#include "upnp.h"
#include "interface.h"
#include "now_playing.h"
#include "browser.h"
#include "helper.h"

extern UPnPBrowserWidgets* widgets;

/**
 * Create a control point and start it.
 */
int init_upnp_controlpoint()
{
	/* Create the cybergarage control point */
	controlPoint = cg_upnp_controlpoint_new();
	cg_upnp_controlpoint_setdevicelistener(controlPoint, device_listener);
	
	/* Start the control point */
        if (cg_upnp_controlpoint_start(controlPoint) == FALSE)
	{
                fprintf(stderr, "Unable to start control point!!\n");
                exit(1);
        }
	else
	{
                fprintf(stderr, "Control point started\n");
	}

	return 0;
}


/**
 * Stop a control point and destroy it.
 */
int destroy_upnp_controlpoint()
{
        cg_upnp_controlpoint_stop(controlPoint);
	fprintf(stderr, "Control point stopped\n");

        cg_upnp_controlpoint_delete(controlPoint);
	fprintf(stderr, "Control point destroyed\n");

	return 0;
}

BOOL get_upnp_content_directory(GtkTreeStore* model,
			        GtkTreeIter* parent,
			        gchar* udn,
			        gchar *parent_id)
{
	CgXmlNodeList* nodelist = NULL;

	char* browseFlag = CG_UPNP_AV_CDS_BROWSE_FLAG_DIRECTCHILDREN;
	char* filter = CG_UPNP_AV_CDS_BROWSE_FILTER_ALL;
	char* startingIndex = CG_UPNP_AV_CDS_BROWSE_STARTINGINDEX_FIRST;
	char* requestedCount = CG_UPNP_AV_CDS_BROWSE_REQUESTEDCOUNT_ALL;
	char* sortCriteria = CG_UPNP_AV_CDS_BROWSE_SORTCRITERIA_NONE;

	char* result = NULL;
	char* numberReturned = NULL;
	char* totalMatches = NULL;
	char* updateID = NULL;

	int action_result = 0;
	BOOL retval = FALSE;
	
	DEBUG("BEGIN: get_upnp_content_directory\n");
	
	g_return_val_if_fail(model != NULL, 0);
	g_return_val_if_fail(parent != NULL, 0);

	/* Invoke a browse action */
	cg_upnp_controlpoint_lock(controlPoint);
	retval = cg_upnp_av_cds_control_browse(
			cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn),
			NULL,
			&parent_id,
			&browseFlag,
			&filter,
			&startingIndex,
			&requestedCount,
			&sortCriteria,
			&result,
			&numberReturned,
			&totalMatches,
			&updateID);
			
	cg_upnp_controlpoint_unlock(controlPoint);

	if (cg_upnp_av_control_iserrorcodesuccessful(action_result))
	{
		/* Create an XML nodelist */
		nodelist = cg_xml_nodelist_new();
		retval = cg_upnp_av_cds_create_cg_xml(result, nodelist);
		
		/* Parse the XML nodelist to the tree model */
		if (retval == TRUE)
		{			
			retval = parse_didllite_nodelist_to_model(model,
								  parent,
								  nodelist,
								  udn);
		}
		
		/* Get rid of the XML nodelist */
		cg_xml_nodelist_delete(nodelist);
	}
	
	free(result);
	free(numberReturned);
	free(totalMatches);
	free(updateID);
	
	/* Set parent node as browsed */
	gtk_tree_store_set(model, parent, FOLDERMODEL_COLUMN_BROWSED, TRUE, -1);

	DEBUG("END: get_upnp_content_directory\n");
	
	return retval;
}

BOOL parse_didllite_nodelist_to_model(GtkTreeStore* model,
				      GtkTreeIter* parent,
				      CgXmlNodeList* nodelist,
				      gchar* udn)
{
	GtkTreeIter iter, child_iter;
	CgXmlNodeList* children = NULL;
	CgXmlNode *node = NULL;
	
	DEBUG("BEGIN: parse_didllite_nodelist_to_model\n");
	
	if (model == NULL || nodelist == NULL || udn == NULL)
	{
		DEBUG("Invalid parameters!\n");
		DEBUG("END: parse_didllite_nodelist_to_model\n");
		return FALSE;
	}

	/* Remove old children */
	while(gtk_tree_model_iter_children(GTK_TREE_MODEL(model), &iter, parent))
	{
		gtk_tree_store_remove(model, &iter);
	}
	
	/* Check the validity of the DIDL-Lite document and get the
	 * child items that actually contain the interesting information */
	children = cg_upnp_av_cds_didllite_getchildren(nodelist, FALSE);
	if (children == NULL)
	{
		DEBUG("END: parse_didllite_nodelist_to_model\n");
		return FALSE;
	}
	
	/* Put the result to the tree */
	for (node = cg_upnp_av_cds_didllite_nodelist_gets(children);
	     node != NULL;
	     node = cg_upnp_av_cds_didllite_node_next(node))
	{		
		if (cg_upnp_av_cds_didllite_node_isname(node,
						    DIDL_LITE_ITEM))
		{
			/* <item> ...  </item> */
			
			gtk_tree_store_append(model, &iter, parent);

			/* gtk_tree_store_set is a REALLY slow function, 
			 * so these have to be set in one function call */
			gtk_tree_store_set(model, &iter,
				FOLDERMODEL_COLUMN_CONTAINER,
					FALSE,
				FOLDERMODEL_COLUMN_DEVUDN,
					udn,
				FOLDERMODEL_COLUMN_ID,
					cg_upnp_av_cds_didllite_node_getattribute(
						node,
						DIDL_LITE_OBJECT_ID),
				-1);

			/* Parse the children of an <item> tag */
			parse_didllite_node_to_model(model, &iter, node);
		}
		else if (cg_upnp_av_cds_didllite_node_isname(node,
							 DIDL_LITE_CONTAINER))
		{
			/* <container> ...  </container> */
			
			gtk_tree_store_append(model, &iter, parent);

			/* gtk_tree_store_set is a REALLY slow function, 
			 * so these have to be set in one function call */
			gtk_tree_store_set(model, &iter,
				FOLDERMODEL_COLUMN_CONTAINER,
					TRUE,
				FOLDERMODEL_COLUMN_DEVUDN,
					udn,
				FOLDERMODEL_COLUMN_ID,
					cg_upnp_av_cds_didllite_node_getattribute(
						node,
						DIDL_LITE_OBJECT_ID),
				-1);

			/* Parse the children of a <container> tag */
			parse_didllite_node_to_model(model, &iter, node);
			
			/* Make dummy child to be able to expand the node */
			gtk_tree_store_append(model, &child_iter, &iter);
			gtk_tree_store_set(model, &child_iter,
				    FOLDERMODEL_COLUMN_CONTAINER, FALSE, -1);
		}
		/*
		else if (cg_upnp_av_cds_didllite_node_isname(node,
							 DIDL_LITE_DESC))
		{
			continue;
		}
		*/
		else
		{
			continue;
		}
	}
	
	DEBUG("END: parse_didllite_nodelist_to_model\n");
	
	return TRUE;
}


BOOL parse_didllite_node_to_model(GtkTreeStore* model,
				  GtkTreeIter* iter,
				  CgXmlNode* node)
{
	CgXmlNodeList* nodelist = NULL;
	CgXmlNodeList* child = NULL;
	gchar* class = NULL;
	gchar* title = NULL;
	gchar* protocol = NULL;
	CgString* res = NULL;
	
	res = cg_string_new();
	
	DEBUG("BEGIN: parse_didllite_node_to_model\n");

	if (model == NULL || iter == NULL || node == NULL)
	{
		return FALSE;
	}
	
	nodelist = cg_upnp_av_cds_didllite_node_getchildnodelist(node);
	if (nodelist == NULL)
	{
		return FALSE;
	}
	
	for (child = cg_upnp_av_cds_didllite_nodelist_gets(nodelist);
	     child != NULL;
	     child = cg_upnp_av_cds_didllite_node_next(child))
	{
		if (cg_upnp_av_cds_didllite_node_isname(child,
						    DIDL_LITE_UPNP_CLASS))
		{
			/* <upnp:class> ... <upnp:class> */
			class = cg_upnp_av_cds_didllite_node_getvalue(child);
		}
		else if (cg_upnp_av_cds_didllite_node_isname(child,
							 DIDL_LITE_DC_TITLE))
		{
			/* <dc:title> ... <dc:title> */
			title = cg_upnp_av_cds_didllite_node_getvalue(child);
		}
		else if (cg_upnp_av_cds_didllite_node_isname(child,
							 DIDL_LITE_RES))
		{
			protocol = 
				cg_upnp_av_cds_didllite_res_getprotocol(child);
                        if (cg_strcmp(protocol, "http-get") == 0)
			{
				if (cg_string_length(res) == 0)
				{
					/* Start a new res list */
					cg_string_addvalue(res, "<");
				}
				else
				{
					/* Continue a res list */
					cg_string_addvalue(res, "><");
				}
			
				/* Add the resource to the res list */
				cg_string_addvalue(res, 
				   cg_upnp_av_cds_didllite_node_getvalue(child));
			}
			
			g_free(protocol);
		}
	}

	if (cg_string_length(res) > 0)
	{
		/* End the res list */
		cg_string_addvalue(res, ">");
	}

	/* gtk_tree_store_set is a REALLY slow function, so these have
	 * to be set in one function call */
	gtk_tree_store_set(model, iter,
			FOLDERMODEL_COLUMN_URI,
			cg_string_getvalue(res),
			FOLDERMODEL_COLUMN_CLASS,
			class,
			FOLDERMODEL_COLUMN_NAME,
			title,
			-1);

	cg_string_delete(res);
	
	DEBUG("END: parse_didllite_node_to_model\n");
	
	return TRUE;
}

/**
 * Update the text displayed in the "Now Playing" area
 * from a remote renderer device
 *
 * @param buffer The GtkTextBuffer* to update
 * @param renderer_udn The UDN of the renderer device
 *
 */
void update_remote_now_playing_text(GtkTextBuffer* buffer,
				    gchar* renderer_udn)
{
	char* instanceID = "0";
	char* transportState = NULL;
	char* trackUri = NULL;
	char* track = NULL;
	char* duration = NULL;
	char* reltime = NULL;
	
	gchar* cached_udn = NULL;
	gchar* cached_uri = NULL;
	gchar* cached_name = NULL;
	
	long dur = 0;
	long rel = 0;
	
	GtkTextIter text_iter;
	int result = 0;
	
	/* Clear text in the now playing bar */
	gtk_text_buffer_set_text(buffer, "\0", -1);
	
	/* Set the text buffer iterator to start  */
	gtk_text_buffer_get_iter_at_offset(buffer, 
					   &text_iter, 0);

	cg_upnp_controlpoint_lock(controlPoint);

	/* Query current transport state */
	result = cg_upnp_av_avt_control_invokeaction(
		cg_upnp_controlpoint_getdevicebyudn(controlPoint, renderer_udn),
		CG_UPNP_AVT_ACTION_GET_TRANSPORTINFO,
		NULL,
		CG_UPNP_AVT_ARG_INSTANCEID,
		&instanceID,
		CG_UPNP_AVT_ARG_CURRENTTRANSPORTSTATE,
		&transportState,
		NULL);

	cg_upnp_controlpoint_unlock(controlPoint);

	/* Set transport status */
	insert_remote_transport_state(buffer,
				      &text_iter,
				      transportState);

	cg_upnp_controlpoint_lock(controlPoint);

	/* Query track name (well, current uri for now) */
	result = cg_upnp_av_avt_control_invokeaction(
		cg_upnp_controlpoint_getdevicebyudn(controlPoint, renderer_udn),
		CG_UPNP_AVT_ACTION_GET_POSITIONINFO,
		NULL,
		CG_UPNP_AVT_ARG_INSTANCEID,
		&instanceID,
		CG_UPNP_AVT_ARG_TRACK,
		&track,
		CG_UPNP_AVT_ARG_TRACKURI,
		&trackUri,
		CG_UPNP_AVT_ARG_TRACKDURATION,
		&duration,
		CG_UPNP_AVT_ARG_RELTIME,
		&reltime,
		NULL);

	cg_upnp_controlpoint_unlock(controlPoint);

	/* Set the maximum value for the seek bar */
	dur = track_duration_to_long(duration);
	rel = track_duration_to_long(reltime);

	if (dur == 0)
	{
		/* If the renderer doesn't know duration, use current pos */
		set_seek_bar_limit_long(rel);
	}
	else
	{
		set_seek_bar_limit_long(dur);
	}

	/* Set the seek bar position */
	set_seek_bar_position_long(rel);
       
	/* Open brackets for reltime */
	gtk_text_buffer_insert_with_tags_by_name(buffer,
					&text_iter, 
					" (\0", -1,
					"play_status",
					NULL);

	/* Insert reltime */
	if (cg_upnp_av_control_iserrorcodesuccessful(result)
	    && 
	    cg_strlen(reltime) > 0)
	{
		/* Set reltime */
		gtk_text_buffer_insert_with_tags_by_name(buffer,
				&text_iter,
				reltime, -1,
				"track_name", NULL);
	}
	else
	{
		/* Set unknown reltime */
		gtk_text_buffer_insert_with_tags_by_name(buffer,
				&text_iter,
				"--:--:--\0", -1,
				"track_name", NULL);
	}

	/* Close reltime brackets */
	gtk_text_buffer_insert_with_tags_by_name(buffer,
					&text_iter, 
					")\0", -1,
					"play_status",
					NULL);

	/* Insert newline */
	gtk_text_buffer_insert(buffer, &text_iter, "\n\0", -1);

	/* Set the current track number */
	if (cg_upnp_av_control_iserrorcodesuccessful(result)
	    &&
	    cg_strlen(track) > 0)
	{
		gtk_text_buffer_insert_with_tags_by_name(buffer,
				&text_iter,
				track, -1,
				"track_name", NULL);

		gtk_text_buffer_insert_with_tags_by_name(buffer,
				&text_iter,
				": \0", -1,
				"track_name", NULL);
	}

	get_current_renderer(&cached_udn, &cached_uri, &cached_name);
	
	/* Set the current track name */
	if (cg_upnp_av_control_iserrorcodesuccessful(result)
	    &&
	    cg_strlen(trackUri) > 0)
	{
		if (cg_strcmp(trackUri, cached_uri) == 0
			&& cached_name != NULL)
		{
			/* Still playing the same track we know */
			gtk_text_buffer_insert_with_tags_by_name(buffer,
				&text_iter,
				cached_name, -1,
				"track_name", NULL);

		}
		else
		{
			/* Unknown track / outside intervention / 
			   renderers have been updated -> get name from CDS */
			if (get_track_name_from_cds(trackUri, &cached_name) 
				== TRUE)
			{
				cache_current_track_for_current_renderer(
						trackUri,
						cached_name);
				
				gtk_text_buffer_insert_with_tags_by_name(
					buffer,
					&text_iter,
					cached_name, -1,
					"track_name", NULL);
			}
			else
			{
				gtk_text_buffer_insert_with_tags_by_name(
					buffer,
					&text_iter,
					trackUri, -1,
					"track_name", NULL);
			}
		}
	}
	else
	{
		/* Unknown track name */
		gtk_text_buffer_insert_with_tags_by_name(buffer,
				&text_iter,
				"Unknown track\0", -1,
				"track_name", NULL);
	}

	free(transportState);
	free(trackUri);
	free(track);
	free(duration);
	free(reltime);
	
	g_free(cached_udn);
	g_free(cached_uri);
	g_free(cached_name);
}

/**
 * Insert remote transport state to text buffer
 *
 * @param buffer The text buffer to insert state to
 * @param text_iter Insert point in the buffer
 * @param state UPnP transport state string
 */
void insert_remote_transport_state(GtkTextBuffer* buffer, 
				   GtkTextIter* text_iter,
				   char* state)
{
	if (cg_strcmp(state, CG_UPNP_AVT_TRANSPORT_STATE_PLAYING) == 0)
	{
		gtk_text_buffer_insert_with_tags_by_name(buffer,
						text_iter,
						"Playing\0", -1,
						"play_status",
						NULL);
	}
	else if (cg_strcmp(state, CG_UPNP_AVT_TRANSPORT_STATE_STOPPED) == 0)
	{
		gtk_text_buffer_insert_with_tags_by_name(buffer,
						text_iter,
						"Stopped\0", -1,
						"play_status",
						NULL);
	}
	else if (cg_strcmp(state, CG_UPNP_AVT_TRANSPORT_STATE_TRANSITIONING) == 0)
	{
		gtk_text_buffer_insert_with_tags_by_name(buffer,
						text_iter,
						"Transitioning\0", -1,
						"play_status",
						NULL);
	}
	else if (cg_strcmp(state, CG_UPNP_AVT_TRANSPORT_STATE_PAUSED_PLAYBACK) == 0 ||
		 cg_strcmp(state, CG_UPNP_AVT_TRANSPORT_STATE_PAUSED_RECORDING) == 0)
	{
		gtk_text_buffer_insert_with_tags_by_name(buffer,
						text_iter,
						"Paused\0", -1,
						"play_status",
						NULL);
	}
	else if (cg_strcmp(state, CG_UPNP_AVT_TRANSPORT_STATE_RECORDING) == 0)
	{
		gtk_text_buffer_insert_with_tags_by_name(buffer,
						text_iter,
						"Recording\0", -1,
						"play_status",
						NULL);
	}
	else if (cg_strcmp(state, CG_UPNP_AVT_TRANSPORT_STATE_NO_MEDIA_PRESENT) == 0)
	{
		gtk_text_buffer_insert_with_tags_by_name(buffer,
						text_iter,
						"No Media Present\0", -1,
						"play_status",
						NULL);
	}
	else
	{
		gtk_text_buffer_insert_with_tags_by_name(buffer,
						text_iter,
						"State not available!\0", -1,
						"play_status",
						NULL);
	}
}

/**
 * Fetch the track name from a CDS service using the resource's URI
 *
 * @param uri The URI of the resource
 * @param name The "friendly" name of the resource pointed by uri
 */
gboolean get_track_name_from_cds(gchar* uri, gchar** name)
{
/*
	CgUpnpDevice* device = NULL;
	
	g_return_val_if_fail(uri != NULL, FALSE);
	
	cg_upnp_controlpoint_lock(controlPoint);

	for (device = cg_upnp_controlpoint_getdevices(controlPoint);
	     device != NULL;
	     device = cg_upnp_device_next(device))
	{
		if (cg_strstr(uri,
		    cg_upnp_device_getinterfaceaddressfromssdppacket(device))
		    != -1)
		{
			break;
		}
	}
	
	g_return_val_if_fail(device != NULL, FALSE);

	cg_upnp_controlpoint_unlock(controlPoint);
*/	
	/* TODO: CDS Search */
	
	return FALSE;
}

/**
 * Start playing media in a remote renderer device
 *
 * @param udn The UDN of the UPnP renderer device
 * @param uri The URI of the selected media
 * @param media_class The class of the media
 * @param objectID The object ID of the media to be played
 * @param srv_udn UDN of the content server (for metadata extraction)
 */
gboolean play_remote_media(gchar* udn, gchar* uri, gchar* media_class,
				gchar* objectID, gchar* srv_udn)
{
	int result = 0;
	
	char* instanceID = "0";
	char* speed = "1";
	char* metaData = NULL;
	
	g_return_val_if_fail(udn != NULL, FALSE);
	
	cg_upnp_controlpoint_lock(controlPoint);

	/* Get meta data */
	if (widgets->settings->sendMetaData == TRUE)
	{
		get_meta_data(cg_upnp_controlpoint_getdevicebyudn(controlPoint, 
								  srv_udn), 
				objectID,
				&metaData);
	}

	/* Set the URI for the media to be played */
	result = cg_upnp_av_avt_control_setavtransporturi(
		cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn),
		NULL,
		&instanceID,
		&uri,
		&metaData);
	
	/* Start play */
	if (cg_upnp_av_control_iserrorcodesuccessful(result))
	{
		result = cg_upnp_av_avt_control_play(
			cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn),
			NULL,
			&instanceID,
			&speed);
	
	}
	
	cg_upnp_controlpoint_unlock(controlPoint);

	return cg_upnp_av_control_iserrorcodesuccessful(result);
}

gboolean get_meta_data(CgUpnpDevice* device, gchar* id, gchar** metaData)
{
	char* browseFlag = CG_UPNP_AV_CDS_BROWSE_FLAG_METADATA;
	char* filter = CG_UPNP_AV_CDS_BROWSE_FILTER_ALL;
	char* startingIndex = CG_UPNP_AV_CDS_BROWSE_STARTINGINDEX_FIRST;
	char* requestedCount = CG_UPNP_AV_CDS_BROWSE_REQUESTEDCOUNT_ALL;
	char* sortCriteria = CG_UPNP_AV_CDS_BROWSE_SORTCRITERIA_NONE;

	char* numberReturned = NULL;
	char* totalMatches = NULL;
	char* updateID = NULL;
	int retval = 0;
	
	DEBUG("BEGIN: get_meta_data\n");

	retval = cg_upnp_av_cds_control_browse(device, 
					       NULL,
					       &id,
					       &browseFlag,
					       &filter,
					       &startingIndex, 
					       &requestedCount,
					       &sortCriteria,
					       metaData,
					       &numberReturned,
					       &totalMatches,
					       &updateID);

	free(numberReturned);
	free(totalMatches);
	free(updateID);

	DEBUG("END: get_meta_data\n");

	return cg_upnp_av_control_iserrorcodesuccessful(retval);
}

/**
 * Stop playing remote media
 *
 * @param udn The UDN of the UPnP renderer device
 */
gboolean stop_remote_media(gchar* udn)
{
	int result = 0;
	char* instanceID = "0";

	g_return_val_if_fail(udn != NULL, FALSE);

	cg_upnp_controlpoint_lock(controlPoint);
	
	result = cg_upnp_av_avt_control_stop(
		cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn), 
		NULL,
		&instanceID);

	cg_upnp_controlpoint_unlock(controlPoint);

	return cg_upnp_av_control_iserrorcodesuccessful(result);
}

/**
 * Pause playing remote media
 *
 * @param udn The UDN of the UPnP renderer device
 */
gboolean pause_remote_media(gchar* udn)
{
	int result = 0;
	char* instanceID = "0";
	char* state = NULL;
	char* speed = "1";

	g_return_val_if_fail(udn != NULL, FALSE);
	
	cg_upnp_controlpoint_lock(controlPoint);
	
	/* Query current transport state */
	result = cg_upnp_av_avt_control_invokeaction(
		cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn),
		CG_UPNP_AVT_ACTION_GET_TRANSPORTINFO,
		NULL,
		CG_UPNP_AVT_ARG_INSTANCEID,
		&instanceID,
		CG_UPNP_AVT_ARG_CURRENTTRANSPORTSTATE,
		&state,
		NULL);
	
	if (cg_upnp_av_control_iserrorcodesuccessful(result)
	    && 
	    (cg_strcmp(state, CG_UPNP_AVT_TRANSPORT_STATE_PAUSED_PLAYBACK) == 0 
	     ||
	     cg_strcmp(state, CG_UPNP_AVT_TRANSPORT_STATE_PAUSED_RECORDING) == 0))
	{
		result = cg_upnp_av_avt_control_play(
			cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn),
			NULL,
			&instanceID,
			&speed);
	}
	else
	{
		result = cg_upnp_av_avt_control_pause(
			cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn),
			NULL,
			&instanceID);
	}

	cg_upnp_controlpoint_unlock(controlPoint);

	free(state);
	
	return cg_upnp_av_control_iserrorcodesuccessful(result);
}

/**
 * Skip to the next remote media
 *
 * @param udn The UDN of the UPnP renderer device
 */
gboolean next_remote_media(gchar* udn)
{
	int result = 0;
	char* instanceID = "0";

	g_return_val_if_fail(udn != NULL, FALSE);

	cg_upnp_controlpoint_lock(controlPoint);
	
	result = cg_upnp_av_avt_control_next(
		cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn),
		NULL,
		&instanceID);

	cg_upnp_controlpoint_unlock(controlPoint);

	return cg_upnp_av_control_iserrorcodesuccessful(result);
}

/**
 * Skip to the previous remote media
 *
 * @param udn The UDN of the UPnP renderer device
 */
gboolean previous_remote_media(gchar* udn)
{
	int result = 0;
	char* instanceID = "0";

	g_return_val_if_fail(udn != NULL, FALSE);

	cg_upnp_controlpoint_lock(controlPoint);
	
	result = cg_upnp_av_avt_control_previous(
		cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn),
		NULL,
		&instanceID);

	cg_upnp_controlpoint_unlock(controlPoint);

	return cg_upnp_av_control_iserrorcodesuccessful(result);
}

/**
 * Check, whether the remote server is muted
 *
 * @param udn The UDN of the UPnP renderer device
 * @param mute The renderer's mute status
 * @return TRUE if successful; otherwise FALSE
 */
gboolean get_remote_mute(gchar* udn, gboolean* mute)
{
	int result = 0;
	char* instanceID = "0";
	char* channel = CG_UPNP_RCS_ARG_CHANNEL_MASTER;
	char* mute_string = NULL;
       
	g_return_val_if_fail(udn != NULL, FALSE);
       
	cg_upnp_controlpoint_lock(controlPoint);
       
	result = cg_upnp_av_rcs_control_getmute(
			cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn),
			NULL,
			&instanceID,
		        &channel,
		        &mute_string);
       
	cg_upnp_controlpoint_unlock(controlPoint);

	if (cg_upnp_av_control_iserrorcodesuccessful(result))
	{
		if (cg_strcmp(mute_string, "1") == 0 ||
		    cg_strcmp(mute_string, "TRUE") == 0 ||
		    cg_strcmp(mute_string, "true") == 0)
		{
			*mute = TRUE;
		}
		else
		{
			*mute = FALSE;
		}
	}

	free(mute_string);

	return cg_upnp_av_control_iserrorcodesuccessful(result);
}

/**
 * Un/Mute a remote media renderer
 *
 * @param udn The UDN of the UPnP renderer device
 * @param mute TRUE to mute; FALSE to unmute the device
 */
gboolean mute_remote_media(gchar* udn, gboolean mute)
{
	int result = 0;
	char* instanceID = "0";
	char* channel = CG_UPNP_RCS_ARG_CHANNEL_MASTER;
	char* mute_string = (mute == TRUE) ? "1" : "0";
	
	g_return_val_if_fail(udn != NULL, FALSE);
	
	cg_upnp_controlpoint_lock(controlPoint);
	
	result = cg_upnp_av_rcs_control_setmute(
		cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn),
		NULL,
		&instanceID,
		&channel,
		&mute_string);
	
	cg_upnp_controlpoint_unlock(controlPoint);

	return cg_upnp_av_control_iserrorcodesuccessful(result);
}

/**
 * Increase volume on a remote media renderer
 *
 * @param udn The UDN of the UPnP renderer device
 */
gboolean increase_remote_volume(gchar* udn)
{
	int result = 0;
	char* instanceID = "0";
	char* channel = CG_UPNP_RCS_ARG_CHANNEL_MASTER;
	char* volume = NULL;

	g_return_val_if_fail(udn != NULL, FALSE);
	
	cg_upnp_controlpoint_lock(controlPoint);
	
	result = cg_upnp_av_rcs_control_getvolume(
		cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn),
		NULL,
		&instanceID,
		&channel,
		&volume);
	
	if (cg_upnp_av_control_iserrorcodesuccessful(result))
	{
		int value = atoi(volume);
		
		value += 1;
		
		free(volume);
		
		volume = (char*) malloc(sizeof(char) * (value / 10) + 2);
		sprintf(volume, "%d", value);
		
		result = cg_upnp_av_rcs_control_setvolume(
			cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn),
			NULL,
			&instanceID,
			&channel,
			&volume);
	}

	cg_upnp_controlpoint_unlock(controlPoint);

	free(volume);
	
	return cg_upnp_av_control_iserrorcodesuccessful(result);
}

/**
 * Decrease volume on a remote media renderer
 *
 * @param udn The UDN of the UPnP renderer device
 */
gboolean decrease_remote_volume(gchar* udn)
{
	int result = 0;
	char* instanceID = "0";
	char* channel = CG_UPNP_RCS_ARG_CHANNEL_MASTER;
	char* volume = NULL;
	
	cg_upnp_controlpoint_lock(controlPoint);
	
	result = cg_upnp_av_rcs_control_getvolume(
		cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn),
		NULL,
		instanceID,
		channel,
		volume);

	if (cg_upnp_av_control_iserrorcodesuccessful(result))
	{
		int value = atoi(volume);
		
		if (value == 0)
			value = 0;
		else
			value -= 1;
		
		free(volume);
		
		volume = (char*) malloc(sizeof(char) * (value / 10) + 2);
		sprintf(volume, "%d", value);
		
		result = cg_upnp_av_rcs_control_setvolume(
			cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn),
			NULL,
			&instanceID,
			&channel,
			&volume);
	}

	cg_upnp_controlpoint_unlock(controlPoint);

	free(volume);
	
	return cg_upnp_av_control_iserrorcodesuccessful(result);
}

/**
 * Get remote renderer presets
 *
 * @param udn The UDN of the UPnP renderer device
 * @param presets A comma-separated-value of available presets
 */
gboolean get_remote_renderer_presets(gchar* udn, gchar** presets)
{
	char* instanceID = "0";
	int result = 0;

	g_return_val_if_fail(udn != NULL, FALSE);
	
	cg_upnp_controlpoint_lock(controlPoint);

	result = cg_upnp_av_rcs_control_listpresets(
		cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn),
		NULL,
		&instanceID,
		presets);
	
	cg_upnp_controlpoint_unlock(controlPoint);

	return cg_upnp_av_control_iserrorcodesuccessful(result);
}

/**
 * Seek remote media to the given position
 *
 * @param udn The UDN of the renderer device
 * @param seekto Seek to this position
 */
void seek_remote_media(gchar* udn, long seekto)
{
	int result = 0;
	char* instanceID = "0";
	char* seekMode = "REL_TIME";
	char* seekTarget = NULL;
       
	g_return_if_fail(udn != NULL);

	seekTarget = long_to_track_duration(seekto);
	
	cg_upnp_controlpoint_lock(controlPoint);
       
	result = cg_upnp_av_avt_control_seek(
			cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn),
			NULL,
			&instanceID,
			&seekMode,
			&seekTarget);

	cg_upnp_controlpoint_unlock(controlPoint);

	free(seekTarget);
}

/**
 * Check, whether a UPnP device supports a certain action
 *
 * @param udn The UDN of the device to check
 * @param serviceType Type of the service to look for from the device
 * @param actionName Name of the action to look for from the service
 */
gboolean device_hasaction(gchar* udn,
			  gchar* serviceType,
			  gchar* actionName)
{
	gboolean result = FALSE;
	CgUpnpDevice* device = NULL;
	CgUpnpService* service = NULL;

	cg_upnp_controlpoint_lock(controlPoint);

	device = cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn);

	if (device == NULL)
	{
		cg_upnp_controlpoint_unlock(controlPoint);
		return FALSE;
	}
	
	service = cg_upnp_device_getservicebytype(device, serviceType);

	if (service == NULL)
	{
		cg_upnp_controlpoint_unlock(controlPoint);
		return FALSE;
	}

	if (cg_upnp_service_getactionbyname(service, actionName) != NULL)
	{
		result = TRUE;
	}
	else
	{
		result = FALSE;
	}

	cg_upnp_controlpoint_unlock(controlPoint);

	return result;
}

gboolean add_device_idle(gpointer user_data)
{
	add_device((gchar*) user_data);
	g_free(user_data);
	return FALSE;
}

gboolean remove_device_idle(gpointer user_data)
{
	remove_device((gchar*) user_data);
	g_free(user_data);
	return FALSE;
}


/**
 * Device listener callback that is called by the clinkc stack whenever
 * a device leaves or enters the network.
 *
 * @param udn The UDN of the device entering/leaving the network
 * @param status The device's new status
 */
void device_listener(char* udn, CgUpnpDeviceStatus status)
{	/*
	static GStaticMutex mutex = G_STATIC_MUTEX_INIT;
	g_static_mutex_lock(&mutex);
	*/
	if (status == CgUpnpDeviceStatusAdded)
	{
		/* A device has entered the network */
		/* add_device(udn); */
		g_idle_add(add_device_idle, (gpointer) g_strdup(udn));
	}
	else if (status == CgUpnpDeviceStatusRemoved ||
		 status == CgUpnpDeviceStatusInvalid)
	{
		/* Device left the network or is unusable */
		/* remove_device(udn); */
		g_idle_add(remove_device_idle, (gpointer) g_strdup(udn));
	}
	else if (status == CgUpnpDeviceStatusUpdated)
	{
		/* TODO: Update? */
		
		/* Brainless remove + add didn't work here, it just
		   causes annoyed users */ 
	}
	/*
	g_static_mutex_unlock(&mutex);
	*/
}

/**
 * Remove a device from CDS or renderer models
 *
 * @param udn The UDN of the device to remove
 */
void remove_device(gchar* udn)
{
	GtkTreeIter iter;

	if (udn == NULL)
	{
		return;
	}
	
	if (find_device_from_model(
			GTK_TREE_MODEL(widgets->folder_model),
			udn, FOLDERMODEL_COLUMN_DEVUDN, &iter) == TRUE)
	{
		/* The device was found from CDS servers */
		gtk_tree_store_remove(widgets->folder_model, &iter);
	}
	else if (find_device_from_model(
			GTK_TREE_MODEL(widgets->renderer_model),
			udn, RENDERERMODEL_COLUMN_DEVUDN, &iter) == TRUE)
	{
		/* The device was found from renderers */
		gtk_list_store_remove(widgets->renderer_model, &iter);
	}
}

/**
 * Add a device either to CDS servers or renderers
 *
 * @param udn The UDN of the device to add
 */
void add_device(gchar* udn)
{
	GtkTreeIter iter;
	CgUpnpDevice* dev = NULL;
	gchar* tmp_name = NULL;
	gchar* tmp_udn = NULL;
	
	cg_upnp_controlpoint_lock(controlPoint);
	
	/* Add a new device to the model */
	dev = cg_upnp_controlpoint_getdevicebyudn(controlPoint, udn);
	if (dev == NULL)
	{
		cg_upnp_controlpoint_unlock(controlPoint);
		return;
	}		
	
	if (cg_upnp_device_getservicebytype(dev,
					    CG_UPNP_CDS_SERVICE_TYPE))
	{
		if (find_device_from_model(
			GTK_TREE_MODEL(widgets->folder_model),
			udn, FOLDERMODEL_COLUMN_DEVUDN, &iter) == FALSE)
		{				
			/* Create deep copies so we can unlock */
			tmp_name = cg_strdup(
				cg_upnp_device_getfriendlyname(dev));
			tmp_udn = cg_strdup(udn);
			
			/* Must unlock before adding the device */
			cg_upnp_controlpoint_unlock(controlPoint);
			
			add_cds_device(tmp_name, tmp_udn,
				       widgets->folder_model);
				
			g_free(tmp_name);
			g_free(tmp_udn);
		}
		else
		{
			cg_upnp_controlpoint_unlock(controlPoint);
		}
	}
	else if (cg_upnp_device_getservicebytype(dev,
					CG_UPNP_RCS_SERVICE_TYPE) &&
		 cg_upnp_device_getservicebytype(dev,
					CG_UPNP_AVT_SERVICE_TYPE))
	{		
		if (find_device_from_model(
			GTK_TREE_MODEL(widgets->renderer_model),
			udn, RENDERERMODEL_COLUMN_DEVUDN, &iter) == FALSE)
		{
			/* Create deep copies so we can unlock */
			tmp_name = cg_strdup(
				cg_upnp_device_getfriendlyname(dev));
			tmp_udn = cg_strdup(udn);
				
			/* Must unlock before adding the device */
			cg_upnp_controlpoint_unlock(controlPoint);

			add_renderer_device(tmp_name, tmp_udn,
					    widgets->renderer_model);

			g_free(tmp_name);
			g_free(tmp_udn);
		}
		else
		{
			cg_upnp_controlpoint_unlock(controlPoint);
		}
	}
	else
	{
		cg_upnp_controlpoint_unlock(controlPoint);
	}
}

/**
 * Add a device to the list of CDS devices.
 *
 * @param dev The UPnP device to add
 * @param store The GtkTreeStore to manipulate
 */
void add_cds_device(gchar* name, gchar* udn, GtkTreeStore* store)
{
	GtkTreeIter iter;
	
	if (name == NULL || udn == NULL || store == NULL)
	{
		return;
	}
	
	/* Get iterator */
	gtk_tree_store_append(store, &iter, NULL);
	
	/* Set the contents of the current tree node */
	gtk_tree_store_set(store, &iter,
			FOLDERMODEL_COLUMN_NAME,
				name,
			FOLDERMODEL_COLUMN_CONTAINER,
				TRUE,
			FOLDERMODEL_COLUMN_DEVUDN,
				udn,
			FOLDERMODEL_COLUMN_ID,
				"0",
			FOLDERMODEL_COLUMN_BROWSED,
				FALSE,
			-1);
}

/**
 * Add the device to the list of renderers.
 *
 * @param dev The device to add
 * @param store The GtkListStore to manipulate
 */
void add_renderer_device(gchar* name, gchar* udn, GtkListStore* store)
{
	GtkTreeIter iter;
	
	if (name == NULL || udn == NULL || store == NULL)
	{
		return;
	}
	
	/* Get iterator */
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter, 
			RENDERERMODEL_COLUMN_NAME,
				name,
			RENDERERMODEL_COLUMN_DEVUDN,
				udn,
			-1);
}

