/*
 * UPnP Browser for Maemo
 *
 * now_playing.c
 *
 * Copyright 2005 Nokia Corporation. All rights reserved.
 *
 * This is licensed under BSD-style license with patent exclusion,
 * see file COPYING.
 */

#ifdef MAEMO
#include <hildon-widgets/gtk-infoprint.h>
#include <hildon-widgets/hildon-defines.h>
#include <osso-mime.h>
#include "player.h"
#endif

#include <gtk/gtk.h>

#include "now_playing.h"
#include "interface.h"
#include "helper.h"
#include "upnp.h"

#include <cybergarage/renderingcontrol/crcscontrolpoint.h>
#include <cybergarage/avdebug.h>

extern UPnPBrowserWidgets* widgets;

/**
 * Create the "Now Playing" tuuba
 */
int create_now_playing_bar()
{
	GtkToolItem *item = NULL;
	GtkWidget* frame = NULL;
	
	/* Create a new toolbar */
	widgets->now_playing_bar = gtk_toolbar_new();

	frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_IN);
	gtk_container_add(GTK_CONTAINER(frame), widgets->now_playing_bar); 

	gtk_container_add(GTK_CONTAINER(widgets->toolbar_vbox),
					frame);
	
	/* Now playing text view */
	create_now_playing_text_view_model();

	/* Separator */
	item = gtk_separator_tool_item_new();
	gtk_separator_tool_item_set_draw(GTK_SEPARATOR_TOOL_ITEM(item), 
					 FALSE);
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->now_playing_bar),
					item, -1);

	/* Renderer combo */
	create_renderer_combo_model();

	return 0;
}

/**
 * Create the renderer combo & model
 */
int create_renderer_combo_model()
{
	GtkTreeIter iter;
	GtkToolItem *item = NULL;
	GtkCellRenderer* renderer = NULL;
	
	/* Create renderer model */
	widgets->renderer_model = gtk_list_store_new(RENDERERMODEL_COLUMNS,
				/* dev name */	G_TYPE_STRING,
				/* dev udn */	G_TYPE_STRING,
				/* curr uri */	G_TYPE_STRING,
				/* curr name */	G_TYPE_STRING);
	/* Create a combo for renderer devices */
	widgets->renderer_combo = 
	 gtk_combo_box_new_with_model(GTK_TREE_MODEL(widgets->renderer_model));

	GTK_WIDGET_UNSET_FLAGS(widgets->renderer_combo, GTK_CAN_FOCUS);
	
	/* Renderer tool item */
	item = gtk_tool_item_new();
	GTK_WIDGET_UNSET_FLAGS(item, GTK_CAN_FOCUS);
	gtk_tool_item_set_expand(item, FALSE);
	gtk_container_add(GTK_CONTAINER(item), widgets->renderer_combo);
	gtk_tool_item_set_tooltip(item, widgets->tips,
				  "Renderer",/* tooltip */
				  "Renderer" /* private tip */);
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->now_playing_bar), /* toolbar */
			   item, -1);

	/* Attach a cell renderer to the combo */
	renderer = gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(widgets->renderer_combo), 
					renderer,
					TRUE);

        gtk_cell_layout_set_attributes(GTK_CELL_LAYOUT(widgets->renderer_combo),
					renderer,
					"text",
					RENDERERMODEL_COLUMN_NAME,
                                        NULL);
	
	gtk_cell_renderer_set_fixed_size(renderer, RENDERER_COMBO_WIDTH, -1);
	
	/* Put "Local" as a default to the renderer combo */
	gtk_list_store_append(widgets->renderer_model, &iter);
#ifdef MAEMO
	gtk_list_store_set(widgets->renderer_model, &iter,
		RENDERERMODEL_COLUMN_NAME, "Nokia 770",
		RENDERERMODEL_COLUMN_DEVUDN, NULL, -1);
#else
	gtk_list_store_set(widgets->renderer_model, &iter,
		RENDERERMODEL_COLUMN_NAME, "Local Player",
		RENDERERMODEL_COLUMN_DEVUDN, NULL, -1);
#endif
	gtk_combo_box_set_active(GTK_COMBO_BOX(widgets->renderer_combo),
				RENDERERMODEL_ROW_LOCAL);

	/* Catch selection changes */
	g_signal_connect_after(G_OBJECT(widgets->renderer_combo),
			 "changed",
			 G_CALLBACK(renderer_combo_selection_changed),
			 widgets);
	
	return 0;
}

/** 
 * Create the "Now Playing" text view
 */
int create_now_playing_text_view_model()
{
	GtkToolItem *item = NULL;
	
	/* Create the text buffer */
	widgets->now_playing_text_buffer = gtk_text_buffer_new(NULL);
	
	/* Create the text view with buffer */
	widgets->now_playing_text_view = 
	    gtk_text_view_new_with_buffer(widgets->now_playing_text_buffer);

	/* Deny editing */
	gtk_text_view_set_editable(GTK_TEXT_VIEW(widgets->now_playing_text_view), 
				   FALSE);

	/* Create some tags to display different kinds of text */
	gtk_text_buffer_create_tag(widgets->now_playing_text_buffer, 
				"play_status",
				"weight", PANGO_WEIGHT_BOLD,
				"size", 12 * PANGO_SCALE,
				"wrap_mode", GTK_WRAP_WORD,
				NULL);

	gtk_text_buffer_create_tag(widgets->now_playing_text_buffer, 
				"track_name",
				"weight", PANGO_WEIGHT_NORMAL,
				"size", 10 * PANGO_SCALE,
				"wrap_mode", GTK_WRAP_WORD,
				NULL);

	/* Renderer tool item */
	item = gtk_tool_item_new();
	GTK_WIDGET_UNSET_FLAGS(item, GTK_CAN_FOCUS);
	gtk_tool_item_set_expand(item, TRUE);
	gtk_container_add(GTK_CONTAINER(item), widgets->now_playing_text_view);
	gtk_tool_item_set_tooltip(item, widgets->tips,
				  "Now Playing",
				  "Now Playing");
	gtk_toolbar_insert(GTK_TOOLBAR(widgets->now_playing_bar), item, -1);

	return 0;
}

/**
 * Callback for selecting an item from the renderer combo
 */
void renderer_combo_selection_changed(GtkComboBox *widget,
					gpointer user_data)
{
	update_now_playing_text();
	update_active_renderer_widgets();
}

/**
 * Update the text displayed in the "Now Playing" area
 *
 */
void update_now_playing_text()
{
	GtkTreeIter renderer_iter;
	gchar* renderer_udn = NULL;

	g_return_if_fail(widgets->renderer_combo != NULL);
	
	/* Get the current selection from the combo box */
	if (gtk_combo_box_get_active_iter(
				GTK_COMBO_BOX(widgets->renderer_combo),
				&renderer_iter))
	{
		/* Get the renderer device */
		gtk_tree_model_get(GTK_TREE_MODEL(widgets->renderer_model),
				&renderer_iter, 
				RENDERERMODEL_COLUMN_DEVUDN,
				&renderer_udn, -1);
	}

	if (renderer_udn == NULL)
	{
		update_local_now_playing_text(widgets->now_playing_text_buffer);
	}
	else
	{
		/* Remote renderer */
		update_remote_now_playing_text(widgets->now_playing_text_buffer,
						renderer_udn);
		g_free(renderer_udn);
	}
}

/**
 * Update the text displayed in the "Now Playing" area
 * from local renderer
 *
 */
void update_local_now_playing_text(GtkTextBuffer* buffer)
{
	GtkTextIter text_iter;
	gint position = 0;
	gchar *buf = NULL;
	gboolean result = FALSE;
	gint length = 0;
	
	gchar* udn = NULL;
	gchar* uri = NULL;
	gchar* name = NULL;
	
	/* Clear text in the now playing bar */
	gtk_text_buffer_set_text(buffer, "\0", -1);

	gtk_text_buffer_get_iter_at_offset(widgets->now_playing_text_buffer, 
					   &text_iter, 0);
#ifdef MAEMO	
	result = player_get_position(widgets->osso, &position, &length, NULL);
#endif
	if( (position / 1000) > 0 ) {
		gint msec = position;
		gint hours = 0;
		gint minutes = 0;
		gint seconds = 0;
		
		hours = msec / 3600000;
		msec -= 3600000 * hours;
		minutes = msec / 60000;
		msec -= 60000 * minutes;
		seconds = msec / 1000;
		
		if( hours ) {
			buf = g_strdup_printf("%d:%02d:%02d", hours, minutes,
					      seconds);
		} else {
			buf = g_strdup_printf("%02d:%02d", minutes, seconds);
		}
	} else {
		buf = g_strdup("00:00");
	}

	/* Set maximum limit for seek bar */
	if (length == 0)
	{
		set_seek_bar_limit_long(position);
	}
	else
	{
		set_seek_bar_limit_long(length);
	}

	set_seek_bar_position_long(position);

	if (result == TRUE && position > 0)
	{
		gtk_text_buffer_insert_with_tags_by_name(
					widgets->now_playing_text_buffer,
					&text_iter, 
					"Playing \0", -1,
					"play_status",
					NULL);		
	}
	else
	{
		gtk_text_buffer_insert_with_tags_by_name(
					widgets->now_playing_text_buffer,
					&text_iter, 
					"Stopped \0", -1,
					"play_status",
					NULL);
	}
	
	/* Open brackets for duration */
	gtk_text_buffer_insert_with_tags_by_name(
					widgets->now_playing_text_buffer,
					&text_iter, 
					" (\0", -1,
					"play_status",
					NULL);
	
	/* Insert track duration */
	if (result == TRUE && position > 0)
	{
		gtk_text_buffer_insert_with_tags_by_name(
					widgets->now_playing_text_buffer,
					&text_iter, 
					buf, -1,
					"track_name",
					NULL);
	}
	else
	{
		gtk_text_buffer_insert_with_tags_by_name(
					widgets->now_playing_text_buffer,
					&text_iter, 
					"--:--:--", -1,
					"track_name",
					NULL);
	}

	if (buf != NULL) g_free(buf);
	
	/* Close duration brackets */
	gtk_text_buffer_insert_with_tags_by_name(
					widgets->now_playing_text_buffer,
					&text_iter, 
					")\0", -1,
					"play_status",
					NULL);

	/* Insert newline */
	gtk_text_buffer_insert(buffer, &text_iter, "\n\0", -1);
	
	/* Get cached track information */
	get_current_renderer(&udn, &uri, &name);
	
	/* Last locally played URI */
	if (name != NULL)
	{
		gtk_text_buffer_insert_with_tags_by_name(
					widgets->now_playing_text_buffer,
					&text_iter, 
					"1: \0", -1,
					"track_name",
					NULL);
	
		gtk_text_buffer_insert_with_tags_by_name(
					widgets->now_playing_text_buffer,
					&text_iter, 
					name, -1,
					"track_name",
					NULL);
	}
	else
	{
		gtk_text_buffer_insert_with_tags_by_name(
					widgets->now_playing_text_buffer,
					&text_iter, 
					"0: \0", -1,
					"track_name",
					NULL);

		gtk_text_buffer_insert_with_tags_by_name(
					widgets->now_playing_text_buffer,
					&text_iter, 
					"Unknown track", -1,
					"track_name",
					NULL);
	}
	
	g_free(udn);
	g_free(uri);
	g_free(name);
}

void reset_now_playing_update_timeout(guint seconds)
{
	if (widgets->now_playing_timeout > 0)
		g_source_remove(widgets->now_playing_timeout);

	if (seconds > 0)
	{
		widgets->now_playing_timeout =
		    g_timeout_add(1000 * seconds, 
				  now_playing_update_timeout, NULL);
	}
}

gboolean now_playing_update_timeout(gpointer data)
{
	update_now_playing_text();

	return TRUE;
}

/**
 * Helper function to get the currently selected renderer device's UDN
 *
 * @param udn The Renderer device's UDN
 * @return TRUE if UDN was set; otherwise FALSE
 */
gboolean get_current_renderer(gchar** udn, gchar** uri, gchar** name)
{
	GtkTreeIter iter;
	
	/* Get the current selection from the combo box */
	if (gtk_combo_box_get_active_iter(
				GTK_COMBO_BOX(widgets->renderer_combo),
				&iter) == TRUE)
	{
		/* Get the renderer device */
		gtk_tree_model_get(GTK_TREE_MODEL(widgets->renderer_model),
				&iter,
				RENDERERMODEL_COLUMN_DEVUDN,
				udn,
				RENDERERMODEL_COLUMN_CURRENT_URI,
				uri,
				RENDERERMODEL_COLUMN_CURRENT_NAME,
				name,
				-1);
		
		if (*udn == NULL) return FALSE;
		
		return TRUE;
	}
	else
	{
		return FALSE;
	}
}

/**
 * Cache the currently playing URI and track name to the renderer model
 *
 * @param uri The URI of the currently playing track (used for comparison)
 * @param trackName The name of the currently playing track
 * @return TRUE if current renderer was updated; otherwise FALSE
 */
gboolean cache_current_track_for_current_renderer(gchar* uri, gchar* trackName)
{
	GtkTreeIter iter;
	
	/* Get the current selection from the combo box */
	if (gtk_combo_box_get_active_iter(
				GTK_COMBO_BOX(widgets->renderer_combo),
				&iter) == TRUE)
	{
		/* Get the renderer device */
		gtk_list_store_set(GTK_LIST_STORE(widgets->renderer_model),
				&iter, 
				RENDERERMODEL_COLUMN_CURRENT_URI,
				uri,
				RENDERERMODEL_COLUMN_CURRENT_NAME,
				trackName,
				-1);

		return TRUE;
	}
	else
	{
		return FALSE;
	}
}

/**
 * Checks for some actions from the selected renderer device and
 * de/activates widgets according to their presence/absence
 */
void update_active_renderer_widgets()
{
	GtkTreeIter renderer_iter;
	gchar* renderer_udn = NULL;
	
	/* Get the current selection from the combo box */
	if (gtk_combo_box_get_active_iter(
				GTK_COMBO_BOX(widgets->renderer_combo),
				&renderer_iter))
	{
		/* Get the renderer device */
		gtk_tree_model_get(GTK_TREE_MODEL(widgets->renderer_model),
				&renderer_iter, 
				RENDERERMODEL_COLUMN_DEVUDN,
				&renderer_udn, -1);
	}

	if (renderer_udn == NULL)
	{
		gtk_widget_set_sensitive(GTK_WIDGET(widgets->mute_button),
					 TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(widgets->volume_inc_button),
					TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(widgets->volume_dec_button),
					TRUE);
		gtk_widget_set_sensitive(GTK_WIDGET(widgets->presets_button),
					FALSE);
	}
	else
	{
		/* SetMute */
		if (device_hasaction(renderer_udn,
				     CG_UPNP_RCS_SERVICE_TYPE,
				     CG_UPNP_RCS_ACTION_SETMUTE) == FALSE)
		{
			gtk_widget_set_sensitive(
					GTK_WIDGET(widgets->mute_button),
					FALSE);
		}
		else
		{
			gtk_widget_set_sensitive(
					GTK_WIDGET(widgets->mute_button),
					TRUE);
		}
		
		/* SetVolume */
		if (device_hasaction(renderer_udn,
				     CG_UPNP_RCS_SERVICE_TYPE,
				     CG_UPNP_RCS_ACTION_SETVOLUME) == FALSE)
		{
			gtk_widget_set_sensitive(
					GTK_WIDGET(widgets->volume_inc_button),
					FALSE);
			gtk_widget_set_sensitive(
					GTK_WIDGET(widgets->volume_dec_button),
					FALSE);
		}
		else
		{
			gtk_widget_set_sensitive(
					GTK_WIDGET(widgets->volume_inc_button),
					TRUE);
			gtk_widget_set_sensitive(
					GTK_WIDGET(widgets->volume_dec_button),
					TRUE);
		}
		
		/* Presets */
		if (device_hasaction(renderer_udn,
				     CG_UPNP_RCS_SERVICE_TYPE,
				     CG_UPNP_RCS_ACTION_SELECTPRESET) == FALSE)
		{
			gtk_widget_set_sensitive(
					GTK_WIDGET(widgets->presets_button),
					FALSE);
		}
		else
		{
			gtk_widget_set_sensitive(
					GTK_WIDGET(widgets->presets_button),
					TRUE);
		}

		
		g_free(renderer_udn);
	}
}

void set_seek_bar_limit(gchar* max_limit)
{
       g_return_if_fail(max_limit != NULL);
       set_seek_bar_limit_long(track_duration_to_long(max_limit));
}

/**
 * Set the maximum limit for the seek bar.
 *
 * @param max_limit The limit in seconds (long)
 */
void set_seek_bar_limit_long(long max_limit)
{
       if (max_limit == 0)
       {
               gtk_widget_set_sensitive(widgets->seek_bar, FALSE);
       }
       else
       {
               gtk_widget_set_sensitive(widgets->seek_bar, TRUE);
               gtk_range_set_range(GTK_RANGE(widgets->seek_bar),
                                   0, max_limit);
       }
}

/**
 * Set the seek bar's position as a TrackDuration string
 *
 */
void set_seek_bar_position(gchar* position)
{
       g_return_if_fail(position != NULL);
       set_seek_bar_position_long(track_duration_to_long(position));
}

/**
 * Set the seek bar's position as a long representation (seconds)
 */
void set_seek_bar_position_long(long position)
{
       if (GTK_WIDGET_IS_SENSITIVE(widgets->seek_bar))
       {
               g_signal_handler_block(widgets->seek_bar, 
                                      widgets->seek_bar_signal);
       
               gtk_range_set_value(GTK_RANGE(widgets->seek_bar), 
                                   (gdouble) position);
       
               g_signal_handler_unblock(widgets->seek_bar, 
                                        widgets->seek_bar_signal);
       }
}
