/* Certificate Management library
 * 
 * Copyright (C) 2005 Nokia. All rights reserved.
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/** test.c **/
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "test.h"

#include "cst.h"
#include <glib.h>

#include <openssl/evp.h>
#include <openssl/pem.h>
#include <openssl/x509v3.h>
#include <openssl/err.h>
#include <openssl/objects.h>
#include <openssl/rand.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

//#define NEED_MEMCHECK

#ifdef NEED_MEMCHECK
#include <mcheck.h>
#endif

void test_main(PARAM * param);
void init_param(PARAM * param);
void free_param(PARAM * param);

int main(int argc, char *argv[])
{
#ifdef NEED_MEMCHECK
    mtrace();
#endif

    /* Standard OpenSSL init */
    OpenSSL_add_all_algorithms();
    //ERR_load_crypto_strings();
    //ERR_load_RAND_strings();

    printf("*********************************************************\n");
    printf("   BEGIN TEST\n");
    printf("*********************************************************\n");

    PARAM param;
    memset(&param, 0, sizeof(param));

    int i;
    for (i = 0; i < 1; i++)
    {
        test_main(&param);
    }

    printf("*********************************************************\n");
    printf("   Total \tOK: %i\tERROR: %i\n", param.counter_ok,
           param.counter_err);
    printf("*********************************************************\n");
    exit((0 == param.counter_err ? 0 : 1));

    EVP_cleanup();
    //RAND_cleanup();
}

void init_param(PARAM * param)
{
    TEST_TITLE("Initialize test");

    param->cert = test_load_cert("files/cert.pem");
    TEST_DO_FATAL(NULL != param->cert);

    param->ca = test_load_cert("files/cacert.pem");
    TEST_DO_FATAL(NULL != param->ca);
    param->name = X509_get_subject_name(param->ca);

    param->user_cert = test_load_cert("files/pair/c1.pem");
    TEST_DO_FATAL(NULL != param->user_cert);
    param->user_name = X509_get_subject_name(param->user_cert);
   
    param->hotbox_cert = test_load_cert("files/sites/hotbox.pem");
    TEST_DO_FATAL(NULL != param->hotbox_cert);

    param->jboss_cert = test_load_cert("files/sites/jboss.pem");
    TEST_DO_FATAL(NULL != param->jboss_cert);

    param->private_key = test_load_private_key("files/priv_key.pem", NULL);
    TEST_DO_FATAL(NULL != param->private_key);
   
    param->user_private_key = test_load_private_key("files/pair/k1.pem", "123");
    TEST_DO_FATAL(NULL != param->user_private_key);
    
    param->public_key = test_load_public_key("files/pub_key.pem");
    TEST_DO_FATAL(NULL != param->public_key);
        
}

void free_param(PARAM * param)
{
    TEST_TITLE("Free test param");
    
    X509_free(param->cert);
    X509_free(param->ca);
    X509_free(param->user_cert);
    X509_free(param->hotbox_cert);
    X509_free(param->jboss_cert);
}

void test_main(PARAM * param)
{
    TEST_TITLE("All tests");

    test_open_crashed(param);
    
    /* Remove file storage */
    unlink(DB);
    
    /* Test open empty file storage */
    TEST_TITLE("Open empty file storage");
    param->st = CST_open_file(DB, FALSE, PASSWORD);
    TEST_DO_SIMPLE(CST_last_error() == CST_ERROR_OK);
    TEST_DO_FATAL(NULL != param->st);
    
    /* Set initial param */
    init_param(param);

    /* Test import multiple cert */
    test_import(param);

    /* Verify cert using imported certs */
    test_verify(param);

    /* Get certificate chain */
    test_chain(param);

    /* Assign private key with cert */
    test_assign(param);

    /* Set and default defaults */
    test_set_default(param);
    test_get_default(param);

    /* Set and get additional param for cert */
    test_set_folder(param);
    test_set_purpose(param);
    test_get_folder(param);
    test_get_purpose(param);

    test_open(param);
    
    /* Set S/MIME capabilities */
    test_capab_set(param);
    test_capab_get(param);

    /* Save storage and reopen it. For test store functions*/
    TEST_TITLE("Save file storage");
    CST_save(param->st);
    TEST_DO_SIMPLE(CST_last_error() == CST_ERROR_OK);

    TEST_TITLE("Close storage");
    CST_free(param->st);
    TEST_DO_SIMPLE(CST_last_error() == CST_ERROR_OK);



    
    TEST_TITLE("Open file storage");
    param->st = CST_open_file(DB, FALSE, PASSWORD);
    TEST_DO_SIMPLE(CST_last_error() == CST_ERROR_OK);
    TEST_DO_FATAL(NULL != param->st);

    /* Test backup */
    test_backup(param);

    /* Test search functions */
    test_search(param);

    /* Test some OpenSSL functions */
    test_openssl(param);

    /* Test certificate properties */
    test_serial(param);
    test_fingerprint(param);
    test_date(param);
    test_other_prop(param);

    test_open(param);

    /* Test export */
    test_export(param);
    
    /* Get additional cert properties after save and open */
    test_get_purpose(param);
    test_get_folder(param);
    test_get_default(param);

    /* Get S/MIME capabilities*/
    test_capab_get(param);

    /* Test CRL - NEED ADD revoked certificate */
    test_revoked(param);

    /* Test delete operations */
    test_delete(param);
    test_delete_assigned_key(param);
    
    
    TEST_TITLE("Close storage");
    CST_free(param->st);

    test_multi_process(param);

    free_param(param);
}

void test_backup(PARAM * param)
{
    TEST_TITLE("Backup");
    /* TODO: remove before create */
    /* TODO: check existence and open */
    CST_backup(param->st, "backup.cst", "other password");
    TEST_DO_SIMPLE(CST_last_error() == CST_ERROR_OK);
}

void test_delete(PARAM * param)
{
    TEST_TITLE("Delete (NEED ADD KEY and CRL)");

    X509 * x = NULL;
    x = CST_get_cert(param->st, param->cert_uid);
    TEST_DO_SIMPLE(NULL != x);
    if (x) X509_free(x);
    TEST_DO_SIMPLE(CST_ERROR_OK == CST_delete_cert(param->st, param->cert_uid)); 
    x = CST_get_cert(param->st, param->cert_uid);
    TEST_DO_SIMPLE(NULL == x);
    if (x) X509_free(x);
}

void test_delete_assigned_key(PARAM * param)
{
    TEST_TITLE("Delete assigned key");
    
    guint key_id = CST_get_assigned_key(param->st, param->user_cert_uid);

    X509_NAME *name = CST_get_key_account(param->st, key_id);    
    /* Try to delete key */
    CST_delete_priv_key(param->st, key_id);
    /* Check if assignment link was cleared */
    TEST_DO_SIMPLE(CST_get_assigned_key(param->st, param->user_cert_uid) == 0);
    /* Check if key is not still here */
    TEST_DO_SIMPLE(CST_priv_key_search_by_name(param->st, name) == 0);

    X509_NAME_free(name);
}

void test_export(PARAM * param)
{
    TEST_TITLE("Export all certs and keys");
    FILE *export_fp;
    export_fp = fopen("exported.txt", "w");
    if (export_fp)
    {
        CST_export_all(param->st, export_fp, CST_FOLDER_OTHER);
        CST_export_cert(param->st, param->ca, export_fp);
        //CST_export_all_priv_key(param->st, param->name, export_fp,
        //                        PASSWORD);
        CST_export_all_pub_key(param->st, param->name, export_fp);
        //CST_export_priv_key(st, , export_fp, PASSWORD);
        //CST_export_pub_key(st, , export_fp);
        fclose(export_fp);
    }

}

static int test_date_eq(const time_t time, 
        const int year, const int month, const int day,
        int h, int m, int s)
{
    struct tm *stm;
    stm = localtime(&time);
    
    if (stm->tm_year != (year - 1900)) return FALSE;
    if (stm->tm_mon != (month - 1)) return FALSE;
    if (stm->tm_mday != day) return FALSE;
    if (stm->tm_hour != h) return FALSE;
    if (stm->tm_min != m) return FALSE;
    if (stm->tm_sec != s) return FALSE;
    
    return TRUE;
}

void test_date(PARAM * param)
{
    TEST_TITLE("Valid from and valid to");
    
    TEST_DO_SIMPLE(test_date_eq(CST_get_valid_from(param->ca), 2004, 12, 22, 16, 22, 38));
    TEST_DO_SIMPLE(test_date_eq(CST_get_valid_to(param->ca), 2005, 12, 22, 16, 22, 38));
    
    TEST_DO_SIMPLE(test_date_eq(CST_get_valid_from(param->cert), 2004, 12, 22, 16, 30, 57));
    TEST_DO_SIMPLE(test_date_eq(CST_get_valid_to(param->cert), 2005, 12, 22, 16, 30, 57));
}

void test_revoked(PARAM * param)
{
    TEST_TITLE("CST_is_revoked");
    TEST_DO_SIMPLE(!CST_is_revoked(param->st, param->cert));
    TEST_DO_SIMPLE(!CST_is_revoked(param->st, param->ca));
}

static void test_assign_op(PARAM * param, const guint certID, X509 * x)
{
    g_assert(x);
    GSList *list;
    
    /* Searh key by cert subject name */
    list = CST_priv_key_search_by_name(param->st,
                X509_get_subject_name(x));
    TEST_DO_FATAL(NULL != list);
    
    int key_id = GPOINTER_TO_INT(list->data);
    
    /* Assign cert with key */
    TEST_DO_SIMPLE(CST_assign(param->st, certID, key_id, DEF_PASS) == CST_ERROR_OK);

    /* Try assign incorrect cert with private key */
    if (X509_cmp(param->cert, x) != 0)
    {
        TEST_DO_SIMPLE(CST_assign(param->st, param->cert_uid, key_id, DEF_PASS) != CST_ERROR_OK);
    }

    //free_array_EVP_PKEY(list);
    g_slist_free(list);
}

void test_assign(PARAM * param)
{
    TEST_TITLE("Assign private key and certificate"); 
    test_assign_op(param, param->ca_uid, param->ca);
    test_assign_op(param, param->user_cert_uid, param->user_cert);
}


void test_set_default_op(PARAM * param, X509 * x)
{
    guint  cid = CST_search_by_UID(param->st, 
            X509_get_issuer_name(x), 
            X509_get_serialNumber(x));
    TEST_DO_SIMPLE(cid > 0);
    TEST_DO_SIMPLE(CST_set_default(param->st, cid) ==
                   CST_ERROR_OK);
}

void test_set_default(PARAM * param)
{
    TEST_TITLE("Default certificate");
    test_set_default_op(param, param->ca);
    test_set_default_op(param, param->user_cert);
}

void test_get_default_op(PARAM * param, X509_NAME * xname, char *email)
{
    g_assert(xname && email);
    
    EVP_PKEY *k = CST_get_priv_key_default(param->st, email, DEF_PASS);
    TEST_DO_SIMPLE(k != NULL);
    if (k) 
    {
        EVP_PKEY_free(k);
    }

    X509 *x = CST_default_cert(param->st, email);
    TEST_DO_SIMPLE(x != NULL);
    if (x) 
    {
        TEST_DO_SIMPLE(0 == X509_NAME_cmp(xname, X509_get_subject_name(x)));
        X509_free(x);
    }
}

void test_get_default(PARAM * param)
{
    TEST_TITLE("Get default");
    /* test_get_default_op(param, param->name); */
    test_get_default_op(param, param->user_name, "ivanov@mail.ru");

    TEST_DO_SIMPLE(CST_default_cert_id(param->st, "info@valicert.com") > 0);
    TEST_DO_SIMPLE(CST_default_cert_id(param->st, "admin@digsigtrust.com") > 0);
    TEST_DO_SIMPLE(CST_default_cert_id(param->st, "www@com") == 0);
    TEST_DO_SIMPLE(CST_default_cert_id(param->st, "ips@mail.ips.es") > 0);
}

void test_set_folder(PARAM * param)
{
    TEST_TITLE("Folder");
    
    CST_set_folder(param->st, param->ca_uid, CST_FOLDER_CA);
    TEST_DO_SIMPLE(CST_FOLDER_CA == CST_get_folder(param->st, param->ca_uid));

    CST_set_folder(param->st, param->user_cert_uid, CST_FOLDER_PERSONAL);
    TEST_DO_SIMPLE(CST_FOLDER_PERSONAL ==
                   CST_get_folder(param->st, param->user_cert_uid));
    TEST_DO_SIMPLE(CST_FOLDER_CA !=
                   CST_get_folder(param->st, param->user_cert_uid));
    TEST_DO_SIMPLE(CST_FOLDER_OTHER !=
                   CST_get_folder(param->st, param->user_cert_uid));

}

void test_set_purpose_ca(PARAM * param)
{
    TEST_TITLE("Set purpose (trust) CA for each in folder CA");
    
    GSList * list = CST_search_by_folder(param->st, CST_FOLDER_CA);

    int result = TRUE;
    
    if (list)
    {
        GSList * i;
        guint certID;
        for (i = list; i != NULL; i = i->next)
        {
            certID = GPOINTER_TO_UINT(i->data);
            result = result && (CST_set_purpose(param->st, certID, CST_PURPOSE_CA, TRUE) == CST_ERROR_OK);
        }
        g_slist_free(list);
    }
    else
    {
        result = FALSE;
    }

    TEST_DO_SIMPLE(result);
}

void test_set_purpose(PARAM * param)
{
    TEST_TITLE("Purpose (trust)");
    
    CST_set_purpose(param->st, param->ca_uid, CST_PURPOSE_CA, TRUE);
    TEST_DO_SIMPLE(CST_is_purpose(param->st, param->ca_uid, CST_PURPOSE_CA) >= 0);
    TEST_DO_SIMPLE(CST_is_purpose(param->st, param->ca_uid, CST_PURPOSE_CA));
    TEST_DO_SIMPLE(!CST_is_purpose(param->st, param->ca_uid, CST_PURPOSE_SSL_CLIENT));

    CST_set_purpose(param->st, param->cert_uid, CST_PURPOSE_SSL_CLIENT | CST_PURPOSE_SSL_SERVER, TRUE);
    TEST_DO_SIMPLE(CST_is_purpose(param->st, param->cert_uid, CST_PURPOSE_SSL_SERVER));
    CST_set_purpose(param->st, param->cert_uid, CST_PURPOSE_SSL_SERVER, FALSE);

    CST_set_purpose(param->st, param->user_cert_uid, CST_PURPOSE_SSL_CLIENT, TRUE);
}



void test_get_folder(PARAM * param)
{
    TEST_TITLE("Folder");
    
    TEST_DO_SIMPLE(CST_FOLDER_CA == CST_get_folder(param->st, param->ca_uid));
    TEST_DO_SIMPLE(CST_FOLDER_CA !=
                   CST_get_folder(param->st, param->user_cert_uid));
    TEST_DO_SIMPLE(CST_FOLDER_PERSONAL ==
                   CST_get_folder(param->st, param->user_cert_uid));
}

void test_get_purpose(PARAM * param)
{
    TEST_TITLE("Purpose (trust)");
    TEST_DO_SIMPLE(CST_is_purpose(param->st, param->ca_uid, CST_PURPOSE_CA) >= 0);
    TEST_DO_SIMPLE(CST_is_purpose(param->st, param->ca_uid, CST_PURPOSE_CA));
    TEST_DO_SIMPLE(!CST_is_purpose(param->st, param->ca_uid, CST_PURPOSE_SSL_CLIENT));

    TEST_DO_SIMPLE(CST_is_purpose(param->st, param->cert_uid, CST_PURPOSE_SSL_CLIENT));
    TEST_DO_SIMPLE(!CST_is_purpose(param->st, param->cert_uid, CST_PURPOSE_SSL_SERVER));
    TEST_DO_SIMPLE(!CST_is_purpose(param->st, param->cert_uid, CST_PURPOSE_ALL));
}

static void free_stack_ASN1_OBJECT(STACK_OF(ASN1_OBJECT) * list)
{
    if (list)
    {
        int i;
        for (i = 0; i < sk_ASN1_OBJECT_num(list); i++)
        {
            ASN1_OBJECT_free(sk_ASN1_OBJECT_value(list, i));
        }
        sk_ASN1_OBJECT_free(list);
    }
}

void test_capab_set(PARAM * param)
{
    TEST_TITLE("S/MIME Capabilities [set]");
    
    TEST_DO_SIMPLE(CST_set_capability(param->st, param->user_cert_uid, 
            OBJ_nid2obj(NID_sha1), NULL, 0) == CST_ERROR_OK);
    
    TEST_DO_SIMPLE(CST_set_capability(param->st, param->user_cert_uid, 
            OBJ_nid2obj(NID_rsaEncryption), NULL, 0) == CST_ERROR_OK);

    TEST_DO_SIMPLE(CST_set_capability(param->st, param->user_cert_uid, 
            OBJ_nid2obj(NID_sha1WithRSAEncryption), NULL, 0) == CST_ERROR_OK);

    TEST_DO_SIMPLE(TRUE == CST_is_capability(param->st, param->user_cert_uid, 
        OBJ_nid2obj(NID_sha1WithRSAEncryption)));

    
    TEST_DO_SIMPLE(CST_ERROR_OK == CST_delete_capability(param->st, param->user_cert_uid,
        OBJ_nid2obj(NID_sha1WithRSAEncryption)));

    TEST_DO_SIMPLE(FALSE == CST_is_capability(param->st, param->user_cert_uid, 
        OBJ_nid2obj(NID_sha1WithRSAEncryption)));
    
}

void test_capab_get(PARAM * param)
{
    TEST_TITLE("S/MIME Capabilities [get]");

    TEST_DO_SIMPLE(TRUE == CST_is_capability(param->st, param->user_cert_uid, 
        OBJ_nid2obj(NID_rsaEncryption)));

    TEST_DO_SIMPLE(TRUE == CST_is_capability(param->st, param->user_cert_uid, 
        OBJ_nid2obj(NID_sha1)));
    
    STACK_OF(ASN1_OBJECT) * list = CST_get_capabilities(param->st,param->user_cert_uid);
    TEST_DO_SIMPLE(sk_ASN1_OBJECT_num(list) == 2);
    free_stack_ASN1_OBJECT(list);
    
}

void test_open(PARAM * param)
{
    TEST_TITLE("Try open more instances");

    CST * st = CST_open_file(DB, FALSE, NULL);

    TEST_DO_SIMPLE(NULL != st);

    if (st)
    {
        CST_free(st);
    }
}

void test_open_crashed(PARAM * param)
{
    TEST_TITLE("Try open crashed database");
    int fd = open(DB_CRASHED, O_RDWR | O_CREAT | O_TRUNC, 0640);
    close(fd);
    CST * st = CST_open_file(DB_CRASHED, FALSE, NULL);    
    TEST_DO_SIMPLE(NULL == st);
    TEST_DO_SIMPLE(CST_ERROR_DBSTRUCTURE_CORRUPT == CST_last_error());
}
