/* Certificate Management library
 * 
 * Copyright (C) 2005 Nokia. All rights reserved.
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/**
    @file cst_cert_idx.c

    Certificate Management Library

    Certificate functions (indexes)
*/

#include "cst_t.h"

/**
    Check existence cert 
*/
gboolean cert_is_exist(CST * st, CERT * c)
{
    g_assert(st && c);

    if (g_tree_lookup(st->certs, c))
    {
        return TRUE;
    } else
    {
        return FALSE;
    }
}

static void cert_put_idx_main(CST * st, CERT * c)
{
    g_tree_insert(st->certs, c, c);
}

static void cert_remove_idx_main(CST * st, CERT * c)
{
    g_tree_remove(st->certs, c);
}

static void cert_put_idx_name(CST * st, CERT * c)
{
    GSList *list = (GSList *) g_tree_lookup(st->idx_cert_name, c->name);
    list = g_slist_append(list, c);
    g_tree_replace(st->idx_cert_name, c->name, list);
}

static void cert_remove_idx_name(CST * st, CERT * c)
{
    GSList *list = (GSList *) g_tree_lookup(st->idx_cert_name, c->name);

    list = g_slist_remove(list, c);

    if (!list)
    {
        g_tree_remove(st->idx_cert_name, c->name);
    } else
    {
        CERT *a = (CERT *) list->data;
        g_tree_replace(st->idx_cert_name, a->name, list);
    }
}


static void cert_put_idx_domain_name(CST * st, CERT * c)
{
    if (c->domain_name)
    {
        GSList *list =
            (GSList *) g_tree_lookup(st->idx_cert_dns, c->domain_name);
        list = g_slist_append(list, c);
        g_tree_replace(st->idx_cert_dns, c->domain_name, list);
    }
}

static void cert_remove_idx_domain_name(CST * st, CERT * c)
{
    if (c->domain_name)
    {
        GSList *list =
            (GSList *) g_tree_lookup(st->idx_cert_dns, c->domain_name);

        list = g_slist_remove(list, c);

        if (!list)
        {
            g_tree_remove(st->idx_cert_dns, c->domain_name);
        } else
        {
            CERT *a = (CERT *) list->data;
            g_tree_replace(st->idx_cert_dns, a->domain_name, list);
        }
    }
}

static void cert_put_idx_email(CST * st, CERT * c)
{
    if (c->email)
    {
        GSList *list =
            (GSList *) g_tree_lookup(st->idx_cert_email, c->email);
        list = g_slist_append(list, c);
        g_tree_replace(st->idx_cert_email, c->email, list);
    }
}

static void cert_remove_idx_email(CST * st, CERT * c)
{
    if (c->email)
    {
        GSList *list =
            (GSList *) g_tree_lookup(st->idx_cert_email, c->email);

        list = g_slist_remove(list, c);

        if (!list)
        {
            g_tree_remove(st->idx_cert_email, c->email);
        } else
        {
            CERT *a = (CERT *) list->data;
            g_tree_replace(st->idx_cert_email, a->email, list);
        }
    }
}

static void cert_put_idx_uid(CST * st, CERT * c)
{
    g_assert(st && c);
    g_assert(c->uid > 0);
    g_hash_table_insert(st->idx_cert_uid, SEQNUM_TO_POINTER(c->uid), c);
}

static void cert_remove_idx_uid(CST * st, CERT * c)
{
    g_assert(st && c);
    g_assert(c->uid > 0);
    g_hash_table_remove(st->idx_cert_uid, GUINT_TO_POINTER(c->uid));
}

static void cert_put_idx_str(GHashTable * hash, char *key, const t_seqnum value)
{
    g_assert(hash && key && value);
    GSList *list = (GSList *) g_hash_table_lookup(hash, key);
    list = g_slist_append(list, SEQNUM_TO_POINTER(value));
    char *key_copy = g_strdup(key);
    g_hash_table_insert(hash, key_copy, list);
}

static void cert_remove_idx_str(GHashTable * hash, char *key, const t_seqnum value)
{
    g_assert(hash && key && value);
    GSList *list = (GSList *) g_hash_table_lookup(hash, key);
    list = g_slist_remove(list, SEQNUM_TO_POINTER(value)); 
    if (!list) 
    { 
        g_hash_table_remove(hash, key);
    } 
    else 
    { 
        char *key_copy = g_strdup(key);
        g_hash_table_insert(hash, key_copy, list);
    }
}

static void cert_put_idx_serial(CST * st, CERT * c)
{
    g_assert(st && c);
    g_assert(c->serial);
    cert_put_idx_str(st->idx_cert_serial, c->serial, c->uid);
}

static void cert_remove_idx_serial(CST * st, CERT * c)
{
    g_assert(st && c);
    g_assert(c->serial);
    cert_remove_idx_str(st->idx_cert_serial, c->serial, c->uid);
}

static void cert_put_idx_fingerprint(CST * st, CERT * c)
{
    g_assert(st && c);
    g_assert(c->fingerprint);
    cert_put_idx_str(st->idx_cert_fingerprint, c->fingerprint, c->uid);
}

static void cert_remove_idx_fingerprint(CST * st, CERT * c)
{
    g_assert(st && c);
    g_assert(c->fingerprint);
    cert_remove_idx_str(st->idx_cert_fingerprint, c->fingerprint, c->uid);
}

/**
    Append cert to list and update indexes
    See: cert_remove()
*/
gboolean cert_put(CST * st, CERT * c)
{
    g_assert(st && c);
    g_assert(c->uid > 0);
    
    if (!cert_is_exist(st, c))
    {
        cert_put_idx_main(st, c);
        cert_put_idx_name(st, c);
        cert_put_idx_domain_name(st, c);
        cert_put_idx_email(st, c);
        cert_put_idx_uid(st, c);
        cert_put_idx_serial(st, c);
        cert_put_idx_fingerprint(st, c);
        return TRUE;
    } else
    {
        return FALSE;
    }
}

int cert_remove(CST * st, const t_seqnum id)
{
    g_assert(st && (id > 0));

    CERT * c = cert_get_by_id(st, id);

    if (c)
    {
        g_assert(c != NULL);
        
        cert_remove_idx_main(st, c);
        cert_remove_idx_name(st, c);
        cert_remove_idx_domain_name(st, c);
        cert_remove_idx_email(st, c);
        cert_remove_idx_uid(st, c);
        cert_remove_idx_serial(st, c);
        cert_remove_idx_fingerprint(st, c);
                
        cert_free(st, c);

        return CST_error(CST_ERROR_OK);
    }
    return CST_error(CST_ERROR_CERT_NOTFOUND);
}

int cert_remove_x(CST * st, X509 * x)
{
    g_assert(st && x);
    return cert_remove_uid(st, X509_get_issuer_name(x), X509_get_serialNumber(x));
}

int cert_remove_uid(CST * st, X509_NAME * iname, ASN1_INTEGER * sn)
{
    g_assert(st && iname && sn);
    t_seqnum certID = cert_search_by_UID(st, iname, sn);

    if (certID > 0)
    {
        return cert_remove(st, certID);
    }
    else
    {
        return CST_error(CST_ERROR_CERT_NOTFOUND);
    }
}

CERT * cert_get_by_id(CST * st, const t_seqnum id)
{
    TRACE("*");
    g_assert(st); 
    
    CERT * result = NULL;
    
    if ((st->idx_cert_uid) && (id > 0))
    {
        result = g_hash_table_lookup(st->idx_cert_uid, SEQNUM_TO_POINTER(id));

        if (!result)
        {
            CST_error(CST_ERROR_CERT_NOTFOUND);
        }
        else
        {
            CST_error(CST_ERROR_OK);
        }
    }

    return result;
}
