/**
  @file main.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
    
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <string.h>
#include <glib.h>
#include <glib-object.h>
#include <openobex/obex.h>

#include "obc-main.h"
#include "main.h"

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

static char *program_name;

/* For getopt */
static struct option const long_options[] = {
    { "help",     no_argument,       0, 'h' },
    { "device",   required_argument, 0, 'd' },
    { "bda",      required_argument, 0, 'b' },
    { "role",     required_argument, 0, 'r' },
    { "command",  required_argument, 0, 'c' },
    { "channel",  required_argument, 0, 'C' },
    { "auth",     no_argument,       0, 'a' },
    { "encrypt",  no_argument,       0, 'e' },
    { "ftp",      no_argument,       0, 'f' },
    { "opp",      no_argument,       0, 'p' },
    { "pbap",     no_argument,       0, 'P' },
    { "version",  no_argument,       0, 'V' },
#ifdef USE_BTCOND
    { "nobtcond", no_argument,       0, 'b' },
#endif
    { NULL, 0, NULL, 0}
};

/* Print usage information and exit with status */
static void usage(int status)
{
    printf("%s - OBEX Bluetooth client %s\n", program_name, VERSION);

    printf("Compilation flags: ");
#ifdef DEBUG
    printf("+DEBUG ");
#else
    printf("-DEBUG ");
#endif

#ifdef USE_BTCOND
    printf("+USE_BTCOND ");
#else
    printf("-USE_BTCOND ");
#endif

#ifdef HAVE_LIBREADLINE
    printf("+HAVE_LIBREADLINE ");
#else
    printf("-HAVE_LIBREADLINE ");
#endif

    printf(
     "\nUsage: %s [OPTION]...\n"
     "Options:\n"
     "-h, --help                 Display this help and exit\n"
     "-V, --version              Output version information and exit\n"
     "-d, --device               Last argument is device node, not BDA\n"
     "-r, --role                 Switch role (e.g. \"slave\" or \"master\")\n"
     "-a, --auth                 Authenticate connection\n"
     "-e, --encrypt              Encrypt connection\n"
     "-f, --ftp                  Use OBEX File Transfer Profile (default)\n"
     "-p, --opp                  Use OBEX Object Push Profile\n"
     "-P, --pbap                 Use Phone Book Access Profile\n"
     "-c, --command              Run given command and exit\n"
     "-C, --channel              Use specific RFCOMM channel\n"
#ifdef USE_BTCOND
     "-b, --nobtcond             Don't use btcond to connect\n"
#endif
     "\n", program_name);

    exit(status);
}

/* Process commandline options. Returns index of first
 * non-option argument */
static int decode_switches(int argc, char *argv[], ObcCfg *cfg)
{
    int c;

    /* Set defaults */
    memset(cfg, 0, sizeof(ObcCfg));
    cfg->target = g_malloc(OBEX_FTP_UUID_LEN);
    memcpy(cfg->target, OBEX_FTP_UUID, OBEX_FTP_UUID_LEN);
    cfg->target_len = OBEX_FTP_UUID_LEN;
    cfg->chan = -1;
#ifdef USE_BTCOND
    cfg->use_btcond = TRUE;
#else
    cfg->use_btcond = FALSE;
#endif

    while ((c = getopt_long(argc, argv,
                    "h"   /* help      */
                    "V"   /* version   */
                    "d"   /* device    */
                    "f"   /* ftp       */
                    "p"   /* opp       */
                    "P"   /* pbap      */
                    "a"   /* auth      */
                    "e"   /* encrypt   */
                    "r:"  /* role      */
                    "c:"  /* command   */
#ifdef USE_BTCOND
                    "b"   /* nobtcond  */
#endif
                    ,long_options, (int *) 0)) != EOF) {
        switch (c) {
            case 'V':
                printf("OBEX Bluetooth client %s\n", VERSION);
                exit(EXIT_SUCCESS);

            case 'h':
                usage(EXIT_SUCCESS);

            case 'P':
                g_free(cfg->target);
                cfg->target = g_malloc(OBEX_PBAP_UUID_LEN);
                memcpy(cfg->target, OBEX_PBAP_UUID, OBEX_PBAP_UUID_LEN);
                cfg->target_len = OBEX_PBAP_UUID_LEN;
                break;

            case 'f':
                g_free(cfg->target);
                cfg->target = g_malloc(OBEX_FTP_UUID_LEN);
                memcpy(cfg->target, OBEX_FTP_UUID, OBEX_FTP_UUID_LEN);
                cfg->target_len = OBEX_FTP_UUID_LEN;
                break;

            case 'p':
                g_free(cfg->target);
                cfg->target = NULL;
                cfg->target_len = 0;
                break;

            case 'd':
                cfg->is_dev = TRUE;
                break;

            case 'a':
                cfg->auth = TRUE;
                break;

            case 'e':
                cfg->encrypt = TRUE;
                break;

#ifdef USE_BTCOND
            case 'b':
                cfg->use_btcond = FALSE;
                break;
#endif

            case 'r':
                cfg->role = g_strdup(optarg);
                break;

            case 'c':
                cfg->cmd = g_strdup(optarg);
                break;

            case 'C':
                cfg->chan = atoi(optarg);
                break;

            default:
                usage(EXIT_FAILURE);
        }
    }

    return optind;
}

static void exit_cleanup(void)
{
    printf("Exiting.\n");
}

int main(int argc, char *argv[])
{
    ObcCfg cfg;
    int i;

    program_name = argv[0];

    i = decode_switches(argc, argv, &cfg);

    /* So getopt can be used again for internal commands */
    optind = 0;

    atexit(exit_cleanup);

    g_type_init();

    printf("OBEX Bluetooth client\n");

    cfg.dst = argv[i];

    obc_init(&cfg);
#if 0
    obc_init(argv[i], cfg.is_dev, cfg.use_ftp,
             cfg.auth, cfg.encrypt, cfg.role,
             cfg.cmd, cfg.chan, cfg.use_btcond);
#endif

    exit(EXIT_SUCCESS);
}

