/**
  @file async.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
    
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/
#include <stdlib.h>
#include <stdio.h>
#include <getopt.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>
#include <errno.h>

#include <glib.h>

#include <gw-obex.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include "obc-main.h"
#include "async.h"

typedef struct {
    int         offset;
    char       *name;
    GwObexXfer *xfer;
} XferHandle;

static GSList *handles = NULL;

static XferHandle *find_handle(const char *name)
{
    GSList *l;

    for (l = handles; l != NULL; l = l->next) {
        XferHandle *handle = l->data;

        if (g_str_equal(handle->name, name))
            return handle;
    }

    return NULL;
}

gboolean cmd_open(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    XferHandle *handle;
    GwObexXfer *xfer;

    if (argc < 3)
        return FALSE;

    if (find_handle(argv[1]))
        return FALSE;

    xfer = gw_obex_get_async(ctx->obex, argv[2], NULL, err);
    if (!xfer)
        return FALSE;

    gw_obex_xfer_set_blocking(xfer, TRUE);

    handle = g_new0(XferHandle, 1);

    handle->name = g_strdup(argv[1]);
    handle->xfer = xfer;

    handles = g_slist_append(handles, handle);

    printf("Handle %s opened\n", handle->name);

    return TRUE;
}

gboolean cmd_close(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    XferHandle *handle;

    if (argc < 2)
        return FALSE;

    handle = find_handle(argv[1]);
    if (!handle)
        return FALSE;

    gw_obex_xfer_close(handle->xfer, NULL);
    gw_obex_xfer_free(handle->xfer);

    printf("Handle %s closed\n", handle->name);

    handles = g_slist_remove(handles, handle);

    g_free(handle->name);
    g_free(handle);

    return TRUE;
}

gboolean cmd_abort(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    XferHandle *handle;

    if (argc < 2)
        return FALSE;

    handle = find_handle(argv[1]);
    if (!handle)
        return FALSE;

    gw_obex_xfer_abort(handle->xfer, NULL);
    gw_obex_xfer_free(handle->xfer);

    printf("Handle %s aborted\n", handle->name);

    handles = g_slist_remove(handles, handle);

    g_free(handle->name);
    g_free(handle);

    return TRUE;
}

gboolean cmd_read(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    XferHandle *handle;
    char *buf;
    int bytes, ret;

    if (argc < 3)
        return FALSE;

    handle = find_handle(argv[1]);
    
    if (!handle)
        return FALSE;

    bytes = atoi(argv[2]);

    if (bytes < 1)
        return FALSE;

    buf = g_malloc0(bytes);

    if (!gw_obex_xfer_read(handle->xfer, buf, bytes, &ret, err)) {
        g_free(buf);
        return FALSE;
    }

    handle->offset += ret;

    printf("Read %d bytes (total %d/%d) from handle %s\n", ret,
            handle->offset, gw_obex_xfer_object_size(handle->xfer), handle->name);

    g_free(buf);

    return TRUE;
}

void open_help(ObcContext *ctx, const char *name)
{
    printf("%s <handle name> <file>\n"
           "Open a remote file for reading\n", name);
}

void close_help(ObcContext *ctx, const char *name)
{
    printf("%s <handle name>\n"
           "Close opened file\n", name);
}

void abort_help(ObcContext *ctx, const char *name)
{
    printf("%s <handle name>\n"
           "Abort the transfer of an opened file\n", name);
}

void read_help(ObcContext *ctx, const char *name)
{
    printf("%s [-d] <handle name> <bytes>\n"
           "Read specified number of bytes from file\n"
           "Options:\n"
           " -d Show read data\n", name);
}
