/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/* Simple test "UI". */

#include <config.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gsf/gsf-utils.h>
#include <libgnomevfs/gnome-vfs.h>

#include "ob-backend.h"
#include "ob-backup-info.h"
#include "ob-backup-locations.h"
#include "ob-archiver-zip.h"
#include "ob-vfs-utils.h"
#include "ob-log.h"

#define RESPONSE_YES_TO_ALL 1


typedef struct {
	GtkWidget         *window;
	GtkWidget         *simulate_errors_button;
	GtkWidget         *simulate_conflicts_button;
	GtkWidget         *progress_bar;

	ObBackend         *backend;

	ObBackupInfo      *backup_info;
} TestData;


static void reset_ui_state (TestData *data);

static char *
get_time_string (time_t t)
{
       struct tm   *tm;
       static char  buf[64];

       tm = localtime (&t);
       strftime (buf, 64, "%Y-%m-%d %H:%M:%S", tm);

       return g_strdup (buf);
}

static void
backend_progress_cb (ObBackend *backend,
		     int        files,
		     int        total,
		     int        complete,
		     TestData  *data)
{
	char  *str;
	float  fraction;

	fraction = complete / 100.0;

	if (fraction < 0 || fraction > 1.0) {
		/*g_warning ("Bug in progress reporting, not between 0-1.0.");*/
	} else {
		gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (data->progress_bar), fraction);
	}
	
	str = g_strdup_printf ("Backed up: %d of %d files", files, total);
	gtk_progress_bar_set_text (GTK_PROGRESS_BAR (data->progress_bar), str);
	g_free (str);
}

static void
backend_error_cb (ObBackend  *backend,
		  GError     *error,
		  TestData   *data)
{
	GtkWidget *dialog;

	dialog = gtk_message_dialog_new (GTK_WINDOW (data->window),
					 0,
					 GTK_MESSAGE_ERROR,
					 GTK_BUTTONS_OK,
					 "An error occured:\n%s",
					 error->message);

	gtk_dialog_run (GTK_DIALOG (dialog));

	gtk_widget_destroy (dialog);

	reset_ui_state (data);
}

/* File in way of file. */
static int
handle_conflict_file_file (GnomeVFSURI *uri,
			   time_t       old_time,
			   time_t       new_time,
			   TestData    *data)
{
	GtkWidget *dialog;
	char      *old_str, *new_str;
	int        response, retval;	
	char      *name;
	
	old_str = get_time_string (old_time);
	new_str = get_time_string (new_time);

	name = gnome_vfs_uri_extract_short_name (uri);
	
	dialog = gtk_message_dialog_new (GTK_WINDOW (data->window),
					 0,
					 GTK_MESSAGE_QUESTION,
					 GTK_BUTTONS_NONE,
					 "Your device contains a newer version of this file:\n%s\n"
					 "Replace the newer version with the backup version?\n"
					 "Newer version: %s\n"
					 "Backed up version: %s",
					 name, old_str, new_str);

	g_free (name);
	g_free (old_str);
	g_free (new_str);

	gtk_dialog_add_buttons (GTK_DIALOG (dialog),
				"Yes", GTK_RESPONSE_YES,
				"Yes to all", RESPONSE_YES_TO_ALL,
				"No", GTK_RESPONSE_NO,
				"Cancel", GTK_RESPONSE_CANCEL,
				NULL);
	
	response = gtk_dialog_run (GTK_DIALOG (dialog));

	switch (response) {
	case GTK_RESPONSE_YES:
		retval = OB_CONFLICT_RESPONSE_YES;
		break;
	case RESPONSE_YES_TO_ALL:
		retval = OB_CONFLICT_RESPONSE_YES_ALL;
		break;

	case GTK_RESPONSE_NO:
		retval = OB_CONFLICT_RESPONSE_NO;
		break;
	case GTK_RESPONSE_CANCEL:
	case GTK_RESPONSE_DELETE_EVENT:
		retval = OB_CONFLICT_RESPONSE_CANCEL;
		break;
	default:
		g_assert_not_reached ();
		retval = OB_CONFLICT_RESPONSE_CANCEL;
	}

	gtk_widget_destroy (dialog);

	return retval;
}

/* Dir in way of file. */
static int
handle_conflict_dir_file (GnomeVFSURI *uri,
			  time_t       old_time,
			  time_t       new_time,
			  TestData    *data)
{
	GtkWidget *dialog;
	int        response, retval;	
	char      *name;
	
	/* "Rename" renames the file, "No" does not overwrite the existing file,
	 * "Cancel" cancels the restore process.
	 */

	name = gnome_vfs_uri_extract_short_name (uri);

	dialog = gtk_message_dialog_new (GTK_WINDOW (data->window),
					 0,
					 GTK_MESSAGE_QUESTION,
					 GTK_BUTTONS_NONE,
					 "Your device contains a "
					 "folder: %s. Do you want to rename backed up "
					 "file in order to avoid data loss?",
					 name);

	g_free (name);
	
	gtk_dialog_add_buttons (GTK_DIALOG (dialog),
				"Rename", GTK_RESPONSE_YES,
				"No", GTK_RESPONSE_NO,
				"Cancel", GTK_RESPONSE_CANCEL,
				NULL);
	
	response = gtk_dialog_run (GTK_DIALOG (dialog));

	switch (response) {
	case GTK_RESPONSE_YES:
		retval = OB_CONFLICT_RESPONSE_YES;
		break;

	case GTK_RESPONSE_NO:
		retval = OB_CONFLICT_RESPONSE_NO;
		break;
	case GTK_RESPONSE_CANCEL:
	case GTK_RESPONSE_DELETE_EVENT:
		retval = OB_CONFLICT_RESPONSE_CANCEL;
		break;
	default:
		g_assert_not_reached ();
		retval = OB_CONFLICT_RESPONSE_CANCEL;
	}

	gtk_widget_destroy (dialog);

	return retval;
}

/* File in way of dir. */
static int
handle_conflict_file_dir (GnomeVFSURI *uri,
			  time_t       old_time,
			  time_t       new_time,
			  TestData    *data)
{
	GtkWidget *dialog;
	int        response, retval;
	char      *name;
	
	/* "Rename" renames the folder, "No" does not restore the folder,
	 * "Cancel" cancels the restore process.
	 */

	name = gnome_vfs_uri_extract_short_name (uri);
	
	dialog = gtk_message_dialog_new (GTK_WINDOW (data->window),
					 0,
					 GTK_MESSAGE_QUESTION,
					 GTK_BUTTONS_NONE,
					 "Your device contains a "
					 "file: %s. Do you want to rename backed up "
					 "folder in order to avoid data loss?",
					 name);

	g_free (name);
	
	gtk_dialog_add_buttons (GTK_DIALOG (dialog),
				"Rename", GTK_RESPONSE_YES,
				"No", GTK_RESPONSE_NO,
				"Cancel", GTK_RESPONSE_CANCEL,
				NULL);
	
	response = gtk_dialog_run (GTK_DIALOG (dialog));

	switch (response) {
	case GTK_RESPONSE_YES:
		retval = OB_CONFLICT_RESPONSE_YES;
		break;

	case GTK_RESPONSE_NO:
		retval = OB_CONFLICT_RESPONSE_NO;
		break;
	case GTK_RESPONSE_CANCEL:
	case GTK_RESPONSE_DELETE_EVENT:
		retval = OB_CONFLICT_RESPONSE_CANCEL;
		break;
	default:
		g_assert_not_reached ();
		retval = OB_CONFLICT_RESPONSE_CANCEL;
	}

	gtk_widget_destroy (dialog);

	return retval;
}

static int
backend_conflict_cb (ObBackend      *backend,
		     ObConflictType  type,
		     GnomeVFSURI    *uri,
		     long            old_time,
		     long            new_time,
		     TestData       *data)
{
	switch (type) {
	case OB_CONFLICT_TYPE_FILE_FILE:
		return handle_conflict_file_file (uri, old_time, new_time, data);
	case OB_CONFLICT_TYPE_DIR_FILE:
		return handle_conflict_dir_file (uri, old_time, new_time, data);
	case OB_CONFLICT_TYPE_FILE_DIR:
		return handle_conflict_file_dir (uri, old_time, new_time, data);
	default:
		g_assert_not_reached ();
		return OB_CONFLICT_RESPONSE_CANCEL;
	}
}	

static void
backend_finished_cb (ObBackend *backend,
		     TestData  *data)
{
	GtkWidget *dialog;

	/* Set the progress to 1.0 in case we got rounding a errors. */
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (data->progress_bar), 1.0);

	dialog = gtk_message_dialog_new (GTK_WINDOW (data->window),
					 0,
					 GTK_MESSAGE_INFO,
					 GTK_BUTTONS_OK,
					 "Backup is completed.");

	gtk_dialog_run (GTK_DIALOG (dialog));

	gtk_widget_destroy (dialog);

	reset_ui_state (data);
}

static void
backend_cancelled_cb (ObBackend *backend,
		      TestData  *data)
{
	g_print ("UI: Cancelled\n");
	reset_ui_state (data);
}

static void
quit_clicked_cb (GtkWidget *button, TestData *data)
{
	gtk_main_quit ();
}

static void
window_destroy_cb (GtkWidget *button, TestData *data)
{
	gtk_main_quit ();
}

static void
backup_clicked_cb (GtkWidget *button, TestData *data)
{
	GError           *error = NULL;
        time_t            timestamp;
	int               num_files;
	GnomeVFSFileSize  size;
	ObCategoryFiles  *files;
	
	/* Start a simulated backup operation. */

	/*simulate_errors = gtk_toggle_button_get_active (
	  GTK_TOGGLE_BUTTON (data->simulate_errors_button));*/

        timestamp = time (NULL);

	ob_backend_count_data (data->backend,
			       OB_CATEGORY_ALL,
			       &num_files,
			       &size);

	/* Get the files managed under "MyDocs". */
	files = ob_category_files_new ();
	ob_category_files_read (files, OB_CATEGORY_ALL);

	size += ob_category_files_get_size (files, OB_CATEGORY_ALL);
	num_files += ob_category_files_get_num_files (files, OB_CATEGORY_ALL);

	if (!ob_backend_start_backup (data->backend,
				      OB_MEMORY_CARD_INTERNAL,
				      files,
				      "Backup", NULL,
				      timestamp,
				      OB_CATEGORY_ALL,
				      num_files, size,
				      NULL, /* replaced_backup_info */
				      &error)) {
		g_print ("Couldn't start test backup: %s\n", error->message);
		g_error_free (error);
	}
}

static void
restore_clicked_cb (GtkWidget *button, TestData *data)
{
	gboolean     simulate_errors;
	gboolean     simulate_conflicts;
	GnomeVFSURI *uri;
	GError      *error = NULL;

	/* Start a simulated restore operation. */

	uri = ob_backup_info_get_uri (data->backup_info);
	
	g_print ("Simulate restore to URI: %s\n",
		 gnome_vfs_uri_get_path (uri));
	
	simulate_errors = gtk_toggle_button_get_active (
		GTK_TOGGLE_BUTTON (data->simulate_errors_button));

	simulate_conflicts = gtk_toggle_button_get_active (
		GTK_TOGGLE_BUTTON (data->simulate_conflicts_button));

	if (!ob_backend_start_restore (data->backend,
				       data->backup_info,
				       NULL, /* password */
                                       OB_CATEGORY_ALL,
				       &error)) {
		g_print ("Couldn't start test restore: %s\n", error->message);
		g_error_free (error);
	}
}

static void
cancel_clicked_cb (GtkWidget *button, TestData *data)
{
	if (ob_backend_cancel (data->backend)) {
		g_print ("UI: Cancelled manually.\n");
	}
}

static void
reset_ui_state (TestData *data)
{
	gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (data->progress_bar), 0);
	gtk_progress_bar_set_text (GTK_PROGRESS_BAR (data->progress_bar), " ");
}

static void
setup_gui (TestData *data)
{
	GtkWidget *vbox;
	GtkWidget *label;
	GtkWidget *bbox;
	GtkWidget *button;

	data->window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title (GTK_WINDOW (data->window), "Test UI");
	g_signal_connect (data->window,
			  "destroy",
			  G_CALLBACK (window_destroy_cb),
			  data);

	vbox = gtk_vbox_new (FALSE, 6);
	gtk_container_add (GTK_CONTAINER (data->window), vbox);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 6);

	label = gtk_label_new ("Test UI");
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
	
	data->simulate_errors_button = gtk_check_button_new_with_label ("Simulate error");
	gtk_box_pack_start (GTK_BOX (vbox), data->simulate_errors_button, FALSE, TRUE, 0);
	
	data->simulate_conflicts_button = gtk_check_button_new_with_label ("Simulate conflicts");
	gtk_box_pack_start (GTK_BOX (vbox), data->simulate_conflicts_button, FALSE, TRUE, 0);
	
	label = gtk_label_new ("");
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
	
	label = gtk_label_new ("<b>Progress:</b>");
	gtk_label_set_use_markup (GTK_LABEL (label), TRUE);
	gtk_misc_set_alignment (GTK_MISC (label), 0, 0);
	gtk_box_pack_start (GTK_BOX (vbox), label, TRUE, TRUE, 0);
	
	data->progress_bar = gtk_progress_bar_new ();
	gtk_box_pack_start (GTK_BOX (vbox), data->progress_bar, FALSE, TRUE, 0);
	
	bbox = gtk_hbutton_box_new ();
	gtk_box_pack_start (GTK_BOX (vbox), bbox, FALSE, TRUE, 0);

	button = gtk_button_new_with_label ("Backup");
	gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 0);
	g_signal_connect (button,
			  "clicked",
			  G_CALLBACK (backup_clicked_cb),
			  data);

	button = gtk_button_new_with_label ("Restore");
	gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 0);
	g_signal_connect (button,
			  "clicked",
			  G_CALLBACK (restore_clicked_cb),
			  data);
	
	button = gtk_button_new_with_label ("Cancel");
	gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 0);
	g_signal_connect (button,
			  "clicked",
			  G_CALLBACK (cancel_clicked_cb),
			  data);
	
	button = gtk_button_new_with_label ("Quit");
	gtk_box_pack_start (GTK_BOX (bbox), button, FALSE, TRUE, 0);
	g_signal_connect (button,
			  "clicked",
			  G_CALLBACK (quit_clicked_cb),
			  data);

	g_signal_connect (data->backend,
			  "progress",
			  G_CALLBACK (backend_progress_cb),
			  data);
	
	g_signal_connect (data->backend,
			  "error",
			  G_CALLBACK (backend_error_cb),
			  data);

	g_signal_connect (data->backend,
			  "conflict",
			  G_CALLBACK (backend_conflict_cb),
			  data);

	g_signal_connect (data->backend,
			  "finished",
			  G_CALLBACK (backend_finished_cb),
			  data);

	g_signal_connect (data->backend,
			  "cancelled",
			  G_CALLBACK (backend_cancelled_cb),
			  data);

	reset_ui_state (data);
	
	gtk_widget_show_all (data->window);
}

static void
test_data_free (TestData *data)
{
	g_object_unref (data->backend);
	ob_backup_info_unref (data->backup_info);
	
	g_free (data);
}

static void
print_usage (const char *prgname)
{
	g_printerr ("Usage: %s <BASEDIR> <MMCDIR>\n"
		    "BASEDIR is a test directory to use for testing.\n"
		    "The data in in <basedir>/data/.\n"
		    "MMCDIR is the path where to store backups.\n"
		    "Note: both paths must be absolute.\n",
		    prgname);
}

int
main (int argc, char **argv)
{
	TestData          *data;
	const char        *basedir;
	char              *basedir_uri_string;
	const char        *mountpoint;
	char              *mountpoint_uri_string;
	char              *tmp;
	GnomeVFSURI       *uri;
	ObConfig          *config;
	ObBackupLocations *locations;
	GList             *backups, *l;
	ObBackupInfo      *info;
	char              *name;

	gtk_init (&argc, &argv);
	gnome_vfs_init ();
	gsf_init ();

	if (argc > 2) {
		if (argv[1][0] != '/' ||  argv[2][0] != '/') {
			print_usage (argv[0]);
			return 1;
		}

		basedir = argv[1];
		basedir_uri_string = g_strconcat ("file://", basedir, NULL);
		
		mountpoint = argv[2];
		mountpoint_uri_string = g_strconcat ("file://", mountpoint, NULL);
	} else {
		print_usage (argv[0]);
		return 1;
	}

	ob_log_start ();
	
	data = g_new0 (TestData, 1);

	/* Create a fake configuration. */
	_ob_config_init_empty ();
	config = ob_config_get ();

	uri = gnome_vfs_uri_new (mountpoint_uri_string);
	_ob_config_set_memory_card_mountpoint (config, OB_MEMORY_CARD_EXTERNAL, uri);
	gnome_vfs_uri_unref (uri);
	
	/* Create a fake configuration for locations to backup. */
	_ob_backup_locations_init_empty ();
	locations = ob_backup_locations_get ();

	tmp = g_build_filename (basedir, "data", NULL);
	_ob_backup_locations_add_location (locations,
					   tmp,
					   OB_CATEGORY_OTHER,
					   TRUE,   /* is_dir */
					   FALSE,  /* is_auto */
					   FALSE); /* is_excluded */
	g_free (tmp);

	_ob_backup_locations_remove_redundancies (locations);

	data->backend = ob_backend_new ();

	/* Test */
	ob_backend_space_on_memory_card (data->backend, 
					 OB_MEMORY_CARD_INTERNAL,
					 OB_CATEGORY_ALL,
					 0);
	
	/* Get the list of backups in the mmc dir. */
	uri = gnome_vfs_uri_append_path (
		ob_config_get_memory_card_mountpoint (config, OB_MEMORY_CARD_INTERNAL), "backups");
	
	backups = ob_backup_info_list_get (uri);
	gnome_vfs_uri_unref (uri);

	/* Find the backup named "Backup" that we use to test with, if it
	 * doesn't exist, create fake one.
	 */
	for (l = backups; l; l = l->next) {
		info = l->data;

		name = ob_backup_info_get_name (info);
		
		if (strcmp (name, "Backup") == 0) {
			g_print ("Found \"Backup\", using that.\n");
			data->backup_info = ob_backup_info_ref (info);
			break;
		}
		
		g_free (name);
	}

	ob_backup_info_list_free (backups);

	if (!data->backup_info) {
		/* Create fake backup info. */
		uri = gnome_vfs_uri_new ("file:///foo/bar/baz/backups");
			
		data->backup_info = ob_backup_info_new (uri,
							time (NULL),
							FALSE,
							NULL,
							NULL,
                                                        OB_CATEGORY_ALL);
		
		gnome_vfs_uri_unref (uri);
	}

	setup_gui (data);
	
	gtk_main ();

	test_data_free (data);

	g_print ("UI: Done\n");

	gsf_shutdown ();
	gnome_vfs_shutdown ();

	g_free (basedir_uri_string);
	g_free (mountpoint_uri_string);

	ob_config_shutdown ();
	ob_backup_locations_shutdown ();
	
	ob_log_stop ();
	
	return 0;
}


