/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/* Some code that is useful for debugging the crypt code. */

#include <config.h>
#include <string.h>

#include <libgnomevfs/gnome-vfs.h>

#include <gsf/gsf-utils.h>
#include <gsf/gsf-output-stdio.h>
#include <gsf/gsf-outfile.h>
#include <gsf/gsf-outfile-zip.h>

#include <gsf/gsf-input-stdio.h>
#include <gsf/gsf-infile.h>
#include <gsf/gsf-infile-zip.h>

#include "gsf-output-crypt.h"
#include "gsf-input-crypt.h"


#define TEST_FILE_1 "file:///tmp/file1.txt"
#define TEST_FILE_2 "file:///tmp/file2.txt"

#define BUF_SIZE 2048

static GsfOutput *
add_dir (GsfOutput *dir, const char *subdirname)
{
	GsfOutput *subdir;
	
	subdir  = gsf_outfile_new_child  (GSF_OUTFILE (dir), subdirname, TRUE);
	if (!subdir) {
		g_print ("couldn't create subdir: %s\n", subdirname);
		return NULL;
	}

	return subdir;
}

static gboolean
add_file (GsfOutput *dir, const char *source)
{
	GnomeVFSURI      *uri;
	char             *filename;
	GsfOutput        *file;
	GnomeVFSResult    result;
	GnomeVFSHandle   *handle;
	GnomeVFSFileSize  bytes_read;
	char             *buf;

	uri = gnome_vfs_uri_new (source);
	
	result = gnome_vfs_open_uri (&handle,
				     uri,
				     GNOME_VFS_OPEN_READ);
	if (result != GNOME_VFS_OK) {
		return FALSE;
	}
	
	filename = gnome_vfs_uri_extract_short_name (uri);
	file = gsf_outfile_new_child (GSF_OUTFILE (dir), filename, FALSE);
	g_free (filename);
	
	if (!file) {
		g_print ("Couldn't create file in zip file\n");
		return FALSE;
	}

	buf = g_malloc (BUF_SIZE);
	
	while (1) {
		result = gnome_vfs_read (handle,
					 buf, BUF_SIZE,
					 &bytes_read);
		
		if (result == GNOME_VFS_ERROR_EOF) {
			break;
		}
		else if (result != GNOME_VFS_OK) {
			g_print ("couldn't read vfs file\n");
			return FALSE;
		}
		
		g_print ("read %d bytes: [%s]\n", (int) bytes_read, g_strndup (buf, bytes_read));
		
		if (!gsf_output_write (file, bytes_read, buf)) {
			g_print ("couldn't write output\n");
			return FALSE;
		}
	}

	g_print ("done writing\n");
	
	gnome_vfs_close (handle);
	gnome_vfs_uri_unref (uri);

	g_free (buf);
	
	if (!gsf_output_close (file)) {
		return FALSE;
	}
	
        g_object_unref (file);

	return TRUE;
}

static gboolean
create_zip (gboolean protected)
{
	const char  *filename;
	const char  *password;
        GsfOutput   *output;
        GsfOutput   *output_crypt;
        GsfOutfile  *outfile_zip;
        GError      *error = NULL;
	GsfOutput   *dir;

	if (protected) {
		password = "password";
		filename = "crypt.data";
	} else {
		password = NULL;	
		filename = "non-crypt2.data";
	}
	
        output = gsf_output_stdio_new (filename, &error);
        if (!output) {
		g_print ("Couldn't create test file: %s\n", error->message);
		g_error_free (error);
		return FALSE;
        }

	output_crypt = gsf_output_crypt_new (output, password, &error);
        g_object_unref (output);
        if (!output_crypt) {
		g_print ("Couldn't create crypt output: %s\n", error->message);
                g_error_free (error);
		return FALSE;
        }

        outfile_zip = gsf_outfile_zip_new (output_crypt, &error);
        g_object_unref (output_crypt);
        if (!outfile_zip) {
		g_print ("Couldn't create zip outfile: %s\n", error->message);
                g_error_free (error);
		return FALSE;
        }

	if (!add_file (GSF_OUTPUT (outfile_zip), TEST_FILE_1)) {
		g_print ("Couldn't add file\n");
		return FALSE;
	}
	
	/* Test dir */
	dir = add_dir (GSF_OUTPUT (outfile_zip), "subdir");
	if (!dir) {
		g_print ("couldn't create subdir\n");
		return FALSE;
	}
	
	/* test file in dir */
	if (!add_file (dir, TEST_FILE_2)) {
		g_print ("Couldn't add file\n");
		return FALSE;
	}
	
	/* close the dir */
	if (!gsf_output_close (GSF_OUTPUT (dir))) {
		g_print ("couldn't close dir\n");
		return FALSE;
	}			

	g_object_unref (dir);
	
	if (!gsf_output_close (GSF_OUTPUT (outfile_zip))) {
		g_print ("Couldn't close zip outfile\n");
		return FALSE;
	}

	if (!gsf_output_close (GSF_OUTPUT (output_crypt))) {
		g_print ("Couldn't close crypt outfile\n");
		return FALSE;
	}

	g_object_unref (outfile_zip);

	return TRUE;
}

static gboolean
get_file (GsfInput *dir, const char *path)
{
	char     *filename;
	GsfInput *file;
	char     *buf;
	size_t    bytes_read, total_bytes;

	filename = g_path_get_basename (path);
	file = gsf_infile_child_by_name (GSF_INFILE (dir), filename);
	g_free (filename);

	if (!file) {
		g_print ("Couldn't find file in zip file\n");
		return FALSE;
	}

	buf = g_malloc (BUF_SIZE);
	
	total_bytes = gsf_input_size (file);
	
	while (total_bytes > 0) {
		bytes_read = MIN (total_bytes, BUF_SIZE);
		
		if (!gsf_input_read (file, bytes_read, buf)) {
			g_print ("Couldn't read input\n");
			return FALSE;
		}
		
		g_print ("Test got %d bytes: [%s]\n", (int) bytes_read, g_strndup (buf, bytes_read));

		total_bytes -= bytes_read;
	}

        g_object_unref (file);

	g_free (buf);

	return TRUE;
}

static gboolean
read_zip (gboolean protected)
{
	const char *filename;
	const char *password;
	GsfInput   *input;
        GsfInput   *input_crypt;
        GsfInfile  *infile_zip;
        GError     *error = NULL;
	GsfInput   *dir;

	if (protected) {
		filename = "crypt.data";
		password = "password";
	} else {
		filename = "non-crypt.data";
		password = NULL;
	}

	g_print ("Reading: %s\n", filename);
	
        input = gsf_input_stdio_new (filename, &error);
        if (!input) {
		g_print ("Couldn't open test file: %s\n", error->message);
		g_error_free (error);
		return FALSE;
        }

	input_crypt = gsf_input_crypt_new (input, password, &error);
        g_object_unref (input);
        if (!input_crypt) {
		g_print ("Couldn't create crypt input: %s\n", error->message);
                g_error_free (error);
		return FALSE;
        }

        infile_zip = gsf_infile_zip_new (input_crypt, &error);
        g_object_unref (input_crypt);
        if (!infile_zip) {
		g_print ("Couldn't create zip infile: %s\n", error->message);
                g_error_free (error);
		return FALSE;
        }

	if (!get_file (GSF_INPUT (infile_zip), TEST_FILE_1)) {
		g_print ("Couldn't get file\n");
		return FALSE;
	}

	/* get the subdir */
	dir = gsf_infile_child_by_name (GSF_INFILE (infile_zip), "subdir");
	if (!dir) {
		g_print ("Couldn't get subdir\n");
		return FALSE;
	}

	g_print ("got dir!\n");

	if (!get_file (GSF_INPUT (dir), TEST_FILE_2)) {
		g_print ("Couldn't get file in subdir\n");
		return FALSE;
	}

	g_object_unref (infile_zip);
	
	return TRUE;
}

int
main (int argc, char *argv[])
{
	gboolean create, read;
	gboolean crypted, plaintext;

	gnome_vfs_init ();
	gsf_init ();

	create = FALSE;/*TRUE;*/
	read = TRUE;

	crypted = TRUE;
	plaintext = FALSE;
	
	if (create && plaintext) {
		unlink ("non-crypt.data");

		if (!create_zip (FALSE)) {
			g_print ("FAILED: Creating non-crypt failed\n");
			return 1;
		}
		
		if (!g_file_test ("non-crypt.data", G_FILE_TEST_EXISTS)) {
			g_print ("FAILED: no non-crypt file found\n");
			return 0;
		}
	}

	if (create && crypted) {
		unlink ("crypt.data");

		if (!create_zip (TRUE)) {
			g_print ("FAILED: Creating failed\n");
			return 1;
		}
		
		if (!g_file_test ("crypt.data", G_FILE_TEST_EXISTS)) {
			g_print ("FAILED: no crypt file found\n");
			return 0;
		}
	}

	g_print ("---------------\n");

	if (read && plaintext) {
		if (!read_zip (FALSE)) {
			g_print ("FAILED: Reading noncrypt failed\n");
			return 1;
		}
	}

	if (read && crypted) {
		if (!read_zip (TRUE)) {
			g_print ("FAILED: Reading failed\n");
			return 1;
		}
	}

	gsf_shutdown ();
	gnome_vfs_shutdown ();
		
	return 0;
}
