/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <stdio.h>
#include <string.h>
#include <libgnomevfs/gnome-vfs.h>
#include <gsf/gsf-utils.h>
#include <gtk/gtkmain.h>

#include "ob-backend.h"
#include "ob-backup-info.h"
#include "ob-backup-locations.h"
#include "ob-vfs-utils.h"


typedef struct {
	ObBackend         *backend;

	ObBackupInfo      *backup_info;
	ObBackupLocations *locations;
} TestData;


static const char *
get_time_string (time_t t)
{
	struct tm   *tm;
	static char  buf[32];

	tm = localtime (&t);
	strftime (buf, 32, "%Y-%m-%d %H:%M:%S", tm);

	return buf;
}

static void
backend_progress_cb (ObBackend *backend,
		     int        files,
		     int        total,
		     int        complete,
		     gpointer   user_data)
{
	g_print ("- Processed %d of %d files: %d%%\n", files, total, complete);
}

static void
backend_error_cb (ObBackend  *backend,
		  GError     *error,
		  gpointer    user_data)
{
	g_print ("- Error (%d): %s\n", error->code, error->message);
}

static int
handle_conflict_file_file (GnomeVFSURI *uri,
			   long         old_time,
			   long         new_time,
			   TestData    *data)
{
	char *name;
	char  response[128];
	int   retval;	

	name = gnome_vfs_uri_extract_short_name (uri);
		
	g_print ("Conflict, file in way of file: %s (%s, %s)\n",
		 name, get_time_string (old_time), get_time_string (new_time));

	g_free (name);
	
	g_print ("Replace? ");

 retry:
	g_print ("yes/all/no/cancel: ");

	fgets (response, sizeof (response), stdin);

	switch (response[0]) {
	case 'y':
		retval = OB_CONFLICT_RESPONSE_YES;
		break;
	case 'a':
		retval = OB_CONFLICT_RESPONSE_YES_ALL;
		break;

	case 'n':
		retval = OB_CONFLICT_RESPONSE_NO;
		break;
	case 'c':
		retval = OB_CONFLICT_RESPONSE_CANCEL;
		break;
	default:
		g_print ("Invalid response, try again. ");
		fflush (stdin);
		goto retry;
	}

	return retval;
}

static int
handle_conflict_dir_file (GnomeVFSURI *uri,
			  long         old_time,
			  long         new_time,
			  TestData    *data)
{
	char *name;
	char  response[128];
	int   retval;	

	/* "Rename" renames the file, "No" does not restore the file, "Cancel"
	 * cancels the restore process.
	 */

	name = gnome_vfs_uri_extract_short_name (uri);
	g_print ("Conflict, dir in way of file: %s\n", name);
	g_free (name);
	
	g_print ("Rename and restore the file? ");

 retry:
	g_print ("yes/no/cancel: ");

	fgets (response, sizeof (response), stdin);

	switch (response[0]) {
	case 'y':
		retval = OB_CONFLICT_RESPONSE_YES;
		break;
	case 'n':
		retval = OB_CONFLICT_RESPONSE_NO;
		break;
	case 'c':
		retval = OB_CONFLICT_RESPONSE_CANCEL;
		break;
	default:
		g_print ("Invalid response, try again. ");
		fflush (stdin);
		goto retry;
	}

	return retval;
}

static int
handle_conflict_file_dir (GnomeVFSURI *uri,
			  long         old_time,
			  long         new_time,
			  TestData    *data)
{
	char *name;
	char  response[128];
	int   retval;	

	/* "Rename" renames the folder, "No" does not restore the folder,
	 * "Cancel" cancels the restore process.
	 */

	name = gnome_vfs_uri_extract_short_name (uri);
	g_print ("conflict, file in way of dir: %s\n", name);
	g_free (name);
	
	g_print ("Rename and restore the folder? ");

 retry:
	g_print ("yes/no/cancel: ");

	fgets (response, sizeof (response), stdin);

	switch (response[0]) {
	case 'y':
		retval = OB_CONFLICT_RESPONSE_YES;
		break;
	case 'n':
		retval = OB_CONFLICT_RESPONSE_NO;
		break;
	case 'c':
		retval = OB_CONFLICT_RESPONSE_CANCEL;
		break;
	default:
		g_print ("Invalid response, try again. ");
		fflush (stdin);
		goto retry;
	}

	return retval;
}

static int
backend_conflict_cb (ObBackend      *backend,
		     ObConflictType  type,
		     GnomeVFSURI    *uri,
		     long            old_time,
		     long            new_time,
		     TestData       *data)
{
	switch (type) {
	case OB_CONFLICT_TYPE_FILE_FILE:
		return handle_conflict_file_file (uri, old_time, new_time, data);
	case OB_CONFLICT_TYPE_DIR_FILE:
		return handle_conflict_dir_file (uri, old_time, new_time, data);
	case OB_CONFLICT_TYPE_FILE_DIR:
		return handle_conflict_file_dir (uri, old_time, new_time, data);
	default:
		g_assert_not_reached ();
		return OB_CONFLICT_RESPONSE_CANCEL;
	}
}	

static void
backend_finished_cb (ObBackend *backend,
		     gpointer   user_data)
{
	g_print ("- Finished\n");

	gtk_main_quit ();
}

static void
backend_cancelled_cb (ObBackend *backend,
		      gpointer   user_data)
{
	g_print ("- Cancelled\n");

	gtk_main_quit ();
}

static gboolean
test_unpack (TestData *data)
{
	GError *error = NULL;

	if (!ob_backend_start_restore (data->backend,
				       data->backup_info,
				       NULL, /* password */
                                       OB_CATEGORY_ALL,
				       &error)) {
		g_print ("Unpacking failed: %s\n", error->message);
		g_error_free (error);
		return FALSE;
	}

	gtk_main ();
	
	/*g_print ("Unpacked %d files, %d bytes\n",
		 ob_archiver_get_processed_files (archiver),
		 (int) ob_archiver_get_processed_size (archiver));
	*/

	return TRUE;
}

static gboolean
test_pack (TestData *data)
{
	GError           *error = NULL;
        time_t            timestamp;
	int               num_files;
	GnomeVFSFileSize  size;
	ObCategoryFiles  *files;

        timestamp = time (NULL);

	ob_backend_count_data (data->backend,
			       OB_CATEGORY_ALL,
			       &num_files,
			       &size);
	
	/* Get the files managed under "MyDocs". */
	files = ob_category_files_new ();
	ob_category_files_read (files, OB_CATEGORY_ALL);

	size += ob_category_files_get_size (files, OB_CATEGORY_ALL);
	num_files += ob_category_files_get_num_files (files, OB_CATEGORY_ALL);

	if (!ob_backend_start_backup (data->backend,
				      OB_MEMORY_CARD_INTERNAL,
				      files,
				      "Backup", NULL,
				      timestamp,
				      OB_CATEGORY_ALL,
				      num_files, size,
				      NULL, /* replaced_backup_info */
				      &error)) {
		g_print ("Packing failed: %s\n", error->message);
		g_error_free (error);
		return FALSE;
	}
	
	gtk_main ();
	
	/*g_print ("Packed %d files, %d bytes\n",
		 ob_archiver_get_processed_files (archiver),
		 (int) ob_archiver_get_processed_size (archiver));
	*/

	return TRUE;
}

static void
print_usage (const char *prgname)
{
	g_printerr ("Usage: %s OPTION <BASEDIR> <MMCDIR>\n"
		    "OPTION is any combination of -p and -u for packing\n"
		    "and unpacking. BASEDIR is a test directory to use for\n"
		    "testing. MMCDIR is the location to store backup.\n"
		    "Note: the basedir and mmcdir must be absolute paths.\n",
		    prgname);
}

int
main (int argc, char **argv)
{
	gboolean           pack, unpack;
	int                i;
	const char        *basedir;
	char              *basedir_uri_string;
	const char        *mountpoint;
	char              *mountpoint_uri_string;
	char              *tmp;
	GnomeVFSURI       *uri;
	TestData          *data;
	ObBackupLocations *locations;
	ObConfig          *config;
	GList             *backups, *l;
	ObBackupInfo      *info;
	char              *name;
	
	gnome_vfs_init ();
	gsf_init ();
	gtk_init (&argc, &argv);

	pack = FALSE;
	unpack = FALSE;

	if (argc < 3) {
		print_usage (argv[0]);
		return 1;
	}

	basedir = NULL;
	mountpoint = NULL;
	
	for (i = 1; i < argc; i++) {
		if (strcmp (argv[i], "-p") == 0) {
			pack = TRUE;
		}
		else if (strcmp (argv[i], "-u") == 0) {
			unpack = TRUE;
		}
		else if (argv[i][0] != '-') {
			if (!basedir) { 
				basedir = argv[i];
			} else {
				mountpoint = argv[i];
			}
		} else {
			g_printerr ("Invalid option.\n\n");
			print_usage (argv[0]);
			return 1;
		}
	}

	if (basedir == NULL) {
		g_printerr ("No basedir specified.\n");
		print_usage (argv[0]);
		return 1;
	}
	
	if (mountpoint == NULL) {
		g_printerr ("No mmcdir specified.\n");
		print_usage (argv[0]);
		return 1;
	}
	
	basedir_uri_string = g_strconcat ("file://", basedir, NULL);
	mountpoint_uri_string = g_strconcat ("file://", mountpoint, NULL);
	
	data = g_new0 (TestData, 1);

	/* Create a fake configuration. */
	_ob_config_init_empty ();
	config = ob_config_get ();

	uri = gnome_vfs_uri_new (mountpoint_uri_string);
	_ob_config_set_memory_card_mountpoint (config, OB_MEMORY_CARD_EXTERNAL, uri);
	gnome_vfs_uri_unref (uri);
	
	/* Create a fake configuration for locations to backup. */
	_ob_backup_locations_init_empty ();
	locations = ob_backup_locations_get ();

	tmp = g_build_filename (basedir, "data", NULL);
	_ob_backup_locations_add_location (locations,
					   tmp,
					   OB_CATEGORY_OTHER,
					   TRUE,   /* is_dir */
					   FALSE,  /* is_auto */
					   FALSE); /* is_excluded */ 
	g_free (tmp);

	_ob_backup_locations_remove_redundancies (locations);

	data->backend = ob_backend_new ();

	/* Test */
	ob_backend_space_on_memory_card (data->backend, 				      
					 OB_MEMORY_CARD_INTERNAL,
					 OB_CATEGORY_ALL, 
					 0);

	/* Get the list of backups in the mmc dir. */
	uri = gnome_vfs_uri_append_path (
		ob_config_get_memory_card_mountpoint (config, OB_MEMORY_CARD_INTERNAL), "backups");
	
	backups = ob_backup_info_list_get (uri);
	gnome_vfs_uri_unref (uri);

	/* Find the backup named "Backup" that we use to test with, if it
	 * doesn't exist, create fake one.
	 */
	for (l = backups; l; l = l->next) {
		info = l->data;

		name = ob_backup_info_get_name (info);
		
		if (strcmp (name, "Backup") == 0) {
			g_print ("Found \"Backup\", using that.\n");
			data->backup_info = ob_backup_info_ref (info);
			break;
		}
		
		g_free (name);
	}

	ob_backup_info_list_free (backups);

	if (!data->backup_info) {
		/* Create fake backup info. */
		uri = gnome_vfs_uri_new ("file:///foo/bar/baz/backups");
			
		data->backup_info = ob_backup_info_new (uri,
							time (NULL),
							FALSE,
							NULL,
							NULL,
                                                        OB_CATEGORY_ALL);
		
		gnome_vfs_uri_unref (uri);
	}
	
	/* Setup backend signals. */
	g_signal_connect (data->backend,
			  "progress",
			  G_CALLBACK (backend_progress_cb),
			  data);
	
	g_signal_connect (data->backend,
			  "error",
			  G_CALLBACK (backend_error_cb),
			  data);

	g_signal_connect (data->backend,
			  "conflict",
			  G_CALLBACK (backend_conflict_cb),
			  data);
	
	g_signal_connect (data->backend,
			  "finished",
			  G_CALLBACK (backend_finished_cb),
			  data);
	
	g_signal_connect (data->backend,
			  "cancelled",
			  G_CALLBACK (backend_cancelled_cb),
			  data);

	if (pack) {
		test_pack (data);
	}

	if (unpack) {
		test_unpack (data);
	}	
	
	g_print ("Test done.\n");

	gsf_shutdown ();
	gnome_vfs_shutdown ();

	g_free (basedir_uri_string);
	g_free (mountpoint_uri_string);

	ob_config_shutdown ();
	ob_backup_locations_shutdown ();
	
	return 0;
}
