/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <string.h>
#include <sys/statfs.h>
#include <glib.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <osso-mime.h>

#include "ob-vfs-utils.h"
#include "ob-category.h"

#define d(x) 

static char * vfs_utils_match_extension      (char   *basename, 
					      GSList *extensions);
static char * vfs_utils_file_rename_template (char   *name);

static char * 
vfs_utils_match_extension (char *basename, GSList *extensions)
{
	GSList *next;
	int     length;
	int     base_length;

	base_length = strlen (basename);

	for (next=extensions; next != NULL; next = g_slist_next (next)) {
		if (g_pattern_match_simple (next->data, basename)) {
			length = strlen (next->data);
			return (basename + (base_length - length + 1));
		}
	}

	return (basename + base_length);
}

static char *
vfs_utils_file_rename_template (gchar *basename)
{
	const gchar *mimetype;
	char       *extension;
	char       *template;
	char       *tmp;
	GSList      *mime_extensions;
	
	mimetype = gnome_vfs_mime_type_from_name_or_default (basename, NULL);
	
	if (!mimetype) {
		template= g_strdup_printf ("%s(%%d)", basename);
	} else {
		mime_extensions = 
			osso_mime_patterns_get_for_mime_type (mimetype, NULL);
		if (!mime_extensions) {
			template = g_strdup_printf ("%s(%%d)", basename);

		} else {
			extension = 
				vfs_utils_match_extension (basename, 
							   mime_extensions);

			tmp = g_strndup (basename, extension - basename);

			template = g_strdup_printf ("%s(%%d)%s", tmp, extension);
			g_free (tmp);

			g_slist_foreach (mime_extensions, (GFunc)g_free, NULL);
			g_slist_free (mime_extensions);
		}
	}

	return template;
}

GnomeVFSResult
ob_vfs_utils_uri_set_timestamp_and_mode (GnomeVFSURI *uri,
					 time_t       stamp,
					 mode_t       mode)
{
	GnomeVFSResult    result;
	GnomeVFSFileInfo *info;

	g_return_val_if_fail (uri != NULL, GNOME_VFS_ERROR_BAD_PARAMETERS);

	/* Handle broken time setting. */
	if (stamp < 0) {
		stamp = 0;
	}
	
	/* First get the times so we can change only one of them. */
	info = gnome_vfs_file_info_new ();
	
	result = gnome_vfs_get_file_info_uri (uri,
					      info,
					      GNOME_VFS_FILE_INFO_FOLLOW_LINKS |
					      GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE);
	if (result != GNOME_VFS_OK) {
		gnome_vfs_file_info_unref (info);
		return result;
	}
		
	info->mtime = stamp;
	info->permissions = mode;

	result = gnome_vfs_set_file_info_uri (uri,
					      info,
					      GNOME_VFS_SET_FILE_INFO_TIME |
					      GNOME_VFS_SET_FILE_INFO_PERMISSIONS);

	gnome_vfs_file_info_unref (info);

	return result;
}

GnomeVFSResult
ob_vfs_utils_uri_set_mode (GnomeVFSURI *uri,
			   mode_t       mode)
{
	GnomeVFSResult    result;
	GnomeVFSFileInfo *info;

	g_return_val_if_fail (uri != NULL, GNOME_VFS_ERROR_BAD_PARAMETERS);

	/* First get the times so we can change only one of them. */
	info = gnome_vfs_file_info_new ();
	
	result = gnome_vfs_get_file_info_uri (uri,
					      info,
					      GNOME_VFS_FILE_INFO_FOLLOW_LINKS |
					      GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE);
	if (result != GNOME_VFS_OK) {
		gnome_vfs_file_info_unref (info);
		return result;
	}
		
	info->permissions = mode;

	result = gnome_vfs_set_file_info_uri (uri,
					      info,
					      GNOME_VFS_SET_FILE_INFO_PERMISSIONS);
	
	gnome_vfs_file_info_unref (info);

	return result;
}

GnomeVFSResult
ob_vfs_utils_uri_set_timestamp (GnomeVFSURI *uri, time_t stamp)
{
	GnomeVFSResult    result;
	GnomeVFSFileInfo *info;

	g_return_val_if_fail (uri != NULL, GNOME_VFS_ERROR_BAD_PARAMETERS);

	/* Handle broken time setting. */
	if (stamp < 0) {
		stamp = 0;
	}
	
	/* First get the times so we can change only one of them. */
	info = gnome_vfs_file_info_new ();
	
	result = gnome_vfs_get_file_info_uri (uri,
					      info,
					      GNOME_VFS_FILE_INFO_FOLLOW_LINKS |
					      GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE);
	if (result != GNOME_VFS_OK) {
		gnome_vfs_file_info_unref (info);
		return result;
	}
		
	info->mtime = stamp;

	result = gnome_vfs_set_file_info_uri (uri,
					      info,
					      GNOME_VFS_SET_FILE_INFO_TIME);
	
	gnome_vfs_file_info_unref (info);

	return result;
}

GnomeVFSResult
ob_vfs_utils_uri_get_timestamp (GnomeVFSURI *uri, time_t *stamp)
{
	GnomeVFSResult    result;
	GnomeVFSFileInfo *info;
	
	g_return_val_if_fail (uri != NULL, GNOME_VFS_ERROR_BAD_PARAMETERS);

	info = gnome_vfs_file_info_new ();
	
	result = gnome_vfs_get_file_info_uri (uri,
					      info,
					      GNOME_VFS_FILE_INFO_FOLLOW_LINKS |
					      GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE);

	if (stamp) {
		*stamp = info->mtime;
	}

	gnome_vfs_file_info_unref (info);
	
	return result;
}

GnomeVFSResult
ob_vfs_utils_uri_get_file_type (GnomeVFSURI *uri, GnomeVFSFileType *type)
{
	GnomeVFSResult    result;
	GnomeVFSFileInfo *info;

	g_return_val_if_fail (uri != NULL, GNOME_VFS_ERROR_BAD_PARAMETERS);
	
	info = gnome_vfs_file_info_new ();
	
	result = gnome_vfs_get_file_info_uri (uri,
					      info,
					      GNOME_VFS_FILE_INFO_FOLLOW_LINKS |
					      GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE);

	if (type) {
		*type = info->type;
	}

	gnome_vfs_file_info_unref (info);
	
	return result;
}

/* Append (1), (2), etc until a directory can be created successfully. Assumes
 * that the original URI passed in is not available. The unique URI is returned,
 * or NULL on failure.
 */
GnomeVFSURI *
ob_vfs_utils_create_unique_dir (GnomeVFSURI  *uri,
				guint         mode)
{
	char           *path;
	char           *new_path;
	GnomeVFSURI    *new_uri;
	int             i;
	GnomeVFSResult  result;

	g_return_val_if_fail (uri != NULL, NULL);

	path = gnome_vfs_unescape_string (gnome_vfs_uri_get_path (uri), NULL);
	
	/* If we can't find an available directory name in less than 100 tries,
	 * we give up.
	 */
	i = 1;
	while (i < 100) {
		new_path = g_strdup_printf ("%s(%d)", path, i++);
		new_uri = gnome_vfs_uri_new (new_path);
		
		result = gnome_vfs_make_directory_for_uri (new_uri, mode);
		g_free (new_path);
		
		if (result == GNOME_VFS_OK) {
			/* The umask will mess up the mode if we don't
			 * explicitly set it.
			 */
			result = ob_vfs_utils_uri_set_mode (uri, mode);
			break;
		}

		gnome_vfs_uri_unref (new_uri);
		new_uri = NULL;
		
		if (result == GNOME_VFS_ERROR_FILE_EXISTS ||
		    result == GNOME_VFS_ERROR_IS_DIRECTORY) {
			continue;
		} else {
			/* We can't handle other errors, bail out. */
			break;
		}
	}

	return new_uri;
}



/* Append (1), (2) etc before the extension, until a file can be created
 * successfully. Assumes that the original URI passed in is not available. The
 * unique URI is returned, or NULL on failure.
 */
GnomeVFSURI *
ob_vfs_utils_create_unique_file (GnomeVFSHandle **handle,
				 GnomeVFSURI     *uri,
				 guint            mode)
{
	GnomeVFSURI    *parent_uri;
	char           *basename;
	char           *template;
	char           *tmp;
	GnomeVFSURI    *new_uri;
	int             i;
	GnomeVFSResult  result;

	g_return_val_if_fail (handle != NULL, NULL);
	g_return_val_if_fail (uri != NULL, NULL);

	*handle = NULL;

	parent_uri = gnome_vfs_uri_get_parent (uri);
	basename = gnome_vfs_uri_extract_short_name (uri);

	template = vfs_utils_file_rename_template (basename);
	g_free (basename);

	/* If we can't find an available filename in less than 100 tries, we
	 * give up.
	 */
	i = 1;
	while (i < 100) {
		tmp = g_strdup_printf (template, i++);
		new_uri = gnome_vfs_uri_append_path (parent_uri, tmp);
		g_free (tmp);

		result = gnome_vfs_create_uri (handle,
					       new_uri,
					       GNOME_VFS_OPEN_WRITE,
					       TRUE, /* exclusive */
					       mode);
		
		if (result == GNOME_VFS_OK) {
			break;
		}

		gnome_vfs_uri_unref (new_uri);
		new_uri = NULL;
		
		if (result == GNOME_VFS_ERROR_FILE_EXISTS ||
		    result == GNOME_VFS_ERROR_IS_DIRECTORY) {
			continue;
		} else {
			/* We can't handle other errors, bail out. */
			break;
		}
	}

	gnome_vfs_uri_unref (parent_uri);
	g_free (template);

	return new_uri;
}

/* Append (1), (2) etc before the extension, until a file can be moved
 * successfully. Assumes that the original URI passed in is not available.
 */
GnomeVFSResult
ob_vfs_utils_move_to_unique_uri (GnomeVFSURI      *src_uri,
				 GnomeVFSURI      *base_uri,
				 GnomeVFSURI     **dest_uri_p)
{
	char           *template_fmt;
	char           *basename;
	GnomeVFSResult  result;
	GnomeVFSURI    *new_uri, *parent_uri;
	guint           i;

	g_return_val_if_fail (src_uri != NULL, GNOME_VFS_ERROR_INTERNAL);
	g_return_val_if_fail (base_uri != NULL, GNOME_VFS_ERROR_INTERNAL);

	basename = gnome_vfs_uri_extract_short_name (base_uri);
	template_fmt = vfs_utils_file_rename_template (basename);
	g_free (basename);

	/* If we can't find an available filename in 999 tries, give up. */
	parent_uri = gnome_vfs_uri_get_parent (base_uri);
        i = 1;
	while (i <= 999) {
		char *tmp;

		tmp = g_strdup_printf (template_fmt, i++);
		new_uri = gnome_vfs_uri_append_path (parent_uri, tmp);
		g_free (tmp);
		
		result = gnome_vfs_move_uri (src_uri, new_uri, FALSE);
		if (result == GNOME_VFS_OK) {
			break;
		}

		gnome_vfs_uri_unref (new_uri);
		new_uri = NULL;
	}

	gnome_vfs_uri_unref (parent_uri);
	g_free (template_fmt);

	/* Return results */
	if (dest_uri_p) {
		*dest_uri_p = new_uri;
	}
	else if (new_uri) {
		gnome_vfs_uri_unref (new_uri);
	}
       
	return result;
}

static GnomeVFSURI *
generate_temp_uri (GnomeVFSURI *uri)
{
	int  i;
	char tmp[9];

	for (i = 0; i < 8; i++) {
		if (g_random_int_range (0, 2) == 0) {
			tmp[i] = g_random_int_range ('a', 'z' + 1);
		} else {
			tmp[i] = g_random_int_range ('A', 'Z' + 1);
		}
	}
	tmp[8] = '\0';
	
	return gnome_vfs_uri_append_path (uri, tmp);
}

/* Try to open a temparary file in the specified directory. Returns the URI for
 * the file in file_uri on success.
 */
GnomeVFSResult
ob_vfs_utils_create_temp_file (GnomeVFSHandle **handle,
			       GnomeVFSURI     *dir_uri,
			       guint            mode,
			       GnomeVFSURI    **file_uri)
{
	GnomeVFSURI    *temp_uri;
	int             i;
	GnomeVFSResult  result;

	g_return_val_if_fail (handle != NULL, GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail (dir_uri != NULL, GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail (file_uri != NULL, GNOME_VFS_ERROR_BAD_PARAMETERS);

	*handle = NULL;
	*file_uri = NULL;

	i = 0;
	while (i++ < 100) {
		temp_uri = generate_temp_uri (dir_uri);
		
		result = gnome_vfs_create_uri (handle,
					       temp_uri,
					       GNOME_VFS_OPEN_WRITE,
					       TRUE, /* exclusive */
					       mode);

		if (result == GNOME_VFS_OK) {
			break;
		}

		gnome_vfs_uri_unref (temp_uri);
		temp_uri = NULL;
		
		if (result == GNOME_VFS_ERROR_FILE_EXISTS ||
		    result == GNOME_VFS_ERROR_IS_DIRECTORY) {
			continue;
		} else {
			/* We can't handle other errors, bail out. */
			break;
		}
	}

	if (result == GNOME_VFS_OK) {
		*file_uri = temp_uri;
	}

	return result;
}

static void
vfs_utils_count_files_helper (const gchar      *uri_str,
			      gint             *files,
			      GnomeVFSFileSize *size)
{
	GnomeVFSResult           result;
	GnomeVFSDirectoryHandle *handle;	
	GnomeVFSFileInfo        *info;
	gchar                   *child_uri_str;

	result = gnome_vfs_directory_open (
		&handle, uri_str,
		GNOME_VFS_FILE_INFO_FOLLOW_LINKS |
		GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE);

	if (result != GNOME_VFS_OK) {
		/* Couldn't open, skip. */
		return;
	}
	
	info = gnome_vfs_file_info_new ();

	while (1) {
		result = gnome_vfs_directory_read_next (handle, info);
		
		if (result == GNOME_VFS_ERROR_EOF) {
			break;
		}
		else if (result != GNOME_VFS_OK) {
			/* Couldn't read, skip. */
			continue;
		}

		/* Recurse for directories but skip current and parent entries
		 * ("." and "..").
		 */
		if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY &&
		    strcmp (info->name, ".") != 0 &&
		    strcmp (info->name, "..") != 0) {
			child_uri_str = g_build_filename (G_DIR_SEPARATOR_S, uri_str, info->name, NULL);
			vfs_utils_count_files_helper (child_uri_str, files, size);
			g_free (child_uri_str);
		}
		else if (info->type == GNOME_VFS_FILE_TYPE_REGULAR) {
			(*files)++;
			*size += info->size;
		}
		
		gnome_vfs_file_info_clear (info);	
	}

	gnome_vfs_file_info_unref (info);
	gnome_vfs_directory_close (handle);
}

/* Counts the number of files and their total size in the locations
 * specified. Locations is a list of URIs, non-overlapping. (For example as
 * returned from the ObBackupLocations API).
 */
void
ob_vfs_utils_count_files (GList            *locations,
			  int              *files,
			  GnomeVFSFileSize *size)
{
	GList            *l;
	const gchar      *uri_str;
	GnomeVFSResult    result;
	GnomeVFSFileInfo *info;
	int               files_internal;
	GnomeVFSFileSize  size_internal;

	info = gnome_vfs_file_info_new ();

	files_internal = 0;
	size_internal = 0;

	for (l = locations; l; l = l->next) {
		uri_str = l->data;

		result = gnome_vfs_get_file_info (
			uri_str, info,
			GNOME_VFS_FILE_INFO_FOLLOW_LINKS |
			GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE);
		if (result == GNOME_VFS_OK) {
			if (info->type == GNOME_VFS_FILE_TYPE_REGULAR) {
				files_internal++;
				size_internal += info->size;
			}
			else if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
				vfs_utils_count_files_helper (uri_str,
							      &files_internal,
							      &size_internal);
			}
		}

		gnome_vfs_file_info_clear (info);
	}	

	gnome_vfs_file_info_unref (info);

	if (files) {
		*files = files_internal;
	}
	if (size) {
		*size = size_internal;
	}
}

static void
vfs_utils_get_file_entries_helper (GnomeVFSURI *uri,
				   GHashTable  *entries)
{
	GnomeVFSResult           result;
	GnomeVFSDirectoryHandle *handle;	
	GnomeVFSFileInfo        *info, *tmp_info;
	GnomeVFSURI             *child_uri;
	ObFileEntry             *entry;

	result = gnome_vfs_directory_open_from_uri (
		&handle, uri,
		GNOME_VFS_FILE_INFO_FOLLOW_LINKS |
		GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE);

	if (result != GNOME_VFS_OK) {
		/* Couldn't open, skip. */
		return;
	}
	
	info = gnome_vfs_file_info_new ();

	while (1) {
		result = gnome_vfs_directory_read_next (handle, info);
		
		if (result == GNOME_VFS_ERROR_EOF) {
			break;
		}
		else if (result != GNOME_VFS_OK) {
			/* Couldn't read, skip. */
			continue;
		}

		/* Recurse for directories but skip current and parent entries
		 * ("." and "..").
		 */
		if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY &&
		    strcmp (info->name, ".") != 0 &&
		    strcmp (info->name, "..") != 0) {
			child_uri = gnome_vfs_uri_append_path (uri, info->name);
			vfs_utils_get_file_entries_helper (child_uri, entries);
		}
		else if (info->type == GNOME_VFS_FILE_TYPE_REGULAR) {
			child_uri = gnome_vfs_uri_append_path (uri, info->name);
			tmp_info = gnome_vfs_file_info_dup (info);
		
			entry = ob_file_entry_new (child_uri, tmp_info);
			g_hash_table_insert (entries, child_uri, entry);
			
			gnome_vfs_file_info_unref (tmp_info);
		}
		
		gnome_vfs_file_info_clear (info);	
	}

	gnome_vfs_file_info_unref (info);
	gnome_vfs_directory_close (handle);
}

/* Gets a list of ObFileEntry from a list of URIs. Needs to be freed with
 * ob_file_entry_free_list().
 */

static gboolean 
vfs_utils_get_file_entries_remove_foreach (GnomeVFSURI  *uri,
					   ObFileEntry  *entry,
					   GPatternSpec *spec)
{
	gchar    *unescaped;
	gboolean  found;
	
	unescaped = gnome_vfs_unescape_string (gnome_vfs_uri_get_path (uri), NULL);
	found = g_pattern_match_string (spec, unescaped);
	g_free (unescaped);

	if (found) {
		d(g_print ("\tRemoving entry:'%s'\n", 
			   gnome_vfs_uri_get_path (uri)));
		return TRUE;
	}

	return FALSE;
}

static void
vfs_utils_get_file_entries_remove_func (const gchar *uri_str,
					GHashTable  *entries)
{
	GPatternSpec *spec;

	d(g_print ("\tRemoving entries with pattern:'%s'\n", uri_str));

	spec = g_pattern_spec_new (uri_str);

	g_hash_table_foreach_remove (
		entries, 
		(GHRFunc) vfs_utils_get_file_entries_remove_foreach,
		spec);

	g_pattern_spec_free (spec);
}

static void
vfs_utils_get_file_entries_list_foreach (GnomeVFSURI  *uri,
					 ObFileEntry  *entry,
					 GSList      **list)
{
	d(g_print ("\tAdding file:'%s'\n", gnome_vfs_uri_get_path (uri)));

	*list = g_slist_append (*list, ob_file_entry_dup (entry));
}

GHashTable *
ob_vfs_utils_get_file_entries (GList *locations)
{
	GHashTable       *entries;
	GnomeVFSURI      *uri;
	GnomeVFSResult    result;
	GnomeVFSFileInfo *info, *tmp_info;
	ObFileEntry      *entry;
	GList            *l;
	const gchar      *uri_str;
	gint              options;

	d(g_print ("\tGetting FULL URI list (from %d URIs)\n", 
		   g_list_length (locations)));

	info = gnome_vfs_file_info_new ();

	options = 
		GNOME_VFS_FILE_INFO_FOLLOW_LINKS |
		GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE;

	entries = g_hash_table_new_full (gnome_vfs_uri_hash, 
					 gnome_vfs_uri_hequal,
					 (GDestroyNotify) gnome_vfs_uri_unref,
					 (GDestroyNotify) ob_file_entry_free);
	
	for (l = locations; l; l = l->next) {
		uri_str = l->data;
		d(g_print ("\tChecking file:'%s'\n", uri_str));

		uri = gnome_vfs_uri_new (uri_str);
	
		result = gnome_vfs_get_file_info_uri (uri, info, options);
		if (result == GNOME_VFS_OK) {
			if (info->type == GNOME_VFS_FILE_TYPE_REGULAR) {
				gnome_vfs_uri_ref (uri);

				tmp_info = gnome_vfs_file_info_dup (info);
				entry = ob_file_entry_new (uri, tmp_info);
				gnome_vfs_file_info_unref (tmp_info);

				g_hash_table_insert (entries, uri, entry);
			}
			else if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
				vfs_utils_get_file_entries_helper (uri, entries);
			}
		}

		gnome_vfs_uri_unref (uri);	
		gnome_vfs_file_info_clear (info);
	}	

	gnome_vfs_file_info_unref (info);

	return entries;
}

GSList *
ob_vfs_utils_get_file_entries_filtered (GList *include_uris,
					GList *exclude_uris)
{
	GSList     *uris = NULL;
	GHashTable *entries;

	/* Generate a FULL list of files to be included. */
	d(g_print ("Creating list for included URIs (%d items)\n",
		   g_list_length (include_uris)));

	entries = ob_vfs_utils_get_file_entries (include_uris);

	/* Remove the duplicates found from the exclude list in the include list. */
	d(g_print ("Removing all excluded URIs found (%d items)\n",
		   g_hash_table_size (entries)));

	g_list_foreach (exclude_uris, 
			(GFunc) vfs_utils_get_file_entries_remove_func,
			entries);
	
	/* Get list from entries. */
	d(g_print ("Creating final URI list\n"));

	g_hash_table_foreach (entries, 
			      (GHFunc) vfs_utils_get_file_entries_list_foreach,
			      &uris);

	d(g_print ("Finished final URI list (%d items)\n",
		   g_slist_length (uris)));

	/* Clean up. */
	g_hash_table_destroy (entries);

	d(g_print ("\n"));

	return uris;
}

typedef struct {
	ObRmDirCancelCb callback_func;
	gpointer        callback_data;
} RmDirCancelData;

static int
remove_dir_progress_cb (GnomeVFSXferProgressInfo *info,
			gpointer                  user_data)
{
	RmDirCancelData *data;

	data = user_data;

	if (data->callback_func) {
		return data->callback_func (data->callback_data);
	} 
	
	return TRUE;
}

/* Removes the directory at uri recursively, with the possibility to cancel. */
GnomeVFSResult
ob_vfs_utils_remove_directory (GnomeVFSURI     *uri,
			       ObRmDirCancelCb  cancel_func,
			       gpointer         cancel_data)
{
	GList           *list;
	GnomeVFSResult   result;
	RmDirCancelData  data;

	g_return_val_if_fail (uri != NULL, GNOME_VFS_ERROR_BAD_PARAMETERS);
	
	data.callback_func = cancel_func;
	data.callback_data = cancel_data;
	
	list = g_list_append (NULL, uri);

	result = gnome_vfs_xfer_delete_list (
		list,
		GNOME_VFS_XFER_ERROR_MODE_ABORT,
		GNOME_VFS_XFER_DEFAULT | GNOME_VFS_XFER_RECURSIVE,
		remove_dir_progress_cb,
		&data);
	
	g_list_free (list);

	return result;
}

/* Writes the entire buffer passed in. */
GnomeVFSResult
ob_vfs_utils_write (GnomeVFSHandle   *handle,
		    gconstpointer     buffer,
		    GnomeVFSFileSize  bytes)
{
	GnomeVFSFileSize bytes_written;
	GnomeVFSResult   result;

	g_return_val_if_fail (handle != NULL, GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail (buffer != NULL, GNOME_VFS_ERROR_BAD_PARAMETERS);
	
	while (bytes > 0) {
		result = gnome_vfs_write (handle, buffer,
					  bytes, &bytes_written);
		if (result != GNOME_VFS_OK) {
			return result;
		}

		bytes -= bytes_written;
	}
	
	return GNOME_VFS_OK;
}

#if 0
static GnomeVFSResult
create_dir_and_ancestors (GnomeVFSURI *uri)
{
	GnomeVFSResult  result;
	GnomeVFSURI    *parent;

	/* Try to create each directory in the hierarchy. If the last part can
	 * be created or already exists, returns GNOME_VFS_OK, otherwise an
	 * error.
	 */

	parent = gnome_vfs_uri_parent (uri);
	if (!parent) {
		/* At the root. */
		return GNOME_VFS_OK;
	}
	
	create_dir_and_ancestors (parent);
	gnome_vfs_uri_unref (parent);
	
	return result;
}
#endif
