/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <signal.h>
#include <string.h>
#include <glib/gi18n.h>
#include <gconf/gconf-client.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <openssl/sha.h>
#include <openssl/aes.h>
#include <osso-helplib.h>
#include <hildon-widgets/hildon-dialoghelp.h>
#include <libogs/ogs-file-system.h>
#include <mce/dbus-names.h>
#include <mce/mode-names.h>

#include "ob-log.h"
#include "ob-utils.h"

static osso_context_t *ctx;
static OgsFileSystem  *file_system;

void
ob_utils_set_osso_context (osso_context_t *context)
{
	ctx = context;
}

osso_context_t *
ob_utils_get_osso_context (void)
{
	return ctx;
}

DBusConnection *
ob_utils_get_dbus_connection (void)
{
	if (!ctx) {
		return NULL;
	}
	
	return osso_get_dbus_connection (ctx);
}

void
ob_utils_set_file_system (OgsFileSystem *fs)
{
	file_system = fs;
}

OgsFileSystem *
ob_utils_get_file_system (void)
{
	return file_system;
}

/* Leave this for debugging. */
#if 0
void
ob_utils_hexdump (guchar *data)
{
	g_print ("        "
		 "%02x%02x %02x%02x "
		 "%02x%02x %02x%02x "
		 "%02x%02x %02x%02x "
		 "%02x%02x %02x%02x\n",
		 data[0], data[1],
		 data[2], data[3],
		 data[4], data[5],
		 data[6], data[7],
		 data[8], data[9],
		 data[10], data[11],
		 data[12], data[13],
		 data[14], data[15]);
}
#endif

guchar *
ob_utils_get_128_bits_hash (const char *password)
{
	char   digest[SHA_DIGEST_LENGTH];
	guchar *hash;

	SHA1 (password, strlen (password), digest);

	/* Remove 32 bits to get 128 bits. */
	hash = g_malloc (16);

	memcpy (hash,      digest,      4);
	memcpy (hash + 4,  digest + 5,  4);
	memcpy (hash + 8,  digest + 10, 4);
	memcpy (hash + 12, digest + 15, 4);

	return hash;
}

void
ob_utils_xor_16_bytes (guchar *a, const guchar *b)
{
	int i;
	
	for (i = 0; i < 16; i++) {
		a[i] ^= b[i];
	}
}

guchar *
ob_utils_create_random_16_bytes (void)
{
	guchar *bytes;
	int     i;

	bytes = g_malloc (16);

	for (i = 0; i < 16; i++) {
		bytes[i] = g_random_int_range (0, 256);
	}

	return bytes;
}

gchar *
ob_utils_get_timestamp_string (time_t t)
{
       struct tm   *tm;
       gchar        buf[64];
       const gchar *format = "%c"; /* Gets rid of warning. */
       
       tm = localtime (&t);
       strftime (buf, 64, format, tm);

       return g_strdup (buf);
}

gchar *  
ob_utils_get_time_string (time_t t)
{
       struct tm   *tm;
       gchar        buf[64];
       const gchar *format = "%X"; /* Gets rid of warning. */
       
       tm = localtime (&t);
       strftime (buf, 64, format, tm);

       return g_strdup (buf);
}

gchar *
ob_utils_get_date_string (time_t t)
{
       struct tm   *tm;
       gchar        buf[64];
       const gchar *format = "%x"; /* Gets rid of warning. */
       
       tm = localtime (&t);
       strftime (buf, 64, format, tm);

       return g_strdup (buf);
}


/* Info about the last backup made. */

/*
<last-backup>
  <name>Foo</name
  <timestamp>1234567</timestamp>
  <memory-card>MyCard</memory-card>
</last-backup>
*/

static gboolean
parse_last_backup (char **name, char **memory_card, time_t *timestamp)
{
	char    *path;
	xmlDoc  *doc;
	xmlNode *root, *node;
	char    *str;
	char    *endptr;

	*timestamp = 0;
	*memory_card = NULL;
	*name = NULL;

	path = g_build_filename (g_get_home_dir (),
				 ".osso-backup",
				 "last-backup.xml",
				 NULL); 

	if (!g_file_test (path, G_FILE_TEST_IS_REGULAR)) {
		g_free (path);
		return FALSE;
	}
	
	doc = xmlParseFile (path);
	g_free (path);

	if (!doc) {
		return FALSE;
	}

	root = xmlDocGetRootElement (doc);

	if (!root || strcmp (root->name, "last-backup") != 0) {
		/* Not the right format. */
		xmlFreeDoc (doc);

		return FALSE;
	}
	
	for (node = root->children; node; node = node->next) {
		if (strcmp (node->name, "name") == 0) {
			str = xmlNodeGetContent (node);

			*name = g_strdup (str);
			xmlFree (str);
		}
		else if (strcmp (node->name, "memory-card") == 0) {
			str = xmlNodeGetContent (node);

			*memory_card = g_strdup (str);
			xmlFree (str);
		}
		else if (strcmp (node->name, "timestamp") == 0) {
			str = xmlNodeGetContent (node);
			
			*timestamp = strtol (str, &endptr, 10);
			if (*endptr != '\0') {
				g_warning ("Invalid timestamp in last backup info.");
			}
			xmlFree (str);
		}
	}

	xmlFreeDoc (doc);
	
	return TRUE;
}

gboolean
ob_utils_get_last_backup (char   **name,
			  char   **memory_card_name,
			  time_t  *timestamp)
{
	char   *name_internal;
	char   *memory_card_name_internal;
	time_t  timestamp_internal;

	if (!parse_last_backup (&name_internal,
				&memory_card_name_internal,
				&timestamp_internal)) {
		return FALSE;
	}

	if (name) {
		*name = name_internal;
	} else {
		g_free (name_internal);
	}

	if (memory_card_name) {
		*memory_card_name = memory_card_name_internal;
	} else {
		g_free (memory_card_name);
	}
	
	if (timestamp) {
		*timestamp = timestamp_internal;
	}
	
	return TRUE;
}

gboolean
ob_utils_set_last_backup (const char *name,
			  const char *memory_card_name,
			  time_t      timestamp)
{
	char   *str;
	char   *path;
	FILE   *file;
	size_t  bytes_written;
	
	g_return_val_if_fail (name != NULL, FALSE);
	g_return_val_if_fail (memory_card_name != NULL, FALSE);
	
	/* Make sure the dir is there. */
	path = g_build_filename (g_get_home_dir (), ".osso-backup", NULL);
	if (!g_file_test (path, G_FILE_TEST_EXISTS | G_FILE_TEST_IS_DIR)) {
		if (mkdir (path, 0755) != 0) {
			g_free (path);
			return FALSE;
		}
	}
	g_free (path);
	
	path = g_build_filename (g_get_home_dir (),
				 ".osso-backup",
				 "last-backup.xml",
				 NULL); 
	
	file = fopen (path, "w");
	g_free (path);
	if (!file) {
		return FALSE;
	}

	str = g_strdup_printf ("<last-backup>\n"
			       "  <name>%s</name>\n"
			       "  <memory-card>%s</memory-card>\n"
			       "  <timestamp>%ld</timestamp>\n"
			       "</last-backup>\n",
			       name, memory_card_name, (long) timestamp);

	bytes_written = fwrite (str,  1, strlen (str), file);

	if (bytes_written != strlen (str)) {
		g_free (str);
	
		fclose (file);
		return FALSE;
	}		

	g_free (str);

	fclose (file);
	
	return TRUE;
}

void     
ob_utils_clear_container (GtkContainer *container)
{
	GList *children, *list;

	children = gtk_container_get_children (container);
	for (list = children; list; list = list->next) {
		gtk_container_remove (container, list->data);
	}

	g_list_free (children);
}

void
ob_utils_show_help (const gchar *help_id,
		    gboolean     in_dialog)
{
	osso_return_t result;

	if (!ctx) {
		return;
	}

	result = ossohelp_show (ctx, help_id, 
				in_dialog ? OSSO_HELP_SHOW_DIALOG : 0);
	if (result != OSSO_OK) {
		ob_log_warning ("Could not show help for id:'%s', %d returned\n", 
				help_id, result);
	}
}

static void
help_cb (GtkWidget *dialog)
{
	const gchar *help_id;
	
	help_id = g_object_get_data (G_OBJECT (dialog), "ob-help-id");
	ob_utils_show_help (help_id, TRUE);
}

void
ob_utils_setup_dialog_help (GtkDialog     *dialog,
			    const gchar   *help_id)
{
	static gboolean hack_enabled = FALSE;
	osso_return_t   error;
	
	if (!ctx) {
		return;
	}

	if (!hack_enabled) {
		/* This has to be the worst hack I've ever
		   written. gtk_dialog_help_enable adds a signal on
		   the fly to GtkDialog (which of course is very
		   bad). However, it doesn't check for subclasses so
		   if you call it on a subclass to GtkDialog before
		   calling it on GtkDialog, stuff won't work. That's
		   why we create a GtkDialog, call
		   gtk_dialog_help_enable on it, and then destroy it
		   again */
		GtkWidget *suck;

		suck = gtk_dialog_new ();
		gtk_dialog_help_enable (GTK_DIALOG (suck));
		gtk_widget_destroy (suck);

		hack_enabled = TRUE;

	}

	/* Check for help availability */
	error = ossohelp_show (ctx, help_id, OSSO_HELP_SHOW_JUSTASK);
	if (error == OSSO_OK) {
		g_object_set_data_full (G_OBJECT (dialog), "ob-help-id",
					g_strdup (help_id), g_free);

		gtk_dialog_help_enable (dialog);
		g_signal_connect (dialog, "help", G_CALLBACK (help_cb), NULL);
	}
}

void
ob_utils_decrypt_data (AES_KEY *aes,
		       guchar  *xor,
		       guchar  *data,
		       gsize    num_bytes)
{
	gsize  offset;
	guchar next_xor[16];
	
	offset = 0;
	while (num_bytes > 0) {
		memcpy (next_xor, data + offset, 16);

		/*g_print ("before:\n");
		hexdump (data + offset);
		*/
		AES_decrypt (data + offset,
			     data + offset,
			     aes);

		/*
		g_print ("before xor:\n");
		hexdump (data + offset);

		g_print ("xor:\n");
		hexdump (xor);
		*/
		ob_utils_xor_16_bytes (data + offset, xor);
		/*
		g_print ("after:\n");
		hexdump (data + offset);
		*/
		memcpy (xor, next_xor, 16);

		offset += 16;
		num_bytes -= 16;
	}
}

gboolean
ob_utils_check_password (const char *path, const char *password)
{
	FILE     *file;
	guint8    buf[32];
	size_t    buf_size;
	guchar   *hash;
	AES_KEY   aes;
	gboolean  ret;     

	g_return_val_if_fail (path != NULL, FALSE);
	g_return_val_if_fail (password != NULL, FALSE);
	
	file = fopen (path, "r");
	if (!file) {
		return FALSE;
	}

	buf_size = fread (buf, 1, 32, file);
	fclose (file);
	
	/* We must always have at least 16 bytes for the IV and one block for
	 * data.
	 */
	if (buf_size < 32) {
		return FALSE;
	}

	hash = ob_utils_get_128_bits_hash (password);

	/* Initialize the AES data. */
	AES_set_decrypt_key (hash, 128, &aes);

	ob_utils_decrypt_data (&aes,
			       buf,
			       buf + 16,
			       16);

	if (buf[16] == 'P' && buf[17] == 'K') {
		ret = TRUE;
	} else {
		ret = FALSE;
	}
	
	g_free (hash);

	return ret;
}

const char *
ob_utils_get_device_version (void)
{
	static char     *version = NULL;
	static gboolean  inited = FALSE;
	char            *contents;

	if (inited) {
		return version;
	}

	/* Read once, keep for the life-time of the app. */
	if (g_file_get_contents (SYSCONFDIR "/osso_software_version",
				 &contents, NULL, NULL)) {
		version = g_strstrip (contents);
	} else {
		version = g_strdup ("");
	}
	
	inited = TRUE;
	
	return version;
}

/* Handles $HOME prefix, strips whitespace, removes any trailing slash. */
char *
ob_utils_parse_path (const char *str)
{
	gchar    *dup, *stripped;
	gboolean  home;
	int       len;
	char     *subdir;
	char     *ret;

	if (!str) {
		return NULL;
	}

	dup = g_strdup (str);
	stripped = g_strstrip (dup);
	
	if (g_str_has_prefix (stripped, "$HOME")) {
		home = TRUE;
		subdir = g_strdup (stripped + strlen ("$HOME"));

                if (subdir[0] != '/') {
			g_warning ("Path after $HOME must be absolute.");
                        g_free (dup);
                        g_free (subdir);
                        return NULL;
                }
	} else {
		/* Must be absolute. */
		if (stripped[0] != '/') {
			g_warning ("Path must be absolute.");
			g_free (dup);
			return NULL;
		}
		
		home = FALSE;
		subdir = g_strdup (stripped);
	}

	g_free (dup);
	
	/* Remove any trailing slash. */
	len = strlen (subdir);
	if (len > 0 && subdir[len - 1] == '/') {
		subdir[len - 1] = '\0';
	}

	if (home) {
		ret = g_build_filename (g_get_home_dir (), subdir, NULL);

                g_free (subdir);
                return ret;
	} else {
		return subdir;
	}
}

void
ob_utils_send_reboot_message (void)
{
	DBusConnection *conn;
	DBusMessage    *msg;

	if (g_getenv ("OSSO_BACKUP_DEBUG")) {
		return;
	}

	/* Helps debugging. */
	ob_log_info ("Sending reboot.");

	conn = dbus_bus_get (DBUS_BUS_SYSTEM, NULL);
	if (!conn) {
		ob_log_warning ("Could not get system bus.");
		return;
	}
	
	msg = dbus_message_new_method_call (MCE_SERVICE,
					    MCE_REQUEST_PATH,
					    MCE_REQUEST_IF,
					    MCE_REBOOT_REQ);

	dbus_connection_send (conn, msg, NULL);
	dbus_connection_flush (conn);
	
	ob_log_info ("Sent, quit the application.");
}

void
ob_utils_kill_apps (void)
{
	int       exit_status;
	gboolean  script_retval;

	if (g_getenv ("OSSO_BACKUP_DEBUG")) {
		return;
	}

	/* Ignore SIGTERM from now on since otherwise we'll be killed too. */
	signal (SIGTERM, SIG_IGN);
	
	script_retval = g_spawn_command_line_sync (
		"/usr/sbin/osso-app-killer-restore.sh", 
		NULL, NULL,
		&exit_status,
		NULL);
		
	if (!script_retval || exit_status != 0) {
		ob_log_warning ("Could not run osso-app-killer-restore.sh");
	}

	/* Restore SIGTERM. */
	signal (SIGTERM, SIG_DFL);
}

gboolean
ob_utils_get_flight_mode (void)
{
	DBusConnection *conn;
	DBusMessage    *message, *reply;
	char           *str;
	gboolean        flight;

	if (g_getenv ("OSSO_BACKUP_DEBUG")) {
		return FALSE;
	}
	
	ob_log_info ("Getting device mode.");

	conn = dbus_bus_get (DBUS_BUS_SYSTEM, NULL);
	if (!conn) {
		ob_log_warning ("Could not get system bus.");
		return FALSE;
	}
	
	message = dbus_message_new_method_call (MCE_SERVICE,
						MCE_REQUEST_PATH,
						MCE_REQUEST_IF,
						MCE_DEVICE_MODE_GET);
	
	reply = dbus_connection_send_with_reply_and_block (conn,
							   message, -1,
							   NULL);
  
	dbus_message_unref (message);
	
	if (!reply) {
		ob_log_warning ("Failed to send device mode get message.");
		return FALSE;
	}

	if (dbus_message_get_type (reply) == DBUS_MESSAGE_TYPE_ERROR) {
		ob_log_warning ("Got error reply.");
		dbus_message_unref (reply);
		return FALSE;
	}
	
	flight = FALSE;

	if (dbus_message_get_args (reply, NULL,
				   DBUS_TYPE_STRING, &str,
				   DBUS_TYPE_INVALID)) {
		if (strcmp (str, MCE_FLIGHT_MODE) == 0) {
			ob_log_info ("Got flight mode.");
			flight = TRUE;
		}
		else if (strcmp (str, MCE_NORMAL_MODE) == 0) {
			ob_log_info ("Got normal mode.");
			flight = FALSE;
		} else {
			ob_log_warning ("Got bad reply.");
			flight = FALSE;
		}
	}
	
	dbus_message_unref (reply);

	return flight;
}

void
ob_utils_set_flight_mode (gboolean flight)
{

	DBusConnection *conn;
	DBusMessage    *message;
	const char     *flight_str;
	const char     *normal_str;

	if (g_getenv ("OSSO_BACKUP_DEBUG")) {
		return;
	}

	ob_log_info ("Resetting device mode.");

	conn = dbus_bus_get (DBUS_BUS_SYSTEM, NULL);
	if (!conn) {
		ob_log_warning ("Could not get system bus.");
		return;
	}
	
	message = dbus_message_new_method_call (MCE_SERVICE,
						MCE_REQUEST_PATH,
						MCE_REQUEST_IF,
						MCE_DEVICE_MODE_CHANGE_REQ);

	flight_str = MCE_FLIGHT_MODE;
	normal_str = MCE_NORMAL_MODE;

	dbus_message_append_args (message,
				  DBUS_TYPE_STRING, flight ?
				  &flight_str : &normal_str,
				  DBUS_TYPE_INVALID);

	dbus_connection_send (conn, message, NULL);
	dbus_connection_flush (conn);
	
	dbus_message_unref (message);
}

/* The prefix used for third-party apps. */
char *
ob_utils_get_install_home (void)
{
	const char *dir;
	
	dir = g_getenv ("INSTALLHOME");
	if (!dir) {
		return g_strdup (LOCALSTATEDIR "/lib/install");
	}

	return gnome_vfs_make_path_name_canonical (dir);
}

#define KILOBYTE_FACTOR (1024.0)
#define MEGABYTE_FACTOR (1024.0 * 1024.0)
#define GIGABYTE_FACTOR (1024.0 * 1024.0 * 1024.0)

gchar *
ob_utils_get_size (guint bytes)
{
	gdouble      size_double;
	gint         size_int;
	const gchar *i18n_hack;

	if (bytes < KILOBYTE_FACTOR) {
		i18n_hack = _("back_li_size_max_99kb");
		return g_strdup_printf (i18n_hack, 1);
	} 

	if (bytes >= (1 * KILOBYTE_FACTOR) &&
	    bytes < (100 * KILOBYTE_FACTOR)) {
		size_int = bytes / KILOBYTE_FACTOR;
		i18n_hack =  _("back_li_size_max_99kb");
		return g_strdup_printf (i18n_hack, size_int);
	} 

	if (bytes >= (100 * KILOBYTE_FACTOR) &&
	    bytes < (10 * MEGABYTE_FACTOR)) {
		size_double = bytes / MEGABYTE_FACTOR;
		i18n_hack = _("back_li_size_100kb_10mb");
		return g_strdup_printf (i18n_hack, size_double);
	}
	
	if (bytes >= (10 * MEGABYTE_FACTOR) && 
	    bytes < (1 * GIGABYTE_FACTOR)) {
		size_int = bytes / MEGABYTE_FACTOR;
		i18n_hack = _("back_li_size_10mb_1gb");
		return g_strdup_printf (i18n_hack, size_int);
	} 
	
	size_double = bytes / GIGABYTE_FACTOR;
	i18n_hack = _("back_li_size_larger_than_1gb");
	return g_strdup_printf (i18n_hack, size_double);
}

gboolean
ob_utils_get_is_usb_inserted (void)
{
	GConfClient *gconf_client;
	gboolean     inserted;
	
	gconf_client = gconf_client_get_default();

	inserted = gconf_client_get_bool (gconf_client,
					  "/system/osso/af/usb-cable-attached",
					  NULL);

	g_object_unref (gconf_client);

	return inserted;
}
