/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>

#include <glib/gi18n.h>
#include <gtk/gtk.h>

#include <hildon-widgets/hildon-note.h>

#include "ob-ui.h"

static void
ui_dialog_add_buttons (GtkDialog    *dialog, 
                       const gchar  *first_button_text,
                       va_list       args)
{
        const gchar  *text;
	GtkWidget   **widget;
        gint          resp_id;
        
        text = first_button_text;
        resp_id = va_arg (args, gint);
        widget = va_arg (args, GtkWidget **);

        while (text != NULL) {
		GtkWidget *button;

                button = gtk_dialog_add_button (GTK_DIALOG (dialog), text, resp_id);
		if (widget) {
			*widget = button;
		}

                text = va_arg (args, gchar *);
                if (!text) {
                        break;
                }

                resp_id = va_arg (args, gint);
		widget = va_arg (args, GtkWidget **);
        }
}

GtkWidget *
ob_ui_dialog_new (const gchar *title,
                  GtkWindow   *parent,
                  const gchar *first_button_text,
                  ...)
{
        GtkWidget   *dialog;
        va_list      args;
        
        dialog = gtk_dialog_new ();

        gtk_window_set_title (GTK_WINDOW (dialog), title);
        gtk_window_set_transient_for (GTK_WINDOW (dialog), parent);
        gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
        
        gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);

	gtk_container_set_border_width (GTK_CONTAINER (dialog),
                                        OB_UI_MARGIN_DEFAULT);
	
        va_start (args, first_button_text);
        ui_dialog_add_buttons (GTK_DIALOG (dialog), first_button_text, args);
        va_end (args);

        return dialog;
}

GtkWidget *
ob_ui_message_dialog_new (GtkWindow       *parent,
                          GtkMessageType   type,
			  const gchar     *title,
                          const gchar     *message,
                          const gchar     *first_button_text,
                          ...)
{
        GtkWidget *dialog;
        va_list    args;

        dialog = gtk_message_dialog_new (parent, GTK_DIALOG_MODAL,
                                         type, GTK_BUTTONS_NONE,
                                         message);

	if (title) {
		gtk_window_set_title (GTK_WINDOW (dialog), title);
	}

	gtk_container_set_border_width (GTK_CONTAINER (dialog),
                                        OB_UI_MARGIN_DEFAULT);
	
        va_start (args, first_button_text);
        ui_dialog_add_buttons (GTK_DIALOG (dialog), first_button_text, args);
        va_end (args);

        return dialog;
}

GtkWidget *
ob_ui_label_new (const gchar *str)
{
        GtkWidget *label;

        label = gtk_label_new (str);
        gtk_misc_set_alignment (GTK_MISC (label), 0.0, 0.5);

        return label;
}

void
ob_ui_progress_dialog_destroy (ObProgressDialog *dialog)
{
	gtk_widget_destroy (dialog->dialog);

	if (dialog->error) {
		g_error_free (dialog->error);
	}

	g_free (dialog);
}

ObProgressDialog *
ob_ui_progress_dialog_new (GtkWindow     *parent, 
                           ObUiOperation  operation,
                           gint           total_files)
{
        ObProgressDialog *dialog;
	gchar            *desc;
	const gchar      *i18n_hack;
	
	/* We don't know the category when the dialog is first
	 * created, we know that once the first file is packed or
         * unpacked.
	 */
        if (operation == OB_UI_OPERATION_BACKUP) {
		i18n_hack = _("back_va_backed_up_category");
		desc = g_strdup_printf (i18n_hack, "");
        } else {
		desc = g_strdup (_("back_va_initializing_restore"));
        }
        
        dialog = g_new0 (ObProgressDialog, 1);
        dialog->operation = operation;
        dialog->error = NULL;
        dialog->total_files = total_files;
        dialog->processed_files = -1;
	dialog->complete = -1;

        dialog->progress_bar = gtk_progress_bar_new ();

	/* Apparently they changed the default in the patched GTK+, but we
	 * should have a centered text.
	 */
	g_object_set (dialog->progress_bar, "text-xalign", 0.5, NULL);

        dialog->dialog = hildon_note_new_cancel_with_progress_bar (
		parent,
		desc,
		GTK_PROGRESS_BAR (dialog->progress_bar));
	
	g_free (desc);

	gtk_window_set_default_size (GTK_WINDOW (dialog->dialog), 450, -1);
	gtk_window_set_position (GTK_WINDOW (dialog->dialog), GTK_WIN_POS_CENTER_ON_PARENT);
	
        return dialog;
}

void
ob_ui_progress_dialog_update (ObProgressDialog *dialog,
                              gint              files, 
                              gint              complete, 
			      ObCategory        category)
{
        gchar       *str;
	const gchar *i18n_hack;

	if (dialog->processed_files != files) {
		dialog->processed_files = files;

		/* Add one so we don't start at 0. The number is really number
		 * of finished files, but it looks bad in the UI, so we show the
		 * number of files including the one in progress.
		 */

		if (dialog->operation == OB_UI_OPERATION_BACKUP) {
			/* Formatted logical string. */
			i18n_hack = _("back_va_backed_up_category");
			str = g_strdup_printf (i18n_hack, ob_category_to_string (category));
		} else {
			/* Formatted logical string. */
			i18n_hack = _("back_va_restoring_category");
			str = g_strdup_printf (i18n_hack, ob_category_to_string (category));
		}
		
		g_object_set (dialog->dialog,
			      "description", str,
			      NULL);
		
		g_free (str);
	}

	if (dialog->complete != complete) {
		dialog->complete = complete;

		gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (dialog->progress_bar),
					       complete / 100.0);
		
		/* Formatted logical string. */
		i18n_hack = _("calc_va_percentage");
		str = g_strdup_printf (i18n_hack, complete);
		gtk_progress_bar_set_text (GTK_PROGRESS_BAR (dialog->progress_bar),
					   str);
		g_free (str);
	}
}

void
ob_ui_progress_dialog_set_string (ObProgressDialog *dialog,
				  const gchar      *str)
{
	g_object_set (dialog->dialog,
		      "description", str,
		      NULL);
}


