/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

/* This is a small simple program that is used to restore files that are not
 * writable by the user, or in a directory not writable by the user. It's run by
 * the backup application during restore, using sudo.
 */

#define ROOT_UID 0
#define ROOT_GID 0
#define USER_UID 29999
#define USER_GID 29999

#include <config.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <errno.h>

typedef struct {
	const char *path;
	uid_t       uid;
	gid_t       gid;
	int         is_dir;
} FileEntry;

static const FileEntry entries[] = {
	{ SPECIAL_LOCALE_PATH,    USER_UID, USER_GID, 0 },
	{ SPECIAL_SOURCES_PATH,   ROOT_UID, ROOT_GID, 0 },
	{ SPECIAL_LOCALTIME_PATH, ROOT_UID, USER_GID, 0 },
	{ SPECIAL_BTNAME_PATH,    ROOT_UID, ROOT_GID, 0 },
	{ SPECIAL_BLUETOOTH_PATH, ROOT_UID, ROOT_GID, 1 },
	{ SPECIAL_BOOKMARKS_PATH, ROOT_UID, ROOT_GID, 1 }
};

static int num_entries = sizeof (entries) / sizeof (entries[0]);

/* Make all path components up to, but not including the last. */
static int
special_mkdir (const char *pathname)
{
	char *tmp;
	char *p;
	char *last_slash;

	if (pathname == NULL || *pathname == '\0' || *pathname != '/') {
		return -1;
	}
	
	tmp = calloc (strlen (pathname) + 1, 1);
	strcpy(tmp, pathname);

	last_slash = strrchr (tmp, '/');
	if (*last_slash) {
		*last_slash = '\0';
	}
	
	/* Skip the root. */
	p =  tmp + 1;

	do {
		/* Skip ahead until the next path separator. */
		while (*p && *p != '/') {
			p++;
		}

		if (!*p) {
			p = NULL;
		} else {
			*p = '\0';
		}

		if (mkdir (tmp, 0755) == -1) {
			if (errno != EEXIST) {
				free (tmp);
				return -1;
			}
		}
		
		if (p) {
			*p++ = '/';
			while (*p && *p == '/') {
				p++;
			}
		}
	} while (p);
	
	free (tmp);
	
	return 0;
}

int
main (int argc, char **argv)
{
	const char *temp_path;
	const char *dest_path;
	int         i, this_entry;
	
	if (argc != 3) {
		fprintf (stderr, "Usage: %s <srcpath> <destpath>\n", argv[0]);
		return 1;
	}

	temp_path = argv[1];
	dest_path = argv[2];

	/* Extra check to make this more secure. */
	if (strstr (dest_path, "/..")) {
		fprintf (stderr, "'..' path components are not allowed\n");
		return 1;
	}

	/* Check that the file is one of the allowed ones. */
	this_entry = -1;
	for (i = 0; i < num_entries; i++) {
		if (!entries[i].is_dir) {
			if (strcmp (dest_path, entries[i].path) == 0) {
				this_entry = i;
				break;
			}
		} else {
			size_t dest_len, entry_len;

			dest_len = strlen (dest_path);
			entry_len = strlen (entries[i].path);

			if (dest_len == entry_len) {
				/* Full comparison. */
				if (strcmp (entries[i].path, dest_path) == 0) {
					this_entry = i;
					break;
				}
			}
			else if (dest_len > entry_len) {
				/* Check if dest_path is inside the dir in the
				 * allowed entry.
				 */
				if (strncmp (dest_path, entries[i].path, entry_len) == 0 &&
				    dest_path[entry_len] == '/') {
					this_entry = i;
					break;
				}
			}
		}
	}

	if (this_entry == -1) {
		fprintf (stderr, "Invalid dest path %s\n", dest_path);
		return 1;
	}

	if (entries[i].is_dir) {
		if (special_mkdir (dest_path) == -1) {
			fprintf (stderr, "Couldn't create necessary directories");
			return 1;
		}
	}
	
	/* Note: This requires the two locations to be on the same
	 * filesystem.
	 */
	if (rename (temp_path, dest_path) != 0) {
		perror ("Could not restore file");
		return 1;
	}
	
#ifdef USE_SUDO
	/* Make the file owner the right one. We only do this when built with
	 * sudo support since otherwise this program is not ran as root. This is
	 * mainly to make testing on desktop computers easier.
	 */
	if (chown (dest_path,
		   entries[this_entry].uid,
		   entries[this_entry].gid) != 0) {
		perror ("Could not set the right owner of the restored file");
		return 1;
	}
#endif
	
	return 0;
}


