/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <string.h>
#include <locale.h>
#include <glib/gi18n.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkwidget.h>
#include <libgnomevfs/gnome-vfs.h>
#include <gsf/gsf-utils.h>
#include <libosso.h>
#include <libogs/ogs-file-system.h>
#include <mce/dbus-names.h>

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus-glib-lowlevel.h>

#include "ob-main-window.h"
#include "ob-dbus-service.h"
#include "ob-utils.h"
#include "ob-config.h"
#include "ob-backup-locations.h"
#include "ob-log.h"

static osso_context_t *osso = NULL;
static DBusConnection *connection = NULL;

static gint
osso_rpc_cb (const gchar *interface,
	     const gchar *method,
	     GArray      *arguments,
	     gpointer     data,
	     osso_rpc_t  *retval)
{
	ObMainWindow *window;
	ObBackend    *backend;
	GtkWindow    *parent;

	window = data;
	parent = ob_main_window_get_window (window);

	ob_log_info ("Got dbus method: %s", method);

	if (strcmp (method, OB_DBUS_METHOD_CANCEL) == 0) {
		ob_log_info ("Got cancellation request through D-BUS.");

		backend = ob_main_window_get_backend (window);
		ob_backend_cancel (backend);
	} else /*if (strcmp (method, OB_DBUS_METHOD_ACTIVATE) == 0)*/ {
		gtk_window_present (parent);
	}

	/* No return value. */
	retval->type = DBUS_TYPE_INVALID;

	return OSSO_OK;
}

static DBusHandlerResult
osso_dbus_filter_cb (DBusConnection *connection,
		     DBusMessage    *message,
		     void           *user_data)
{
	/*ob_log_info ("D-Bus: Received message:%p", message);*/

	if (dbus_message_is_signal (message,
				    MCE_SIGNAL_IF,
				    MCE_SHUTDOWN_SIG)) {
		ob_log_info ("D-Bus: Signal to shutdown received!");
		gtk_main_quit ();
	} else {
		/*ob_log_info ("D-Bus: Signal not handled...");*/
	}

	return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

static gboolean
osso_dbus_init (void)
{
	DBusError  error;
	gchar     *rule;

	/*ob_log_info ("D-Bus: Initializing to handle system bus signalling...");*/
	
	dbus_error_init (&error);

	connection = dbus_bus_get (DBUS_BUS_SYSTEM, &error);
	if (connection == NULL) {
		ob_log_error ("D-Bus: Failed to open connection to system bus: %s",
			      error.message);
		dbus_error_free (&error);

		return FALSE;
	}

	dbus_connection_setup_with_g_main (connection, NULL);

	if (!dbus_connection_add_filter (connection, osso_dbus_filter_cb, NULL, NULL)) {
		ob_log_error ("D-Bus: Failed to add filter");

		dbus_connection_unref (connection);
		connection = NULL;

		return FALSE;
	}

	rule = g_strdup_printf ("type='signal', sender='%s', interface='%s'",
				MCE_SERVICE,
				MCE_SIGNAL_IF);

	dbus_bus_add_match (connection, rule, &error);
	g_free (rule);

	if (dbus_error_is_set (&error)) {
		ob_log_error ("D-Bus: Could not set filter on monitoring connection: %s\n",
			      error.message);

		dbus_error_free (&error);
		dbus_connection_unref (connection);
		connection = NULL;

		return FALSE;
	}
	
	/*ob_log_info ("D-Bus: Ready");*/

	return TRUE;
}

static gboolean
osso_dbus_finalize (void)
{
	/*ob_log_info ("D-Bus: Finalizing...\n");*/
	
	if (!connection) {
		return TRUE;
	}

	dbus_connection_remove_filter (connection, osso_dbus_filter_cb, NULL);

	dbus_connection_unref (connection);
	connection = NULL;
}

static void
print_version (void)
{
	g_printerr (PACKAGE " " VERSION "\n");
}

int
main (int argc, char **argv)
{
	ObMainWindow   *main_window;
	ObBackend      *backend;
	osso_return_t   ret;
	OgsFileSystem  *file_system;
	GnomeVFSURI    *uri;
	gchar          *config_dir = NULL;
	gboolean        show_startup = FALSE;
	gboolean        show_version = FALSE;
	GOptionContext *context;
	GOptionEntry    entries[] = {
		{ "config-dir", 'd', 0, G_OPTION_ARG_FILENAME, &config_dir, 
		  "Use an alternate config directory", NULL },
		{ "show-startup", 's', 0, G_OPTION_ARG_NONE, &show_startup, 
		  "Show the startup dialog (for newly flashed devices)", NULL },
		{ "version", 'v', 0, G_OPTION_ARG_NONE, &show_version, 
		  "Display the software version", NULL },
		{ NULL }
	};

	context = g_option_context_new ("- Backup/Restore");
	g_option_context_add_main_entries (context, entries, GETTEXT_PACKAGE);
	g_option_context_add_group (context, gtk_get_option_group (TRUE));
	g_option_context_parse (context, &argc, &argv, NULL);

	if (show_version) {
		print_version ();
		return 0;
	}

	ob_log_start ();

	/* Initialize libraries */
	gtk_init (&argc, &argv);
	gnome_vfs_init ();
	gsf_init ();

#ifdef L10N_WORKAROUND_ENABLED
	bindtextdomain ("hildon-libs", LOCALEDIR);
	bindtextdomain ("hildon-fm", LOCALEDIR);
#endif

	if (config_dir && config_dir[0] != '/') {
		gchar *current_dir;
		gchar *old_config_dir;
		
		old_config_dir = config_dir;

		/* Make sure we have an absolute path. */
		current_dir = g_get_current_dir ();
		config_dir = g_build_filename (current_dir,
					       old_config_dir,
					       NULL);
		g_free (current_dir);
		g_free (old_config_dir);
	}

	bindtextdomain (PACKAGE, LOCALEDIR);
        bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (PACKAGE);

	/* Load configuration. */
	ob_config_init (config_dir);
	ob_backup_locations_init (config_dir);
	g_free (config_dir);

	/* Setup libosso. */
	osso = osso_initialize ("backup", VERSION, TRUE, NULL);

	ob_utils_set_osso_context (osso);

	/* Pass TRUE if it is a start up dialog we are showing and
	 * FALSE for a HildonApp.
	 */
	main_window = ob_main_window_get (show_startup);
	if (!main_window) {
		return 1;
	}

	backend = ob_main_window_get_backend (main_window);

	ret = osso_rpc_set_default_cb_f (osso, osso_rpc_cb, main_window);
	if (ret != OSSO_OK) {
		ob_log_error ("Couldn't register osso rpc callback");
		return 1;
	}

	/* Initialise system bus */
	if (!osso_dbus_init ()) {
		ob_log_error ("Couldn't register with system bus");
		return 1;
	}

	uri = ob_config_get_documents_dir (ob_config_get ());
	file_system = ogs_file_system_new_with_locations (
		OGS_FILE_SYSTEM_DOCUMENTS,
		gnome_vfs_uri_get_path (uri),
		-1);
	ob_utils_set_file_system (file_system);

	ob_main_window_show (main_window);

	gtk_main ();

	/* Shutdown. */
	osso_dbus_finalize ();
	osso_rpc_unset_default_cb_f (osso, osso_rpc_cb, main_window);

	gsf_shutdown ();
	gnome_vfs_shutdown ();

	ob_config_shutdown ();
	ob_backup_locations_shutdown ();

	/* NOTE: libosso unrefs the connection even though you are only allowed
	 * to do it when you are disconnected. So therefore we ref it an extra
	 * time here just so it won't be finalized while connected, to avoid
	 * triggering an assertion in libosso. This will not cause any trouble.
	 */
	dbus_bus_get (DBUS_BUS_SESSION, NULL);
	dbus_bus_get (DBUS_BUS_SYSTEM, NULL);
	
	osso_deinitialize (osso);

	ob_log_stop ();
	
	return 0;
}

