/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <string.h>
#include <glib.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libxml/parser.h>
#include <libxml/tree.h>

#include "ob-backup-locations.h"
#include "ob-vfs-utils.h"
#include "ob-utils.h"

#define d(x)


struct _ObBackupLocations {
	GHashTable *hash; /* hashtable with location entries */
};

typedef struct {
	guint         ref_count;

	char         *path;
	
	ObCategory    category;    /* email, settings, etc */
	gboolean      is_dir;      /* is dir or file */
	gboolean      is_auto;     /* should be auto-restored */
	gboolean      is_excluded; /* inclusion or exclusion */

	/* Auto-files or dirs that are also included through a directory
	 * entry.
	 */
	gboolean     is_redundant;
} LocationEntry; 


static gboolean    backup_locations_read_directory   (ObBackupLocations *locations,
						      const char        *config_dir,
						      gboolean           allow_system);


static ObBackupLocations *the_locations;


static LocationEntry *
location_entry_new (const char *path,
		    ObCategory  category,
		    gboolean    is_dir,
		    gboolean    is_auto,
		    gboolean    is_excluded)
{
	LocationEntry *entry;

	entry = g_new0 (LocationEntry, 1);
	entry->ref_count = 1;

	entry->path = g_strdup (path);
	entry->category = category;
	entry->is_dir = is_dir;
	entry->is_auto = is_auto;
	entry->is_excluded = is_excluded;

	return entry;
}

static void
location_entry_unref (LocationEntry *entry)
{
	entry->ref_count--;

	if (entry->ref_count > 0) {
		return;
	}

	g_free (entry->path);
	g_free (entry);
}

/* Only exposed for testing. */
gboolean
_ob_backup_locations_init_empty (void)
{
	if (the_locations) {
		g_warning ("Locations already inited.");
		return FALSE;
	}
	
	the_locations = g_new0 (ObBackupLocations, 1);

	the_locations->hash = g_hash_table_new_full (
		g_str_hash, g_str_equal,
		g_free, (GDestroyNotify) location_entry_unref);
	
	return TRUE;
}

void
ob_backup_locations_init (const char *dir)
{
	gchar *app_dir, *tmp;
	
	if (!_ob_backup_locations_init_empty ()) {
		return;
	}

	/* First the system configuration. */
	if (!dir) {
		dir = SYSCONFDIR "/osso-backup";
	}
	app_dir = g_build_filename (dir, "applications", NULL);
	backup_locations_read_directory (the_locations, app_dir, TRUE);
	g_free (app_dir);
	
	/* The third-party apps. */
	tmp = ob_utils_get_install_home ();
	app_dir = g_build_filename (tmp,
				    "etc/osso-backup", "applications",
				    NULL);
	backup_locations_read_directory (the_locations, app_dir, FALSE);
	g_free (tmp);
	g_free (app_dir);
	
	/* Handle redundant entries among the locations listed. */
	_ob_backup_locations_remove_redundancies (the_locations);
}

ObBackupLocations *
ob_backup_locations_get (void)
{
	return the_locations;
}

void
ob_backup_locations_shutdown (void)
{
	if (!the_locations) {
		return;
	}

	g_hash_table_destroy (the_locations->hash);
	g_free (the_locations);
	the_locations = NULL;
}

gboolean
_ob_backup_locations_add_location (ObBackupLocations *locations,
				   const char        *path,
				   ObCategory         category,
				   gboolean           is_dir,
				   gboolean           is_auto,
				   gboolean           is_exclusion)
{
	LocationEntry *entry;

	g_return_val_if_fail (locations != NULL, FALSE);
	g_return_val_if_fail (path != NULL, FALSE);

	if (path[0] != '/') {
		g_warning ("Path is not absolute: %s", path);
		return FALSE;
	}
	
	entry = g_hash_table_lookup (locations->hash, path);
	if (entry) {
		/* This is a duplicate, don't add it. */
		g_warning ("Got duplicate path in configuration:\n%s", path);
		return TRUE;
	}

	entry = location_entry_new (path, category, is_dir, is_auto, is_exclusion);
	g_hash_table_insert (locations->hash, g_strdup (path), entry);

	return TRUE;
}

static void
get_all_entries_func (gpointer key,
		      gpointer value,
		      gpointer user_data)
{
	GList **list;

	list = user_data;

	*list = g_list_append (*list, value);
}

/* Only exposed for the tests, otherwise internal function. */
void
_ob_backup_locations_remove_redundancies (ObBackupLocations *locations)
{
	GList         *list, *l;
	LocationEntry *entry;
	LocationEntry *parent_entry;
	char          *parent_path;
	
	g_return_if_fail (locations != NULL);

	list = NULL;
	g_hash_table_foreach (locations->hash, get_all_entries_func, &list);

	for (l = list; l; l = l->next) {
		entry = l->data;
		
		parent_path = g_strdup (entry->path);
		while (1) {
			char *tmp;

			tmp = g_path_get_dirname (parent_path);
			g_free (parent_path);
			parent_path = tmp;
			
			if (strcmp (parent_path, ".") == 0 ||
			    strcmp (parent_path, "/") == 0) {
				g_free (parent_path);
				break;
			}
			
			/* We don't make an entry redundant if the
			 * parent and child entry are not both the
			 * same type (inclusion/exclusion). 
			 */
			parent_entry = g_hash_table_lookup (locations->hash,
							    parent_path);

			if (parent_entry &&
			    parent_entry->is_excluded == entry->is_excluded) {
				/* Keep "auto" files so they override directory
				 * entries.
				 */
				if (entry->is_auto) {
					/* Keep, but mark as redundant so we can
					 * filter it out when returning the list
					 * of locations.
					 */
					entry->is_redundant = TRUE;
				} else {
					/* Redundant, remove it. */
					g_hash_table_remove (locations->hash,
							     entry->path);
				}
				
				g_free (parent_path);
				break;
			}
		}
	}

	g_list_free (list);

	/* At this stage, we have a hash table with all unique directories and
	 * files. Entries that are already covered by ancestor directories have
	 * been filtered out. The resulting table can be used to query if a file
	 * should be saved and if it should be auto-restored.
	 */
}

gboolean
ob_backup_locations_is_included (ObBackupLocations *locations,
				 const char        *path)
{
	char          *parent_path, *tmp;
	LocationEntry *entry;
	
	g_return_val_if_fail (locations != NULL, FALSE);
	g_return_val_if_fail (path != NULL, FALSE);

	parent_path = g_strdup (path);
	while (1) {
		entry = g_hash_table_lookup (locations->hash, parent_path);
		if (entry) {
			return TRUE;
		}
	
		tmp = g_path_get_dirname (parent_path);
		g_free (parent_path);
		parent_path = tmp;
			
		if (strcmp (parent_path, ".") == 0 ||
		    strcmp (parent_path, "/") == 0) {
			g_free (parent_path);
			break;
		}
	}

	return FALSE;
}

typedef struct {
	int       categories;
	gboolean  is_excluded;
	GList    *list;
} GetUrisData;

static void
get_all_uris_no_redundant_func (gpointer key,
				gpointer value,
				gpointer user_data)
{
	GetUrisData   *data;
	LocationEntry *entry;
	gchar         *uri_string;
	gboolean       need_slash = FALSE;
	gboolean       need_asterisk = FALSE;


	entry = (LocationEntry *) value;

	if (entry->is_redundant) {
		return;
	}

	data = (GetUrisData *) user_data;

	/* Only include the entry if the category is right. The category "other"
	 * is only included when all are selected.
	 */
	if (entry->category == OB_CATEGORY_OTHER &&
	    data->categories == OB_CATEGORY_ALL) {
		/* Included. */
	}
	else if (entry->is_excluded != data->is_excluded) {
		return;
	}
	else if ((entry->category & data->categories) == 0) {
		return;
	}
	
	/* For excluded directories, check we have slash asterisk on
	 * the end for the pattern matcher to spot it. 
	 */
	if (entry->is_excluded && 
	    entry->is_dir) {
		gint len;

		len = strlen (entry->path);
		
		if ((entry->path[len - 1] != '*' ||
		     entry->path[len - 2] != G_DIR_SEPARATOR)) {
			need_asterisk = TRUE; 

			if (entry->path[len - 1] != G_DIR_SEPARATOR) {
				need_slash = TRUE;
			}
		}
	}

	/* If entry is an exclusion, we DON'T escape the URI because
	 * it may contain wildcards like * and ? which will then not
	 * work when we use glib pattern matching.
	 */
	if (entry->is_excluded) {
		uri_string = g_strconcat (entry->path, 
					  need_slash ? G_DIR_SEPARATOR_S : "",
					  need_asterisk ? "*" : "",
					  NULL);
	} else {
		/* Adding a slash or asterisk only applies to
		 * exclusions.
		 */
		uri_string = gnome_vfs_escape_host_and_path_string (entry->path);

	}
	
	d(g_print ("Adding '%s' (added slash:%s, added asterisk:%s, exclusion:%s)\n", 
		   uri_string, 
		   need_slash ? "yes" : "no", 
		   need_asterisk ? "yes" : "no",
		   entry->is_excluded ? "yes" : "no"));

	data->list = g_list_prepend (data->list, uri_string);
}

/* Returns a list of URIs, both directories and files. The list does not have
 * duplicates or redundant entries. categories is a bitfield of ObCategory
 * values.
 */
GList *
ob_backup_locations_get_uris (ObBackupLocations *locations,
			      gint               categories,
			      gboolean           is_excluded)
{
	GetUrisData data;

	data.categories = categories;	
	data.is_excluded = is_excluded;
	data.list = NULL;

	g_hash_table_foreach (locations->hash,
			      get_all_uris_no_redundant_func,
			      &data);
	
	return g_list_sort (data.list, (GCompareFunc) strcmp);
}

typedef struct {
	gchar    *path;
	gboolean  is_auto;
} IsAutoForeachData;

static void
backup_locations_is_auto_foreach (gpointer key,
				  gpointer value,
				  gpointer user_data)
{
	IsAutoForeachData *data = user_data;
	LocationEntry     *entry = value;
	char              *key_str = (gchar *) key;
	int                key_len, path_len;

	/* Compare the path to the stored auto paths, check if the path is equal
	 * to or is located below the stored path.
	 */

	if (entry->is_auto) {
		key_len = strlen (key_str);
		path_len = strlen (data->path);
		
		if (strcmp (data->path, key_str) == 0) {
			data->is_auto = TRUE;
		}
		else if (key_len < path_len &&
			 g_str_has_prefix (data->path, key_str) &&
			 data->path[key_len] == '/') {
			data->is_auto = TRUE;
		}
	}
}

gboolean
ob_backup_locations_is_auto_restore (ObBackupLocations *locations,
				     const char        *path)
{
	LocationEntry     *entry;
	IsAutoForeachData  data;
	
	g_return_val_if_fail (locations != NULL, FALSE);
	g_return_val_if_fail (path != NULL, FALSE);

	/* First lookup the direct path. */
	entry = g_hash_table_lookup (locations->hash, path);
	if (entry && entry->is_auto == TRUE && !entry->is_dir) {
		return TRUE;
	}

	/* Then find ancestors to auto paths. */
	data.path = (gchar *) path;
	data.is_auto = FALSE;
	
	g_hash_table_foreach (locations->hash, backup_locations_is_auto_foreach, &data);

	return data.is_auto;
}

/* Checks if the path is in the system, which means that it's in a location
 * other than $HOME or the third-party installation location.
 */
static gboolean
backup_locations_path_is_in_system (const gchar *path)
{
	char     *tmp;
	gboolean  ret;
	
	tmp = ob_utils_get_install_home ();

	if (!g_str_has_prefix (path, g_get_home_dir ()) &&
	    !g_str_has_prefix (path, tmp)) {
		ret = TRUE;
	} else {
		ret = FALSE;
	}

	g_free (tmp);

	return ret;
}

/*
 * The XML parser.
 */
static gboolean
backup_locations_parse_location (ObBackupLocations *locations,
				 xmlNode           *node,
				 gboolean           allow_system)
{
	char       *type_str, *category_str, *is_auto_str;
	char       *path;
	ObCategory  category;
	gboolean    is_dir;
	gboolean    is_auto;
	gboolean    is_exclusion = FALSE;
	gboolean    retval;
	char       *normalized;

	if (strcmp (node->name, "exclusion") == 0) {
		is_exclusion = TRUE;
	}
	
	if (!is_exclusion && 
	    strcmp (node->name, "location") != 0) {
		return FALSE;   
	}

	type_str = xmlGetProp (node, "type");
	category_str = xmlGetProp (node, "category");
	is_auto_str = xmlGetProp (node, "auto");

	if (is_exclusion && is_auto_str) {
		g_warning ("Invalid property auto in locations config for exclusion.");
	}

	path = xmlNodeGetContent (node);

	if (!type_str) {
		g_warning ("No type in locations config.");
		retval = FALSE;
		goto done;
	}
	else if (strcmp (type_str, "dir") == 0) {
		is_dir = TRUE;
	} else {
		is_dir = FALSE;
	}

	if (!category_str) {
		category = OB_CATEGORY_OTHER;
	}
	else if (strcmp (category_str, "emails") == 0) {
		category = OB_CATEGORY_EMAILS;
	}
	else if (strcmp (category_str, "contacts") == 0) {
		category = OB_CATEGORY_CONTACTS;
		
	}
	else if (strcmp (category_str, "bookmarks") == 0) {
		category = OB_CATEGORY_BOOKMARKS;
		
	}
	else if (strcmp (category_str, "settings") == 0) {
		category = OB_CATEGORY_SETTINGS;
		
	}
	else if (strcmp (category_str, "documents") == 0) {
		category = OB_CATEGORY_DOCUMENTS;
		
	}
	else if (strcmp (category_str, "media") == 0) {
		category = OB_CATEGORY_MEDIA;
		
	}
	else if (strcmp (category_str, "other") == 0){
		category = OB_CATEGORY_OTHER;
	} else {
		g_warning ("Invalid property category in locations config.");
		retval = FALSE;
		goto done;
	}

	if (!is_auto_str || strcmp (is_auto_str, "false") == 0) {
		is_auto = FALSE;
	}
	else if (strcmp (is_auto_str, "true") == 0) {
		is_auto = TRUE;
	} else {
		g_warning ("Invalid property auto in locations config.");
		retval = FALSE;
		goto done;
	}
	
	normalized = ob_utils_parse_path (path);
        if (!normalized) {
                retval = FALSE;
                goto done;
        }

	if (!allow_system && backup_locations_path_is_in_system (normalized)) {
		g_warning ("Third-party applications must not have system paths listed:\n %s",
			   normalized);
		g_free (normalized);
		retval = FALSE;
		goto done;
	}
	
	retval = _ob_backup_locations_add_location (locations,
						    normalized,
						    category,
						    is_dir,
						    is_auto, 
						    is_exclusion);

	g_free (normalized);

 done:
	xmlFree (type_str);
	xmlFree (category_str);
	xmlFree (is_auto_str);
	xmlFree (path);
	
	return retval;
}

static gboolean
backup_locations_parse_conf_file (ObBackupLocations *locations,
				  GnomeVFSURI       *uri,
				  gboolean           allow_system)
{
	char    *path;
	xmlDoc  *doc;
	xmlNode *root, *node;

	path = gnome_vfs_unescape_string (gnome_vfs_uri_get_path (uri), NULL);
	doc = xmlParseFile (path);
	g_free (path);
	
	if (!doc) {
		return FALSE;
	}

	root = xmlDocGetRootElement (doc);

	if (!root || strcmp (root->name, "backup-configuration") != 0) {
		xmlFreeDoc (doc);

		return FALSE;
	}

	for (node = root->children; node; node = node->next) {
		if (strcmp (node->name, "locations") == 0) {
			break;
		}
	}

	if (node) {
		for (node = node->children; node; node = node->next) {
			if (node->type == XML_ELEMENT_NODE) {
				backup_locations_parse_location (locations,
								 node,
								 allow_system);
			}
		}
	}
	
	xmlFreeDoc (doc);
	
	return TRUE;
}

static gboolean
backup_locations_read_directory (ObBackupLocations *locations,
				 const char        *config_dir,
				 gboolean           allow_system)
{
	GnomeVFSDirectoryHandle *handle;
	GnomeVFSResult           result;
	GnomeVFSFileInfo        *info;
	GnomeVFSURI             *dir_uri, *uri;

	g_return_val_if_fail (locations != NULL, FALSE);
	g_return_val_if_fail (config_dir != NULL, FALSE);

	/* Must be absolute path. */
	g_return_val_if_fail (config_dir[0] == '/', FALSE);
	
	dir_uri = gnome_vfs_uri_new (config_dir);

	if (!dir_uri) {
		g_warning ("Invalid config path: %s", config_dir);
		return FALSE;
	}

	result = gnome_vfs_directory_open_from_uri (
		&handle, dir_uri, GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE);

	if (result != GNOME_VFS_OK) {
		gnome_vfs_uri_unref (dir_uri);
		return FALSE;
	}

	info = gnome_vfs_file_info_new ();

	while (1) {
		result = gnome_vfs_directory_read_next (handle, info);
		if (result == GNOME_VFS_ERROR_EOF) {
			break;
		}
		else if (result != GNOME_VFS_OK) {
			continue;
		}
		
		if (info->type == GNOME_VFS_FILE_TYPE_REGULAR) {
			if (g_str_has_suffix (info->name, ".conf")) {
				uri = gnome_vfs_uri_append_path (dir_uri,
								 info->name);
				backup_locations_parse_conf_file (locations,
								  uri,
								  allow_system);
				gnome_vfs_uri_unref (uri);
			}
		}
		
		gnome_vfs_file_info_clear (info);	
	}

	gnome_vfs_file_info_unref (info);
	gnome_vfs_directory_close (handle);
	
	gnome_vfs_uri_unref (dir_uri);
	
	return TRUE;
}

void
ob_backup_locations_count_data (ObBackupLocations *locations,
				int                categories,
				int               *num_files,
				GnomeVFSFileSize  *size)
{
	GList            *uris;
	gint              exclude_num = 0;
	GnomeVFSFileSize  exclude_size = 0;

	g_return_if_fail (locations != NULL);
	g_return_if_fail (num_files != NULL);
	
	uris = ob_backup_locations_get_uris (locations, categories, FALSE);
	ob_vfs_utils_count_files (uris, num_files, size);
	g_list_foreach (uris, (GFunc) g_free, NULL);
	g_list_free (uris);

	uris = ob_backup_locations_get_uris (locations, categories, TRUE);
	ob_vfs_utils_count_files (uris, &exclude_num, &exclude_size);
	g_list_foreach (uris, (GFunc) g_free, NULL);
	g_list_free (uris);

	*num_files -= exclude_num;
}

