/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <string.h>
#include <glib.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libxml/parser.h>
#include <libxml/tree.h>

#include "ob-backup-info.h"
#include "ob-types.h"
#include "ob-vfs-utils.h"
#include "ob-utils.h"
#include "ob-category.h"
#include "ob-config.h"

#define d(x) 

typedef struct {
	int              num_files;
	GnomeVFSFileSize size;
} DataAmount;
	
struct _ObBackupInfo {
	guint             ref_count;

	/* Points to the directory where this backup resides. */
	GnomeVFSURI      *uri;

	time_t            timestamp;

	gboolean          is_protected;
	char             *password;

	char             *device_version;

	int               categories;

	int               num_files[OB_NUM_CATEGORIES];
	GnomeVFSFileSize  file_size[OB_NUM_CATEGORIES];
};


static char *
backup_info_get_category_xml (ObBackupInfo *info,
			      ObCategory    category)
{
	int         index;
	const char *category_str;
	char       *xml;

	g_return_val_if_fail (category != OB_CATEGORY_ALL, NULL);

	if (!(info->categories & category)) {
		return g_strdup ("");
	}
	
	index = ob_category_to_index (category);

	if (info->num_files[index] == 0) {
		return g_strdup ("");
	}
	
	category_str = ob_category_to_non_translated_string (category);

	xml = g_strdup_printf ("    <%s>\n"
			       "      <size>%ld</size>\n"
			       "      <number-of-files>%d</number-of-files>\n"
			       "    </%s>\n",
			       category_str,
			       (long) info->file_size[index],
			       info->num_files[index],
			       category_str);
	
	return xml;
}

static char *
backup_info_get_categories_xml (ObBackupInfo *info)
{
	int   categories;
	char *emails;
	char *contacts;
	char *docs;
	char *media;
	char *bookmarks;
	char *settings;
	char *other;
	char *ret_val;

	categories = info->categories;

	emails = backup_info_get_category_xml (info, OB_CATEGORY_EMAILS);
	contacts = backup_info_get_category_xml (info, OB_CATEGORY_CONTACTS);
	docs = backup_info_get_category_xml (info, OB_CATEGORY_DOCUMENTS);
	media = backup_info_get_category_xml (info, OB_CATEGORY_MEDIA);
	bookmarks = backup_info_get_category_xml (info, OB_CATEGORY_BOOKMARKS);
	settings = backup_info_get_category_xml (info, OB_CATEGORY_SETTINGS);
	other = backup_info_get_category_xml (info, OB_CATEGORY_OTHER);
	
	ret_val = g_strconcat (emails, contacts, docs, media,
			       bookmarks, settings, other, NULL);

	g_free (emails);
	g_free (contacts);
	g_free (docs);
	g_free (media);
	g_free (bookmarks);
	g_free (settings);
	g_free (other);
	
	return ret_val;
}

static int
backup_info_get_categories_from_node (xmlNode *cat_node)
{
	xmlNode *node;
	int      ret_val = 0;

	for (node = cat_node->children; node; node = node->next) {
		if (strcmp (node->name, "emails") == 0) {
			ret_val |= OB_CATEGORY_EMAILS;
		}
		else if (strcmp (node->name, "contacts") == 0) {
			ret_val |= OB_CATEGORY_CONTACTS;
		}
		else if (strcmp (node->name, "documents") == 0) {
			ret_val |= OB_CATEGORY_DOCUMENTS;
		}
		else if (strcmp (node->name, "media") == 0) {
			ret_val |= OB_CATEGORY_MEDIA;
		}
		else if (strcmp (node->name, "bookmarks") == 0) {
			ret_val |= OB_CATEGORY_BOOKMARKS;
		}
		else if (strcmp (node->name, "settings") == 0) {
			ret_val |= OB_CATEGORY_SETTINGS;
		}
		else if (strcmp (node->name, "other") == 0) {
			ret_val |= OB_CATEGORY_OTHER;
		}
	}

	return ret_val;
}

static void
backup_info_get_category_data_from_node (ObBackupInfo *info,
					 ObCategory    category,
					 xmlNode      *cat_node)
{
	xmlNode          *node;
	int               num_files;
	GnomeVFSFileSize  size;
	char             *str, *endptr;

	size = 0;
	num_files = 0;
	
	for (node = cat_node->children; node; node = node->next) {
		if (strcmp (node->name, "size") == 0) {
			str = xmlNodeGetContent (node);
		
			size = strtol (str, &endptr, 10);
			if (*endptr != '\0') {
				g_warning ("Invalid size in metadata.");
				xmlFree (str);
				break;
			}
			xmlFree (str);
		}
		else if (strcmp (node->name, "number-of-files") == 0) {
			str = xmlNodeGetContent (node);
			
			num_files = strtol (str, &endptr, 10);
			if (*endptr != '\0') {
				g_warning ("Invalid number of files in metadata.");
				xmlFree (str);
				break;
			}
			xmlFree (str);
		}
	}

	ob_backup_info_set_data_amount (info,
					category, size,
					num_files);
}

static void
backup_info_get_categories_data_from_node (ObBackupInfo *info,
					   xmlNode      *cat_node)
{
	xmlNode *node;

	for (node = cat_node->children; node; node = node->next) {
		if (strcmp (node->name, "emails") == 0) {
			backup_info_get_category_data_from_node (
				info, OB_CATEGORY_EMAILS, node);
		}
		else if (strcmp (node->name, "contacts") == 0) {
			backup_info_get_category_data_from_node (
				info, OB_CATEGORY_CONTACTS, node);
		}
		else if (strcmp (node->name, "documents") == 0) {
			backup_info_get_category_data_from_node (
				info, OB_CATEGORY_DOCUMENTS, node);
		}
		else if (strcmp (node->name, "media") == 0) {
			backup_info_get_category_data_from_node (
				info, OB_CATEGORY_MEDIA, node);
		}
		else if (strcmp (node->name, "bookmarks") == 0) {
			backup_info_get_category_data_from_node (
				info, OB_CATEGORY_BOOKMARKS, node);
		}
		else if (strcmp (node->name, "settings") == 0) {
			backup_info_get_category_data_from_node (
				info, OB_CATEGORY_SETTINGS, node);
		}
		else if (strcmp (node->name, "other") == 0) {
			backup_info_get_category_data_from_node (
				info, OB_CATEGORY_OTHER, node);
		}
	}
}

ObBackupInfo *
ob_backup_info_new (GnomeVFSURI      *uri,
		    time_t            timestamp,
		    gboolean          is_protected,
		    const char       *password,
		    const char       *device_version,
		    int               categories)
{
	ObBackupInfo *info;

	g_return_val_if_fail (uri != NULL, NULL);
	
	info = g_new0 (ObBackupInfo, 1);
	info->ref_count = 1;

	info->uri = gnome_vfs_uri_ref (uri);

	info->timestamp = timestamp;
	info->is_protected = is_protected;
	info->password = g_strdup (password);
	info->device_version = g_strdup (device_version);
	info->categories = categories;

	return info;
}

ObBackupInfo *
ob_backup_info_dup (ObBackupInfo *info)
{
	ObBackupInfo *new_info;
	int           i;
	
	g_return_val_if_fail (info != NULL, NULL);
	
	new_info = ob_backup_info_new (info->uri,
				       info->timestamp,
				       info->is_protected,
				       info->password,
				       info->device_version,
				       info->categories);

	for (i = 0; i < OB_NUM_CATEGORIES; i++) {
		new_info->num_files[i] = info->num_files[i];
		new_info->file_size[i] = info->file_size[i];
	}

	return new_info;
}

ObBackupInfo *
ob_backup_info_ref (ObBackupInfo *info)
{
	info->ref_count++;

	return info;
}

void
ob_backup_info_unref (ObBackupInfo *info)
{
	info->ref_count--;

	if (info->ref_count > 0) {
		return;
	}

	gnome_vfs_uri_unref (info->uri);

	g_free (info->password);
	g_free (info->device_version);
	
	g_free (info);
}

void
ob_backup_info_set_timestamp (ObBackupInfo *info,
			      time_t        timestamp)
{
	g_return_if_fail (info != NULL);
	
	info->timestamp = timestamp;
}

void
ob_backup_info_set_categories (ObBackupInfo *info,
			       int           categories)
{
	g_return_if_fail (info != NULL);
	
	info->categories = categories;
}

void
ob_backup_info_set_is_protected (ObBackupInfo *info,
				 gboolean      is_protected)
{
	g_return_if_fail (info != NULL);
	
	info->is_protected = is_protected;
}

void
ob_backup_info_set_password (ObBackupInfo *info,
			     const char   *password)
{
	g_return_if_fail (info != NULL);

	g_free (info->password);
	info->password = g_strdup (password);
}

/* Set the amount of files and size for a specific category. */ 
void
ob_backup_info_set_data_amount (ObBackupInfo     *info,
				ObCategory        category,
				GnomeVFSFileSize  size,
				int               num_files)
{
	int index;
	
	g_return_if_fail (info != NULL);
	g_return_if_fail (category != OB_CATEGORY_ALL);
	
	index = ob_category_to_index (category);

	info->num_files[index] = num_files;
	info->file_size[index] = size;
}

/* Get the amount of files and size for a specific category. */ 
void
ob_backup_info_get_data_amount (ObBackupInfo     *info,
				ObCategory        category,
				GnomeVFSFileSize *size,
				int              *num_files)
{
	int index;
	
	g_return_if_fail (info != NULL);
	g_return_if_fail (category != OB_CATEGORY_ALL);
	
	index = ob_category_to_index (category);

	if (size) {
		*size = info->file_size[index];
	}
	if (num_files) {
		*num_files = info->num_files[index];
	}
}

char *
ob_backup_info_get_name (ObBackupInfo *info)
{
	g_return_val_if_fail (info != NULL, NULL);

	return gnome_vfs_uri_extract_short_name (info->uri);
}

GnomeVFSURI *
ob_backup_info_get_uri (ObBackupInfo *info)
{
	g_return_val_if_fail (info != NULL, NULL);

	return info->uri;
}

GnomeVFSFileSize
ob_backup_info_get_size (ObBackupInfo *info)
{
	return ob_backup_info_get_size_for_categories (info, OB_CATEGORY_ALL);
}

GnomeVFSFileSize
ob_backup_info_get_size_for_categories (ObBackupInfo *info,
					int           categories)
{
	GnomeVFSFileSize size;
	int              i;
	ObCategory       category;

	g_return_val_if_fail (info != NULL, 0);

	size = 0;

	for (i = 0; i < OB_NUM_CATEGORIES; i++) {
		category = ob_category_from_index (i);
		if (categories & category) {
			size += info->file_size[i];
		}
	}
 	
	return size;
}

int
ob_backup_info_get_num_files (ObBackupInfo *info)
{
	return ob_backup_info_get_num_files_for_categories (info, OB_CATEGORY_ALL);
}

int
ob_backup_info_get_num_files_for_categories (ObBackupInfo *info,
					    int            categories)
{
	int        num_files;
	int        i;
	ObCategory category;

	g_return_val_if_fail (info != NULL, 0);

	num_files = 0;

	for (i = 0; i < OB_NUM_CATEGORIES; i++) {
		category = ob_category_from_index (i);
		if (categories & category) {
			num_files += info->num_files[i];
		}
	}
 	
	return num_files;
}

time_t
ob_backup_info_get_timestamp (ObBackupInfo *info)
{
	g_return_val_if_fail (info != NULL, 0);

	return info->timestamp;
}

/* Not avaiable on a parsed backup info, obviously. */
const char *
ob_backup_info_get_password (ObBackupInfo *info)
{
	g_return_val_if_fail (info != NULL, NULL);

	return info->password;
}

const char *
ob_backup_info_get_device_version (ObBackupInfo *info)
{
	g_return_val_if_fail (info != NULL, NULL);

	return info->device_version;
}

int
ob_backup_info_get_categories (ObBackupInfo *info)
{
	g_return_val_if_fail (info != NULL, 0);

	return info->categories;
}

gboolean
ob_backup_info_get_is_protected (ObBackupInfo *info)
{
	g_return_val_if_fail (info != NULL, FALSE);

	return info->is_protected;
}

ObMemoryCardType
ob_backup_info_get_card_type (ObBackupInfo *info)
{
	GnomeVFSURI *internal;
	GnomeVFSURI *external;

	g_return_val_if_fail (info != NULL, OB_MEMORY_CARD_INTERNAL);
	
	internal = ob_config_get_memory_card_mountpoint (
		ob_config_get (), OB_MEMORY_CARD_INTERNAL);

	if (internal) {
		if (gnome_vfs_uri_is_parent (internal, info->uri, TRUE)) {
			return OB_MEMORY_CARD_INTERNAL;
		}
	}

	external = ob_config_get_memory_card_mountpoint (
		ob_config_get (), OB_MEMORY_CARD_EXTERNAL);

	if (external) {
		if (gnome_vfs_uri_is_parent (external, info->uri, TRUE)) {
			return OB_MEMORY_CARD_EXTERNAL;
		}
	}

	return OB_MEMORY_CARD_INTERNAL;
}

static int
backup_info_compare_func (gconstpointer a,
			  gconstpointer b,
			  gpointer      data)
{
	ObBackupInfo *info_a, *info_b;

	info_a = (ObBackupInfo *) a;
	info_b = (ObBackupInfo *) b;

	if (info_a->timestamp > info_b->timestamp) {
		return -1;
	}
	else if (info_a->timestamp < info_b->timestamp) {
		return 1;
	}

	return 0;
}


/*
 * XML parser.
 */
static ObBackupInfo *
parse_metadata_file (GnomeVFSURI *uri)
{
	GnomeVFSURI      *file_uri;
	char             *path;
	xmlDoc           *doc;
	xmlNode          *root, *node;
	xmlNode          *categories_node;
	ObBackupInfo     *info;
	GnomeVFSFileSize  size;
	int               files;
	time_t            timestamp;
	gboolean          is_protected;
	gboolean          ret;
	char             *str;
	char             *endptr;
	int               categories;
	char             *version;

	file_uri = gnome_vfs_uri_append_path (uri, "backup.metadata");

	path = gnome_vfs_unescape_string (gnome_vfs_uri_get_path (file_uri), NULL);
	doc = xmlParseFile (path);
	g_free (path);

	gnome_vfs_uri_unref (file_uri);

	if (!doc) {
		return NULL;
	}

	root = xmlDocGetRootElement (doc);

	if (!root || strcmp (root->name, "backup-metadata") != 0) {
		/* Not a metadata file. */
		xmlFreeDoc (doc);

		return NULL;
	}
	
	size = 0;
	files = 0;
	timestamp = 0;
	is_protected = FALSE;
	version = NULL;
	categories = 0;

	categories_node = NULL;
	
	ret = TRUE;
	for (node = root->children; node; node = node->next) {
		if (strcmp (node->name, "size") == 0) {
			str = xmlNodeGetContent (node);

			size = strtol (str, &endptr, 10);
			if (*endptr != '\0') {
				g_warning ("Invalid size in metadata.");
				xmlFree (str);
				ret = FALSE;
				break;
			}
			xmlFree (str);
		}
		else if (strcmp (node->name, "number-of-files") == 0) {
			str = xmlNodeGetContent (node);
			
			files = strtol (str, &endptr, 10);
			if (*endptr != '\0') {
				g_warning ("Invalid number of files in metadata.");
				xmlFree (str);
				ret = FALSE;
				break;
			}
			xmlFree (str);
		}
		else if (strcmp (node->name, "timestamp") == 0) {
			str = xmlNodeGetContent (node);
			
			timestamp = strtol (str, &endptr, 10);
			if (*endptr != '\0') {
				g_warning ("Invalid timestamp in metadata.");
				xmlFree (str);
				ret = FALSE;
				break;
			}
			xmlFree (str);
		}
		else if (strcmp (node->name, "protected") == 0) {
			str = xmlNodeGetContent (node);

			if (strcmp (str, "true") == 0) {
				is_protected = TRUE;
			}
			else if (strcmp (str, "false") == 0) {
				is_protected = FALSE;
			} else {
				g_warning ("Invalid is-protected in metadata.");
				xmlFree (str);
				ret = FALSE;
				break;
			}
			xmlFree (str);
		}
		else if (strcmp (node->name, "device-version") == 0) {
			version = xmlNodeGetContent (node);
		}
		else if (strcmp (node->name, "categories") == 0) {
			categories_node = node;
			categories = backup_info_get_categories_from_node (node);
		}
	}

	if (ret) {
		info = ob_backup_info_new (uri,
					   timestamp,
					   is_protected,
					   NULL,
					   version,
					   categories);
		
		/* Get the category size/num_files values. */
		if (categories_node) {
			backup_info_get_categories_data_from_node (
				info, categories_node);
		}
	} else {
		info = NULL;
	}

	if (version) {
		xmlFree (version);
	}
	
	xmlFreeDoc (doc);
 
	return info;
}

static ObBackupInfo *
get_backup_info (GnomeVFSURI *uri)
{
	GnomeVFSDirectoryHandle *handle;
	GnomeVFSResult           result;
	GnomeVFSFileInfo        *file_info;
	ObBackupInfo            *backup_info;

	result = gnome_vfs_directory_open_from_uri (
		&handle, uri, GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE);
	
	if (result != GNOME_VFS_OK) {
		return NULL;
	}

	file_info = gnome_vfs_file_info_new ();

	/* Check for the backup metadata file. */
	backup_info = NULL;
	while (1) {
		result = gnome_vfs_directory_read_next (handle, file_info);
		if (result == GNOME_VFS_ERROR_EOF) {
			break;
		}
		else if (result != GNOME_VFS_OK) {
			/* Couldn't read. */
			continue;
		}

		if (file_info->type == GNOME_VFS_FILE_TYPE_REGULAR) {
			if (strcmp (file_info->name, "backup.metadata") == 0) {
				backup_info = parse_metadata_file (uri);
				break;
			}
		}
		
		gnome_vfs_file_info_clear (file_info);	
	}

	gnome_vfs_file_info_unref (file_info);
	gnome_vfs_directory_close (handle);
	
	return backup_info;
}

/* Get the available backups at the location, which is equal to subdirectories
 * that contain a backup metadata file.
 */
GList *
ob_backup_info_list_get (GnomeVFSURI *uri)
{
	GList                   *list;
	GnomeVFSDirectoryHandle *handle;
	GnomeVFSResult           result;
	GnomeVFSFileInfo        *file_info;
	ObBackupInfo            *backup_info;
	GnomeVFSURI             *child_uri;

	g_return_val_if_fail (uri != NULL, NULL);

	result = gnome_vfs_directory_open_from_uri (
		&handle, uri,
		GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE);
	
	if (result != GNOME_VFS_OK) {
		/* Couldn't open the location directory. */
		return NULL;
	}

	file_info = gnome_vfs_file_info_new ();

	list = NULL;
	while (1) {
		result = gnome_vfs_directory_read_next (handle, file_info);
		if (result == GNOME_VFS_ERROR_EOF) {
			break;
		}
		else if (result != GNOME_VFS_OK) {
			/* Couldn't read. */
			continue;
		}

		if (file_info->type == GNOME_VFS_FILE_TYPE_DIRECTORY &&
		    strcmp (file_info->name, ".") != 0 &&
		    strcmp (file_info->name, "..")) {
			child_uri = gnome_vfs_uri_append_path (
				uri, file_info->name);
			
			backup_info = get_backup_info (child_uri);
			if (backup_info) {
				list = g_list_prepend (list, backup_info);
			}
			
			gnome_vfs_uri_unref (child_uri);
		}
		
		gnome_vfs_file_info_clear (file_info);	
	}

	gnome_vfs_file_info_unref (file_info);
	gnome_vfs_directory_close (handle);

	return g_list_sort_with_data (list, backup_info_compare_func, NULL);
}

void
ob_backup_info_list_free (GList *list)
{
	GList *l;
	
	for (l = list; l; l = l->next) {
		ob_backup_info_unref (l->data);
	}

	g_list_free (list);
}

/* Creates a directory for the backup to be placed in. The parent directory must
 * exist. It's not an error if the directory already exists.
 */
GnomeVFSResult
ob_backup_info_create_dir (ObBackupInfo *info)
{
	GnomeVFSResult   result;
	GnomeVFSFileType type;

	g_return_val_if_fail (info != NULL, GNOME_VFS_ERROR_BAD_PARAMETERS);
	
	result = gnome_vfs_make_directory_for_uri (info->uri, 0700);

	if (result == GNOME_VFS_ERROR_FILE_EXISTS) {
		/* If it's a dir, we're OK. */
		if (ob_vfs_utils_uri_get_file_type (info->uri, &type) == GNOME_VFS_OK &&
		    type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
			return GNOME_VFS_OK;
		}
	}
	
	return result;
}

/* Removes the directory corresponding to the backup info, including
 * contents. Also removes the parent backup directory if empty.
 */
GnomeVFSResult
ob_backup_info_remove_dir (ObBackupInfo *info)
{
	GnomeVFSResult  result;
	GnomeVFSURI    *parent;
	
	g_return_val_if_fail (info != NULL, GNOME_VFS_ERROR_BAD_PARAMETERS);

	result = ob_vfs_utils_remove_directory (info->uri, NULL, NULL);

	parent = gnome_vfs_uri_get_parent (info->uri);
	if (parent) {
		/* We don't care about the result here. */
		gnome_vfs_remove_directory_from_uri (parent);
		gnome_vfs_uri_unref (parent);
	}

	return result;
}

/* Writes the metadata file for the backup. The directory must exist. */
GnomeVFSResult
ob_backup_info_write_metadata (ObBackupInfo *info)
{
	char           *xml;
	const char     *protected;
	GnomeVFSURI    *uri;
	GnomeVFSResult  result;
	GnomeVFSHandle *handle;
	const char     *version;
	char           *categories_string;

	g_return_val_if_fail (info != NULL, GNOME_VFS_ERROR_BAD_PARAMETERS);

	protected = ob_backup_info_get_is_protected (info) ? "true" : "false";
	
	version = ob_backup_info_get_device_version (info);
	if (!version) {
		version = "";
	}

	categories_string = backup_info_get_categories_xml (info);

	xml = g_strdup_printf ("<backup-metadata>\n"
			       "  <size>%ld</size>\n"
			       "  <number-of-files>%d</number-of-files>\n"
			       "  <timestamp>%ld</timestamp>\n"
			       "  <protected>%s</protected>\n"
			       "  <device-version>%s</device-version>\n"
			       "  <categories>\n"
			       "%s"
			       "  </categories>\n"
			       "</backup-metadata>\n",
			       (long) ob_backup_info_get_size (info),
			       ob_backup_info_get_num_files (info),
			       (long) ob_backup_info_get_timestamp (info),
			       protected,
			       version,
			       categories_string);
	g_free (categories_string);

	uri = gnome_vfs_uri_append_path (info->uri, "backup.metadata");
	
	result = gnome_vfs_create_uri (&handle,
				       uri,
				       GNOME_VFS_OPEN_WRITE,
				       FALSE, /* exclusive */
				       0600);
	gnome_vfs_uri_unref (uri);

	if (result != GNOME_VFS_OK) {
		g_free (xml);
		return result;
	}
	
	result = ob_vfs_utils_write (handle, xml, strlen (xml));
	g_free (xml);
	if (result != GNOME_VFS_OK) {
		return result;
	}
	
	return gnome_vfs_close (handle);
}

/* Gets the path to the archive zip file. */
char *
ob_backup_info_get_category_archive_path (ObBackupInfo *info,
					  ObCategory    category)
{
	GnomeVFSURI *uri;
	char        *path, *full_path;
	char        *zip_string;

	uri = ob_backup_info_get_uri (info);

	zip_string = g_strdup_printf ("%s.zip",
				      ob_category_to_non_translated_string (category));
	
	path = gnome_vfs_unescape_string (gnome_vfs_uri_get_path (uri), NULL);
	full_path = g_build_filename (path, zip_string, NULL);
	g_free (path);
	g_free (zip_string);

	return full_path;
}

void
ob_backup_info_reset_categories (ObBackupInfo *info,
				 int           categories)
{
	int        i;
	ObCategory category;

	g_return_if_fail (info != NULL);

	for (i = 0; i < OB_NUM_CATEGORIES; i++) {
		category = ob_category_from_index (i);
		if (categories & category) {
			info->num_files[i] = 0;
			info->file_size[i] = 0;
		}
	}	
}

void
ob_backup_info_print (ObBackupInfo *info)
{
	GnomeVFSFileSize size;
	int              num_files, i;
	ObCategory       category;

	for (i = 0; i < OB_NUM_CATEGORIES; i++) {
		category = ob_category_from_index (i);

		if (info->categories & category) {
			ob_backup_info_get_data_amount (info,
							category,
							&size,
							&num_files);

			g_print ("%9s: %2d files\n", ob_category_to_non_translated_string (category), num_files);
		}
	}
}

/* Checks if the password is the correct one for the categories that are stored
 * in the specified backup.
 */
gboolean
ob_backup_info_check_password (ObBackupInfo *info, const char *password)
{
	gboolean    retval;
	int         i;
	ObCategory  category;
	char       *path;
	
	if (!info->is_protected) {
		/* This is not protected so it's always OK. */ 
		return TRUE;
	}

	retval = TRUE;
	for (i = 0; i < OB_NUM_CATEGORIES; i++) {
		category = ob_category_from_index (i);
		if (info->categories & category) {
			/*g_print ("Checking password for '%s'\n",
			  ob_category_to_non_translated_string (category));*/

			path = ob_backup_info_get_category_archive_path (info, category);
			
			if (!ob_utils_check_password (path, password)) {
				/*g_print ("Failed\n");*/
				g_free (path);
				retval = FALSE;
				break;
			}

			g_free (path);
		}
	}
		
	return retval;
}

GType
ob_backup_info_get_type (void)
{
	static GType type = 0;

	if (type == 0) {
		type = g_boxed_type_register_static ("ObBackupInfoType",
						     (GBoxedCopyFunc) ob_backup_info_ref,
						     (GBoxedFreeFunc) ob_backup_info_unref);
	}

	return type;
}
