/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#ifndef OB_BACKEND_H_
#define OB_BACKEND_H_

#include <time.h>
#include <glib.h>
#include <glib-object.h>

#include "ob-types.h"
#include "ob-event.h"
#include "ob-memory-card.h"
#include "ob-backup-info.h"
#include "ob-backup-locations.h"
#include "ob-config.h"
#include "ob-category.h"

G_BEGIN_DECLS

#define OB_TYPE_BACKEND            (ob_backend_get_type ())
#define OB_BACKEND(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), \
					OB_TYPE_BACKEND, ObBackend))
#define OB_BACKEND_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass), \
					OB_TYPE_BACKEND, ObBackendClass))
#define OB_IS_BACKEND(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), \
					OB_TYPE_BACKEND))
#define OB_IS_BACKEND_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), \
					OB_TYPE_BACKEND))
#define OB_BACKEND_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj), \
					OB_TYPE_BACKEND, ObBackendClass))

typedef struct _ObBackend      ObBackend;
typedef struct _ObBackendClass ObBackendClass;

#include "ob-archiver.h"

struct _ObBackend {
	GObject        parent;
};

struct _ObBackendClass {
	GObjectClass   parent_class;
};

GType            ob_backend_get_type               (void) G_GNUC_CONST;
ObBackend *      ob_backend_new                    (void);
gboolean         ob_backend_remove_backup          (ObBackend           *backend,
						    ObBackupInfo        *info);
gboolean         ob_backend_cancel                 (ObBackend           *backend);
ObState          ob_backend_get_state              (ObBackend           *backend);
gboolean         ob_backend_name_is_available      (ObBackend           *backend,
						    ObMemoryCardType     type,
						    const char          *name);
gchar *          ob_backend_get_available_name     (ObBackend           *backend,
						    ObMemoryCardType     type);
void             ob_backend_count_data             (ObBackend           *backend,
						    int                  categories,
						    int                 *num_files,
						    GnomeVFSFileSize    *size);
gboolean         ob_backend_start_backup           (ObBackend           *backend,
						    ObMemoryCardType     type,
						    ObCategoryFiles     *category_files,
						    const char          *name,
						    const char          *password,
						    time_t               timestamp,
						    int                  categories,
						    int                  num_files,
						    GnomeVFSFileSize     total_size,
						    ObBackupInfo        *replaced_backup_info,
						    GError             **error);
gboolean         ob_backend_start_restore          (ObBackend           *backend,
						    ObBackupInfo        *backup_info,
						    const char          *password,
						    gint                 categories,
						    GError             **error);

/*
 * Private API between backend and archiver.
 */
void             ob_backend_push_event             (ObBackend           *backend,
						    ObEvent             *event);
void             ob_backend_confirm_cancel         (ObBackend           *backend);

/*
 * Conflict resolving API.
 */
ObConflictResponse 
                 ob_backend_wait_for_conflict_response (ObBackend           *backend);
void             ob_backend_respond_to_conflict    (ObBackend           *backend,
						    ObConflictResponse   response);

/*
 * Memory card API.
 */
ObMemoryCard *   ob_backend_get_memory_card        (ObBackend           *backend,
						    ObMemoryCardType     type);
gboolean         ob_backend_space_on_memory_card   (ObBackend           *backend,
						    ObMemoryCardType     type,
						    GnomeVFSFileSize     size,
						    GnomeVFSFileSize    *available_size);
guint            ob_backend_count_memory_cards     (ObBackend           *backend);
guint            ob_backend_count_backups          (ObBackend           *backend,
	                                            ObMemoryCardType     type);
int              ob_backend_get_elapsed_time       (ObBackend           *backend);
GnomeVFSFileSize ob_backend_get_processed_size     (ObBackend           *backend);
int              ob_backend_get_processed_files    (ObBackend           *backend);
GError *         ob_backend_get_error              (ObBackend           *backend);

void             ob_backend_add_restored_flag      (void);
void             ob_backend_remove_startup_flag    (void);

G_END_DECLS

#endif /* OB_BACKEND_H_ */
