/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <glib.h>
#include <gtk/gtkmain.h>
#include <glib/gi18n.h>
#include <glib/gstdio.h>

#include "ob-marshal.h"
#include "ob-types.h"
#include "ob-backend.h"
#include "ob-memory-card.h"
#include "ob-archiver-zip.h"
#include "ob-config.h"
#include "ob-vfs-utils.h"
#include "ob-log.h"
#include "ob-error.h"
#include "ob-utils.h"

#define d(x) 

/* Some comments about how the interaction between the UI, backend and archiver
 * works.
 *
 * The backend is the glue between the UI and the archiver thread. Events such
 * as progress updates and "finished" and "error", are pushed from the archiver
 * thread to a queue in the backend. The backend then emits signals for those
 * events from the main thread by taking events from the queue in an idle
 * handler.
 *
 * When the user cancels, the backend is told to cancel. It sets the cancelled
 * flag on the archiver, and then runs a recursive main loop. The archiver,
 * which regularly checks for cancellation, stops what it's doing, and unwinds
 * until and finally confirms the cancellation, by calling
 * ob_backend_confirm_cancel(). When this happens, the backend quits the
 * recursive main loop and control is handed back to the UI.
 *
 */

static void backend_finalize                     (GObject                  *object);
static void backend_volume_mounted_cb            (GnomeVFSVolumeMonitor    *monitor,
						  GnomeVFSVolume           *volume,
						  ObBackend                *backend);
static void backend_volume_unmounted_cb          (GnomeVFSVolumeMonitor    *monitor,
						  GnomeVFSVolume           *volume,
						  ObBackend                *backend);
static void backend_volume_pre_unmount_cb        (GnomeVFSVolumeMonitor    *monitor,
						  GnomeVFSVolume           *volume,
						  ObBackend                *backend);
static void backend_backup_dir_monitor_cb        (GnomeVFSMonitorHandle    *handle,
						  const gchar              *monitor_uri,
						  const gchar              *info_uri,
						  GnomeVFSMonitorEventType  event_type,
						  ObBackend                *backend);
static void backend_setup_backup_dir_monitor     (ObBackend                *backend,
						  ObMemoryCardType          type,
						  gboolean                  enable);
static void backend_setup_volume_monitoring      (ObBackend                *backend);

G_DEFINE_TYPE (ObBackend, ob_backend, G_TYPE_OBJECT)

#define GET_PRIV(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), OB_TYPE_BACKEND, ObBackendPriv))


/* Signals for the UI to listen to for updates. */
enum {
	FINALIZING,
	PROGRESS,
	CONFLICT,
	ERROR,
	FINISHED,
	CANCELLED,
	CONFLICT_ABORTED, /* When we get a cancel request during a conflict. */
	MEMORY_CARD_INSERTED,
	MEMORY_CARD_REMOVED,
	BACKUPS_CHANGED,
	LAST_SIGNAL
};

typedef struct {
	ObState            state;
	ObArchiver        *active_archiver;
	ObMemoryCard      *internal_mem_card;
	ObMemoryCard      *external_mem_card;
	ObConfig          *config;
	ObBackupLocations *locations;

	gboolean	   blame_unmount;   /* Set after we got ::pre-unmount */
	gboolean	   in_cancellation; /* Waiting for a cancellation to take effect */
	gboolean           conflict_aborted;

	ObMemoryCardType   active_mem_card;

	GError            *error;
	
	guint              reboot_timeout_id;
	
	/* Stats. */
	GTimer            *timer;
	GnomeVFSFileSize   processed_size;
	int                processed_files;

	/* Monitioring */
	GnomeVFSMonitorHandle *internal_monitor;
	GnomeVFSMonitorHandle *external_monitor;
} ObBackendPriv;

static guint        signals[LAST_SIGNAL];


/* Singleton instance. */
static ObBackend   *backend;


/* For communicating from the archiver thread to the main thread. */
static GAsyncQueue *event_queue;


/* For conflict response handling, from the main thread to the archiver. */
static GCond       *conflict_cond;
static GMutex      *conflict_mutex;
static int          conflict_response;


static void
ob_backend_class_init (ObBackendClass *klass)
{
	GObjectClass *object_class;

	object_class = (GObjectClass*) klass;
	object_class->finalize = backend_finalize;

	signals[FINALIZING] =
		g_signal_new ("finalizing",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      g_cclosure_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	signals[PROGRESS] =
		g_signal_new ("progress",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      ob_marshal_VOID__INT_INT_INT_INT,
			      G_TYPE_NONE, 4,
			      G_TYPE_INT, G_TYPE_INT, G_TYPE_INT, G_TYPE_INT);

	signals[CONFLICT] =
		g_signal_new ("conflict",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      ob_marshal_INT__ENUM_POINTER_LONG_LONG,
			      G_TYPE_INT, 4,
			      OB_TYPE_CONFLICT_TYPE,
			      G_TYPE_POINTER,
			      G_TYPE_LONG, G_TYPE_LONG);

	signals[ERROR] =
		g_signal_new ("error",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      ob_marshal_VOID__POINTER,
			      G_TYPE_NONE, 1,
			      G_TYPE_INT, G_TYPE_POINTER);

	signals[FINISHED] =
		g_signal_new ("finished",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      ob_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	signals[CANCELLED] =
		g_signal_new ("cancelled",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      ob_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	signals[CONFLICT_ABORTED] =
		g_signal_new ("conflict_aborted",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      ob_marshal_VOID__VOID,
			      G_TYPE_NONE, 0);

	signals[MEMORY_CARD_INSERTED] =
		g_signal_new ("memory_card_inserted",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      ob_marshal_VOID__POINTER_INT,
			      G_TYPE_NONE, 
			      2, G_TYPE_POINTER, G_TYPE_INT);

	signals[MEMORY_CARD_REMOVED] =
		g_signal_new ("memory_card_removed",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      ob_marshal_VOID__POINTER_INT,
			      G_TYPE_NONE, 
			      2, G_TYPE_POINTER, G_TYPE_INT);

	signals[BACKUPS_CHANGED] =
		g_signal_new ("backups_changed",
			      G_TYPE_FROM_CLASS (klass),
			      G_SIGNAL_RUN_LAST,
			      0,
			      NULL, NULL,
			      ob_marshal_VOID__POINTER_INT,
			      G_TYPE_NONE, 
			      2, G_TYPE_POINTER, G_TYPE_INT);

	g_type_class_add_private (object_class, sizeof (ObBackendPriv));
}

static void
ob_backend_init (ObBackend *backend)
{
	ObBackendPriv *priv;

	priv = GET_PRIV (backend);

	event_queue = g_async_queue_new ();

	priv->blame_unmount = FALSE;
	priv->in_cancellation = FALSE;
	priv->conflict_aborted = FALSE;

	priv->state = OB_STATE_READY;
	priv->timer = g_timer_new ();
}

static void
backend_finalize (GObject *object)
{
	ObBackendPriv *priv;
	
	priv = GET_PRIV (object);

	if (priv->state != OB_STATE_READY) {
		ob_log_error ("Finalizing backend when not ready.");
	}

	if (priv->error) {
		g_error_free (priv->error);
	}

	if (priv->reboot_timeout_id) {
		g_source_remove (priv->reboot_timeout_id);
	}

	g_timer_destroy (priv->timer);
	
	g_async_queue_unref (event_queue);
	event_queue = NULL;

	G_OBJECT_CLASS (ob_backend_parent_class)->finalize (object);
}

/* Creates a singleton instance of the backup backend. No more than one can be
 * created per application.
 */
ObBackend *
ob_backend_new (void)
{
	ObBackendPriv *priv;
	
	backend = g_object_new (OB_TYPE_BACKEND, NULL);

	priv = GET_PRIV (backend);
		
	conflict_cond = g_cond_new ();
	conflict_mutex = g_mutex_new ();
	conflict_response = OB_CONFLICT_RESPONSE_NONE;

	priv->config = ob_config_get ();
	priv->locations = ob_backup_locations_get ();

	priv->active_mem_card = OB_MEMORY_CARD_INTERNAL;

	backend_setup_volume_monitoring (backend);

	return backend;
}

static void
backend_set_active_memory_card (ObBackend        *backend, 
				ObMemoryCardType  type)
{
	ObBackendPriv *priv = GET_PRIV (backend);

	priv->active_mem_card = type;
}

static ObMemoryCard *
backend_get_active_memory_card (ObBackend *backend)
{
	ObBackendPriv *priv = GET_PRIV (backend);
	
	if (priv->active_mem_card == OB_MEMORY_CARD_INTERNAL) {
		return priv->internal_mem_card;
	} else {
		return priv->external_mem_card;
	}
}

static void
ob_backend_emit_cancelled (ObBackend *backend)
{
	
	ObBackendPriv *priv = GET_PRIV (backend);
	ObMemoryCard  *mem_card;
	GError        *error = NULL;

	mem_card = backend_get_active_memory_card (backend);

	if ((priv->blame_unmount || !mem_card) &&
	    priv->state == OB_STATE_RESTORE_CONFLICT) {
		/* A conflict must be specially treated since the user interface
		 * has another dialog waiting to be cancelled, and not just the
		 * progress dialog.
		 */
		g_set_error (&error, OB_ERROR,
			     OB_ERROR_RESTORE_MEMORY_CARD_REMOVED,
			     "memory card removed during conflict");
	}
	else if ((priv->blame_unmount || !mem_card) &&
		 (priv->state == OB_STATE_BACKUP ||
		  priv->state == OB_STATE_RESTORE)) {
		/* We apparently got cancelled during restore or backup with the
		 * memory card (about to be) removed. We're supposed to report
		 * that, so we cast the cancellation notification into a
		 * memory-card-not-present error.
		 */
		if (priv->state == OB_STATE_BACKUP) {
			g_set_error (&error, OB_ERROR,
				     OB_ERROR_BACKUP_MEMORY_CARD_REMOVED,
				     "memory card removed");
		} else { /* OB_STATE_RESTORE */
			g_set_error (&error, OB_ERROR,
				     OB_ERROR_RESTORE_MEMORY_CARD_REMOVED,
				     "memory card removed");
		}
	}
	
	/* Reset state */
	priv->state = OB_STATE_READY;
	priv->in_cancellation = FALSE;
	
	if (error) {
		g_signal_emit (backend, signals[ERROR], 0, error);
		g_error_free (error);
	} else {
		g_signal_emit (backend, signals[CANCELLED], 0);
	}
}

static gboolean
reboot_timeout_func (ObBackend *backend)
{
	ObBackendPriv *priv;

	priv = GET_PRIV (backend);

	ob_log_warning ("Got reboot timeout... reboot.");

	/* Reboot and kill the backup application. */
	ob_utils_send_reboot_message ();
	kill (getpid (), SIGTERM);
	
	priv->reboot_timeout_id = 0; 
	
	return FALSE;
}

static ObMemoryCardType
backend_get_backup_memory_card_type (ObBackend    *backend, 
				     ObBackupInfo *info)
{
	ObBackendPriv *priv;
	GList         *backups;

	priv = GET_PRIV (backend);

	if (priv->internal_mem_card) {
		backups = ob_memory_card_get_backups (priv->internal_mem_card);
		if (g_list_find (backups, info)) {
			return OB_MEMORY_CARD_INTERNAL;
		}
	}

	return OB_MEMORY_CARD_EXTERNAL;
}

static gboolean
backup_cleanup_directory (const gchar *root_location)
{
	GDir        *dir;
	const gchar *filename;
	gchar       *location;

	gboolean     ok = TRUE;

	dir = g_dir_open (root_location, 0, NULL);
	if (!dir) {
		d(g_print ("Backend: Could not open directory:'%s'\n", root_location));
		return FALSE;
	}

	d(g_print ("Backend: Cleaning up all files in directory:'%s'\n", root_location));

	while (ok && (filename = g_dir_read_name (dir)) != NULL) {
		location = g_build_filename (root_location, filename, NULL);

		if (g_file_test (location, G_FILE_TEST_IS_DIR)) {
			/* Open directory */
			ok = backup_cleanup_directory (location);
		} else {
			d(g_print ("Backend: Cleaning up file:'%s'...", location));

			if (g_unlink (location) != 0) {
				ok = FALSE;
			}

			d(g_print ("%s\n", ok ? "OK" : "Failed!"));
 		}

		g_free (location);
	}

	g_dir_close (dir);

	/* Remove the directory */
 	if (g_file_test (root_location, G_FILE_TEST_IS_DIR)) {
		d(g_print ("Backend: Cleaning up directory:'%s'...", root_location));
		
		if (g_rmdir (root_location) != 0) {
			ok = FALSE;
		}
		
		d(g_print ("%s\n", ok ? "OK" : "Failed!"));
	}

	return ok;
}

gboolean
ob_backend_remove_backup (ObBackend    *backend, 
			  ObBackupInfo *info)
{
	ObBackendPriv    *priv;
	ObMemoryCardType  type;
	GnomeVFSURI      *mountpoint_uri;
	GnomeVFSURI      *backups_uri;
	GnomeVFSURI      *uri;
	gchar            *uri_str;
	gchar            *name;
	gboolean          ok;

	priv = GET_PRIV (backend);

	name = ob_backup_info_get_name (info);
	d(g_print ("Backend: Removing backup:'%s'...\n", name));

	type = backend_get_backup_memory_card_type (backend, info);

	mountpoint_uri = ob_config_get_memory_card_mountpoint (priv->config, type);
	backups_uri = gnome_vfs_uri_append_path (mountpoint_uri, "backups");
	uri = gnome_vfs_uri_append_path (backups_uri, name);

	g_free (name);

	uri_str = gnome_vfs_unescape_string (gnome_vfs_uri_get_path (uri), NULL);
	gnome_vfs_uri_unref (uri);
	ok = backup_cleanup_directory (uri_str);
	g_free (uri_str);

	d(g_print ("Backend: Removal %s\n", ok ? "successful" : "failed"));

	if (!ok) {
		gnome_vfs_uri_unref (backups_uri);
		return ok; 
	}

	/* See if we can clean up top level "backups" directory. */
	gnome_vfs_remove_directory_from_uri (backups_uri);

	gnome_vfs_uri_unref (backups_uri);

	return ok;
}

gboolean
ob_backend_cancel (ObBackend *backend)
{
	ObBackendPriv *priv;

	g_return_val_if_fail (OB_IS_BACKEND (backend), FALSE);
	
	priv = GET_PRIV (backend);

	if (priv->in_cancellation) {
		d(g_print ("Backend: Already waiting for cancellation.\n"));
		return FALSE;
	}

	switch (priv->state) {
	case OB_STATE_READY:
		d(g_print ("Backend: Nothing to cancel.\n"));
		return FALSE;

	case OB_STATE_BACKUP:
		d(g_print ("Backend: Cancel backup operation.\n"));
		priv->in_cancellation = TRUE;
		break;

	case OB_STATE_RESTORE:
		d(g_print ("Backend: Cancel restore operation.\n"));
		priv->in_cancellation = TRUE;
		break;

	case OB_STATE_RESTORE_CONFLICT:
		/* If there is a conflict waiting for the user to resolve, and
		 * we get a cancellation request, like when the card is
		 * unmounted or a cancellation is received through D-BUS, we
		 * emit a signal so the UI can cancel the conflict dialog.
		 *
		 * The UI will then cancel the operation and we go through
		 * cancellation the regular way.
		 */
		if (!priv->conflict_aborted) {
			d(g_print ("Backend: Received cancel during conflict\n"));
			g_signal_emit (backend, signals[CONFLICT_ABORTED], 0);
			return FALSE;
		}

		d(g_print ("Backend: Received cancel during conflict (2)\n"));
		
		priv->in_cancellation = TRUE;
		break;

	default:
		g_assert_not_reached ();
		break;
	}

	/* We request a cancellation from the archiver, and then run a recursive
	 * mainloop to block until the archiver is done. The archiver responds
	 * by calling ob_backend_confirm_cancel() which quits the recursive
	 * mainloop from an idle handler.
	 *
	 * If there was a conflict dialog shown during the cancellation, we need
	 * to respond to it instead, so that the archiver doesn't block waiting
	 * for that.
	 */
	if (priv->active_archiver) {
		if (!priv->conflict_aborted) {
			ob_archiver_cancel (priv->active_archiver);
		} else {
			ob_backend_respond_to_conflict (
				backend, OB_CONFLICT_RESPONSE_CANCEL);
		}
	}

	/* Setup a timeout so that we can reboot if the archiver doesn't
	 * respond. Only for restoring.
	 */
	if (priv->state == OB_STATE_RESTORE) {
		priv->reboot_timeout_id = g_timeout_add (
			30*1000,
			(GSourceFunc) reboot_timeout_func,
			backend);
	}
	
	/* Run the recursive mainloop. */
	gtk_main ();

	if (priv->reboot_timeout_id) {
		g_source_remove (priv->reboot_timeout_id);
		priv->reboot_timeout_id = 0;
	}

	/* Emit notification and reset state to OB_STATE_READY. */
	ob_backend_emit_cancelled (backend);
	
	/* The cancel confirmation is done as the last thing before leaving the
	 * archiver so it's safe to unref it here.
	 */
	if (priv->active_archiver) {
		priv->processed_size =
			ob_archiver_get_processed_size (priv->active_archiver);
		priv->processed_files =
			ob_archiver_get_processed_files (priv->active_archiver);
		
		g_object_unref (priv->active_archiver);
		priv->active_archiver = NULL;
	}

	if (priv->internal_mem_card) {
		ob_memory_card_clear_cache (priv->internal_mem_card);
	}

	if (priv->external_mem_card) {
		ob_memory_card_clear_cache (priv->external_mem_card);
	}
	
	return TRUE;
}

ObState
ob_backend_get_state (ObBackend *backend)
{
	ObBackendPriv *priv;

	g_return_val_if_fail (OB_IS_BACKEND (backend), OB_STATE_READY);

	priv = GET_PRIV (backend);

	return priv->state;
}

/*
 * Backup API
 */

static GnomeVFSURI *
backend_get_backup_uri (ObBackend        *backend,
			ObMemoryCardType  type,
			const char       *name)
{
	ObBackendPriv *priv;
	GnomeVFSURI   *backups_uri, *backup_uri;

	priv = GET_PRIV (backend);

	/* The backup files are in the path <mountpoint>/backups/<name>/. */
	backups_uri = gnome_vfs_uri_append_path (
		ob_config_get_memory_card_mountpoint (priv->config, type),
		"backups");

	backup_uri = gnome_vfs_uri_append_path (backups_uri, name);
	
	gnome_vfs_uri_unref (backups_uri);
	
	return backup_uri;
}

/* Creates the directory where backups are kept, if it does not already
 * exist.
 */
static GnomeVFSResult
backend_ensure_backups_dir (ObBackend *backend)
{
	ObBackendPriv    *priv;
	GnomeVFSURI      *backups_uri;
	GnomeVFSResult    result;
	GnomeVFSFileType  type;
	
	priv = GET_PRIV (backend);

	/* The backups are kept in the path <mountpoint>/backups/... */
	backups_uri = gnome_vfs_uri_append_path (
		ob_config_get_memory_card_mountpoint (priv->config, 
						      priv->active_mem_card),
		"backups");

	result = ob_vfs_utils_uri_get_file_type (backups_uri, &type);
	if (result == GNOME_VFS_OK && type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
		/* Already exists, OK. */
		gnome_vfs_uri_unref (backups_uri);
				
		return GNOME_VFS_OK;
	}
	
	/* Try to create the directory. */
	result = gnome_vfs_make_directory_for_uri (backups_uri, 0700);
	gnome_vfs_uri_unref (backups_uri);

	return result;
}

/* Checks if the backup name is available on the inserted memory card. */
gboolean
ob_backend_name_is_available (ObBackend        *backend,
			      ObMemoryCardType  type,
			      const char       *name)
{
	ObBackendPriv  *priv;
	ObMemoryCard   *mem_card;
	GnomeVFSURI    *backup_uri;
	GnomeVFSResult  result;

	g_return_val_if_fail (OB_IS_BACKEND (backend), FALSE);
	g_return_val_if_fail (name != NULL, FALSE);
	
	priv = GET_PRIV (backend);

	/* If specific card, just check that */
	if (type != -1) {
		mem_card = ob_backend_get_memory_card (backend, type);
		if (!mem_card) {
			ob_log_warning ("No memory card inserted.");
			return FALSE;
		}

		backup_uri = backend_get_backup_uri (backend, type, name);
	
		result = ob_vfs_utils_uri_get_file_type (backup_uri, NULL);
		
		gnome_vfs_uri_unref (backup_uri);

		/* If we could get the file type, the name is taken. */
		return (result != GNOME_VFS_OK);
	} 

	if (ob_backend_get_memory_card (backend, OB_MEMORY_CARD_INTERNAL) == NULL &&
	    ob_backend_get_memory_card (backend, OB_MEMORY_CARD_EXTERNAL) == NULL) {
		ob_log_warning ("No memory card inserted.");
		return FALSE;
	}

	/* Try internal card */
	mem_card = ob_backend_get_memory_card (backend, OB_MEMORY_CARD_INTERNAL);
	if (mem_card) {
		backup_uri = backend_get_backup_uri (backend, OB_MEMORY_CARD_INTERNAL, name);
		result = ob_vfs_utils_uri_get_file_type (backup_uri, NULL);
		gnome_vfs_uri_unref (backup_uri);
		
		if (result == GNOME_VFS_OK) {
			return FALSE;
		}
	}

	/* Try external card */
	mem_card = ob_backend_get_memory_card (backend, OB_MEMORY_CARD_EXTERNAL);
	if (mem_card) {
		backup_uri = backend_get_backup_uri (backend, OB_MEMORY_CARD_EXTERNAL, name);
		result = ob_vfs_utils_uri_get_file_type (backup_uri, NULL);
		gnome_vfs_uri_unref (backup_uri);
		
		if (result == GNOME_VFS_OK) {
			return FALSE;
		}
	}
	
	return TRUE;
}

/* Return a string on the form "Backup", "Backup01", "Backup02", whichever is
 * first available on the inserted memory card.
 */
gchar *
ob_backend_get_available_name (ObBackend        *backend,
			       ObMemoryCardType  type)
{
	ObBackendPriv *priv;
	gchar         *name;
	gint           i;

	g_return_val_if_fail (OB_IS_BACKEND (backend), NULL);
	
	priv = GET_PRIV (backend);
	
	if (type == -1) {
		if (ob_backend_get_memory_card (backend, OB_MEMORY_CARD_INTERNAL) == NULL &&
		    ob_backend_get_memory_card (backend, OB_MEMORY_CARD_EXTERNAL) == NULL) {
			ob_log_warning ("No memory card inserted.");
			return NULL;
		}
	} else {
		ObMemoryCard *mem_card;

		mem_card = ob_backend_get_memory_card (backend, type);
		if (!mem_card) {
			ob_log_warning ("No memory card inserted.");
			return NULL;
		}
	}

	/* Try with Backup, Backup1, ... */

	i = 0;
	while (i < 100) {
		if (i == 0) {
			name = g_strdup (_("back_ia_dia002_name"));
		} else {
			name = g_strdup_printf ("%s%2.2d", _("back_ia_dia002_name"), i);
		}
		
		i++;

		if (ob_backend_name_is_available (backend, type, name)) {
			return name;
		}

		g_free (name);
	}

	return NULL;
}

static void
set_error_and_log (GError     **error,
		   int          code,
		   const char  *message)
{
	ob_log_warning (message);
	g_set_error (error, OB_ERROR, code, message);
}

void
ob_backend_count_data (ObBackend        *backend,
		       int               categories,
		       int              *num_files,
		       GnomeVFSFileSize *size)
{
	ObBackendPriv    *priv;
	GList            *uris;
	GnomeVFSURI      *gconf_dir;
	int               gconf_files;
	GnomeVFSFileSize  gconf_size;
	
	priv = GET_PRIV (backend);

	ob_backup_locations_count_data (priv->locations,
					categories,
					num_files,
					size);

	gconf_size = 0;
	gconf_files = 0;
	
	/* Include the gconf dir if there is one and if settings is chosen. */
	if (categories & OB_CATEGORY_SETTINGS) {
		gconf_dir = ob_config_get_gconf_dir (priv->config);
		if (gconf_dir) {
			const gchar *gconf_dir_str;

			gconf_dir_str = gnome_vfs_uri_get_path (gconf_dir);
			uris = g_list_prepend (NULL, g_strdup (gconf_dir_str));
			
			ob_vfs_utils_count_files (uris, &gconf_files, &gconf_size);

			g_list_foreach (uris, (GFunc) g_free, NULL);
			g_list_free (uris);
		}
	}

	*num_files += gconf_files;
	*size += gconf_size;
}


/* This is a little hack that helps catching an unmounted card for the case that
 * the volume monitor doesn't tell us in time that the volume went away. It
 * works by checking if the mountpoint and the root is the same filesystem.
 */
static gboolean
backend_check_memory_card_mounted (ObBackend        *backend, 
				   ObMemoryCardType  type)
{
	GnomeVFSURI *uri;
	const char  *mountpoint;
	struct stat  statbuf;
	dev_t        dev_root, dev_mount;
		
	uri = ob_config_get_memory_card_mountpoint (ob_config_get (), type);
	mountpoint = gnome_vfs_uri_get_path (uri);
	
	dev_root = 0;
	dev_mount = 0;
	
	if (stat ("/", &statbuf) == 0) {
		dev_root = statbuf.st_dev;
	}
	
	if (stat (mountpoint, &statbuf) == 0) {
		dev_mount = statbuf.st_dev;
	}
	
	if (dev_mount == dev_root) {
		ob_log_warning ("Extra check for mounted: no card.");
		return FALSE;
	}

	return TRUE;
}

/* Starts a backup operation. The specified name is used as the directory name
 * for the backup and must already be checked for availibility (and removed if
 * it should be replaced). If password protection should be not be used, pass
 * NULL for password.
 */
gboolean
ob_backend_start_backup (ObBackend         *backend,
			 ObMemoryCardType   type,
			 ObCategoryFiles   *category_files,
			 const char        *name,
			 const char        *password,
			 time_t             timestamp,
			 int                categories,
			 int                num_files,
			 GnomeVFSFileSize   total_size,
			 ObBackupInfo      *replaced_backup_info,
			 GError           **error)
{
	ObBackendPriv  *priv;
	ObArchiver     *archiver;
	GnomeVFSURI    *backup_dir;
	const char     *real_password;
	ObBackupInfo   *info;
	GnomeVFSResult  result;
	int             old_categories;
	
	g_return_val_if_fail (OB_IS_BACKEND (backend), FALSE);
	g_return_val_if_fail (name != NULL, FALSE);

	priv = GET_PRIV (backend);

	if (priv->state != OB_STATE_READY) {
		set_error_and_log (error, OB_ERROR_BACKUP_GENERIC,
				   "Backup, backend is busy");
		return FALSE;
	}

	if (!ob_backend_get_memory_card (backend, type)) {
		set_error_and_log (error, OB_ERROR_BACKUP_NO_MEMORY_CARD,
				   "No memory card found");
		return FALSE;
	}
	
	if (!backend_check_memory_card_mounted (backend, type)) {
		set_error_and_log (error, OB_ERROR_BACKUP_NO_MEMORY_CARD,
				   "No memory card found");
		return FALSE;
	}
	
	backend_set_active_memory_card (backend, type);

	priv->blame_unmount = FALSE;
	priv->conflict_aborted = FALSE;

	if (priv->error) {
		g_error_free (priv->error);
		priv->error = NULL;
	}
	
	/* Create the necessary directory structure on the backup target. */
	result = backend_ensure_backups_dir (backend);

	/* setup backup info (preserve errors in result) */
	if (password && strlen (password) > 0) {
		real_password = password;
	} else {
		real_password = NULL;
	}
	if (replaced_backup_info) {
		/* Create a copy of the backup info and change the info that
		 * differs from this new one.
		 */
		info = ob_backup_info_dup (replaced_backup_info);
		ob_backup_info_set_timestamp (info, timestamp);
		ob_backup_info_set_categories (info, categories);
		ob_backup_info_set_is_protected (info, real_password != NULL);
		ob_backup_info_set_password (info, real_password);

		old_categories = ob_backup_info_get_categories (
			replaced_backup_info);
	} else {		
                backup_dir = backend_get_backup_uri (backend, type, name);

		info = ob_backup_info_new (backup_dir,
					   timestamp,
					   real_password != NULL,
					   real_password,
					   ob_utils_get_device_version (),
					   categories);
		old_categories = 0;
                gnome_vfs_uri_unref (backup_dir);
	}

	/* Wipe out existing data upon full replace (preserve errors in result). */
	if (result == GNOME_VFS_OK &&
	    ob_vfs_utils_uri_get_file_type (ob_backup_info_get_uri (info), NULL) == GNOME_VFS_OK) {
		/* Wipe out an existing backup dir. */
		result = ob_vfs_utils_remove_directory (ob_backup_info_get_uri (info), NULL, NULL);
		if (result != GNOME_VFS_OK) {
			ob_log_error ("Wiping out backup directory: %s: %s\n",
				      gnome_vfs_uri_get_path (ob_backup_info_get_uri (info)),
				      gnome_vfs_result_to_string (result));
		}
	}

	/* Handle initialization errors. */
	if (result == GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM ||
	    result == GNOME_VFS_ERROR_NOT_PERMITTED ||
	    result == GNOME_VFS_ERROR_ACCESS_DENIED) {
		/* Read-only filesystem. */
		set_error_and_log (error, OB_ERROR_BACKUP_READONLY,
				   "Memory card read-only");
		ob_backup_info_unref (info);
		return FALSE;
	}
	else if (result != GNOME_VFS_OK) {
		set_error_and_log (error, OB_ERROR_BACKUP_GENERIC,
				   "Failed to access backup dir");
		ob_backup_info_unref (info);
		return FALSE;
	}

	/* Switch to backup state from here on (affects error reporting etc). */
	priv->state = OB_STATE_BACKUP;
	priv->in_cancellation = FALSE;
	
	/* Create an archiver to perform the backup operation. */
	archiver = ob_archiver_zip_new_backup (info,
					       category_files,
					       old_categories,
					       num_files,
					       total_size,
					       real_password);
	priv->active_archiver = archiver;
	ob_archiver_set_backend (archiver, backend);

	ob_backup_info_unref (info);

	g_timer_start (priv->timer);

	if (!ob_archiver_pack_start_thread (archiver)) {
		priv->state = OB_STATE_READY;
		priv->active_archiver = NULL;
		g_object_unref (archiver);

		/* This only happens if the archiver has not implemented the
		 * start function. Any real errors from the archiver are
		 * reported through an error event.
		 */
		set_error_and_log (error, OB_ERROR_BACKUP_GENERIC,
				   "Couldn't start backup");
		return FALSE;
	}

	return TRUE;
}


/*
 * Restore API
 */

/* Starts a restore operation. backup_info is the information about the backup
 * to restore.
 */
gboolean
ob_backend_start_restore (ObBackend     *backend,
			  ObBackupInfo  *backup_info,
			  const char    *password,
                          gint           categories,
			  GError       **error)
{
	ObBackendPriv    *priv;
	ObMemoryCardType  type;
	ObArchiver       *archiver;
	const char       *real_password;

	g_return_val_if_fail (OB_IS_BACKEND (backend), FALSE);
	g_return_val_if_fail (backup_info != NULL, FALSE);

	priv = GET_PRIV (backend);
	
	type = backend_get_backup_memory_card_type (backend, backup_info);
	backend_set_active_memory_card (backend, type);

	if (priv->state != OB_STATE_READY) {
		set_error_and_log (error, OB_ERROR_RESTORE_GENERIC,
				   "restore, backend is busy");
		return FALSE;
	}

	if (!ob_backend_get_memory_card (backend, type)) {
		set_error_and_log (error, OB_ERROR_BACKUP_NO_MEMORY_CARD,
				   "No memory card found");
		return FALSE;
	}

	if (!backend_check_memory_card_mounted (backend, type)) {
		set_error_and_log (error, OB_ERROR_RESTORE_NO_MEMORY_CARD,
				   "No memory card found");
		return FALSE;
	}
	
	priv->blame_unmount = FALSE;
	priv->conflict_aborted = FALSE;

	if (priv->error) {
		g_error_free (priv->error);
		priv->error = NULL;
	}
	
	if (password && strlen (password) > 0) {
		real_password = password;
	} else {
		real_password = NULL;
	}
	
        /* switch to restore state from here on (affects error reporting etc.) */
	priv->state = OB_STATE_RESTORE;
	priv->in_cancellation = FALSE;
	
	/* create archiver to perform restore operation */
	archiver = ob_archiver_zip_new_restore (backup_info,
						categories, 
						real_password);
	priv->active_archiver = archiver;
	ob_archiver_set_backend (archiver, backend);

	g_timer_start (priv->timer);

	if (!ob_archiver_unpack_start_thread (archiver)) {
		priv->state = OB_STATE_READY;
		priv->active_archiver = NULL;
		g_object_unref (archiver);

		/* This only happens if the archiver has not implemented the
		 * start function. Any real errors from the archiver are
		 * reported through an error event.
		 */
		set_error_and_log (error, OB_ERROR_RESTORE_GENERIC,
				   "Couldn't start restore");
		
		return FALSE;
	}

	return TRUE;
}


/*
 * Event handling.
 */

static gboolean
backend_handle_event_idle_func (ObBackend *backend)
{
	ObBackendPriv    *priv;
	ObMemoryCard     *mem_card;
	ObEvent          *event;
	int               files, total, complete;
	ObCategory        category;
	ObConflictType    type;
	ObProgressType    progress_type;
	GnomeVFSFileSize  processed_size;
	int               processed_files;
	time_t            existing_time, backup_time;
	GnomeVFSURI      *uri;
	GError           *error = NULL;

	priv = GET_PRIV (backend);

	mem_card = backend_get_active_memory_card (backend);
	
	event = g_async_queue_pop (event_queue);

	/* If we are in the READY state, we can't handle an event, so just
	 * discard it. This will happend when cancelling through the UI for
	 * example, and the cancellation takes a little bit of time.
	 */
	if (priv->state == OB_STATE_READY) {
		/*ob_log_warning ("Event arrived when state is READY.");*/
		ob_event_free (event);
		return FALSE;
	}
	
	switch (ob_event_get_type (event)) {
	case OB_EVENT_PROGRESS:
		/* Only ever emit the progress signal if an operation is in
		 * progress and there is no error or conflict.
		 */
		if (priv->state != OB_STATE_BACKUP &&
		    priv->state != OB_STATE_RESTORE) {
			break;
		}
		
		if (!ob_event_get_progress (event,
					    &progress_type,
                                            &files,
                                            &total,
                                            &complete,
					    &category)) {
			g_assert_not_reached ();
		}

		if (progress_type == OB_PROGRESS_TYPE_FINALIZING) {
			g_signal_emit (backend, signals[FINALIZING], 0);
		} else {
			complete = CLAMP (complete, 0, 100);
			
			g_signal_emit (backend, signals[PROGRESS], 0,
				       files, total, complete, category);
		}
		break;
		
	case OB_EVENT_ERROR:
		/* At this stage, the archiver should be done and cleaned up any
		 * resources if necessary.
		 */
		if (!ob_event_get_error (event, &error)) {
			g_assert_not_reached ();
		}

		if (mem_card) {
			ob_memory_card_clear_cache (mem_card);
		}

		if ((priv->blame_unmount || !mem_card) &&
		    (priv->state == OB_STATE_BACKUP ||
		     priv->state == OB_STATE_RESTORE) &&
		    error->domain == OB_ERROR &&
		    (error->code == OB_ERROR_BACKUP_GENERIC ||
		     error->code == OB_ERROR_RESTORE_GENERIC ||
		     error->code == OB_ERROR_GENERIC)) {
			/* We apparently ran into an error during restore or
			 * backup with the memory card (about to be)
			 * removed. We're supposed to report that, so we cast
			 * the error into a memory-card-not-present error.
			 */
			if (priv->state == OB_STATE_BACKUP) {
				error->code = OB_ERROR_BACKUP_MEMORY_CARD_REMOVED;
			} else { /* OB_STATE_RESTORE */
				error->code = OB_ERROR_RESTORE_MEMORY_CARD_REMOVED;
			}
		}

		priv->state = OB_STATE_READY;
		priv->in_cancellation = FALSE;

		priv->error = error;

		g_signal_emit (backend, signals[ERROR], 0, priv->error);
		
		if (priv->active_archiver) {
			/* If we were cancelled, we also need to make sure we
			 * don't block in the cancellation procedure.
			 */
			if (ob_archiver_is_cancelled (priv->active_archiver)) {
				ob_backend_confirm_cancel (backend);
			}
			
			g_object_unref (priv->active_archiver);
			priv->active_archiver = NULL;
		}

		break;
		
	case OB_EVENT_CONFLICT:
		/* A conflict event should only be possible to get if a restore
		 * operation is in progress. It can however arrive when waiting
		 * for a cancellation, in which case we reply with a cancel
		 * response.
		 */
		if (priv->state == OB_STATE_RESTORE && !priv->in_cancellation) {
			int response;
			
			if (!ob_event_get_conflict (event,
						    &type,
						    &uri,
						    &existing_time,
						    &backup_time)) {
				g_assert_not_reached ();
			}

			priv->state = OB_STATE_RESTORE_CONFLICT;
			
			/* This will block until the conflict dialog is closed,
			 * either by the user or by cancellation.
			 */
			g_signal_emit (backend, signals[CONFLICT], 0,
				       type, uri,
				       (long) existing_time,
				       (long) backup_time,
				       &response);

			gnome_vfs_uri_unref (uri);

			if (response == OB_CONFLICT_RESPONSE_CANCEL &&
			    priv->blame_unmount) {
				/* When a conflict is cancelled due to removed
				 * card, we set the flag and cancel in the
				 * regular way, to let the backend finish and to
				 * get the card unmounted error emitted from the
				 * backend.
				 */
				priv->conflict_aborted = TRUE;

				ob_backend_cancel (backend);
			} else {
				ob_backend_respond_to_conflict (backend, response);
			}
			
			if (response == OB_CONFLICT_RESPONSE_CANCEL) {
				/* Cancel. */
			} else {
				/* Go back to restoring. */
				priv->state = OB_STATE_RESTORE;
			}
		} else {
			/* If there is a pending cancellation, just reply
			 * directly with "cancel".
			 */
			ob_backend_respond_to_conflict (
				backend, OB_CONFLICT_RESPONSE_CANCEL);
		}

		break;
		
	case OB_EVENT_FINISHED:
		if (!ob_event_get_finished (event,
					    &processed_size,
					    &processed_files)) {
			g_assert_not_reached ();
		}
		
		if (mem_card) {
			ob_memory_card_clear_cache (mem_card);
		}

		g_timer_stop (priv->timer);

		priv->processed_size = processed_size;
		priv->processed_files = processed_files;

		g_signal_emit (backend, signals[FINISHED], 0);

		/* If we were cancelled, we also need to make sure we don't
		 * block in the cancellation procedure. Since we got a finished
		 * event from the archiver, we know that it's done.
		 */
		if (ob_archiver_is_cancelled (priv->active_archiver)) {
			ob_backend_confirm_cancel (backend);
		} else {
			if (priv->state == OB_STATE_RESTORE) {
				ob_backend_add_restored_flag ();
				ob_backend_remove_startup_flag ();
			}
		}
		
		priv->state = OB_STATE_READY;

		if (priv->active_archiver) {
			g_object_unref (priv->active_archiver);
			priv->active_archiver = NULL;
		}

		break;
		
	case OB_EVENT_CANCELLED:
		if (mem_card) {
			ob_memory_card_clear_cache (mem_card);
		}
		
		/* If the conflict was cancelled by the user normally, emit the
		 * normal cancellation notification and reset state to
		 * OB_STATE_READY.
		 *
		 * If the cancellation was due to a removed card, we go through
		 * the cancellation confirmation to make sure that the archiver
		 * is done, and to get the card removed error emitted at the
		 * end.
		 */
		if (!priv->conflict_aborted) {
			ob_backend_emit_cancelled (backend);
		} else {
			ob_backend_confirm_cancel (backend);
		}
		
		if (priv->active_archiver) {
			g_object_unref (priv->active_archiver);
			priv->active_archiver = NULL;
		}

		break;
	}

	ob_event_free (event);
	
	return FALSE;
}

/*
 * API for communication between the backend and archiver.
 */

/* Called from the archiver thread to push an event to the main thread. */
void
ob_backend_push_event (ObBackend *backend,
		       ObEvent   *event)
{
	g_async_queue_push (event_queue, event);
	g_idle_add ((GSourceFunc) backend_handle_event_idle_func,
		    backend);
}

static gboolean
confirm_cancel_idle_func (ObBackend *backend)
{
	ObBackendPriv *priv;

	priv = GET_PRIV (backend);

	gtk_main_quit ();
	return FALSE;
}

/* Called from the archiver thread to confirm that a cancellation request has
 * been noticed and obeyed.
 */
void
ob_backend_confirm_cancel (ObBackend *backend)
{
	/* Do the work in an idle to make it come from the main thread, and so
	 * that the recursive mainloop gets a change to run before we call
	 * gtk_main_quit(). Do this with a high priority to make cancellation
	 * happen more quickly.
	 */
	g_idle_add_full (G_PRIORITY_HIGH_IDLE,
			 (GSourceFunc) confirm_cancel_idle_func,
			 backend,
			 NULL);
}

/* Called from the main thread to respond to a conflict. */
void
ob_backend_respond_to_conflict (ObBackend          *backend,
				ObConflictResponse  response)
{
	ObBackendPriv *priv;

	g_return_if_fail (OB_IS_BACKEND (backend));

	priv = GET_PRIV (backend);

	if (priv->state != OB_STATE_RESTORE_CONFLICT) {
		d(g_print ("Backend: Can't respond when there's no conflict.\n"));
		return;
	}
	
	g_mutex_lock (conflict_mutex);
	conflict_response = response;
	g_cond_signal (conflict_cond);
	g_mutex_unlock (conflict_mutex);
}

/* Called from the archiver thread to wait for a conflict response. */
ObConflictResponse
ob_backend_wait_for_conflict_response (ObBackend *backend)
{				       
	int response;
		
	g_mutex_lock (conflict_mutex);

	while (conflict_response == OB_CONFLICT_RESPONSE_NONE) {
		g_cond_wait (conflict_cond, conflict_mutex);
	}

	response = conflict_response;
	conflict_response = OB_CONFLICT_RESPONSE_NONE;

	g_mutex_unlock (conflict_mutex);

	return response;
}


/*
 * Memory card handling.
 */

ObMemoryCard *
ob_backend_get_memory_card (ObBackend        *backend,
			    ObMemoryCardType  type)
{
	ObBackendPriv *priv;
	
	g_return_val_if_fail (OB_IS_BACKEND (backend), NULL);

	priv = GET_PRIV (backend);

	if (type == OB_MEMORY_CARD_INTERNAL) {
		return priv->internal_mem_card;
	} else {
		return priv->external_mem_card;
	}
}

static gboolean
backend_is_mountpoint (ObBackend      *backend,
		       GnomeVFSVolume *volume,
		       GnomeVFSURI    *mountpoint_uri)
{
	ObBackendPriv *priv;
	char          *activation_uri;
	GnomeVFSURI   *uri;
	gboolean       match;

	if (!mountpoint_uri) {
		return FALSE;
	}

	priv = GET_PRIV (backend);
	
	activation_uri = gnome_vfs_volume_get_activation_uri (volume);
	uri = gnome_vfs_uri_new (activation_uri);

	match = gnome_vfs_uri_equal (uri, mountpoint_uri);

	gnome_vfs_uri_unref (uri);
	g_free (activation_uri);
	
	return match;
}

static gboolean
backend_is_external_mountpoint (ObBackend      *backend,
				GnomeVFSVolume *volume)
{
	GnomeVFSURI *uri;

	uri = ob_config_get_memory_card_mountpoint (ob_config_get (), 
						    OB_MEMORY_CARD_EXTERNAL);
	
	return backend_is_mountpoint (backend,
				      volume,
				      uri);
}

static gboolean
backend_is_internal_mountpoint (ObBackend      *backend,
				GnomeVFSVolume *volume)
{
	GnomeVFSURI *uri;

	uri = ob_config_get_memory_card_mountpoint (ob_config_get (), 
						    OB_MEMORY_CARD_INTERNAL);

	return backend_is_mountpoint (backend,
				      volume,
				      uri);
}

static gchar *
backend_get_volume_name (ObBackend      *backend,
			 GnomeVFSVolume *volume)
{
	gchar    *name;
	gboolean  internal;
	
	internal = backend_is_internal_mountpoint (backend, volume);

	name = gnome_vfs_volume_get_display_name (volume);

	/* Only check the prefix, the internal has -internal appended. */
	if (!name || strncmp (name, "mmc-undefined-name", 18) == 0) {
		if (internal) {
			name = g_strdup (_("back_fi_dia002_memorycard_internal"));
		} else {
			name = g_strdup (_("back_fi_dia002_memorycard_external"));
		}
	}

	return name;
}

static void
backend_volume_mounted_cb (GnomeVFSVolumeMonitor *monitor,
			   GnomeVFSVolume        *volume,
			   ObBackend             *backend)
{
	ObBackendPriv    *priv;
	gchar            *name, *activation_uri;
	GnomeVFSURI      *uri;
	ObMemoryCard     *memory_card;
	ObMemoryCardType  type;

	if (backend_is_internal_mountpoint (backend, volume)) {
		type = OB_MEMORY_CARD_INTERNAL;
	} else {
		type = OB_MEMORY_CARD_EXTERNAL;
	}

	priv = GET_PRIV (backend);

	priv->blame_unmount = FALSE;

	if (priv->state != OB_STATE_READY && type == priv->active_mem_card) {
		/* Not much we can do if we're not ready. */
		d(g_print ("Backend: %s volume mounted but we weren't ready!\n", 
			   type == OB_MEMORY_CARD_INTERNAL ? "Internal" : "External"));
		return;
	}		

	name = backend_get_volume_name (backend, volume);
	d(g_print ("Backend: %s volume mounted:'%s'\n", 
		   type == OB_MEMORY_CARD_INTERNAL ? "Internal" : "External", name));

	activation_uri = gnome_vfs_volume_get_activation_uri (volume);

	uri = gnome_vfs_uri_new (activation_uri);
	g_free (activation_uri);

	memory_card = ob_memory_card_new (uri, name);
	g_free (name);

	if (type == OB_MEMORY_CARD_INTERNAL) {
		priv->internal_mem_card = memory_card;
	} else {
		priv->external_mem_card = memory_card;
	}

	g_signal_emit (backend, signals[MEMORY_CARD_INSERTED], 0,
		       memory_card, type);

	backend_setup_backup_dir_monitor (backend, type, TRUE);
}

static void
backend_volume_unmounted_cb (GnomeVFSVolumeMonitor *monitor,
			     GnomeVFSVolume        *volume,
			     ObBackend             *backend)
{
	ObBackendPriv    *priv;
	ObMemoryCard     *mem_card;
	ObMemoryCardType  type;

	priv = GET_PRIV (backend);

	if (backend_is_internal_mountpoint (backend, volume)) {
		mem_card = priv->internal_mem_card;
		type = OB_MEMORY_CARD_INTERNAL;
	} else {
		mem_card = priv->external_mem_card;
		type = OB_MEMORY_CARD_EXTERNAL;
	}

	priv->blame_unmount = FALSE;

	if (!mem_card) {
		return;
	}

	if (priv->state != OB_STATE_READY) {
		d(g_print ("Backend: %s volume unmounted during operation, cancel.\n", 
			   type == OB_MEMORY_CARD_INTERNAL ? "Internal" : "External"));

		if (priv->active_mem_card == type) {
			ob_backend_cancel (backend);
		}
	} else {
		d(g_print ("Backend: %s volume unmounted\n", 
			   type == OB_MEMORY_CARD_INTERNAL ? "Internal" : "External"));
	}

	/* Note: we do this first because when the signal is emitted, calls to
	 * ob_backend_count_memory_cards() need to be accurate.
	 */
	if (type == OB_MEMORY_CARD_INTERNAL) {
		priv->internal_mem_card = NULL;
	} else {
		priv->external_mem_card = NULL;
	}

	/* Remove GnomeVFS monitor */
	backend_setup_backup_dir_monitor (backend, type, FALSE);

	/* Make sure when we obtain a new list because the card was removed it
	 * is up to date.
	 */
	ob_memory_card_clear_cache (mem_card);

	g_signal_emit (backend, signals[MEMORY_CARD_REMOVED], 0,
		       mem_card, type);

	ob_memory_card_unref (mem_card);
}

static void
backend_volume_pre_unmount_cb (GnomeVFSVolumeMonitor *monitor,
			       GnomeVFSVolume        *volume,
			       ObBackend             *backend)
{
	ObBackendPriv    *priv;
	ObMemoryCard     *mem_card;
	ObMemoryCardType  type;

	priv = GET_PRIV (backend);

	if (backend_is_internal_mountpoint (backend, volume)) {
		type = OB_MEMORY_CARD_INTERNAL;
	} else {
		type = OB_MEMORY_CARD_EXTERNAL;
	}

	mem_card = ob_backend_get_memory_card (backend, type);

	if (!mem_card) {
		return;
	}

 
 	/* Just return, since the behavior has changed from IT-2005, now the
 	 * system warns about the cover being open or a USB cable being
 	 * inserted. At some point, we need to remove this code and also clean
 	 * up the rest of the code in the backend that handles cancelling on
 	 * unmount.
 	 */
 	d(g_print ("Backend: Got pre-unmount, doing nothing...\n"));
 	return;
	
	priv->blame_unmount = TRUE;

	/* If we get a pre-unmount signal, cancel any ongoing operation. */
	if (priv->state != OB_STATE_READY) {
		d(g_print ("Backend: Pre-unmount during operation, cancel.\n"));
		ob_backend_cancel (backend);
	} else {
		d(g_print ("Backend: Pre-unmount\n"));
	}	
}

static void
backend_backup_dir_monitor_cb (GnomeVFSMonitorHandle    *handle,
			       const gchar              *monitor_uri,
			       const gchar              *info_uri,
			       GnomeVFSMonitorEventType  event_type,
			       ObBackend                *backend)
{
	ObBackendPriv *priv;

	d(g_printerr ("Backend: Monitor event for URI:'%s'...\n", 
		      monitor_uri));

	/* Ignore events that are not about changing, deleting or
	 * creating a directory.
	 */
	if (event_type != GNOME_VFS_MONITOR_EVENT_CHANGED &&
	    event_type != GNOME_VFS_MONITOR_EVENT_DELETED &&
	    event_type != GNOME_VFS_MONITOR_EVENT_CREATED) {
		return;
	}

	d(g_printerr ("Backend: URI:'%s' updated, created or deleted...\n", 
		      monitor_uri));

	priv = GET_PRIV (backend);

	if (handle == priv->internal_monitor) {
		ob_memory_card_clear_cache (priv->internal_mem_card);
		g_signal_emit (backend, signals[BACKUPS_CHANGED], 0, 
			       priv->internal_mem_card,
			       OB_MEMORY_CARD_INTERNAL);
	} else {
		ob_memory_card_clear_cache (priv->external_mem_card);
		g_signal_emit (backend, signals[BACKUPS_CHANGED], 0, 
			       priv->external_mem_card, 
			       OB_MEMORY_CARD_EXTERNAL);
	}
}

static void
backend_setup_backup_dir_monitor (ObBackend        *backend, 
				  ObMemoryCardType  type,
				  gboolean          enabled)
{
	ObBackendPriv         *priv;
	GnomeVFSURI           *uri;
	GnomeVFSMonitorHandle *handle;
	GnomeVFSResult         result;

	priv = GET_PRIV (backend);
	
	if (!enabled) {
		if (type == OB_MEMORY_CARD_INTERNAL && priv->internal_monitor) {
			d(g_printerr ("Backend: Removing monitor for internal URI.\n"));
			gnome_vfs_monitor_cancel (priv->internal_monitor);
			priv->internal_monitor = NULL;
		} 
		else if (type == OB_MEMORY_CARD_EXTERNAL && priv->external_monitor) {
			d(g_printerr ("Backend: Removing monitor for external URI.\n"));
			gnome_vfs_monitor_cancel (priv->external_monitor);
			priv->external_monitor = NULL;
		}

		return;
	}

	uri = gnome_vfs_uri_append_path (
		ob_config_get_memory_card_mountpoint (priv->config, type),
		"backups");

	d(g_printerr ("Backend: Adding monitoring for URI:'%s'.\n", 
		      gnome_vfs_uri_get_path (uri)));

	result = gnome_vfs_monitor_add (&handle, 
					gnome_vfs_uri_get_path (uri), 
					GNOME_VFS_MONITOR_DIRECTORY, 
					(GnomeVFSMonitorCallback)
					backend_backup_dir_monitor_cb,
					backend);
	
	if (result == GNOME_VFS_OK && handle != NULL) {
		if (type == OB_MEMORY_CARD_INTERNAL) {
			priv->internal_monitor = handle;
		} else {
			priv->external_monitor = handle;
		}
	} else {
		ob_log_warning ("Could not set up URI monitor for uri:'%s'.", 
				gnome_vfs_uri_get_path (uri));
	}
	
	gnome_vfs_uri_unref (uri);
}

static void
backend_setup_volume_monitoring (ObBackend *backend)
{
	ObBackendPriv         *priv;
	GnomeVFSVolumeMonitor *monitor;
	GnomeVFSURI           *internal_uri;
	GnomeVFSURI           *external_uri;
	GList                 *volumes, *l;
	GnomeVFSVolume        *volume;
	char                  *name;

	priv = GET_PRIV (backend);
	
	monitor = gnome_vfs_get_volume_monitor ();

	g_signal_connect (monitor,
			  "volume_mounted",
			  G_CALLBACK (backend_volume_mounted_cb),
			  backend);

	g_signal_connect (monitor,
			  "volume_unmounted",
			  G_CALLBACK (backend_volume_unmounted_cb),
			  backend);

	g_signal_connect (monitor,
			  "volume_pre_unmount",
			  G_CALLBACK (backend_volume_pre_unmount_cb),
			  backend);

	internal_uri = ob_config_get_memory_card_mountpoint (priv->config, 
							     OB_MEMORY_CARD_INTERNAL);
	external_uri = ob_config_get_memory_card_mountpoint (priv->config, 
							     OB_MEMORY_CARD_EXTERNAL);

	if (!internal_uri && !external_uri) {
		ob_log_warning ("No mountpoint configured.");
		return;
	}
	
	volumes = gnome_vfs_volume_monitor_get_mounted_volumes (monitor);
	for (l = volumes; l; l = l->next) {
		volume = l->data;

		if (backend_is_external_mountpoint (backend, volume)) {
			/* Found mountpoint. */
			name = backend_get_volume_name (backend, volume);
			d(g_printerr ("Backend: External MMC:'%s', URI:'%s'\n", 
				      name, gnome_vfs_uri_get_path (external_uri)));
			
			priv->external_mem_card = ob_memory_card_new (external_uri, name);
			g_free (name);

			/* Add GnomeVFS monitor */
			backend_setup_backup_dir_monitor (backend, OB_MEMORY_CARD_EXTERNAL, TRUE);
		}
		else if (backend_is_internal_mountpoint (backend, volume)) {
			/* Found internal mountpoint. */
			name = backend_get_volume_name (backend, volume);
			d(g_printerr ("Backend: Internal MMC:'%s', URI:'%s'\n", 
				      name, gnome_vfs_uri_get_path (internal_uri)));

			priv->internal_mem_card = ob_memory_card_new (internal_uri, name);
			g_free (name);

			/* Add GnomeVFS monitor */
			backend_setup_backup_dir_monitor (backend, OB_MEMORY_CARD_INTERNAL, TRUE);
		}
	}

	g_list_foreach (volumes, (GFunc) gnome_vfs_volume_unref, NULL);
	g_list_free (volumes);
}

/* Checks if the backup data will fit on the inserted backup target. Note that
 * this check is not fully accurate, due to compression and overhead from the
 * archiver format. The replaced_size will be subtracted from the memory used on
 * the card, which is used when replacing a backup with the same name.
 */
gboolean
ob_backend_space_on_memory_card (ObBackend        *backend,
				 ObMemoryCardType  type,
				 GnomeVFSFileSize  size,
				 GnomeVFSFileSize *available_size)
{
	ObMemoryCard   *mem_card;
	GnomeVFSResult  result;
	
	*available_size = 0;

	g_return_val_if_fail (OB_IS_BACKEND (backend), FALSE);

	mem_card = ob_backend_get_memory_card (backend, type);
	if (!mem_card) {
		/* No memory card inserted. */
		return FALSE;
	}

	/* Make sure we clean up any old .gsf* files laying around
	 * first other wise the space remaining is incorrect.
	 */
	ob_memory_card_clear_tmp_files (mem_card);
	
	result = ob_memory_card_get_free_space (mem_card, available_size);
	if (result != GNOME_VFS_OK) {
		return FALSE;
	}

	/* Do an estimated check with some margin. */
	if (size >= *available_size) {
		return FALSE;
	}
	
	return TRUE;
}

guint
ob_backend_count_memory_cards (ObBackend *backend)
{
	ObBackendPriv *priv = GET_PRIV (backend);
	guint          count = 0;

	if (priv->internal_mem_card) {
		count++;
	}

	if (priv->external_mem_card) {
		count++;
	}

	d(g_print ("Backend: Found %d memory cards.\n", count);)

	return count;
}

guint
ob_backend_count_backups (ObBackend        *backend,
			  ObMemoryCardType  type)
{
	ObBackendPriv *priv;
	ObMemoryCard  *mem_card;
	GList         *backups = NULL;

	g_return_val_if_fail (OB_IS_BACKEND (backend), 0);

	priv = GET_PRIV (backend);

	if (type == OB_MEMORY_CARD_INTERNAL) {
		mem_card = priv->internal_mem_card;
	} else {
		mem_card = priv->external_mem_card;
	}	

	if (mem_card) {
		backups = ob_memory_card_get_backups (mem_card);
	}

	return g_list_length (backups);
}

int
ob_backend_get_elapsed_time (ObBackend *backend)
{
	ObBackendPriv *priv;

	g_return_val_if_fail (OB_IS_BACKEND (backend), 0);

	priv = GET_PRIV (backend);

	return 0.5 + g_timer_elapsed (priv->timer, NULL);
}

GnomeVFSFileSize
ob_backend_get_processed_size (ObBackend *backend)
{
	ObBackendPriv *priv;

	g_return_val_if_fail (OB_IS_BACKEND (backend), 0);

	priv = GET_PRIV (backend);

	return priv->processed_size;
}

int
ob_backend_get_processed_files (ObBackend *backend)
{
	ObBackendPriv *priv;

	g_return_val_if_fail (OB_IS_BACKEND (backend), 0);

	priv = GET_PRIV (backend);

	return priv->processed_files;
}

GError *
ob_backend_get_error (ObBackend *backend)
{
	ObBackendPriv *priv;

	g_return_val_if_fail (OB_IS_BACKEND (backend), NULL);
	
	priv = GET_PRIV (backend);

	return priv->error;
}

void
ob_backend_add_restored_flag (void)
{
	const gchar *content;
	gchar       *filename;

	d(g_print ("Backend: Adding restore flag\n"));

	content = "This file is generated automatically, do not delete it.";
	
	/* Create startup-restore-flag */
	filename = g_build_filename (g_get_home_dir (), 
				     ".osso-backup", 
				     "startup-restored-flag",
				     NULL);
	
	g_file_set_contents (filename, content, -1, NULL);
	
	g_free (filename);
}

void
ob_backend_remove_startup_flag (void)
{
	gchar *filename;

	d(g_print ("Backend: Removing startup flag\n"));
	
	/* Remove first startup flag */
	filename = g_build_filename (g_get_home_dir (), 
				     "first-boot-flag",
				     NULL);

	if (g_remove (filename) != 0) {
		/*ob_log_warning ("Could not remove startup flag filename:'%s'.", filename);*/
	}

	g_free (filename);

	/* Remove second startup flag */
	filename = g_build_filename (g_get_home_dir (), 
				     ".suw_first_run",
				     NULL);

	if (g_remove (filename) != 0) {
		/*ob_log_warning ("Could not remove startup flag filename:'%s'.", filename);*/
	}

	g_free (filename);
}
