/*
Print currently active locks in the system held by libgpsmgr library.

Copyright (C) 2006 Nokia Corporation. All rights reserved.

Author: Jukka Rissanen <jukka.rissanen@nokia.com>

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
The name of the author may not be used to endorse or promote products derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/


#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <errno.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <string.h>

#include "gpsmgr.h"

/* ----------------------------------------------------------------------- */
static int get_lock(int fd, int type, off_t offset, off_t len, pid_t *pid,
		    off_t *start, off_t *stop)
{
  struct flock fl;
  int ret;

  memset(&fl, 0, sizeof(fl));

  fl.l_type = type;
  fl.l_start = offset;
  fl.l_whence = SEEK_SET;
  fl.l_len = len;
  
  ret = fcntl(fd, F_GETLK, &fl);

  if (fl.l_type == F_UNLCK)
    ret = 0;  /* is not locked */
  else {
    if (fl.l_pid != 0)
      ret = 1; /* is locked */
    else
      ret = 0; /* pid 0 cannot lock anything, this was seen when
		* the program was executed in scrachbox in transparency
		* mode
		*/

    *pid = fl.l_pid;
    *start = fl.l_start;
    *stop = fl.l_start + fl.l_len - 1;

#if 0
    printf("ret=%d, pid=%d, %d[%d], type=%s\n", ret, fl.l_pid, 
	   fl.l_start, fl.l_len,
	   fl.l_type==F_UNLCK ? "F_UNLCK" : (fl.l_type==F_WRLCK ? "F_WRLCK" : "?"));
#endif
  }

  return ret;
}


static int file_open(char *file)
{
  int fd;
  fd = open(file, O_RDWR);
  return fd;
}


/* ----------------------------------------------------------------------- */
int main(int argc, char **argv)
{
  int ret, fd, pid = 0;
  char *file;
  off_t start = 0, stop = 0;
  FILE *pid_file;

  file = GPSMGR_GPSD_LOCK;
  pid_file = fopen(file, "r");
  if (pid_file) {
    char pid_buf[10+1];
    memset(pid_buf, 0, 11);
    if (!fgets(pid_buf, 10, pid_file)) {
      printf("Cannot get GPS daemon pid from %s\n", file);
    } else {
      pid_t pid;
      int pid_len = strlen(pid_buf);
      if (pid_buf[pid_len-1]=='\n')
	pid_buf[pid_len-1]='\0';
      pid = atoi(pid_buf);
      if (pid) {
	if (!kill(pid, 0))
	  printf("GPS daemon pid %s\n", pid_buf);
	else {
	  if (errno==ESRCH)
	    printf("GPS daemon (pid=%d) is already dead\n", pid);
	  else
	    printf("GPS daemon (pid=%d) status unknown [%s/%d]\n", pid, strerror(errno), errno);
	}
      } else
	printf("GPS daemon pid %s invalid\n", pid_buf);
    }
    fclose(pid_file);
  }

  file = GPSMGR_GPSD_LOCK;
  fd = file_open(file);
  if (fd<0)
    printf("Cannot open %s [%s/%d]\n", file, strerror(errno), errno);
  else {
    ret = get_lock(fd, F_WRLCK, 0, 0, &pid, &start, &stop);
    if (ret<0) {
      printf("Cannot lock %s [%s/%d]\n", file, strerror(errno), errno);
    } else if (ret == 0) {
      printf("File %s is not locked\n", file);
    } else {
      printf("File %s already locked by pid %d\n", file, pid);
    }
  }
  close(fd);


  file = GPSMGR_LOCK2;
  fd = file_open(file);
  if (fd<0)
    printf("Cannot open %s [%s/%d]\n", file, strerror(errno), errno);
  else {
    ret = get_lock(fd, F_WRLCK, 0, 0, &pid, &start, &stop);
    if (ret<0) {
      printf("Cannot lock %s [%s/%d]\n", file, strerror(errno), errno);
    } else if (ret == 0) {
      printf("File %s is not locked\n", file);
    } else {
      printf("File %s already locked by pid %d\n", file, pid);
    }
  }
  close(fd);


  file = GPSMGR_LOCK;
  fd = file_open(file);
  if (fd<0)
    printf("Cannot open %s [%s/%d]\n", file, strerror(errno), errno);
  else {
    int i, count=0;
    printf("File %s slot lock status:\n\t", file);
    for (i=0; i<GPSMGR_MAX_APPLICATIONS; i++) {
      ret = get_lock(fd, F_WRLCK, i, 1, &pid, &start, &stop);
      if (ret<0) {
	printf("\nCannot lock %s slot %d [%s/%d]\n", file, i, strerror(errno), errno);
      } else if (ret == 0) {
	//printf("File %s slot %d is not locked\n", file, i);
	printf("[%02d/-] ", i);
      } else {
	//printf("File %s slot %i already locked by pid %d\n", file, i, pid);
	printf("[%02d/%d] ", i, pid);
	count++;
      }
      if (!((i+1) % 4))
	printf("\n\t");
    }
    if (!count)
      printf("All slots are unlocked\n");
    else
      printf("\n");
  }
  close(fd);

}

