/*
 * Optional support for a non-standard TUSB6010 loopback test
 *
 * Copyright (C) 2006 Nokia Corporation
 * Tony Lindgren <tony@atomide.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 * In order for the test to work, the dongle musb connect DM to DP over a 1k
 * resistor when VBUS is enabled. By using direct OTG PHY register access
 * DM line is raised by the test function, which then also raises DP line if
 * the dongle is connected.
 *
 * To run the test, connect the loopback dongle to the USB interface on the
 * board and type the following on the board:
 *
 * # cat /sys/devices/platform/musb_hdrc/looptest
 *
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/irq.h>
#include <linux/platform_device.h>
#include <linux/delay.h>

#include "musbdefs.h"
#include "tusb6010.h"

#define TUSB_TEST_PASSED		0
#define TUSB_TEST_FAILED		-1
#define TUSB_TEST_INIT_ERROR		-2

#define TUSB_TEST_DM_HIGH_MASK		(TUSB_PHY_OTG_CTRL_WRPROTECT	\
					| TUSB_PHY_OTG_CTRL_TX_DATA2	\
					| TUSB_PHY_OTG_CTRL_TX_GZ2	\
					| TUSB_PHY_OTG_CTRL_TX_ENABLE2 \
					| TUSB_PHY_OTG_CTRL_DP_PULLDOWN)
#ifdef TEST_OTG
#define test_otg()	1
#define TUSB_PIN_TEST_MASK	(TUSB_DEV_OTG_STAT_ID_STATUS |		\
					TUSB_DEV_OTG_STAT_DP_ENABLE |	\
					TUSB_DEV_OTG_STAT_DM_ENABLE)
#else
#define test_otg()	0
#define TUSB_PIN_TEST_MASK	(TUSB_DEV_OTG_STAT_DP_ENABLE |	\
					TUSB_DEV_OTG_STAT_DM_ENABLE)
#endif

static inline void tusb_test_debug(u32 otg_stat, char *msg)
{
	DBG(2, "%s %s %s otg_stat: 0x%08x %s\n",
		otg_stat & TUSB_DEV_OTG_STAT_ID_STATUS ? "ID" : "  ",
		otg_stat & TUSB_DEV_OTG_STAT_DP_ENABLE ? "D+" : "  ",
		otg_stat & TUSB_DEV_OTG_STAT_DM_ENABLE ? "D-" : "  ",
		otg_stat, msg);
}

static struct musb *the_musb;

static inline void tusb_test_force_vbus(struct musb *musb, u8 enabled)
{
	void __iomem	*base = musb->ctrl_base;
	u32		reg;

	reg = musb_readl(base, TUSB_DEV_CONF);
	if (enabled)
		reg |= TUSB_DEV_CONF_USB_HOST_MODE;
	else
		reg &= ~TUSB_DEV_CONF_USB_HOST_MODE;
	musb_writel(base, TUSB_DEV_CONF, reg);

	reg = musb_readl(base, TUSB_PRCM_CONF);
	if (enabled)
		reg |= TUSB_PRCM_CONF_SFW_CPEN;
	else
		reg &= ~TUSB_PRCM_CONF_SFW_CPEN;
	musb_writel(base, TUSB_PRCM_CONF, reg);

	reg = musb_readl(base, TUSB_PRCM_MNGMT);
	if (enabled)
		reg |= TUSB_PRCM_MNGMT_5V_CPEN;
	else
		reg &= ~TUSB_PRCM_MNGMT_5V_CPEN;
	musb_writel(base, TUSB_PRCM_MNGMT, reg);
}

static int tusb_do_test(void)
{
	void __iomem	*base = the_musb->ctrl_base;
	void __iomem	*musb_base = the_musb->pRegs;
	unsigned long	flags;
	u32		otg_stat;
	int		retries;
	int		ret;

	if (the_musb == NULL)
		return TUSB_TEST_INIT_ERROR;

	set_irq_type(the_musb->nIrq, IRQ_TYPE_NONE);
	spin_lock_irqsave(&the_musb->Lock, flags);

	/* Enable direct access to OTG registers. Note that we need to
	 * have TUSB_PHY_OTG_CTRL_DM_PULLDOWN access enabled */
	musb_writel(base, TUSB_PHY_OTG_CTRL_ENABLE,
			TUSB_PHY_OTG_CTRL_WRPROTECT |
			TUSB_TEST_DM_HIGH_MASK |
			TUSB_PHY_OTG_CTRL_OTG_ID_PULLUP |
			TUSB_PHY_OTG_CTRL_DM_PULLDOWN);

	/* Sometimes ID line does not go up on it's own. Force it up */
	musb_writel(base, TUSB_PHY_OTG_CTRL, TUSB_PHY_OTG_CTRL_WRPROTECT |
			TUSB_PHY_OTG_CTRL_OTG_ID_PULLUP);

	/* Check ID line. It should be up at this point */
	otg_stat = musb_readl(base, TUSB_DEV_OTG_STAT);
	tusb_test_debug(otg_stat, "Before VBUS");

	if (test_otg() && !(otg_stat & TUSB_DEV_OTG_STAT_ID_STATUS)) {
		tusb_test_debug(otg_stat, "ID down before VBUS is on");
		ret = TUSB_TEST_INIT_ERROR;
		goto restore;
	}

	tusb_test_force_vbus(the_musb, 1);
	msleep(1);

	/* Try to lower all lines */
	musb_writel(base, TUSB_PHY_OTG_CTRL, TUSB_PHY_OTG_CTRL_WRPROTECT);

	/* Check lines. ID, DM and DP lines should all be down */
	otg_stat = musb_readl(base, TUSB_DEV_OTG_STAT);
	tusb_test_debug(otg_stat, "After VBUS");

	if (otg_stat & TUSB_PIN_TEST_MASK) {
		tusb_test_debug(otg_stat, "line(s) up after VBUS is on");
		ret = TUSB_TEST_INIT_ERROR;
		goto restore;
	}

	/* Test dongle should take ID line to ground after VBUS is on */
	if (test_otg() && (otg_stat & TUSB_DEV_OTG_STAT_ID_STATUS)) {
		tusb_test_debug(otg_stat, "ID up after VBUS is on");
		goto restore;
	}

	/* Try to raise DM line. If the loopback dongle is connected, DP
	 * line should go up as well
	 */
	for (retries = 0; retries < 500; retries++) {
		static int	connected = 0;

		musb_writel(base, TUSB_PHY_OTG_CTRL,
				TUSB_PHY_OTG_CTRL_WRPROTECT);

		/* REVISIT: Why do we need SESSION to change DM? */
		musb_writeb(musb_base, MGC_O_HDRC_DEVCTL, MGC_M_DEVCTL_SESSION);
		musb_writel(base, TUSB_PHY_OTG_CTRL, TUSB_TEST_DM_HIGH_MASK);
		msleep(1);
		otg_stat = musb_readl(base, TUSB_DEV_OTG_STAT);

		if (otg_stat & TUSB_DEV_OTG_STAT_DP_ENABLE)
			connected++;
		else
			connected = 0;

		if (connected)
			tusb_test_debug(otg_stat, "D- & D+ connected");
		else
			tusb_test_debug(otg_stat, "D- & D+ disconnected");

		if (connected >= 5) {
			ret = TUSB_TEST_PASSED;
			goto restore;
		}
	}

	/* Failed to raise DP line */
	tusb_test_debug(otg_stat, "DP down after test");
	ret = TUSB_TEST_FAILED;

restore:
	musb_writel(base, TUSB_PHY_OTG_CTRL, 0);
	tusb_test_force_vbus(the_musb, 0);

	/* Disable direct access to OTG registers and restore settings */
	musb_writel(base, TUSB_PHY_OTG_CTRL, TUSB_PHY_OTG_CTRL_WRPROTECT);
	musb_writel(base, TUSB_PHY_OTG_CTRL_ENABLE,
				TUSB_PHY_OTG_CTRL_WRPROTECT);

	musb_platform_try_idle(the_musb);
	spin_unlock_irqrestore(&the_musb->Lock, flags);
	set_irq_type(the_musb->nIrq, IRQ_TYPE_LEVEL_LOW);

	return ret;
}

static ssize_t tusb_show_result(struct device *dev,
				struct device_attribute *attr,
				char *buf)
{
	char		*s;
	int		len;

	switch (tusb_do_test()) {
	case TUSB_TEST_PASSED:
		s = "passed";
		break;
	case TUSB_TEST_FAILED:
		s = "failed";
		break;
	case TUSB_TEST_INIT_ERROR:
		s = "init error";
		break;
	default:
		s = "unknown error";
		break;
	}

	len = sprintf(buf, "%s\n", s);

	return len;
}

static DEVICE_ATTR(looptest, 0440, tusb_show_result, NULL);

int __devinit tusb_test_init(struct musb *musb)
{
	struct platform_device	*pdev;
	struct device		*dev;

	the_musb = musb;

	pdev = to_platform_device(musb->controller);
	dev = &pdev->dev;
	return device_create_file(dev, &dev_attr_looptest);
}

void __devexit tusb_test_release(void)
{
	struct platform_device	*pdev;
	struct device		*dev;

	if (the_musb == NULL)
		return;

	pdev = to_platform_device(the_musb->controller);
	dev = &pdev->dev;
	device_remove_file(dev, &dev_attr_looptest);
}
