/*
 * drivers/media/video/omap/omap24xxcam.c
 *
 * Video-for-Linux (Version 2) camera capture driver for 
 * the OMAP24xx camera controller.
 *
 * Author: Andy Lowe (source@mvista.com)
 *
 * Copyright (C) 2004 MontaVista Software, Inc.
 * Copyright (C) 2004 Texas Instruments.
 *
 * This file is licensed under the terms of the GNU General Public License 
 * version 2. This program is licensed "as is" without any warranty of any 
 * kind, whether express or implied.
 *
 * History:
 * 2005/nov - David Cohen <david.cohen@indt.org.br>
 *            Updated the driver for the Linux Device Model and new version of V4L2.
 */

#include <linux/config.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/delay.h>
#include <linux/timer.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#include <linux/sched.h>
#include <linux/smp_lock.h>
#include <linux/interrupt.h>
#include <linux/kdev_t.h>
#include <linux/types.h>
#include <linux/wait.h>
#include <linux/i2c.h>
#include <linux/videodev2.h>
#include <linux/pci.h>		/* needed for videobufs */
#include <linux/dma-mapping.h>
#include <linux/device.h>
#include <linux/input.h>
#include <linux/version.h>
#include <linux/platform_device.h>
#include <linux/clk.h>

#include <media/video-buf.h>
#include <media/v4l2-common.h>

#include <asm/io.h>
#include <asm/byteorder.h>
#include <asm/scatterlist.h>
#include <asm/irq.h>
#include <asm/semaphore.h>
#include <asm/processor.h>

#include <asm/arch/clock.h>

#include "omap24xxcam.h"
#include "omap24xxcam-dma.h"

#include "sensor_if.h"

/* configuration macros */
#define CAM_NAME 	"omap24xxcam"

//#define DEBUG_CAM
#ifdef DEBUG_CAM
#define DBG		printk(KERN_INFO CAM_NAME ": %s\n", __FUNCTION__)
#define DBG_END 	printk(KERN_INFO CAM_NAME ": %s END!\n", __FUNCTION__)
#define DBG_MID(x)	printk(KERN_INFO CAM_NAME ": %s - %d\n", __FUNCTION__, x)
#else
#define DBG
#define DBG_END
#define DBG_MID(x)
#endif

static struct omap_camera_sensor *camera_sensor_if;
struct omap24xxcam_device * camera_module;

void omap24xxcam_cleanup(void);

/* module parameters */
static int video_nr = -1;	/* video device minor (-1 ==> auto assign) */
/* Maximum amount of memory to use for capture buffers.
 * Default is 4800KB, enough to double-buffer SXGA.
 */
static int capture_mem = 1280 * 960 * 2 * 2;
/* Size of video overlay framebuffer.  This determines the maximum image size 
 * that can be previewed.  Default is 600KB, enough for VGA.
 */
static int overlay_mem = 640 * 480 * 2;
/* Size of video2_mem framebuffer. This determines the maximum image size which
 * video2_layer can support. Default is 300 KB. Size of LCD or 320*240.
*/
static int video2_mem = 320 * 240 * 2;

/* -------------------------------------------------------------------------- */

/* Set the value of the CC_CTRL register in cam->cc_ctrl that is required to 
 * support the currently selected capture format in cam->pix.  The CC_CTRL bits 
 * which must be configured are:  NOBT_SYNCHRO, BT_CORRECT, PAR_ORDERCAM, 
 * PAR_CLK_POL, NOBT_HS_POL, NOBT_VS_POL, PAR_MODE, and CCP_MODE.  The CC_RST, 
 * CC_FRAME_TRIG, and CC_EN bits are actively managed by the driver and should 
 * be set to zero by this routine.
 */
static void omap24xxcam_sensor_cc_ctrl(struct omap24xxcam_device *cam)
{
	struct v4l2_pix_format *pix;

	DBG;

	pix = &cam->pix;

	cam->cc_ctrl = CC_CTRL_NOBT_SYNCHRO | CC_CTRL_PAR_MODE_NOBT8;

	switch (pix->pixelformat) {
	case V4L2_PIX_FMT_UYVY:
	case V4L2_PIX_FMT_RGB565:
	case V4L2_PIX_FMT_RGB555:
	default:
		/* These formats need a 16-bit byte swap */
		//cam->cc_ctrl |= CC_CTRL_PAR_ORDERCAM;
	case V4L2_PIX_FMT_YUYV:
	case V4L2_PIX_FMT_RGB565X:
	case V4L2_PIX_FMT_RGB555X:
		/* These formats don't need a 16-bit byte swap */
		break;
	}
}

/*
 * camera core register I/O routines
 */

static __inline__ u32
cc_reg_in(const struct omap24xxcam_device *cam, u32 offset)
{
	return readl(cam->cam_mmio_base + CC_REG_OFFSET + offset);
}

static __inline__ u32
cc_reg_out(const struct omap24xxcam_device *cam, u32 offset, u32 val)
{
	writel(val, cam->cam_mmio_base + CC_REG_OFFSET + offset);
	return val;
}

static __inline__ u32
cc_reg_merge(const struct omap24xxcam_device *cam, u32 offset,
	     u32 val, u32 mask)
{
	u32 addr = cam->cam_mmio_base + CC_REG_OFFSET + offset;
	u32 new_val = (readl(addr) & ~mask) | (val & mask);

	writel(new_val, addr);
	return new_val;
}

void cc_init(const struct omap24xxcam_device *cam)
{
	DBG;
	/* Setting the camera core AUTOIDLE bit causes problems with frame 
	 * synchronization, so we will clear the AUTOIDLE bit instead.
	 */
	//cc_reg_out(cam, CC_SYSCONFIG, 0);
	cc_reg_out(cam, CC_SYSCONFIG, CC_SYSCONFIG_AUTOIDLE);
}

/*
 * camera MMU register I/O routines
 */

static __inline__ u32
cammmu_reg_in(const struct omap24xxcam_device *cam, u32 offset)
{
	return readl(cam->cam_mmio_base + CAMMMU_REG_OFFSET + offset);
}

static __inline__ u32
cammmu_reg_out(const struct omap24xxcam_device *cam, u32 offset, u32 val)
{
	writel(val, cam->cam_mmio_base + CAMMMU_REG_OFFSET + offset);
	return val;
}

static __inline__ u32
cammmu_reg_merge(const struct omap24xxcam_device *cam, u32 offset,
		 u32 val, u32 mask)
{
	u32 addr = cam->cam_mmio_base + CAMMMU_REG_OFFSET + offset;
	u32 new_val = (readl(addr) & ~mask) | (val & mask);

	writel(new_val, addr);
	return new_val;
}

void cammmu_init(const struct omap24xxcam_device *cam)
{
	DBG;
	/* set the camera MMU autoidle bit */
	cammmu_reg_out(cam, CAMMMU_SYSCONFIG, CAMMMU_SYSCONFIG_AUTOIDLE);
}

/*
 * camera subsystem register I/O routines
 */

static __inline__ u32
cam_reg_in(const struct omap24xxcam_device *cam, u32 offset)
{
	return readl(cam->cam_mmio_base + offset);
}

static __inline__ u32
cam_reg_out(const struct omap24xxcam_device *cam, u32 offset, u32 val)
{
	writel(val, cam->cam_mmio_base + offset);
	return val;
}

static __inline__ u32
cam_reg_merge(const struct omap24xxcam_device *cam, u32 offset,
	      u32 val, u32 mask)
{
	u32 addr = cam->cam_mmio_base + offset;
	u32 new_val = (readl(addr) & ~mask) | (val & mask);

	writel(new_val, addr);
	return new_val;
}

/* Reset the camera subsystem (camera core, camera DMA, and camera MMU) */
static void
cam_reset(const struct omap24xxcam_device *cam, unsigned long timeout_ticks)
{
	unsigned long timeout;

	DBG;

	cam_reg_out(cam, CAM_SYSCONFIG, CAM_SYSCONFIG_SOFTRESET);
	/* wait for reset to complete */
	timeout = jiffies + timeout_ticks;
	while (!(cam_reg_in(cam, CAM_SYSSTATUS) & CAM_SYSSTATUS_RESETDONE)
	       && time_before(jiffies, timeout)) {
		if (!in_atomic()) {
			set_current_state(TASK_INTERRUPTIBLE);
			schedule_timeout(1);
		} else
			udelay(10);
	}
	if (!(cam_reg_in(cam, CAM_SYSSTATUS) & CAM_SYSSTATUS_RESETDONE)) {
		printk(KERN_WARNING CAM_NAME
		       ": timeout waiting for camera subsystem reset\n");
	}

	return;
}

/* Initialize the camera subsystem (camera core, camera DMA, and camera MMU) */
void cam_init(const struct omap24xxcam_device *cam)
{
	DBG;
	/* reset the camera subsystem with a timeout of 200ms */
	cam_reset(cam, HZ / 5);

	/* set the camera subsystem autoidle bit */
	cam_reg_out(cam, CAM_SYSCONFIG, CAM_SYSCONFIG_AUTOIDLE);

	/* initialize the camera MMU */
	cammmu_init(cam);

	/* initialize the camera DMA controller */
	camdma_init(cam);

	/* initialize the camera core module */
	cc_init(cam);
}

/*
 * display controller register I/O routines
 */

static __inline__ u32
dispc_reg_in(const struct omap24xxcam_device *cam, u32 offset)
{
	return readl(cam->dispc_mmio_base + DISPC_REG_OFFSET + offset);
}

static __inline__ u32
dispc_reg_out(const struct omap24xxcam_device *cam, u32 offset, u32 val)
{
	writel(val, cam->dispc_mmio_base + DISPC_REG_OFFSET + offset);
	return val;
}

static __inline__ u32
dispc_reg_merge(const struct omap24xxcam_device *cam, u32 offset,
		u32 val, u32 mask)
{
	u32 addr = cam->dispc_mmio_base + DISPC_REG_OFFSET + offset;
	u32 new_val = (readl(addr) & ~mask) | (val & mask);

	writel(new_val, addr);
	return new_val;
}

/* -------------------------------------------------------------------------- */

/* Turn off the video overlay window. */
static void omap24xxcam_disable_vlayer(struct omap24xxcam_device *cam, int v)
{
	unsigned long vid_attributes;

	DBG;

	vid_attributes = dispc_reg_merge(cam, DISPC_VID_ATTRIBUTES(v), 0,
					 DISPC_VID_ATTRIBUTES_ENABLE);
	if (vid_attributes & DISPC_VID_ATTRIBUTES_VIDCHANNELOUT) {
		/* digital output */
		dispc_reg_merge(cam, DISPC_CONTROL, DISPC_CONTROL_GODIGITAL,
				DISPC_CONTROL_GODIGITAL);
	} else {
		/* LCD */
		dispc_reg_merge(cam, DISPC_CONTROL, DISPC_CONTROL_GOLCD,
				DISPC_CONTROL_GOLCD);
	}
}

/* Flip the video overlay framebuffer.  The video overlay window may initially 
 * be either enabled or disabled.  The overlay window will be enabled by this 
 * routine.  fb_base_phys is the physical base address of the framebuffer for 
 * the video overlay.  The address programmed into the base address register of 
 * the video overlay window is calculated based on the cropped size and the full
 * size of the overlay framebuffer. 
 */
static void
omap24xxcam_flip_overlay(struct omap24xxcam_device *cam,
			 unsigned long fb_base_phys)
{
	unsigned long vid_attributes;
	int v = cam->vid1;
	unsigned long cropped_base_phys;
	struct v4l2_rect *crop = &cam->crop;
	struct v4l2_pix_format *pix = &cam->pix;

	DBG;

	cropped_base_phys = fb_base_phys
	    + pix->bytesperline * crop->top + crop->left * 2;
	dispc_reg_out(cam, DISPC_VID_BA0(v), cropped_base_phys);
	dispc_reg_out(cam, DISPC_VID_BA1(v), cropped_base_phys);

	vid_attributes = dispc_reg_merge(cam, DISPC_VID_ATTRIBUTES(v),
					 DISPC_VID_ATTRIBUTES_ENABLE,
					 DISPC_VID_ATTRIBUTES_ENABLE);
	if (vid_attributes & DISPC_VID_ATTRIBUTES_VIDCHANNELOUT) {
		/* digital output */
		dispc_reg_merge(cam, DISPC_CONTROL, DISPC_CONTROL_GODIGITAL,
				DISPC_CONTROL_GODIGITAL);
	} else {
		/* LCD */
		dispc_reg_merge(cam, DISPC_CONTROL, DISPC_CONTROL_GOLCD,
				DISPC_CONTROL_GOLCD);
	}
}

/* Get the framebuffer parameters by reading the display controller registers 
 * for the graphics window.
 */
static void omap24xxcam_g_fbuf(struct omap24xxcam_device *cam)
{
	struct v4l2_framebuffer *fbuf = &cam->fbuf;
	unsigned long gfx_size, gfx_position;

	DBG;

	/* This routine makes some assumptions about the framebuffer driver.  
	 * First, the entire contents of the graphics framebuffer must be 
	 * displayed, i.e. a configuration in which part of the graphics 
	 * framebuffer is offscreen is not supported.  The graphics 
	 * window must not be resized or repositioned while capture preview is 
	 * active.  The rotation capabilities of the display controller must 
	 * not be used to rotate the graphics window.
	 */
	fbuf->capability = V4L2_FBUF_CAP_EXTERNOVERLAY
		| V4L2_FBUF_CAP_CHROMAKEY;
	gfx_size = dispc_reg_in(cam, DISPC_GFX_SIZE);
	fbuf->fmt.width = 1 + ((gfx_size & DISPC_GFX_SIZE_GFXSIZEX)
			       >> DISPC_GFX_SIZE_GFXSIZEX_SHIFT);
	fbuf->fmt.height = 1 + ((gfx_size & DISPC_GFX_SIZE_GFXSIZEY)
				>> DISPC_GFX_SIZE_GFXSIZEY_SHIFT);
	gfx_position = dispc_reg_in(cam, DISPC_GFX_POSITION);
	cam->gfx_position_x = (gfx_position & DISPC_GFX_POSITION_GFXPOSX)
		>> DISPC_GFX_POSITION_GFXPOSX_SHIFT;
	cam->gfx_position_y = (gfx_position & DISPC_GFX_POSITION_GFXPOSY)
		>> DISPC_GFX_POSITION_GFXPOSY_SHIFT;
	cam->gfx_attributes = dispc_reg_in(cam, DISPC_GFX_ATTRIBUTES);
}

/* Return the default overlay cropping rectangle in crop given the image capture
 * size in cam->pix and the video display size in cam->fbuf.  The default 
 * cropping rectangle is the largest rectangle no larger than the capture size 
 * that will fit on the display.  The default cropping rectangle is centered in 
 * the captured image.  All dimensions and offsets are rounded down to even 
 * numbers.
 */
static void
omap24xxcam_default_crop_rect(struct omap24xxcam_device *cam,
			      struct v4l2_rect *crop)
{
	struct v4l2_pix_format *pix = &cam->pix;
	struct v4l2_framebuffer *fbuf = &cam->fbuf;

	DBG;

	crop->width = (pix->width < fbuf->fmt.width) ?
	    pix->width : fbuf->fmt.width;
	crop->height = (pix->height < fbuf->fmt.height) ?
	    pix->height : fbuf->fmt.height;
	crop->width &= ~1;
	crop->height &= ~1;
	crop->left = ((pix->width - crop->width) >> 1) & ~1;
	crop->top = ((pix->height - crop->height) >> 1) & ~1;
}

/* Given a new cropping rectangle in new_crop, adjust the cropping rectangle to 
 * the nearest supported configuration.  The image preview window cam->win will 
 * also be adjusted if necessary.  The preview window is adjusted such that the 
 * horizontal and vertical rescaling ratios stay constant.  If the preview 
 * window would fall outside the display boundaries, the cropping rectangle will 
 * also be adjusted to maintain the rescaling ratios.  If successful, cam->crop 
 * and cam->win are updated.
 * Returns zero if succesful, or -EINVAL if the requested cropping rectangle is 
 * impossible and cannot reasonably be adjusted.
 */
static int
omap24xxcam_new_crop_rect(struct omap24xxcam_device *cam,
			  const struct v4l2_rect *new_crop)
{
	struct v4l2_pix_format *pix = &cam->pix;
	struct v4l2_window *win = &cam->win;
	struct v4l2_framebuffer *fbuf = &cam->fbuf;
	struct v4l2_rect *crop = &cam->crop;
	struct v4l2_rect try_crop;
	unsigned long vresize, hresize;

	DBG;

	/* make a working copy of the new_crop rectangle */
	try_crop = *new_crop;

	/* adjust the cropping rectangle so it fits in the image */
	if (try_crop.left < 0) {
		try_crop.width += try_crop.left;
		try_crop.left = 0;
	}
	if (try_crop.top < 0) {
		try_crop.height += try_crop.top;
		try_crop.top = 0;
	}
	try_crop.width = (try_crop.width < pix->width) ?
	    try_crop.width : pix->width;
	try_crop.height = (try_crop.height < pix->height) ?
	    try_crop.height : pix->height;
	if (try_crop.left + try_crop.width > pix->width)
		try_crop.width = pix->width - try_crop.left;
	if (try_crop.top + try_crop.height > pix->height)
		try_crop.height = pix->height - try_crop.top;
	try_crop.width &= ~1;
	try_crop.height &= ~1;
	if (try_crop.width <= 0 || try_crop.height <= 0)
		return -EINVAL;

	if (crop->height != win->w.height) {
		/* If we're resizing vertically, we can't support a crop width 
		 * wider than 768 pixels.
		 */
		if (try_crop.width > 768)
			try_crop.width = 768;
	}
	/* vertical resizing */
	vresize = (1024 * crop->height) / win->w.height;
	if (vresize > 2048)
		vresize = 2048;
	else if (vresize == 0)
		vresize = 1;
	win->w.height = ((1024 * try_crop.height) / vresize) & ~1;
	if (win->w.height == 0)
		win->w.height = 2;
	if (win->w.height + win->w.top > fbuf->fmt.height) {
		/* We made the preview window extend below the bottom of the 
		 * display, so clip it to the display boundary and resize the 
		 * cropping height to maintain the vertical resizing ratio.
		 */
		win->w.height = (fbuf->fmt.height - win->w.top) & ~1;
		try_crop.height = ((vresize * win->w.height) / 1024) & ~1;
		if (try_crop.height == 0)
			try_crop.height = 2;
	}
	/* horizontal resizing */
	hresize = (1024 * crop->width) / win->w.width;
	if (hresize > 2048)
		hresize = 2048;
	else if (hresize == 0)
		hresize = 1;
	win->w.width = ((1024 * try_crop.width) / hresize) & ~1;
	if (win->w.width == 0)
		win->w.width = 2;
	if (win->w.width + win->w.left > fbuf->fmt.width) {
		/* We made the preview window extend past the right side of the  
		 * display, so clip it to the display boundary and resize the 
		 * cropping width to maintain the horizontal resizing ratio.
		 */
		win->w.width = (fbuf->fmt.width - win->w.left) & ~1;
		try_crop.width = ((hresize * win->w.width) / 1024) & ~1;
		if (try_crop.width == 0)
			try_crop.width = 2;
	}
	/* update our cropping rectangle and we're done */
	*crop = try_crop;
	return 0;
}

/* Given a new preview window in new_win, adjust the preview window to the 
 * nearest supported configuration.  The adjusted preview window parameters are 
 * returned in new_win.
 * Returns zero if succesful, or -EINVAL if the requested preview window is 
 * impossible and cannot reasonably be adjusted.
 */
static int
omap24xxcam_try_preview_window(struct omap24xxcam_device *cam,
			       struct v4l2_window *new_win)
{
	struct v4l2_framebuffer *fbuf = &cam->fbuf;
	struct v4l2_rect try_win;

	DBG;

	/* make a working copy of the new_win rectangle */
	try_win = new_win->w;

	/* adjust the preview window so it fits on the display by clipping any 
	 * offscreen areas
	 */
	if (try_win.left < 0) {
		try_win.width += try_win.left;
		try_win.left = 0;
	}
	if (try_win.top < 0) {
		try_win.height += try_win.top;
		try_win.top = 0;
	}
	try_win.width = (try_win.width < fbuf->fmt.width) ?
	    try_win.width : fbuf->fmt.width;
	try_win.height = (try_win.height < fbuf->fmt.height) ?
	    try_win.height : fbuf->fmt.height;
	if (try_win.left + try_win.width > fbuf->fmt.width)
		try_win.width = fbuf->fmt.width - try_win.left;
	if (try_win.top + try_win.height > fbuf->fmt.height)
		try_win.height = fbuf->fmt.height - try_win.top;
	try_win.width &= ~1;
	try_win.height &= ~1;
	if (try_win.width <= 0 || try_win.height <= 0)
		return -EINVAL;

	/* We now have a valid preview window, so go with it */
	new_win->w = try_win;
	new_win->field = V4L2_FIELD_NONE;
	return 0;
}

/* Given a new preview window in new_win, adjust the preview window to the 
 * nearest supported configuration.  The image cropping window cam->crop 
 * will also be adjusted if necessary.  Preference is given to keeping the 
 * preview window as close to the requested configuration as possible.  If 
 * successful, new_win, cam->win, and cam->crop are updated.
 * Returns zero if succesful, or -EINVAL if the requested preview window is 
 * impossible and cannot reasonably be adjusted.
 */
static int
omap24xxcam_new_preview_window(struct omap24xxcam_device *cam,
			       struct v4l2_window *new_win)
{
	struct v4l2_window *win = &cam->win;
	struct v4l2_rect *crop = &cam->crop;
	int err;

	DBG;

	err = omap24xxcam_try_preview_window(cam, new_win);
	if (err)
		return err;

	/* update our preview window */
	win->w = new_win->w;
	win->field = new_win->field;
	win->chromakey = new_win->chromakey;

	/* adjust the cropping window to allow for resizing limitations */
	if (crop->height / win->w.height >= 2) {
		/* The maximum vertical downsizing ratio is 2:1 */
		crop->height = win->w.height * 2;
	}
	if (crop->width / win->w.width >= 2) {
		/* The maximum horizontal downsizing ratio is 2:1 */
		crop->width = win->w.width * 2;
	}
	if (crop->width > 768) {
		/* The OMAP2420 vertical resizing line buffer is 768 pixels 
		 * wide.  If the cropped image is wider than 768 pixels then it 
		 * cannot be vertically resized.
		 */
		if (crop->height != win->w.height)
			crop->width = 768;
	}
	return 0;
}

/* Given a capture format in cam->pix, the cam->crop, cam->win, and cam->fbuf 
 * structures are initialized to default values.  cam->fbuf is initialized by 
 * reading the display controller registers for the graphics window.  cam->crop 
 * is initialized to the largest window size that will fit on the display.  The 
 * crop window is centered in the captured image.  cam->win is initialized to 
 * the same size as cam->crop and is centered on the display.
 * All sizes and offsets are constrained to be even numbers.  
 */
static void omap24xxcam_new_capture_format(struct omap24xxcam_device *cam)
{
	struct v4l2_rect *crop = &cam->crop;
	struct v4l2_window *win = &cam->win;
	struct v4l2_framebuffer *fbuf = &cam->fbuf;

	DBG;

	/* get the framebuffer parameters */
	omap24xxcam_g_fbuf(cam);

	/* crop defines the preview source window in the image capture 
	 * buffer
	 */
	omap24xxcam_default_crop_rect(cam, crop);

	/* win defines the preview target window on the display */
	win->w.width = crop->width;
	win->w.height = crop->height;
	win->w.left = ((fbuf->fmt.width - win->w.width) >> 1) & ~1;
	win->w.top = ((fbuf->fmt.height - win->w.height) >> 1) & ~1;
}

/* Program the camera interface xclk for the frequency cam->xclk based on 
 * the functional clock frequency cam->mclk.  If the specifed cam->xclk 
 * frequency is not possible based on the value of cam->mclk, then the 
 * closest xclk frequency lower than the specified xclk will be selected.  
 * The actual xclk frequency is returned in cam->xclk.  If cam->xclk is zero, 
 * then xclk is turned off (stable low value).
 */
static void omap24xxcam_set_xclk(struct omap24xxcam_device *cam)
{
	unsigned long divisor;

	DBG;

	if (cam->mclk == 0)
		cam->mclk = 96000000;	/* supply a default mclk */

	if (cam->xclk == 0) {
		cc_reg_out(cam, CC_CTRL_XCLK, CC_CTRL_XCLK_DIV_STABLE_LOW);
		return;
	}

	if (cam->xclk > cam->mclk)
		cam->xclk = cam->mclk;

	divisor = cam->mclk / cam->xclk;
	if (cam->xclk * divisor < cam->mclk)
		divisor += 1;
	if (divisor > 30)
		divisor = 30;
	cam->xclk = cam->mclk / divisor;
	if (divisor == 1)
		cc_reg_out(cam, CC_CTRL_XCLK, CC_CTRL_XCLK_DIV_BYPASS);
	else
		cc_reg_out(cam, CC_CTRL_XCLK, divisor);
}

/* Write the color space conversion coefficients to the display controller 
 * registers.  Each coefficient is a signed 11-bit integer in the range 
 * [-1024, 1023].  The matrix coefficients are:
 *	[ RY  RCr  RCb ]
 *	[ GY  GCr  GCb ]
 *	[ BY  BCr  BCb ]
 */
static void
omap24xxcam_set_colorconv(struct omap24xxcam_device *cam,
			  const short int mtx[3][3], int v)
{
	unsigned long ccreg;

	DBG;

	ccreg = (mtx[0][0] & 0x7ff) | ((mtx[0][1] & 0x7ff) << 16);
	dispc_reg_out(cam, DISPC_VID_CONV_COEF0(v), ccreg);
	ccreg = (mtx[0][2] & 0x7ff) | ((mtx[1][0] & 0x7ff) << 16);
	dispc_reg_out(cam, DISPC_VID_CONV_COEF1(v), ccreg);
	ccreg = (mtx[1][1] & 0x7ff) | ((mtx[1][2] & 0x7ff) << 16);
	dispc_reg_out(cam, DISPC_VID_CONV_COEF2(v), ccreg);
	ccreg = (mtx[2][0] & 0x7ff) | ((mtx[2][1] & 0x7ff) << 16);
	dispc_reg_out(cam, DISPC_VID_CONV_COEF3(v), ccreg);
	ccreg = mtx[2][2] & 0x7ff;
	dispc_reg_out(cam, DISPC_VID_CONV_COEF4(v), ccreg);
}

/* Write the horizontal and vertical resizing coefficients to the display 
 * controller registers.  Each coefficient is a signed 8-bit integer in the 
 * range [-128, 127] except for the middle coefficient (vc[1][i] and hc[3][i]) 
 * which is an unsigned 8-bit integer in the range [0, 255].  The first index of
 * the matrix is the coefficient number (0 to 2 vertical or 0 to 4 horizontal) 
 * and the second index is the phase (0 to 7).
 */
static void
omap24xxcam_set_resize(struct omap24xxcam_device *cam,
		       const short int vc[3][8], const short int hc[5][8],
		       int v)
{
	int i;
	unsigned long reg;

	DBG;

	for (i = 0; i < 8; i++) {
		reg = (hc[0][i] & 0xff) | ((hc[1][i] & 0xff) << 8)
		    | ((hc[2][i] & 0xff) << 16) | ((hc[3][i] & 0xff) << 24);
		dispc_reg_out(cam, DISPC_VID_FIR_COEF_H(v, i), reg);
		reg = (hc[4][i] & 0xff) | ((vc[0][i] & 0xff) << 8)
		    | ((vc[1][i] & 0xff) << 16) | ((vc[2][i] & 0xff) << 24);
		dispc_reg_out(cam, DISPC_VID_FIR_COEF_HV(v, i), reg);
	}
}

static void
omap24xxcam_configure_overlay(struct omap24xxcam_device *cam,
			      struct omap24xx_vid2_format *vid2_format)
{
	int v;
	struct v4l2_window *win = &cam->win;
	struct v4l2_rect *crop = &cam->crop;
	struct v4l2_pix_format *pix;
	struct v4l2_framebuffer *fbuf = &cam->fbuf;
	int vid_position_x, vid_position_y;
	unsigned long vid_position, vid_size, vid_picture_size;
	unsigned long vid_attributes;
	unsigned long firvinc, firhinc;
	/* color space conversion matrices */
	const static short int cc_bt601[3][3] = {
		{298, 409, 0},
		{298, -208, -100},
		{298, 0, 517}
	};
	const static short int cc_bt709[3][3] = {
		{298, 459, 0},
		{298, -137, -55},
		{298, 0, 541}
	};
	const static short int cc_bt601_full[3][3] = {
		{256, 351, 0},
		{256, -179, -86},
		{256, 0, 443}
	};
	/* vertical resizing matrix */
	const static short int vc[3][8] = {
		{0, 3, 12, 32, 0, 7, 5, 2},
		{128, 123, 111, 89, 64, 89, 111, 123},
		{0, 2, 5, 7, 64, 32, 12, 3}
	};
	/* horizontal resizing matrix */
	const static short int hc[5][8] = {
		{0, -1, -2, -5, 0, -2, -1, 0},
		{0, 13, 30, 51, -9, -11, -11, -8},
		{128, 124, 112, 95, 73, 95, 112, 124},
		{0, -8, -11, -11, 73, 51, 30, 13},
		{0, 0, -1, -2, -9, -5, -2, -1}
	};

	DBG;

	if (vid2_format) {
		pix = &vid2_format->pix;
		v = cam->vid2;
	} else {
		pix = &cam->pix;
		v = cam->vid1;
	}
	/* make sure the video overlay is disabled before we reconfigure it */
	omap24xxcam_disable_vlayer(cam, v);

	/* configure the video attributes register */
	vid_attributes = 0;
	switch (pix->pixelformat) {
	case V4L2_PIX_FMT_YUYV:
		vid_attributes |= DISPC_VID_ATTRIBUTES_VIDFORMAT_YUV2;
		vid_attributes |= DISPC_VID_ATTRIBUTES_VIDCOLORCONVENABLE;
		break;
	case V4L2_PIX_FMT_UYVY:
		vid_attributes |= DISPC_VID_ATTRIBUTES_VIDFORMAT_UYVY;
		vid_attributes |= DISPC_VID_ATTRIBUTES_VIDCOLORCONVENABLE;
		break;
	case V4L2_PIX_FMT_RGB565:
	default:
		vid_attributes |= DISPC_VID_ATTRIBUTES_VIDFORMAT_RGB16;
		vid_attributes |= ((cam->gfx_attributes &
				    DISPC_GFX_ATTRIBUTES_GFXREPLICATIONENABLE)
				   << 5);
		break;
	case V4L2_PIX_FMT_RGB565X:
		vid_attributes |= DISPC_VID_ATTRIBUTES_VIDFORMAT_RGB16;
		vid_attributes |= DISPC_VID_ATTRIBUTES_VIDENDIANNESS;
		vid_attributes |= ((cam->gfx_attributes &
				    DISPC_GFX_ATTRIBUTES_GFXREPLICATIONENABLE)
				   << 5);
		break;
	}
	if (cam->gfx_attributes & DISPC_GFX_ATTRIBUTES_GFXCHANNELOUT)
		vid_attributes |= DISPC_VID_ATTRIBUTES_VIDCHANNELOUT;
	vid_attributes |= ((cam->gfx_attributes
			    & DISPC_GFX_ATTRIBUTES_GFXBURSTSIZE) << 8);
	switch (pix->colorspace) {
	case V4L2_COLORSPACE_SMPTE170M:
	case V4L2_COLORSPACE_SMPTE240M:
	case V4L2_COLORSPACE_BT878:
	case V4L2_COLORSPACE_470_SYSTEM_M:
	case V4L2_COLORSPACE_470_SYSTEM_BG:
		/* luma (Y) range lower limit is 16, BT.601 standard */
		omap24xxcam_set_colorconv(cam, cc_bt601, v);
		break;
	case V4L2_COLORSPACE_REC709:
		/* luma (Y) range lower limit is 16, BT.709 standard */
		omap24xxcam_set_colorconv(cam, cc_bt709, v);
		break;
	case V4L2_COLORSPACE_JPEG:
	case V4L2_COLORSPACE_SRGB:
		/* full luma (Y) range, assume BT.601 standard */
		vid_attributes |= DISPC_VID_ATTRIBUTES_VIDFULLRANGE;
		omap24xxcam_set_colorconv(cam, cc_bt601_full, v);
		break;
	}
	if (!vid2_format) {	/* use win and crop if it is for overlay */
		if (win->w.width != crop->width) {
			vid_attributes |=
			    DISPC_VID_ATTRIBUTES_VIDRESIZEENABLE_HRESIZE;
			if (win->w.width < crop->width)
				vid_attributes |=
				    DISPC_VID_ATTRIBUTES_VIDHRESIZECONF;
		}
		if (win->w.height != crop->height) {
			vid_attributes |=
			    DISPC_VID_ATTRIBUTES_VIDRESIZEENABLE_VRESIZE;
			if (win->w.height < crop->height)
				vid_attributes |=
				    DISPC_VID_ATTRIBUTES_VIDVRESIZECONF;
		}
	}
	dispc_reg_out(cam, DISPC_VID_ATTRIBUTES(v), vid_attributes);

	/* configure transparency color key */
	if (fbuf->flags & V4L2_FBUF_FLAG_CHROMAKEY) {
		/* enable chromakey */
		if (vid_attributes & DISPC_VID_ATTRIBUTES_VIDCHANNELOUT) {
			/* digital output channel */
			dispc_reg_out(cam, DISPC_TRANS_COLOR1, win->chromakey);
			dispc_reg_merge(cam, DISPC_CONFIG,
					DISPC_CONFIG_TCKDIGENABLE,
					DISPC_CONFIG_TCKDIGSELECTION
					| DISPC_CONFIG_TCKDIGENABLE);
		} else {
			/* LCD */
			dispc_reg_out(cam, DISPC_TRANS_COLOR0, win->chromakey);
			dispc_reg_merge(cam, DISPC_CONFIG,
					DISPC_CONFIG_TCKLCDENABLE,
					DISPC_CONFIG_TCKLCDSELECTION
					| DISPC_CONFIG_TCKLCDENABLE);
		}
	} else {
		/* disable chromakey */
		if (vid_attributes & DISPC_VID_ATTRIBUTES_VIDCHANNELOUT) {
			/* digital output channel */
			dispc_reg_merge(cam, DISPC_CONFIG, 0,
					DISPC_CONFIG_TCKDIGSELECTION
					| DISPC_CONFIG_TCKDIGENABLE);
		} else {
			/* LCD */
			dispc_reg_merge(cam, DISPC_CONFIG, 0,
					DISPC_CONFIG_TCKLCDSELECTION
					| DISPC_CONFIG_TCKLCDENABLE);
		}
	}

	/* initialize the resizing filter */
	omap24xxcam_set_resize(cam, vc, hc, v);

	if (!vid2_format) {	/* use win and crop if it is for overlay */
		dispc_reg_out(cam, DISPC_VID_ACCU0(v), 0);
		dispc_reg_out(cam, DISPC_VID_ACCU1(v), 0);
		firhinc = (1024 * (crop->width - 1)) / (win->w.width - 1);
		if (firhinc < 1)
			firhinc = 1;
		else if (firhinc > 2047)
			firhinc = 2047;
		firvinc = (1024 * (crop->height - 1)) / (win->w.height - 1);
		if (firvinc < 1)
			firvinc = 1;
		else if (firvinc > 2047)
			firvinc = 2047;
		dispc_reg_out(cam, DISPC_VID_FIR(v), firhinc | (firvinc << 16));

		/* configure the target window on the display */
		vid_position_x = cam->gfx_position_x + win->w.left;
		vid_position_y = cam->gfx_position_y + win->w.top;
		vid_size =
		    (((win->w.width - 1) << DISPC_VID_SIZE_VIDSIZEX_SHIFT)
		     & DISPC_VID_SIZE_VIDSIZEX)
		    | (((win->w.height - 1) << DISPC_VID_SIZE_VIDSIZEY_SHIFT)
		       & DISPC_VID_SIZE_VIDSIZEY);

		/* configure the source window in the framebuffer */
		vid_picture_size =
		    (((crop->width -
		       1) << DISPC_VID_PICTURE_SIZE_VIDORGSIZEX_SHIFT)
		     & DISPC_VID_PICTURE_SIZE_VIDORGSIZEX)
		    |
		    (((crop->height -
		       1) << DISPC_VID_PICTURE_SIZE_VIDORGSIZEY_SHIFT)
		     & DISPC_VID_PICTURE_SIZE_VIDORGSIZEY);
		dispc_reg_out(cam, DISPC_VID_ROW_INC(v),
			      1 + pix->bytesperline - crop->width * 2);
	} else {		/* video 2 layer configuration */
		struct v4l2_framebuffer *fbuf = &cam->fbuf;
		int row_inc;
		/* in video2 layer we won't be down or up scaling */
		dispc_reg_out(cam, DISPC_VID_FIR(v), 0x400 | (0x400 << 16));
		if (pix->width + vid2_format->left > fbuf->fmt.width) {
			vid_position_x = fbuf->fmt.width < pix->width ? 0 :
			    cam->gfx_position_x + fbuf->fmt.width - pix->width;
		} else {
			vid_position_x = cam->gfx_position_x +
			    vid2_format->left;

		}

		if (pix->height + vid2_format->top > fbuf->fmt.height) {
			vid_position_y = fbuf->fmt.height < pix->height ? 0 :
			    cam->gfx_position_y +
			    fbuf->fmt.height - pix->height;
		} else {
			vid_position_y = cam->gfx_position_y + vid2_format->top;
		}

		vid_size = ((((pix->width > fbuf->fmt.width ? fbuf->fmt.width
			       : pix->width) - 1)
			     << DISPC_VID_SIZE_VIDSIZEX_SHIFT)
			    & DISPC_VID_SIZE_VIDSIZEX)
		    | ((((pix->height > fbuf->fmt.height ? fbuf->fmt.height
			  : pix->height) - 1) << DISPC_VID_SIZE_VIDSIZEY_SHIFT)
		       & DISPC_VID_SIZE_VIDSIZEY);
		vid_picture_size = vid_size;
		row_inc = ((pix->width -
			    ((vid_size >> DISPC_VID_SIZE_VIDSIZEX_SHIFT)
			     & DISPC_VID_SIZE_VIDSIZEX)) * 2)
		    - 1;
		/* we are subtracting 1 instead of adding because vid_size 
		 ** is 1 less than the pix width 
		 */
		dispc_reg_out(cam, DISPC_VID_ROW_INC(v), row_inc);
	}
	vid_position = ((vid_position_x << DISPC_VID_POSITION_VIDPOSX_SHIFT)
			& DISPC_VID_POSITION_VIDPOSX)
	    | ((vid_position_y << DISPC_VID_POSITION_VIDPOSY_SHIFT)
	       & DISPC_VID_POSITION_VIDPOSY);
	dispc_reg_out(cam, DISPC_VID_POSITION(v), vid_position);
	dispc_reg_out(cam, DISPC_VID_SIZE(v), vid_size);
	dispc_reg_out(cam, DISPC_VID_PICTURE_SIZE(v), vid_picture_size);
	dispc_reg_out(cam, DISPC_VID_PIXEL_INC(v), 1);

	if (vid2_format) {
		unsigned long vid2_base_phys;
		vid2_base_phys = cam->video2_base_phys;
		//+ pix->bytesperline*crop->top + crop->left*2;
		dispc_reg_out(cam, DISPC_VID_BA0(v), vid2_base_phys);
		dispc_reg_out(cam, DISPC_VID_BA1(v), vid2_base_phys);
		vid_attributes = dispc_reg_merge(cam, DISPC_VID_ATTRIBUTES(v),
						 DISPC_VID_ATTRIBUTES_ENABLE,
						 DISPC_VID_ATTRIBUTES_ENABLE);
		if (vid_attributes & DISPC_VID_ATTRIBUTES_VIDCHANNELOUT) {
			/* digital output */
			dispc_reg_merge(cam, DISPC_CONTROL,
					DISPC_CONTROL_GODIGITAL,
					DISPC_CONTROL_GODIGITAL);
		} else {
			/* LCD */
			dispc_reg_merge(cam, DISPC_CONTROL, DISPC_CONTROL_GOLCD,
					DISPC_CONTROL_GOLCD);
		}
	}
}

/* DMA completion routine for the scatter-gather DMA fragments. */
static void
omap24xxcam_sg_dma_callback(struct omap24xxcam_device *cam, unsigned long csr,
			    void *arg)
{
	int sgslot = (int)arg;
	struct sgdma_state *sgdma;
	const unsigned long csr_error = CAMDMA_CSR_MISALIGNED_ERR
	    | CAMDMA_CSR_SUPERVISOR_ERR | CAMDMA_CSR_SECURE_ERR
	    | CAMDMA_CSR_TRANS_ERR | CAMDMA_CSR_DROP;

	DBG;

	spin_lock(&cam->sg_lock);

	/* We got an interrupt, we can remove the timer */
	del_timer(&cam->reset_timer);

	sgdma = cam->sgdma + sgslot;
	if (!sgdma->queued_sglist) {
		spin_unlock(&cam->sg_lock);
		printk(KERN_DEBUG CAM_NAME
		       ": sgdma completed when none queued!\n");
		return;
	}

	sgdma->csr |= csr;
	if (!--sgdma->queued_sglist) {
		/* Queue for this sglist is empty, so check to see if we're 
		 * done.
		 */
		if ((sgdma->next_sglist == sgdma->sglen)
		    || (sgdma->csr & csr_error)) {
			dma_callback_t callback = sgdma->callback;
			void *arg = sgdma->arg;
			unsigned long sg_csr = sgdma->csr;
			/* All done with this sglist */
			cam->free_sgdma++;
			if (callback) {
				spin_unlock(&cam->sg_lock);
				(*callback) (cam, sg_csr, arg);
				return;
			}
		}
	}

	spin_unlock(&cam->sg_lock);
}

/* Process the scatter-gather DMA queue by starting queued transfers. */
static void omap24xxcam_sg_dma_process(struct omap24xxcam_device *cam)
{
	unsigned long irqflags;
	int queued_sgdma, sgslot;
	struct sgdma_state *sgdma;
	const unsigned long csr_error = CAMDMA_CSR_MISALIGNED_ERR
	    | CAMDMA_CSR_SUPERVISOR_ERR | CAMDMA_CSR_SECURE_ERR
	    | CAMDMA_CSR_TRANS_ERR | CAMDMA_CSR_DROP;

	DBG;

	spin_lock_irqsave(&cam->sg_lock, irqflags);

	queued_sgdma = NUM_SG_DMA - cam->free_sgdma;
	sgslot = (cam->next_sgdma + cam->free_sgdma) % NUM_SG_DMA;
	while (queued_sgdma > 0) {
		sgdma = cam->sgdma + sgslot;
		while ((sgdma->next_sglist < sgdma->sglen) &&
		       !(sgdma->csr & csr_error)) {
			const struct scatterlist *sglist;
			unsigned int len;

			sglist = sgdma->sglist + sgdma->next_sglist;
			/* try to start the next DMA transfer */
			if ( sgdma->next_sglist + 1 == sgdma->sglen ) {
				/* 
				 *  On the last sg, we handle the case where
				 *  cam->pix.sizeimage % PAGE_ALIGN != 0
				 */
				len = cam->pix.sizeimage - sgdma->bytes_read;
			} else {
				len = sg_dma_len(sglist);
			}

			if (omap24xxcam_dma_start(cam, sg_dma_address(sglist),
						  len,
						  omap24xxcam_sg_dma_callback,
						  (void *)sgslot)) {
				/* DMA start failed */
				spin_unlock_irqrestore(&cam->sg_lock, irqflags);
				return;
			} else {
				unsigned long expires;
				/* DMA start was successful */
				sgdma->next_sglist++;
				sgdma->bytes_read += len;
				sgdma->queued_sglist++;

				/* We start the reset timer */
				expires = jiffies + HZ;
				mod_timer(&cam->reset_timer, expires);
			}
		}
		queued_sgdma--;
		sgslot = (sgslot + 1) % NUM_SG_DMA;
	}

	spin_unlock_irqrestore(&cam->sg_lock, irqflags);
	DBG_END;
}

/* Queue a scatter-gather DMA transfer from the camera to memory.
 * Returns zero if the transfer was successfully queued, or 
 * non-zero if all of the scatter-gather slots are already in use.
 */
static int
omap24xxcam_sg_dma_queue(struct omap24xxcam_device *cam,
			 const struct scatterlist *sglist, int sglen,
			 dma_callback_t callback, void *arg)
{
	unsigned long irqflags;
	struct sgdma_state *sgdma;

	DBG;

	if ((sglen < 0) || ((sglen > 0) & !sglist))
		return -EINVAL;

	spin_lock_irqsave(&cam->sg_lock, irqflags);

	if (!cam->free_sgdma) {
		spin_unlock_irqrestore(&cam->sg_lock, irqflags);
		return -EBUSY;
	}

	sgdma = cam->sgdma + cam->next_sgdma;

	sgdma->sglist = sglist;
	sgdma->sglen = sglen;
	sgdma->next_sglist = 0;
	sgdma->bytes_read = 0;
	sgdma->queued_sglist = 0;
	sgdma->csr = 0;
	sgdma->callback = callback;
	sgdma->arg = arg;

	cam->next_sgdma = (cam->next_sgdma + 1) % NUM_SG_DMA;
	cam->free_sgdma--;

	spin_unlock_irqrestore(&cam->sg_lock, irqflags);

	omap24xxcam_sg_dma_process(cam);

	return 0;
}

/* Abort all chained DMA transfers.  After all transfers have been aborted and 
 * the DMA controller is idle, the completion routines for any aborted transfers
 * will be called in sequence.  The DMA controller may not be idle after this 
 * routine completes, because the completion routines might start new transfers.
 */
static void
omap24xxcam_dma_abort(struct omap24xxcam_device *cam, unsigned long csr)
{
	unsigned long irqflags;
	int dmach, i, free_dmach;
	dma_callback_t callback;
	void *arg;

	DBG;

	spin_lock_irqsave(&cam->dma_lock, irqflags);

	/* stop any DMA transfers in progress */
	dmach = (cam->next_dmach + cam->free_dmach) % NUM_CAMDMA_CHANNELS;
	for (i = 0; i < NUM_CAMDMA_CHANNELS; i++) {
		omap24xxcam_dma_abort_ch(cam, dmach);
		dmach = (dmach + 1) % NUM_CAMDMA_CHANNELS;
	}

	/* We have to be careful here because the callback routine might start 
	 * a new DMA transfer, and we only want to abort transfers that were 
	 * started before this routine was called.
	 */
	free_dmach = cam->free_dmach;
	while ((cam->free_dmach < NUM_CAMDMA_CHANNELS) &&
	       (free_dmach < NUM_CAMDMA_CHANNELS)) {
		dmach = (cam->next_dmach + cam->free_dmach)
		    % NUM_CAMDMA_CHANNELS;
		callback = cam->camdma[dmach].callback;
		arg = cam->camdma[dmach].arg;
		cam->free_dmach++;
		free_dmach++;
		if (callback) {
			/* leave interrupts disabled during callback */
			spin_unlock(&cam->dma_lock);
			(*callback) (cam, csr, arg);
			spin_lock(&cam->dma_lock);
		}
	}

	spin_unlock_irqrestore(&cam->dma_lock, irqflags);
}

/* Abort all chained DMA transfers.  After all transfers have been aborted and 
 * the DMA controller is idle, the completion routines for any aborted transfers
 * will be called in sequence.  If the completion routines attempt to start a 
 * new DMA transfer it will fail, so the DMA controller will be idle after this 
 * routine completes.
 */
static void
omap24xxcam_dma_stop(struct omap24xxcam_device *cam, unsigned long csr)
{
	unsigned long irqflags;

	DBG;

	spin_lock_irqsave(&cam->dma_lock, irqflags);
	cam->dma_stop++;
	spin_unlock_irqrestore(&cam->dma_lock, irqflags);
	omap24xxcam_dma_abort(cam, csr);
	spin_lock_irqsave(&cam->dma_lock, irqflags);
	cam->dma_stop--;
	spin_unlock_irqrestore(&cam->dma_lock, irqflags);
}

/* Sync scatter-gather DMA by aborting any DMA transfers currently in progress.
 * Any queued scatter-gather DMA transactions that have not yet been started 
 * will remain queued.  The DMA controller will be idle after this routine 
 * completes.  When the scatter-gather queue is restarted, the next 
 * scatter-gather DMA transfer will begin at the start of a new transaction.
 */
static void
omap24xxcam_sg_dma_sync(struct omap24xxcam_device *cam, unsigned long csr)
{
	unsigned long irqflags;
	int sgslot;
	struct sgdma_state *sgdma;

	DBG;

	/* stop any DMA transfers in progress */
	omap24xxcam_dma_stop(cam, csr);

	spin_lock_irqsave(&cam->sg_lock, irqflags);

	if (cam->free_sgdma < NUM_SG_DMA) {
		sgslot = (cam->next_sgdma + cam->free_sgdma) % NUM_SG_DMA;
		sgdma = cam->sgdma + sgslot;
		if (sgdma->next_sglist != 0) {
			/* This DMA transfer was in progress, so abort it. */
			dma_callback_t callback = sgdma->callback;
			void *arg = sgdma->arg;
			cam->free_sgdma++;
			if (callback) {
				/* leave interrupts masked */
				spin_unlock(&cam->sg_lock);
				(*callback) (cam, csr, arg);
				spin_lock(&cam->sg_lock);
			}
		}
	}

	spin_unlock_irqrestore(&cam->sg_lock, irqflags);
}

/* This routine is not needed at the moment, but we'll keep it around just in 
 * case.
 */
#if 0
/* Stop the DMA controller, aborting any DMA transfers in progress.  Any queued 
 * scatter-gather DMA transactions are cancelled.  The DMA controller will be 
 * idle after this routine completes, and the scatter-gather DMA queue will 
 * be empty.  csr is the completion status passed to the completion routines 
 * for any DMA transfers that are aborted.  It should have at least one error 
 * bit set.
 */
static void
omap24xxcam_sg_dma_stop(struct omap24xxcam_device *cam, unsigned long csr)
{
	unsigned long irqflags;
	int queued_sgdma, sgslot;
	struct sgdma_state *sgdma;
	dma_callback_t callback;
	void *arg;

	/* stop any DMA transfers in progress */
	omap24xxcam_dma_stop(cam, csr);

	spin_lock_irqsave(&cam->sg_lock, irqflags);

	queued_sgdma = NUM_SG_DMA - cam->free_sgdma;
	sgslot = (cam->next_sgdma + cam->free_sgdma) % NUM_SG_DMA;
	while (queued_sgdma > 0) {
		sgdma = cam->sgdma + sgslot;
		callback = sgdma->callback;
		arg = sgdma->arg;
		cam->free_sgdma++;
		sgdma->queued_sglist = 0;
		if (callback) {
			/* leave interrupts masked */
			spin_unlock(&cam->sg_lock);
			(*callback) (cam, csr, arg);
			spin_lock(&cam->sg_lock);
		}
		queued_sgdma--;
		sgslot = (sgslot + 1) % NUM_SG_DMA;
	}

	spin_unlock_irqrestore(&cam->sg_lock, irqflags);
}
#endif				/* if 0 */

/* Register a routine to be called once immediately after a DMA transfer is 
 * started.  The purpose of this is to allow the camera interface to be 
 * started only after a DMA transaction has been queued in order to avoid 
 * DMA overruns.  The registered callback routine will only be called one 
 * time and then discarded.  Only one callback routine may be registered at a 
 * time.
 * Returns zero if successful, or a non-zero error code if a different callback 
 * routine has already been registered.
 */
static int
omap24xxcam_dma_notify(struct omap24xxcam_device *cam,
		       void (*callback) (struct omap24xxcam_device * cam))
{
	unsigned long irqflags;

	DBG;

	spin_lock_irqsave(&cam->dma_lock, irqflags);

	if (cam->dma_notify && (cam->dma_notify != callback)) {
		spin_unlock_irqrestore(&cam->dma_lock, irqflags);
		return -EBUSY;
	}

	cam->dma_notify = callback;

	spin_unlock_irqrestore(&cam->dma_lock, irqflags);

	return 0;
}

/* Camera DMA interrupt service routine. */
static void omap24xxcam_dma_isr(struct omap24xxcam_device *cam)
{
	int dmach;
	dma_callback_t callback;
	void *arg;
	unsigned long csr;
	const unsigned long csr_error = CAMDMA_CSR_MISALIGNED_ERR
	    | CAMDMA_CSR_SUPERVISOR_ERR | CAMDMA_CSR_SECURE_ERR
	    | CAMDMA_CSR_TRANS_ERR | CAMDMA_CSR_DROP;

	DBG;

	spin_lock(&cam->dma_lock);

	if (cam->free_dmach == NUM_CAMDMA_CHANNELS) {
		/* A camera DMA interrupt occurred while all channels are idle, 
		 * so we'll acknowledge the interrupt in the IRQSTATUS register 
		 * and exit.
		 */
		omap24xxcam_dma_ack_all(cam);
		spin_unlock(&cam->dma_lock);
		return;
	}

	while (cam->free_dmach < NUM_CAMDMA_CHANNELS) {
		dmach = (cam->next_dmach + cam->free_dmach)
		    % NUM_CAMDMA_CHANNELS;
		if (omap24xxcam_dma_running(cam, dmach)) {
			/* This buffer hasn't finished yet, so we're done. */
			break;
		}
		csr = omap24xxcam_dma_ack_ch(cam, dmach);
		if (csr & csr_error) {
			/* A DMA error occurred, so stop all DMA transfers in 
			 * progress.
			 */
			spin_unlock(&cam->dma_lock);
			omap24xxcam_dma_stop(cam, csr);
			return;
		} else {
			callback = cam->camdma[dmach].callback;
			arg = cam->camdma[dmach].arg;
			cam->free_dmach++;
			if (callback) {
				spin_unlock(&cam->dma_lock);
				(*callback) (cam, csr, arg);
				spin_lock(&cam->dma_lock);
			}
		}
	}

	spin_unlock(&cam->dma_lock);

	omap24xxcam_sg_dma_process(cam);
}

/* Initialize the camera DMA driver. */
static void omap24xxcam_dma_init(struct omap24xxcam_device *cam)
{
	int ch, sg;

	DBG;

	/* group all channels on DMA IRQ0 and unmask irq */
	omap24xxcam_dma_unmask_irq0(cam);

	spin_lock_init(&cam->dma_lock);
	cam->free_dmach = NUM_CAMDMA_CHANNELS;
	cam->next_dmach = 0;
	for (ch = 0; ch < NUM_CAMDMA_CHANNELS; ch++) {
		cam->camdma[ch].callback = NULL;
		cam->camdma[ch].arg = NULL;
	}

	spin_lock_init(&cam->sg_lock);
	cam->free_sgdma = NUM_SG_DMA;
	cam->next_sgdma = 0;
	for (sg = 0; sg < NUM_SG_DMA; sg++) {
		cam->sgdma[sg].sglen = 0;
		cam->sgdma[sg].next_sglist = 0;
		cam->sgdma[sg].bytes_read = 0;
		cam->sgdma[sg].queued_sglist = 0;
		cam->sgdma[sg].csr = 0;
		cam->sgdma[sg].callback = NULL;
		cam->sgdma[sg].arg = NULL;
	}
}

/* -------------------------------------------------------------------------- */

/* Callback routine for overlay DMA completion.  We just start another DMA 
 * transfer unless overlay has been turned off.
 */
static void
omap24xxcam_overlay_callback(struct omap24xxcam_device *cam, unsigned long csr,
			     void *arg)
{
	int err;
	unsigned long irqflags;
	const unsigned long csr_error = CAMDMA_CSR_MISALIGNED_ERR
	    | CAMDMA_CSR_SUPERVISOR_ERR | CAMDMA_CSR_SECURE_ERR
	    | CAMDMA_CSR_TRANS_ERR | CAMDMA_CSR_DROP;

	DBG;

	spin_lock_irqsave(&cam->overlay_lock, irqflags);

	if (cam->overlay_cnt > 0)
		--cam->overlay_cnt;

	if (!cam->previewing || (csr & csr_error)) {
		printk(KERN_INFO CAM_NAME
		       ": overlay_callback error. csr = 0x%08lx\n", csr);
		spin_unlock_irqrestore(&cam->overlay_lock, irqflags);
		return;
	}

	sg_dma_address(&cam->overlay_sglist) = cam->overlay_base_phys;
	sg_dma_len(&cam->overlay_sglist) = cam->pix.sizeimage;

	while (cam->overlay_cnt < 2) {
		err = omap24xxcam_sg_dma_queue(cam, &cam->overlay_sglist, 1,
					       omap24xxcam_overlay_callback,
					       NULL);
		if (err)
			break;
		++cam->overlay_cnt;
	}

	spin_unlock_irqrestore(&cam->overlay_lock, irqflags);
}

/* Begin DMA'ing into the camera overlay framebuffer.  We queue up two DMA 
 * transfers so that all frames will be captured.
 */
static void omap24xxcam_start_overlay_dma(struct omap24xxcam_device *cam)
{
	int err;
	unsigned long irqflags;

	DBG;

	if (!cam->previewing)
		return;

	if (cam->pix.sizeimage > cam->overlay_size)
		return;

	spin_lock_irqsave(&cam->overlay_lock, irqflags);

	sg_dma_address(&cam->overlay_sglist) = cam->overlay_base_phys;
	sg_dma_len(&cam->overlay_sglist) = cam->pix.sizeimage;

	while (cam->overlay_cnt < 2) {
		err = omap24xxcam_sg_dma_queue(cam, &cam->overlay_sglist, 1,
					       omap24xxcam_overlay_callback,
					       NULL);
		if (err)
			break;
		++cam->overlay_cnt;
	}

	spin_unlock_irqrestore(&cam->overlay_lock, irqflags);
}

/* Enable the camera core interface. */
static void omap24xxcam_cc_enable(struct omap24xxcam_device *cam)
{
	DBG;
	/* Get the CC_CTRL register value for the current capture format. */
	omap24xxcam_sensor_cc_ctrl(cam);

	/* program the camera interface DMA packet size */
	cc_reg_out(cam, CC_CTRL_DMA, CC_CTRL_DMA_EN | (DMA_THRESHOLD / 4 - 1));

	/* enable camera core error interrupts */
	cc_reg_out(cam, CC_IRQENABLE, CC_IRQENABLE_FW_ERR_IRQ
		   | CC_IRQENABLE_FSC_ERR_IRQ | CC_IRQENABLE_SSC_ERR_IRQ
		   | CC_IRQENABLE_FIFO_OF_IRQ /*| CC_IRQENABLE_FIFO_FULL_IRQ */
		   );

	/* enable the camera interface */
	cc_reg_out(cam, CC_CTRL, cam->cc_ctrl | CC_CTRL_CC_EN);
}

/* Error recovery for DMA errors and camera interface errors. */
static void omap24xxcam_error_handler(struct omap24xxcam_device *cam)
{
	DBG;
	/* Get the CC_CTRL register value for the current capture format. */
	omap24xxcam_sensor_cc_ctrl(cam);

	/* Disable and reset the camera interface. */
	cc_reg_out(cam, CC_CTRL,
		   cam->cc_ctrl & ~(CC_CTRL_CC_EN | CC_CTRL_CC_FRAME_TRIG));
	cc_reg_out(cam, CC_CTRL, (cam->cc_ctrl | CC_CTRL_CC_RST)
		   & ~(CC_CTRL_CC_EN | CC_CTRL_CC_FRAME_TRIG));

	/* Stop the DMA controller and frame sync scatter-gather DMA. */
	omap24xxcam_sg_dma_sync(cam, CAMDMA_CSR_TRANS_ERR);

	/* Reset and re-initialize the entire camera subsystem.
	 * Resetting the camera FIFO via the CC_RST bit in the CC_CTRL 
	 * register is supposed to be sufficient to recover from a 
	 * camera interface error, but it doesn't seem to be enough.  If 
	 * we only do that then subsequent image captures are out of sync 
	 * by either one or two times DMA_THRESHOLD bytes.  Resetting and 
	 * re-initializing the entire camera subsystem prevents the problem 
	 * with frame synchronization.  Calling cam_init() from an ISR is
	 * undesirable since it waits an indeterminate amount of time for the 
	 * camera subsystem reset to complete.  If we ever figure out exactly 
	 * what needs to be reset to recover from a camera interface error, 
	 * maybe we can replace this global reset with something less drastic.
	 */
	cam_init(cam);
	omap24xxcam_set_xclk(cam);
	/* group all channels on DMA IRQ0 and unmask irq */
	omap24xxcam_dma_unmask_irq0(cam);

	if (cam->previewing || cam->streaming) {
		/* Preview or capture is in progress, so we need to register 
		 * our routine to restart the camera interface the next time a 
		 * DMA transfer is queued.
		 */
		omap24xxcam_dma_notify(cam, omap24xxcam_cc_enable);
	}

	/* Restart overlay DMA if preview is enabled. */
	omap24xxcam_start_overlay_dma(cam);

	/* Restart the scatter-gather DMA queue. */
	omap24xxcam_sg_dma_process(cam);
}

/* Interrupt service routine for camera core interrupts. */
static void omap24xxcam_cc_isr(struct omap24xxcam_device *cam)
{
	unsigned long cc_irqstatus;
	const unsigned long cc_irqstatus_err = CC_IRQSTATUS_FW_ERR_IRQ
		| CC_IRQSTATUS_FSC_ERR_IRQ | CC_IRQSTATUS_SSC_ERR_IRQ
		| CC_IRQSTATUS_FIFO_OF_IRQ;

	DBG;

	cc_irqstatus = cc_reg_in(cam, CC_IRQSTATUS);
	cc_reg_out(cam, CC_IRQSTATUS, cc_irqstatus);

	if (cc_irqstatus & cc_irqstatus_err)
		omap24xxcam_error_handler(cam);
}

static irqreturn_t omap24xxcam_isr(int irq, void *arg, struct pt_regs *regs)
{
	struct omap24xxcam_device *cam = (struct omap24xxcam_device *)arg;
	unsigned long irqstatus;
	unsigned int irqhandled = 0;

	DBG;

	irqstatus = cam_reg_in(cam, CAM_IRQSTATUS);

	if (irqstatus &
	    (CAM_IRQSTATUS_DMA_IRQ2 | CAM_IRQSTATUS_DMA_IRQ1
	     | CAM_IRQSTATUS_DMA_IRQ0)) {
		omap24xxcam_dma_isr(cam);
		irqhandled = 1;
	}
	if (irqstatus & CAM_IRQSTATUS_CC_IRQ) {
		omap24xxcam_cc_isr(cam);
		irqhandled = 1;
	}
	if (irqstatus & CAM_IRQSTATUS_MMU_IRQ)
		printk(KERN_ERR CAM_NAME ": Unhandled camera MMU interrupt!\n");

	return IRQ_RETVAL(irqhandled);
}

static void omap24xxcam_adjust_xclk(struct omap24xxcam_device *cam)
{
	unsigned long divisor;

	DBG;

	if (cam->xclk > cam->mclk)
		cam->xclk = cam->mclk;
	divisor = cam->mclk / cam->xclk;
	if (cam->xclk * divisor < cam->mclk)
		divisor += 1;
	if (divisor > 30)
		divisor = 30;

	cam->xclk = cam->mclk / divisor;
}

/* -------------------------------------------------------------------------- */

/* This routine is called from interrupt context when a scatter-gather DMA 
 * transfer of a videobuf_buffer completes.
 */
static void
omap24xxcam_vbq_complete(struct omap24xxcam_device *cam, unsigned long csr,
			 void *arg)
{
	struct videobuf_buffer *vb = (struct videobuf_buffer *)arg;
	const unsigned long csr_error = CAMDMA_CSR_MISALIGNED_ERR
	    | CAMDMA_CSR_SUPERVISOR_ERR | CAMDMA_CSR_SECURE_ERR
	    | CAMDMA_CSR_TRANS_ERR | CAMDMA_CSR_DROP;

	DBG;

	spin_lock(&cam->vbq_lock);

	do_gettimeofday(&vb->ts);
	vb->field_count = cam->field_count;
	cam->field_count += 2;
	if (csr & csr_error) {
		vb->state = STATE_ERROR;
		omap24xxcam_error_handler(cam);
	} else
		vb->state = STATE_DONE;
	wake_up(&vb->done);
	spin_unlock(&cam->vbq_lock);
}

static void
omap24xxcam_vbq_release(struct videobuf_queue *vbq, struct videobuf_buffer *vb)
{
	DBG;

	videobuf_waiton(vb, 0, 0);	
	videobuf_dma_unmap(vbq, &vb->dma);
	videobuf_dma_free(&vb->dma);

	vb->state = STATE_NEEDS_INIT;
}

/* Limit the number of available kernel image capture buffers based on the 
 * number requested, the currently selected image size, and the maximum 
 * amount of memory permitted for kernel capture buffers.
 */
static int
omap24xxcam_vbq_setup(struct videobuf_queue *vbq, unsigned int *cnt,
		      unsigned int *size)
{
	struct omap24xxcam_fh *fh = vbq->priv_data;
	struct omap24xxcam_device *cam = fh->cam;

	DBG;

	if (*cnt <= 0)
		*cnt = VIDEO_MAX_FRAME;	/* supply a default number of buffers */

	if (*cnt > VIDEO_MAX_FRAME)
		*cnt = VIDEO_MAX_FRAME;

	spin_lock(&cam->img_lock);

	*size = cam->pix.sizeimage;

	spin_unlock(&cam->img_lock);

	while (*size * *cnt > capture_mem)
		(*cnt)--;

	return 0;
}

static int
omap24xxcam_vbq_prepare(struct videobuf_queue *vbq, struct videobuf_buffer *vb,
			enum v4l2_field field)
{
	struct omap24xxcam_fh *fh = vbq->priv_data;
	struct omap24xxcam_device *cam = fh->cam;
	int err = 0;

	DBG;

	spin_lock(&cam->img_lock);

	if (vb->baddr) {
		/* This is a userspace buffer. */
		if (cam->pix.sizeimage > vb->bsize) {
			/* The buffer isn't big enough. */
			err = -EINVAL;
		} else
			vb->size = cam->pix.sizeimage;
	} else if (!vb->baddr) {
		if (vb->state != STATE_NEEDS_INIT) {
			/* We have a kernel bounce buffer that has already been 
			 * allocated.
			 */
			if (cam->pix.sizeimage > vb->size) {
				/* The image size has been changed to a larger 
				 * size since this buffer was allocated, so we 
				 * need to free and reallocate it.
				 */
				spin_unlock(&cam->img_lock);
				omap24xxcam_vbq_release(vbq, vb);
				spin_lock(&cam->img_lock);
				vb->size = cam->pix.sizeimage;
			}
		} else {
			/* We need to allocate a new kernel bounce buffer. */
			vb->size = cam->pix.sizeimage;
		}
	}

	vb->width = cam->pix.width;
	vb->height = cam->pix.height;
	vb->field = field;

	spin_unlock(&cam->img_lock);

	if (err)
		return err;

	if (vb->state == STATE_NEEDS_INIT)
		err = videobuf_iolock(vbq, vb, NULL);

	if (!err)
		vb->state = STATE_PREPARED;
	else
		omap24xxcam_vbq_release(vbq, vb);

	return err;
}

static void
omap24xxcam_vbq_queue(struct videobuf_queue *vbq, struct videobuf_buffer *vb)
{
	struct omap24xxcam_fh *fh = vbq->priv_data;
	struct omap24xxcam_device *cam = fh->cam;
	enum videobuf_state state = vb->state;
	int err;

	DBG;

	vb->state = STATE_QUEUED;
	err = omap24xxcam_sg_dma_queue(cam, vb->dma.sglist, vb->dma.sglen,
				       omap24xxcam_vbq_complete, vb);
	if (err) {
		/* Oops.  We're not supposed to get any errors here.  The only 
		 * way we could get an error is if we ran out of scatter-gather 
		 * DMA slots, but we are supposed to have at least as many 
		 * scatter-gather DMA slots as video buffers so that can't 
		 * happen.
		 */
		printk(KERN_DEBUG CAM_NAME
		       ": Failed to queue a video buffer for DMA!\n");
		vb->state = state;
	}
}

/* -------------------------------------------------------------------------- */

static int cam_get_format(void *arg, struct omap24xxcam_device *cam, struct omap_camera_sensor *sensor)
{
	struct v4l2_format *f = (struct v4l2_format *)arg;

	switch (f->type) {
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
	{
		struct v4l2_pix_format *pix = &f->fmt.pix;
		memset(pix, 0, sizeof(*pix));
		spin_lock(&cam->img_lock);
		*pix = cam->pix;
		spin_unlock(&cam->img_lock);
		return 0;
	}

	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
	{
		struct v4l2_window *win = &f->fmt.win;
		memset(win, 0, sizeof(*win));
		/* The API has a bit of a problem here. 
		 * We're returning a v4l2_window 
		 * structure, but that structure 
		 * contains pointers to variable-sized 
		 * objects for clipping rectangles and 
		 * clipping bitmaps.  We will just 
		 * return NULLs for those pointers.
		 */
		spin_lock(&cam->img_lock);
		win->w = cam->win.w;
		win->field = cam->win.field;
		win->chromakey = cam->win.chromakey;
		spin_unlock(&cam->img_lock);
		return 0;
	}

	default:
		return -EINVAL;
	}
}

static int cam_try_format(void *arg, struct omap24xxcam_device *cam,
			  struct omap_camera_sensor *sensor, int err)
{

	struct v4l2_format *f = (struct v4l2_format *)arg;

	switch (f->type) {
	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
	{
		struct v4l2_window *win = &f->fmt.win;
		spin_lock(&cam->img_lock);
		err = omap24xxcam_try_preview_window(cam, win);
		spin_unlock(&cam->img_lock);
		return err;
		}

	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
	{
		return cam->cam_sensor->try_format(&f->fmt.pix, cam->sensor);
	}

	default:
					return -EINVAL;
	}
}

static int cam_set_format(void *arg, struct omap24xxcam_device *cam,
			  struct omap_camera_sensor *sensor, int err)
{

	struct v4l2_format *f = (struct v4l2_format *)arg;

	switch (f->type) {
	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
	{
		struct v4l2_window *win = &f->fmt.win;

		spin_lock(&cam->img_lock);
		if (cam->previewing) {
			spin_unlock(&cam->img_lock);
			return -EBUSY;
		}
		err = omap24xxcam_new_preview_window(cam, win);
		spin_unlock(&cam->img_lock);
		return err;
	}

	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
	{
		spin_lock(&cam->img_lock);
		if (cam->streaming || cam->previewing) {
			spin_unlock(&cam->img_lock);
			return -EBUSY;
		}
		cam->cam_sensor->try_format(&f->fmt.pix, cam->sensor);
		/* set the new capture format */
		cam->pix = f->fmt.pix;
		/* adjust the capture frame rate */
		cam->xclk = cam->cam_sensor->calc_xclk(&cam->pix,
				&cam->nominal_timeperframe, cam->sensor);
		omap24xxcam_adjust_xclk(cam);
		cam->cparm.timeperframe =
		cam->nominal_timeperframe;
		/* intialize the preview parameters */
		omap24xxcam_new_capture_format(cam);
		spin_unlock(&cam->img_lock);
		/* program xclk */
		omap24xxcam_set_xclk(cam);
		/* program the sensor */
		err = cam->cam_sensor->configure(&cam->pix, cam->xclk,
				&cam->cparm.timeperframe, cam->sensor);
		return err;
	}

	default:
		return -EINVAL;
	}
}

static int cam_get_crop(void *arg, struct omap24xxcam_device *cam)
{

	struct v4l2_crop *crop = (struct v4l2_crop *)arg;

	switch (crop->type) {
	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
	{
		spin_lock(&cam->img_lock);
		crop->c = cam->crop;
		spin_unlock(&cam->img_lock);
		return 0;
	}

	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
		/* We don't support cropping of captured images. */
		return -EINVAL;

	default:
		return -EINVAL;
	}
}

static int cam_set_crop(void *arg, struct omap24xxcam_device *cam, int err)
{

	struct v4l2_crop *crop = (struct v4l2_crop *)arg;

	switch (crop->type) {
	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
	{
		spin_lock(&cam->img_lock);
		if (cam->previewing) {
			spin_unlock(&cam->img_lock);
			return -EBUSY;
		}
		err = omap24xxcam_new_crop_rect(cam, &crop->c);
		spin_unlock(&cam->img_lock);
		return err;
	}

	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
		/* We don't support cropping of captured images. */
		return -EINVAL;

	default:
		return -EINVAL;
	}
}

static int cam_cropcap(void *arg, struct omap24xxcam_device *cam)
{

	struct v4l2_cropcap *cropcap = (struct v4l2_cropcap *)arg;
	enum v4l2_buf_type type = cropcap->type;

	memset(cropcap, 0, sizeof(*cropcap));
	cropcap->type = type;
	switch (type) {
	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
	{
		struct v4l2_pix_format *pix = &cam->pix;

		spin_lock(&cam->img_lock);
		cropcap->bounds.width = pix->width & ~1;
		cropcap->bounds.height = pix->height & ~1;
		omap24xxcam_default_crop_rect(cam, &cropcap->defrect);
		spin_unlock(&cam->img_lock);
		cropcap->pixelaspect.numerator = 1;
		cropcap->pixelaspect.denominator = 1;
		return 0;
	}

	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
	{
		/* We're required to support the CROPCAP
		 * ioctl even though the G_CROP/S_CROP 
		 * ioctls are optional, which seems a 
		 * little strange.  We don't support 
		 * cropping of captured images.
		 */
		spin_lock(&cam->img_lock);
		cropcap->bounds.width = cam->pix.width;
		cropcap->bounds.height = cam->pix.height;
		spin_unlock(&cam->img_lock);
		cropcap->defrect.width = cropcap->bounds.width;
		cropcap->defrect.height = cropcap->bounds.height;
		cropcap->pixelaspect.numerator = 1;
		cropcap->pixelaspect.denominator = 1;
		return 0;
	}

	default:
		return -EINVAL;
	}
}

static int
omap24xxcam_do_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		     void *arg)
{
	struct omap24xxcam_fh *fh = file->private_data;
	struct omap24xxcam_device *cam = fh->cam;
	struct omap_camera_sensor *sensor = cam->cam_sensor;
	int err;

	switch (cmd) {
		/* for time being below IOCTL cmd is here */

	case VIDIOC_ENUMINPUT:
		{
			/* default handler assumes 1 video input (the camera) */
			struct v4l2_input *input = (struct v4l2_input *)arg;
			int index = input->index;

			memset(input, 0, sizeof(*input));
			input->index = index;

			if (index > 0)
				return -EINVAL;

			strlcpy(input->name, "camera", sizeof(input->name));
			input->type = V4L2_INPUT_TYPE_CAMERA;

			return 0;
		}

	case VIDIOC_G_INPUT:
		{
			unsigned int *input = arg;
			*input = 0;

			return 0;
		}

	case VIDIOC_S_INPUT:
		{
			unsigned int *input = arg;

			if (*input > 0)
				return -EINVAL;

			return 0;
		}

	case VIDIOC_QUERYCAP:
		{
			struct v4l2_capability *cap =
			    (struct v4l2_capability *)arg;

			memset(cap, 0, sizeof(*cap));
			strlcpy(cap->driver, CAM_NAME, sizeof(cap->driver));
			strlcpy(cap->card, cam->vfd->name, sizeof(cap->card));
			cap->bus_info[0] = '\0';
			cap->version = KERNEL_VERSION(0, 0, 0);
			cap->capabilities =
			    V4L2_CAP_VIDEO_CAPTURE |
			    V4L2_CAP_VIDEO_OVERLAY |
			    V4L2_CAP_READWRITE | V4L2_CAP_STREAMING;
			return 0;
		}
		
	case VIDIOC_G_FMT:
		return cam_get_format(arg, cam, sensor);
		
	case VIDIOC_TRY_FMT:
		return cam_try_format(arg, cam, sensor, err);
		
	case VIDIOC_S_FMT:
		return cam_set_format(arg, cam, sensor, err);
		
	case VIDIOC_G_FBUF:
		{
			struct v4l2_framebuffer *fbuf =
			    (struct v4l2_framebuffer *)arg;

			spin_lock(&cam->img_lock);
			*fbuf = cam->fbuf;
			spin_unlock(&cam->img_lock);
			return 0;
		}

	case VIDIOC_S_FBUF:
		{
			struct v4l2_framebuffer *fbuf =
			    (struct v4l2_framebuffer *)arg;
			unsigned int flags = fbuf->flags;

			/* The only field the user is allowed to change is 
			 * fbuf->flags.
			 */
			spin_lock(&cam->img_lock);
			if (cam->previewing) {
				spin_unlock(&cam->img_lock);
				return -EBUSY;
			}
			if (flags & V4L2_FBUF_FLAG_CHROMAKEY)
				cam->fbuf.flags |= V4L2_FBUF_FLAG_CHROMAKEY;
			else
				cam->fbuf.flags &= ~V4L2_FBUF_FLAG_CHROMAKEY;
			spin_unlock(&cam->img_lock);
			return 0;
		}
		
	case VIDIOC_CROPCAP:
		return cam_cropcap(arg, cam);

	case VIDIOC_G_CROP:
		return cam_get_crop(arg, cam);

	case VIDIOC_S_CROP:
		return cam_set_crop(arg, cam, err);
	case VIDIOC_G_PARM:
		{
			struct v4l2_streamparm *parm =
			    (struct v4l2_streamparm *)arg;
			enum v4l2_buf_type type = parm->type;

			memset(parm, 0, sizeof(*parm));
			parm->type = type;
			if (type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
				return -EINVAL;
			spin_lock(&cam->img_lock);
			parm->parm.capture = cam->cparm;
			spin_unlock(&cam->img_lock);
			return 0;
		}

	case VIDIOC_S_PARM:
		{
			struct v4l2_streamparm *parm =
			    (struct v4l2_streamparm *)arg;
			struct v4l2_captureparm *cparm = &parm->parm.capture;

			if (parm->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
				return -EINVAL;
			spin_lock(&cam->img_lock);
			if (cam->streaming || cam->previewing) {
				spin_unlock(&cam->img_lock);
				return -EBUSY;
			}
			cam->cparm.capturemode = cparm->capturemode;
			if (cparm->timeperframe.numerator
			    && cparm->timeperframe.denominator) {
				cam->nominal_timeperframe = cparm->timeperframe;
				/* adjust the capture frame rate */
				cam->xclk =
				    cam->cam_sensor->calc_xclk(
					    &cam->pix,
					    &cam->nominal_timeperframe,
					    cam->sensor);
				omap24xxcam_adjust_xclk(cam);
				cam->cparm.timeperframe =
				    cam->nominal_timeperframe;
				spin_unlock(&cam->img_lock);
				/* program xclk */
				omap24xxcam_set_xclk(cam);
				/* program the sensor */
				err =
				    cam->cam_sensor->configure(
					    &cam->pix,
					    cam->xclk,
					    &cam->cparm.timeperframe,
					    cam->sensor);
			} else {
				spin_unlock(&cam->img_lock);
			}
			return 0;
		}

	case VIDIOC_OVERLAY:
		{
			int *on = arg;

			spin_lock(&cam->img_lock);

			if (!cam->previewing && *on) {
				if (cam->pix.sizeimage <= cam->overlay_size) {
					/* Turn on the overlay window */
					omap24xxcam_configure_overlay(cam,
								      NULL);
					omap24xxcam_flip_overlay(
						cam,
						cam->overlay_base_phys);
					cam->previewing = fh;
					spin_unlock(&cam->img_lock);
					/* start the camera interface */
					omap24xxcam_dma_notify(cam,
							       omap24xxcam_cc_enable);
					omap24xxcam_start_overlay_dma(cam);
				} else {
					/* Image size is bigger than overlay 
					 * buffer.
					 */
					omap24xxcam_disable_vlayer(cam,
								   cam->vid1);
					spin_unlock(&cam->img_lock);
					return -EINVAL;
				}
			} else if (!*on) {
				/* turn overlay off */
				omap24xxcam_disable_vlayer(cam, cam->vid1);
				cam->previewing = NULL;
				spin_unlock(&cam->img_lock);
			} else
				spin_unlock(&cam->img_lock);

			return 0;
		}

	case VIDIOC_REQBUFS:
		return videobuf_reqbufs(&fh->vbq, arg);

	case VIDIOC_QUERYBUF:
		return videobuf_querybuf(&fh->vbq, arg);

	case VIDIOC_QBUF:
		return videobuf_qbuf(&fh->vbq, arg);

	case VIDIOC_DQBUF:
		return videobuf_dqbuf(&fh->vbq, arg,
				      file->f_flags & O_NONBLOCK);

	case VIDIOC_STREAMON:
		{
			spin_lock(&cam->img_lock);
			if (cam->streaming) {
				spin_unlock(&cam->img_lock);
				return -EBUSY;
			} else
				cam->streaming = fh;
			spin_unlock(&cam->img_lock);

			omap24xxcam_dma_notify(cam, omap24xxcam_cc_enable);
			return videobuf_streamon(&fh->vbq);
		}

	case VIDIOC_STREAMOFF:
		{
			struct videobuf_queue *q = &fh->vbq;
			int i;

			/* video-buf lib has trouble to turn off streaming while
			   any buffer is still in QUEUED state. Let's wait until
			   all queued buffers are filled.
                        */
			for (i = 0; i < VIDEO_MAX_FRAME; i++) {
				if (NULL == q->bufs[i])
					continue;
				while (q->bufs[i]->state == STATE_QUEUED) ;
			}

			err = videobuf_streamoff(q);

			if (err < 0)
				return err;

			for (i = 0; i < VIDEO_MAX_FRAME; i++) {
				if (NULL == q->bufs[i])
					continue;
				if (q->bufs[i]->memory == V4L2_MEMORY_USERPTR)
					omap24xxcam_vbq_release(q,
								q->bufs[i]);
			}

			spin_lock(&cam->img_lock);
			if (cam->streaming == fh)
				cam->streaming = NULL;
			spin_unlock(&cam->img_lock);
			return 0;
		}

	case VIDIOC_G_CTRL:
	        {
			struct v4l2_control *vc = arg;
			return cam->cam_sensor->get_control(vc, cam->sensor);
	        }
	case VIDIOC_S_CTRL:
		{
			struct v4l2_control *vc = arg;
			return cam->cam_sensor->set_control(vc, cam->sensor);
	        }
	case VIDIOC_QUERYCTRL:
		{
			struct v4l2_queryctrl *qc = arg;
			return cam->cam_sensor->query_control(qc, cam->sensor);
	        }
	case VIDIOC_QUERYMENU:
		/* no controls have been implemented yet */
		return -EINVAL;

	case VIDIOC_ENUMSTD:
	case VIDIOC_G_STD:
	case VIDIOC_S_STD:
	case VIDIOC_QUERYSTD:
		/* Digital cameras don't have an analog video standard,
		 * so we don't need to implement these ioctls.
		 */
		return -EINVAL;

	case VIDIOC_G_AUDIO:
	case VIDIOC_S_AUDIO:
	case VIDIOC_G_AUDOUT:
	case VIDIOC_S_AUDOUT:
		/* we don't have any audio inputs or outputs */
		return -EINVAL;


	case VIDIOC_G_JPEGCOMP:
	case VIDIOC_S_JPEGCOMP:
		/* JPEG compression is not supported */
		return -EINVAL;

	case VIDIOC_G_TUNER:
	case VIDIOC_S_TUNER:
	case VIDIOC_G_MODULATOR:
	case VIDIOC_S_MODULATOR:
	case VIDIOC_G_FREQUENCY:
	case VIDIOC_S_FREQUENCY:
		/* we don't have a tuner or modulator */
		return -EINVAL;

	case VIDIOC_ENUMOUTPUT:
	case VIDIOC_G_OUTPUT:
	case VIDIOC_S_OUTPUT:
		/* we don't have any video outputs */
		return -EINVAL;

	case VIDIOC_ENUM_FMT:
		{
			struct v4l2_fmtdesc *fmt = arg;
			return cam->cam_sensor->enum_pixformat(fmt,
							       cam->sensor);
		}
	case VIDIOC_S_VID2:
		{
			/* this api will set the parameters of video layer 2
			 ** of DISPC to display to use video 2 layer pipeline
			 ** to use hardware color conversion block to display
			 ** YUV422 format data on LCD
			 **/
			struct omap24xx_vid2_format *vid2_format =
			    (struct omap24xx_vid2_format *)arg;

			omap24xxcam_configure_overlay(cam, vid2_format);
			return 0;
		}

	case VIDIOC_S_VID2_DISABLE:
		{
			omap24xxcam_disable_vlayer(cam, cam->vid2);
			return 0;
		}

	default:
		/* unrecognized ioctl */
		return -ENOIOCTLCMD;
	}

	return 0;
}

/* -------------------------------------------------------------------------- */

	/*
	 *  file operations
	 */

static unsigned
int omap24xxcam_poll(struct file *file, struct poll_table_struct *wait)
{
	struct omap24xxcam_fh *fh = file->private_data;
	struct omap24xxcam_device *cam = fh->cam;
	struct videobuf_buffer *vb;
	enum v4l2_field field;

	DBG;

	spin_lock(&cam->img_lock);

	if (cam->streaming == fh) {
		spin_unlock(&cam->img_lock);
		/* streaming capture */
		if (list_empty(&fh->vbq.stream))
			return POLLERR;
		vb = list_entry(fh->vbq.stream.next, struct videobuf_buffer,
				stream);
	} else if (cam->streaming) {
		/* streaming I/O is in progress on another file descriptor */
		spin_unlock(&cam->img_lock);
		return POLLERR;
	} else {
		/* read() capture */
		spin_unlock(&cam->img_lock);
		mutex_lock(&fh->vbq.lock);
		if (fh->vbq.read_buf == NULL) {
			/* need to capture a new image */
			fh->vbq.read_buf = videobuf_alloc(fh->vbq.msize);
			if (fh->vbq.read_buf == NULL) {
				mutex_unlock(&fh->vbq.lock);
				return POLLERR;
			}
			fh->vbq.read_buf->memory = V4L2_MEMORY_USERPTR;
			field = videobuf_next_field(&fh->vbq);
			if (fh->vbq.ops->buf_prepare(&fh->vbq, fh->vbq.read_buf,
						     field) != 0) {
				mutex_unlock(&fh->vbq.lock);
				return POLLERR;
			}
			
			omap24xxcam_dma_notify(cam, omap24xxcam_cc_enable);
			fh->vbq.ops->buf_queue(&fh->vbq, fh->vbq.read_buf);
			fh->vbq.read_off = 0;
		}
		mutex_unlock(&fh->vbq.lock);
		vb = (struct videobuf_buffer *)fh->vbq.read_buf;
	}

	omap24xxcam_dma_notify(cam, omap24xxcam_cc_enable);
	poll_wait(file, &vb->done, wait);
	if (vb->state == STATE_DONE || vb->state == STATE_ERROR)
		return POLLIN | POLLRDNORM;

	return 0;
}

static ssize_t
omap24xxcam_read(struct file *file, char *data, size_t count, loff_t * ppos)
{
	struct omap24xxcam_fh *fh = file->private_data;
	struct omap24xxcam_device *cam = fh->cam;
	int err;

	DBG;

	spin_lock(&cam->img_lock);
	if (cam->streaming) {
		spin_unlock(&cam->img_lock);
		return -EBUSY;
	}
	spin_unlock(&cam->img_lock);

	omap24xxcam_dma_notify(cam, omap24xxcam_cc_enable);

	err =
		videobuf_read_one(&fh->vbq, data, count, ppos,
				  file->f_flags & O_NONBLOCK);

	DBG_END;

	return err;
}

static ssize_t
omap24xxcam_write(struct file *file, const char *data, size_t count,
		  loff_t * ppos)
{
	unsigned long vid_attributes;
	struct omap24xxcam_fh *fh = file->private_data;
	struct omap24xxcam_device *cam = fh->cam;

	DBG;

	vid_attributes = dispc_reg_merge(cam, DISPC_VID_ATTRIBUTES(1),
					 DISPC_VID_ATTRIBUTES_ENABLE,
					 DISPC_VID_ATTRIBUTES_ENABLE);
	if (vid_attributes & DISPC_VID_ATTRIBUTES_VIDCHANNELOUT) {
		/* digital output */
		dispc_reg_merge(cam, DISPC_CONTROL, DISPC_CONTROL_GODIGITAL,
				DISPC_CONTROL_GODIGITAL);
	} else {
		/* LCD */
		dispc_reg_merge(cam, DISPC_CONTROL, DISPC_CONTROL_GOLCD,
				DISPC_CONTROL_GOLCD);
	}
	if (copy_from_user((void *)cam->video2_base, data, count)) {
		printk("error in copying data");
	}
	return count;
}

static int omap24xxcam_mmap(struct file *file, struct vm_area_struct *vma)
{
	struct omap24xxcam_fh *fh = file->private_data;

	return videobuf_mmap_mapper(&fh->vbq, vma);
}

static int
omap24xxcam_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		  unsigned long arg)
{
	return video_usercopy(inode, file, cmd, arg, omap24xxcam_do_ioctl);
}


static int omap24xxcam_clock_get(struct omap24xxcam_device *cam)
{
	cam->fck = clk_get(0, "cam_fck");
	if (IS_ERR((cam->fck))) {
		dev_err(cam->dev, "can't get cam_fck");
		return PTR_ERR(cam->fck);
	}

	cam->ick = clk_get(0, "cam_ick");
	if (IS_ERR((cam->ick))) {
		dev_err(cam->dev, "can't get cam_ick");
		clk_put(cam->fck);
		return PTR_ERR(cam->ick);
	}

	return 0;
}

static void omap24xxcam_clock_put(struct omap24xxcam_device *cam)
{
	clk_put(cam->ick);
	clk_put(cam->fck);
}

/*
 * Do some sanity check, set clock rate, starts it and
 *  turn codec audio on 
 */
static int omap24xxcam_clock_on(struct omap24xxcam_device *cam)
{

	clk_enable(cam->fck);
	clk_enable(cam->ick);

	printk(KERN_DEBUG
	       "FCLK = %d [%d], usecount = %d\n",
	       (uint) clk_get_rate(cam->fck), CAM_CLOCK,
	       clk_get_usecount(cam->fck));

	return 0;
}

/*
 * Do some sanity check, turn clock off and then turn
 *  codec audio off
 */
static int omap24xxcam_clock_off(struct omap24xxcam_device *cam)
{
	if (clk_get_usecount(cam->fck) > 0) {
		if (clk_get_rate(cam->fck) != CAM_CLOCK) {
			printk(KERN_WARNING
			       "FCLK for camera should be %d Hz. But is %d Hz\n",
			       (uint) clk_get_rate(cam->fck), CAM_CLOCK);
		}

		clk_disable(cam->fck);
		clk_disable(cam->ick);
	}

	return 0;
}

static int omap24xxcam_dispc_clock_get(struct omap24xxcam_device *cam)
{
	cam->dss_ick = clk_get(cam->dev, "dss_ick");
	if (IS_ERR((cam->dss_ick))) {
		dev_err(cam->dev, "can't get dss_ick");
		return PTR_ERR(cam->dss_ick);
	}

	cam->dss1_fck = clk_get(cam->dev, "dss1_fck");
	if (IS_ERR((cam->dss1_fck))) {
		dev_err(cam->dev, "can't get dss1_fck");
		clk_put(cam->dss_ick);
		return PTR_ERR(cam->dss1_fck);
	}

	return 0;
}

static void omap24xxcam_dispc_clock_put(struct omap24xxcam_device *cam)
{
	clk_put(cam->dss1_fck);
	clk_put(cam->dss_ick);
}

static void omap24xxcam_dispc_clock_on(struct omap24xxcam_device *cam)
{
	clk_enable(cam->dss_ick);
	clk_enable(cam->dss1_fck);
}

static void omap24xxcam_dispc_clock_off(struct omap24xxcam_device *cam)
{
	clk_disable(cam->dss1_fck);
	clk_disable(cam->dss_ick);
}

/* Power-up and configure camera sensor */
static int omap24xxcam_sensor_enable(struct omap24xxcam_device *cam)
{
	if (cam->cam_sensor->power_on)
		cam->cam_sensor->power_on(cam->sensor);

	/* calculate xclk based on the default capture format and default 
	 * frame rate
	 */
	cam->xclk = cam->cam_sensor->calc_xclk(&cam->pix,
					       &cam->nominal_timeperframe,
					       cam->sensor);
	omap24xxcam_adjust_xclk(cam);
	cam->cparm.timeperframe = cam->nominal_timeperframe;

	/* program the camera interface to generate the new xclk frequency */
	omap24xxcam_set_xclk(cam);

	/* initialize the image preview parameters based on the default capture 
	 * format
	 */
	omap24xxcam_new_capture_format(cam);

	/* program the sensor for the default capture format and rate */
	cam->cam_sensor->configure(&cam->pix, cam->xclk,
				   &cam->cparm.timeperframe, cam->sensor);

	return 0;
}

static void omap24xxcam_reset_work(void * data)
{
	struct omap24xxcam_device *cam = (struct omap24xxcam_device *)data;

	if (cam->cam_sensor->power_off)
		cam->cam_sensor->power_off(cam->sensor);
	omap24xxcam_sensor_enable(cam);
}

/* This gets called whenever a DMA transfer is stalled */
static void omap24xxcam_reset_timer(unsigned long data)
{
	struct omap24xxcam_device *cam = (struct omap24xxcam_device *)data;

	printk("%s()\n", __FUNCTION__);

	/*
	 * Sensor reset must not be called from
	 * interrupt context as it may sleep
	 */
	schedule_work(&cam->reset_work);
}

static int omap24xxcam_release(struct inode *inode, struct file *file)
{
	struct omap24xxcam_fh *fh = file->private_data;
	struct omap24xxcam_device *cam = fh->cam;

	DBG;

	spin_lock(&cam->img_lock);
	/* turn off overlay */
	if (cam->previewing == fh) {
		cam->previewing = NULL;
		spin_unlock(&cam->img_lock);
		omap24xxcam_disable_vlayer(cam, cam->vid1);
		spin_lock(&cam->img_lock);
	}

	/* stop streaming capture */
	if (cam->streaming == fh) {
		cam->streaming = NULL;
		spin_unlock(&cam->img_lock);
		videobuf_streamoff(&fh->vbq);
		spin_lock(&cam->img_lock);
	}
	spin_unlock(&cam->img_lock);

	/* release read_buf videobuf_buffer struct */
	if (fh->vbq.read_buf) {
		omap24xxcam_vbq_release(&fh->vbq, fh->vbq.read_buf);
		kfree(fh->vbq.read_buf);
	}

	mutex_lock(&cam->mutex);
	if (--cam->users == 0) {
		/* power down the sensor */
		cc_reg_out(cam, CC_CTRL_XCLK, CC_CTRL_XCLK_DIV_STABLE_LOW);
		if (cam->cam_sensor->power_off)
			cam->cam_sensor->power_off(cam->sensor);
		omap24xxcam_dispc_clock_off(cam);
		omap24xxcam_clock_off(cam);
	}
	mutex_unlock(&cam->mutex);

	file->private_data = NULL;
	kfree(fh);


	return 0;
}

static int omap24xxcam_open(struct inode *inode, struct file *file)
{
	int minor = iminor(inode);
	struct omap24xxcam_device *cam = camera_module;
	struct omap24xxcam_fh *fh;

	DBG;

	if (!cam || !cam->vfd || (cam->vfd->minor != minor))
		return -ENODEV;

	/* allocate per-filehandle data */
	fh = kmalloc(sizeof(*fh), GFP_KERNEL);
	if (NULL == fh)
		return -ENOMEM;
	file->private_data = fh;
	fh->cam = cam;
	fh->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

	mutex_lock(&cam->mutex);
	if (cam->users == 0) {
		omap24xxcam_clock_on(cam);
		omap24xxcam_dispc_clock_on(cam);
		/* power up and configure the sensor */
		omap24xxcam_sensor_enable(cam);
	}
	cam->users++;
	mutex_unlock(&cam->mutex);

	videobuf_queue_init(&fh->vbq, &cam->vbq_ops, NULL, &cam->vbq_lock,
			    fh->type, V4L2_FIELD_NONE,
			    sizeof(struct videobuf_buffer), fh);

	return 0;
}

static struct file_operations omap24xxcam_fops = {
	.owner = THIS_MODULE,
	.llseek = no_llseek,
	.read = omap24xxcam_read,
	.write = omap24xxcam_write,
	.poll = omap24xxcam_poll,
	.ioctl = omap24xxcam_ioctl,
	.mmap = omap24xxcam_mmap,
	.open = omap24xxcam_open,
	.release = omap24xxcam_release,
};

/* -------------------------------------------------------------------------- */
#ifdef CONFIG_PM
static int omap24xxcam_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct omap24xxcam_device *cam = platform_get_drvdata(pdev);

	DBG;

	mutex_lock(&cam->mutex);
	if (cam->users == 0) {
		mutex_unlock(&cam->mutex);
		return 0;
	}
	mutex_unlock(&cam->mutex);

	/* disable previewing */
	spin_lock(&cam->img_lock);
	if (cam->previewing) {
		/* turn overlay off */
		omap24xxcam_disable_vlayer(cam, cam->vid1);
		cam->previewing = NULL;
	}
	spin_unlock(&cam->img_lock);

	/* Get the CC_CTRL register value for the current capture format. */
	omap24xxcam_sensor_cc_ctrl(cam);

	/* Disable the camera interface. */
	cc_reg_out(cam, CC_CTRL,
		   cam->cc_ctrl & ~(CC_CTRL_CC_EN | CC_CTRL_CC_FRAME_TRIG));

	/* Stop the DMA controller and frame sync scatter-gather DMA. */
	omap24xxcam_sg_dma_sync(cam, CAMDMA_CSR_TRANS_ERR);

	/* stop XCLK */
	cc_reg_out(cam, CC_CTRL_XCLK, CC_CTRL_XCLK_DIV_STABLE_LOW);

	/* power down the sensor */
	if (cam->cam_sensor->power_off)
		cam->cam_sensor->power_off(cam->sensor);

	omap24xxcam_clock_off(cam);

	return 0;
}
static int omap24xxcam_resume(struct platform_device *pdev)
{
	struct omap24xxcam_device *cam = platform_get_drvdata(pdev);

	DBG;

	mutex_lock(&cam->mutex);
	if (cam->users == 0) {
		mutex_unlock(&cam->mutex);
		return 0;
	}
	mutex_unlock(&cam->mutex);

	omap24xxcam_clock_on(cam);
	/* power-up and re-configure the sensor */
	omap24xxcam_sensor_enable(cam);

	/* set XCLK */
	omap24xxcam_set_xclk(cam);

	if (cam->streaming) {
		/* capture was in progress, so we need to register 
		 * our routine to restart the camera interface the next time a 
		 * DMA transfer is queued.
		 */
		omap24xxcam_dma_notify(cam, omap24xxcam_cc_enable);
	}

	/* Restart the scatter-gather DMA queue. */
	omap24xxcam_sg_dma_process(cam);

	/* camera interface will be enabled through dma_notify function
	 ** automatically when new dma starts 
	 */

	return 0;
}
#endif /* CONFIG_PM */

static int omap24xxcam_init_sensor(struct omap24xxcam_device *cam)
{
	int err = 0;

	omap24xxcam_clock_on(camera_module);
	omap24xxcam_dispc_clock_on(camera_module);
	/* power up sensor during sensor initialization */
	if (cam->cam_sensor->power_on)
		cam->cam_sensor->power_on(cam->sensor);

	/* initialize the sensor and define a default capture format cam->pix */
	cam->sensor = cam->cam_sensor->init(&cam->pix);
	if (!cam->sensor) {
		printk(KERN_ERR CAM_NAME ": cannot initialize sensor\n");
		err = -ENODEV;
		goto out;
	}
	printk(KERN_INFO "Sensor is %s\n", cam->cam_sensor->name);

out:
	/* power down sensor until camera device is open */
	cc_reg_out(cam, CC_CTRL_XCLK, CC_CTRL_XCLK_DIV_STABLE_LOW);
	if (cam->cam_sensor->power_off)
		cam->cam_sensor->power_off(cam->sensor);
	omap24xxcam_dispc_clock_off(camera_module);
	omap24xxcam_clock_off(camera_module);

	return err;
}

/* This should be called by the sensor code whenever it's ready */
int omap24xxcam_register_sensor(struct omap_camera_sensor *sensor)
{
	BUG_ON(sensor == NULL);

	if (camera_module == NULL) {
		BUG_ON(camera_sensor_if != NULL);
		camera_sensor_if = sensor;
		return 0;
	}

	camera_module->cam_sensor = sensor;
	return omap24xxcam_init_sensor(camera_module);
}
EXPORT_SYMBOL(omap24xxcam_register_sensor);

static int omap24xxcam_probe(struct platform_device *pdev)
{
	struct video_device *vfd;

	DBG;

	camera_module = kmalloc(sizeof(struct omap24xxcam_device), GFP_KERNEL);
	if (!camera_module) {
		printk(KERN_ERR CAM_NAME ": could not allocate memory\n");
		goto error_out_0;
	}
	memset(camera_module, 0, sizeof(struct omap24xxcam_device));
	camera_module->dev = &pdev->dev;

	/* initialize the video_device struct */
	vfd = camera_module->vfd = video_device_alloc();
	if (!vfd) {
		printk(KERN_ERR CAM_NAME
		       ": could not allocate video device struct\n");
		goto error_out_1;
	}
	vfd->release = video_device_release;

	strlcpy(vfd->name, CAM_NAME, sizeof(vfd->name));
	vfd->type = VID_TYPE_CAPTURE | VID_TYPE_OVERLAY | VID_TYPE_CHROMAKEY;
	/* need to register for a VID_HARDWARE_* ID in videodev.h */
	vfd->hardware = 0;
	vfd->fops = &omap24xxcam_fops;
	/* FIXME: need to use the full v4l2 API */
	vfd->priv = camera_module;
//	video_set_drvdata(vfd, camera_module);
	vfd->minor = -1;

	/* initialize the videobuf queue ops */
	camera_module->vbq_ops.buf_setup = omap24xxcam_vbq_setup;
	camera_module->vbq_ops.buf_prepare = omap24xxcam_vbq_prepare;
	camera_module->vbq_ops.buf_queue = omap24xxcam_vbq_queue;
	camera_module->vbq_ops.buf_release = omap24xxcam_vbq_release;
	spin_lock_init(&camera_module->vbq_lock);

	/* Impose a lower limit on the amount of memory allocated for capture.
	 * We require at least enough memory to double-buffer QVGA (300KB).
	 */
	if (capture_mem < 320 * 240 * 2 * 2)
		capture_mem = 320 * 240 * 2 * 2;

	/* request the mem region for the camera registers */
	if (!request_mem_region(CAM_REG_BASE, CAM_REG_SIZE, vfd->name)) {
		printk(KERN_ERR CAM_NAME
		       ": cannot reserve camera register I/O region\n");
		goto error_out_2;
	}
	camera_module->cam_mmio_base_phys = CAM_REG_BASE;
	camera_module->cam_mmio_size = CAM_REG_SIZE;

	/* map the region */
	camera_module->cam_mmio_base = (unsigned long)
	    ioremap(camera_module->cam_mmio_base_phys, camera_module->cam_mmio_size);
	if (!camera_module->cam_mmio_base) {
		printk(KERN_ERR CAM_NAME
		       ": cannot map camera register I/O region\n");
		goto error_out_3;
	}

	/* Map the display controller registers.
	 * Note that we do not request the memory region first, because the 
	 * framebuffer driver will already have claimed this region and the 
	 * request would fail.
	 */
	camera_module->dispc_mmio_base_phys = DSS_REG_BASE;
	camera_module->dispc_mmio_size = DSS_REG_SIZE;
	camera_module->dispc_mmio_base = (unsigned long)
	    ioremap(camera_module->dispc_mmio_base_phys, camera_module->dispc_mmio_size);
	if (!camera_module->dispc_mmio_base) {
		printk(KERN_ERR CAM_NAME
		       ": cannot map display controller register I/O region\n");
		goto error_out_4;
	}

	/* allocate coherent memory for the overlay framebuffer */
	camera_module->overlay_size = overlay_mem;
	if (camera_module->overlay_size > 0) {
		camera_module->overlay_base = 
			(unsigned long)dma_alloc_coherent(NULL,
							  camera_module->overlay_size,
							  (dma_addr_t*) &camera_module->overlay_base_phys,
							  GFP_KERNEL | GFP_DMA);
		if (!camera_module->overlay_base) {
			printk(KERN_ERR CAM_NAME
			       ": cannot allocate overlay framebuffer\n");
			goto error_out_5;
		}
	}
	memset((void *)camera_module->overlay_base, 0, camera_module->overlay_size);

	camera_module->video2_size = video2_mem;
	if (camera_module->video2_size > 0) {
		camera_module->video2_base = 
			(unsigned long) dma_alloc_coherent(NULL,
							   camera_module->video2_size,
							   (dma_addr_t*) & camera_module->video2_base_phys,
							   GFP_KERNEL | GFP_DMA);
		if (!camera_module->video2_base) {
			printk(KERN_ERR CAM_NAME
			       ": cannot allocate video2 layer memory\n");
			goto error_out_6;

		}
	}
	memset((void *)camera_module->video2_base, 0, camera_module->video2_size);

	/* initialize the overlay spinlock  */
	spin_lock_init(&camera_module->overlay_lock);

	/* initialize the camera interface */
	cam_init(camera_module);

	init_timer(&camera_module->reset_timer);
	camera_module->reset_timer.function = omap24xxcam_reset_timer;
	camera_module->reset_timer.data = (unsigned long)camera_module;
	INIT_WORK(&camera_module->reset_work,
		  omap24xxcam_reset_work, camera_module);

	/* initialize the spinlock used to serialize access to the image 
	 * parameters
	 */
	spin_lock_init(&camera_module->img_lock);

	if (omap24xxcam_dispc_clock_get(camera_module) < 0)
		goto error_out_6;
	/* initialize the camera interface functional clock frequency */
	camera_module->mclk = 96000000;	/* 96MHz */
	if (omap24xxcam_clock_get(camera_module) < 0) {
		omap24xxcam_dispc_clock_put(camera_module);
		goto error_out_6;
	}
	omap24xxcam_dispc_clock_on(camera_module);
	omap24xxcam_clock_on(camera_module);

	/* initialize the streaming capture parameters */
	camera_module->cparm.readbuffers = 1;
	camera_module->cparm.capability = V4L2_CAP_TIMEPERFRAME;

	/* Enable the xclk output.  The sensor may (and does, in the case of 
	 * the OV9640) require an xclk input in order for its initialization 
	 * routine to work.
	 */
	camera_module->xclk = 12000000;	/* choose an arbitrary xclk frequency */
	omap24xxcam_set_xclk(camera_module);

	/* get the framebuffer parameters in case the sensor init routine 
	 * needs them
	 */
	omap24xxcam_g_fbuf(camera_module);

	/* select an arbitrary default capture frame rate of 5fps */
	camera_module->nominal_timeperframe.numerator = 1;
	camera_module->nominal_timeperframe.denominator = 15;

	/* Sensor has registered but we were not here yet */
	if (camera_sensor_if != NULL) {
			camera_module->cam_sensor = camera_sensor_if;	
			if (omap24xxcam_init_sensor(camera_module) < 0)
				goto error_out_7;

	}

	/* use display controller video window 0 for preview */
	camera_module->vid1 = 0;
	camera_module->vid2 = 1;

	/* initialize the DMA driver */
	omap24xxcam_dma_init(camera_module);

	/* install the interrupt service routine */
	if (request_irq(INT_24XX_CAM_IRQ, omap24xxcam_isr, 0, CAM_NAME, camera_module)) {
		printk(KERN_ERR CAM_NAME
		       ": could not install interrupt service routine\n");
		goto error_out_8;
	}
	camera_module->irq = INT_24XX_CAM_IRQ;

	if (video_register_device(vfd, VFL_TYPE_GRABBER, video_nr) < 0) {
		printk(KERN_ERR CAM_NAME
		       ": could not register Video for Linux device\n");
		vfd->minor = -1;
		goto error_out_9;
	}
	/* set driver specific data to use in power management functions */
	platform_set_drvdata(pdev, camera_module);

	omap24xxcam_dispc_clock_off(camera_module);
	omap24xxcam_clock_off(camera_module);

	mutex_init(&camera_module->mutex);
	camera_module->users = 0;

	printk(KERN_INFO CAM_NAME
	       ": registered device video%d [v4l2]\n", vfd->minor);
	
	return 0;

 error_out_9:
	free_irq(camera_module->irq, camera_module);
	camera_module->irq = 0;
 error_out_8:
	omap24xxcam_dma_exit(camera_module);
	camera_module->cam_sensor->cleanup(camera_module->sensor);
	/* sensor allocated private data is gone */
	camera_module->sensor = NULL;
 error_out_7:
	omap24xxcam_clock_off(camera_module);
	omap24xxcam_dispc_clock_off(camera_module);
	omap24xxcam_clock_put(camera_module);
	omap24xxcam_dispc_clock_put(camera_module);

	dma_free_coherent(NULL, camera_module->video2_size,
			  (void *)camera_module->video2_base,
			  camera_module->video2_base_phys);
	camera_module->video2_base = 0;
 error_out_6:
	dma_free_coherent(NULL, camera_module->overlay_size,
			  (void *)camera_module->overlay_base,
			  camera_module->overlay_base_phys);
	camera_module->overlay_base = 0;
 error_out_5:
	iounmap((void *)camera_module->dispc_mmio_base);
	camera_module->dispc_mmio_base = 0;
 error_out_4:
	iounmap((void *)camera_module->cam_mmio_base);
	camera_module->cam_mmio_base = 0;
 error_out_3:
	release_mem_region(camera_module->cam_mmio_base_phys, camera_module->cam_mmio_size);
	camera_module->cam_mmio_base_phys = 0;
 error_out_2:
	video_device_release(vfd);
 error_out_1:
	kfree(camera_module);
 error_out_0:	
	return -ENODEV;
}

static int omap24xxcam_remove(struct platform_device *pdev)
{
	return 0;
}

static struct platform_driver omap24xxcam_driver = {
	.probe = omap24xxcam_probe,
	.remove = omap24xxcam_remove,
#ifdef CONFIG_PM
	.suspend = omap24xxcam_suspend,
	.resume = omap24xxcam_resume,
#else
	.suspend = NULL,
	.resume = NULL,
#endif
	.shutdown = NULL,
	.driver = {
		.name = CAM_NAME,
	},
};

static struct platform_device omap24xxcam_dev = {
	.name = CAM_NAME,
	.dev = {
		.release = NULL,
		},
	.id = 0,
};

int __init omap24xxcam_init(void)
{
	int ret;

	DBG;

	ret = platform_driver_register(&omap24xxcam_driver);
	if (ret != 0)
		return ret;

	ret = platform_device_register(&omap24xxcam_dev);
	if (ret != 0) {
		platform_driver_unregister(&omap24xxcam_driver);
		return ret;
	}

	return 0;
}

void omap24xxcam_cleanup(void)
{
	struct video_device *vfd;

	DBG;

	if (!camera_module)
		return;
	vfd = camera_module->vfd;

	if (vfd) {
		if (vfd->minor == -1) {
			/* The device was never registered, so release the 
			 * video_device struct directly.
			 */
			video_device_release(vfd);
		} else {
			/* The unregister function will release the video_device
			 * struct as well as unregistering it.
			 */
			video_unregister_device(vfd);
		}
		camera_module->vfd = NULL;
	}

	if (camera_module->irq) {
		free_irq(camera_module->irq, camera_module);
		camera_module->irq = 0;
	}

	omap24xxcam_dma_exit(camera_module);
	camera_module->cam_sensor->cleanup(camera_module->sensor);
	/* sensor allocated private data is gone */
	camera_module->sensor = NULL;

	if (camera_module->video2_base) {
		dma_free_coherent(NULL, camera_module->video2_size,
				  (void *)camera_module->video2_base,
				  camera_module->video2_base_phys);
		camera_module->video2_base = 0;
	}
	camera_module->video2_base_phys = 0;

	if (camera_module->overlay_base) {
		dma_free_coherent(NULL, camera_module->overlay_size,
				  (void *)camera_module->overlay_base,
				  camera_module->overlay_base_phys);
		camera_module->overlay_base = 0;
	}
	camera_module->overlay_base_phys = 0;

	if (camera_module->dispc_mmio_base) {
		iounmap((void *)camera_module->dispc_mmio_base);
		camera_module->dispc_mmio_base = 0;
	}
	camera_module->dispc_mmio_base_phys = 0;

	if (camera_module->cam_mmio_base) {
		iounmap((void *)camera_module->cam_mmio_base);
		camera_module->cam_mmio_base = 0;
	}
	if (camera_module->cam_mmio_base_phys) {
		release_mem_region(camera_module->cam_mmio_base_phys, camera_module->cam_mmio_size);
		camera_module->cam_mmio_base_phys = 0;
	}

	platform_device_unregister(&omap24xxcam_dev);
	platform_driver_unregister(&omap24xxcam_driver);

	kfree(camera_module);
}

MODULE_AUTHOR("MontaVista Software, Inc.");
MODULE_DESCRIPTION("OMAP24xx Video for Linux camera driver");
MODULE_LICENSE("GPL");
module_param(video_nr, int, 0);
MODULE_PARM_DESC(video_nr,
		 "Minor number for video device (-1 ==> auto assign)");
module_param(capture_mem, int, 0);
MODULE_PARM_DESC(capture_mem,
		 "Maximum amount of memory for capture buffers (default 4800KB)");
module_param(overlay_mem, int, 0);
MODULE_PARM_DESC(overlay_mem,
		 "Preview overlay framebuffer size (default 600KB)");

module_init(omap24xxcam_init);
module_exit(omap24xxcam_cleanup);
