/*
 * drivers/media/video/omap/omap24xxcam-dma.c
 *
 * Video-for-Linux (Version 2) camera capture driver for 
 * the OMAP24xx camera controller.
 *
 * Author: David Cohen <david.cohen@indt.org.br>
 * Based on: omap24xxcam.c by Andy Lowe (source@mvista.com)
 *
 * Copyright (C) 2004 MontaVista Software, Inc.
 * Copyright (C) 2004 Texas Instruments.
 *
 * This file is licensed under the terms of the GNU General Public License 
 * version 2. This program is licensed "as is" without any warranty of any 
 * kind, whether express or implied.
 *
 * History:
 * 2005/nov - David Cohen <david.cohen@indt.org.br>
 *            Updated the driver for the Linux Device Model and new version of V4L2.
 */

#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/kernel.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#include <linux/sched.h>
#include <linux/smp_lock.h>
#include <linux/types.h>
#include <linux/wait.h>
#include <linux/videodev.h>
#include <linux/pci.h>		/* needed for videobufs */
#include <media/video-buf.h>
#include <linux/dma-mapping.h>
#include <linux/input.h>
#include <linux/version.h>

#include <asm/io.h>
#include <asm/byteorder.h>
#include <asm/scatterlist.h>
#include <asm/irq.h>
#include <asm/semaphore.h>

#include "omap24xxcam.h"

/* configuration macros */
#define CAM_NAME 	"omap24xxcam"
#define CONFIG_H4

/*
 * camera DMA register I/O routines
 */

__inline__ u32 
camdma_reg_in(const struct omap24xxcam_device *cam, u32 offset)
{
	return readl(cam->cam_mmio_base + CAMDMA_REG_OFFSET + offset);
}

__inline__ u32 
camdma_reg_out(const struct omap24xxcam_device *cam, u32 offset, u32 val)
{
	writel(val, cam->cam_mmio_base + CAMDMA_REG_OFFSET + offset);
	return val;
}

__inline__ u32 
camdma_reg_merge(const struct omap24xxcam_device *cam, u32 offset, 
		u32 val, u32 mask)
{
	u32 addr = cam->cam_mmio_base + CAMDMA_REG_OFFSET + offset;
	u32 new_val = (readl(addr) & ~mask) | (val & mask);

	writel(new_val, addr);
	return new_val;
}

void
camdma_init(const struct omap24xxcam_device *cam)
{
	DBG;
	camdma_reg_out(cam, CAMDMA_OCP_SYSCONFIG, 
		CAMDMA_OCP_SYSCONFIG_MIDLEMODE_FSTANDBY
	      | CAMDMA_OCP_SYSCONFIG_SIDLEMODE_FIDLE
	      | CAMDMA_OCP_SYSCONFIG_AUTOIDLE);

	camdma_reg_merge(cam, CAMDMA_GCR, 0x10, 
		CAMDMA_GCR_MAX_CHANNEL_FIFO_DEPTH);
}

/* Ack all interrupt on CSR and IRQSTATUS_L0 */
void omap24xxcam_dma_ack_all(struct omap24xxcam_device *cam)
{
	unsigned long csr;
	int i;
	
	for (i = 0; i< 4; ++i) {
		csr = camdma_reg_in(cam, CAMDMA_CSR(i));
		/* ack interrupt in CSR */
		camdma_reg_out(cam, CAMDMA_CSR(i), csr);
	}
	camdma_reg_out(cam, CAMDMA_IRQSTATUS_L0, 0xffffffff);
}

/* Ack dmach on CSR and IRQSTATUS_L0 */
unsigned long omap24xxcam_dma_ack_ch(struct omap24xxcam_device *cam, int dmach)
{
	unsigned long csr;
	
	csr = camdma_reg_in(cam, CAMDMA_CSR(dmach));
	/* ack interrupt in CSR */
	camdma_reg_out(cam, CAMDMA_CSR(dmach), csr);
	/* ack interrupt in IRQSTATUS */
	camdma_reg_out(cam, CAMDMA_IRQSTATUS_L0, (1 << dmach));

	return csr;
}

int omap24xxcam_dma_running(struct omap24xxcam_device *cam, int dmach)
{
	return camdma_reg_in(cam, CAMDMA_CCR(dmach)) & CAMDMA_CCR_ENABLE;
}

/* Start a DMA transfer from the camera to memory.
 * Returns zero if the transfer was successfully started, or non-zero if all 
 * DMA channels are already in use or starting is currently inhibited.
 */
int
omap24xxcam_dma_start(struct omap24xxcam_device *cam, dma_addr_t start, 
	unsigned long len, dma_callback_t callback, void *arg)
{
	unsigned long irqflags;
	int dmach;
	void (*dma_notify)(struct omap24xxcam_device *cam);

	DBG;

	spin_lock_irqsave(&cam->dma_lock, irqflags);

	if (!cam->free_dmach || cam->dma_stop) {
		spin_unlock_irqrestore(&cam->dma_lock, irqflags);
		DBG_MID(3);
		return -EBUSY;
	}

	dmach = cam->next_dmach;

	cam->camdma[dmach].callback = callback;
	cam->camdma[dmach].arg = arg;

	camdma_reg_out(cam, CAMDMA_CCR(dmach), 
		CAMDMA_CCR_SEL_SRC_DST_SYNC
	      | CAMDMA_CCR_BS
	      | CAMDMA_CCR_DST_AMODE_POST_INC
	      | CAMDMA_CCR_SRC_AMODE_POST_INC
	      | CAMDMA_CCR_FS
	      | CAMDMA_CCR_WR_ACTIVE
	      | CAMDMA_CCR_RD_ACTIVE
	      | CAMDMA_CCR_SYNCHRO_CAMERA);
	camdma_reg_out(cam, CAMDMA_CLNK_CTRL(dmach), 0);
	camdma_reg_out(cam, CAMDMA_CEN(dmach), len);
	camdma_reg_out(cam, CAMDMA_CFN(dmach), 1);
	camdma_reg_out(cam, CAMDMA_CSDP(dmach), 
		CAMDMA_CSDP_WRITE_MODE_POSTED 
	      | CAMDMA_CSDP_DST_BURST_EN_64
	      | CAMDMA_CSDP_DST_PACKED
	      | CAMDMA_CSDP_SRC_BURST_EN_64
	      | CAMDMA_CSDP_SRC_PACKED
	      | CAMDMA_CSDP_DATA_TYPE_8BITS);
	camdma_reg_out(cam, CAMDMA_CSSA(dmach), 0);
	camdma_reg_out(cam, CAMDMA_CDSA(dmach), start);
	camdma_reg_out(cam, CAMDMA_CSEI(dmach), 0);
	camdma_reg_out(cam, CAMDMA_CSFI(dmach), DMA_THRESHOLD);
	camdma_reg_out(cam, CAMDMA_CDEI(dmach), 0);
	camdma_reg_out(cam, CAMDMA_CDFI(dmach), 0);
	camdma_reg_out(cam, CAMDMA_CSR(dmach), 
		CAMDMA_CSR_MISALIGNED_ERR
	      | CAMDMA_CSR_SUPERVISOR_ERR
	      | CAMDMA_CSR_SECURE_ERR
	      | CAMDMA_CSR_TRANS_ERR
	      | CAMDMA_CSR_BLOCK
	      | CAMDMA_CSR_DROP);
	camdma_reg_out(cam, CAMDMA_CICR(dmach), 
		CAMDMA_CICR_MISALIGNED_ERR_IE
	      | CAMDMA_CICR_SUPERVISOR_ERR_IE
	      | CAMDMA_CICR_SECURE_ERR_IE
	      | CAMDMA_CICR_TRANS_ERR_IE
	      | CAMDMA_CICR_BLOCK_IE
	      | CAMDMA_CICR_DROP_IE);

	/* We're ready to start the DMA transfer. */

	if (cam->free_dmach < NUM_CAMDMA_CHANNELS) {
		/* A transfer is already in progress, so try to chain to it. */
		int prev_dmach, ch;

		if (dmach == 0)
			prev_dmach = NUM_CAMDMA_CHANNELS - 1;
		else
			prev_dmach = dmach - 1;
		camdma_reg_out(cam, CAMDMA_CLNK_CTRL(prev_dmach), 
			CAMDMA_CLNK_CTRL_ENABLE_LNK | dmach);
		/* Did we chain the DMA transfer before the previous one 
		 * finished?
		 */
		ch = (dmach + cam->free_dmach) % NUM_CAMDMA_CHANNELS;
		DBG_MID(1);
		while (!(camdma_reg_in(cam, CAMDMA_CCR(ch))
			 & CAMDMA_CCR_ENABLE))
		{
			if (ch == dmach) {
				/* The previous transfer has ended and this one 
				 * hasn't started, so we must not have chained 
				 * to the previous one in time.  We'll have to 
				 * start it now.
				 */
				camdma_reg_out(cam, CAMDMA_CCR(dmach), 
					CAMDMA_CCR_SEL_SRC_DST_SYNC
				      | CAMDMA_CCR_BS
				      | CAMDMA_CCR_DST_AMODE_POST_INC
				      | CAMDMA_CCR_SRC_AMODE_POST_INC
				      | CAMDMA_CCR_ENABLE
				      | CAMDMA_CCR_FS
				      | CAMDMA_CCR_SYNCHRO_CAMERA);
				break;
			} else
				ch = (ch + 1) % NUM_CAMDMA_CHANNELS;
		}
		DBG_MID(2);
	}
	else {
		/* No transfer is in progress, so we'll just start this one 
		 * now.
		 */
		camdma_reg_out(cam, CAMDMA_CCR(dmach), 
			CAMDMA_CCR_SEL_SRC_DST_SYNC
		      | CAMDMA_CCR_BS
		      | CAMDMA_CCR_DST_AMODE_POST_INC
		      | CAMDMA_CCR_SRC_AMODE_POST_INC
		      | CAMDMA_CCR_ENABLE
		      | CAMDMA_CCR_FS
		      | CAMDMA_CCR_SYNCHRO_CAMERA);
	}

	cam->next_dmach = (cam->next_dmach + 1) % NUM_CAMDMA_CHANNELS;
	cam->free_dmach--;

	dma_notify = cam->dma_notify;
	cam->dma_notify = NULL;

	spin_unlock_irqrestore(&cam->dma_lock, irqflags);

	if (dma_notify)
		(*dma_notify)(cam);

	DBG_END;

	return 0;
}

/* Abort all chained DMA transfers.  After all transfers have been aborted and 
 * the DMA controller is idle, the completion routines for any aborted transfers
 * will be called in sequence.  The DMA controller may not be idle after this 
 * routine completes, because the completion routines might start new transfers.
 */
void
omap24xxcam_dma_abort_ch(struct omap24xxcam_device *cam, int dmach)
{
	/* mask all interrupts from this channel */
	camdma_reg_out(cam, CAMDMA_CICR(dmach), 0);
	/* unlink this channel */
	camdma_reg_merge(cam, CAMDMA_CLNK_CTRL(dmach), 0,
			CAMDMA_CLNK_CTRL_ENABLE_LNK);
	/* disable this channel */
	camdma_reg_merge(cam, CAMDMA_CCR(dmach), 0, CAMDMA_CCR_ENABLE);
}

/* Mask all DMA interrupts */
void omap24xxcam_dma_mask_all(struct omap24xxcam_device *cam)
{
	camdma_reg_out(cam, CAMDMA_IRQENABLE_L0, 0);
	camdma_reg_out(cam, CAMDMA_IRQENABLE_L1, 0);
	camdma_reg_out(cam, CAMDMA_IRQENABLE_L2, 0);
	camdma_reg_out(cam, CAMDMA_IRQENABLE_L3, 0);
}

/* group all channels on DMA IRQ0 and unmask irq */
void omap24xxcam_dma_unmask_irq0(struct omap24xxcam_device *cam)
{
	camdma_reg_out(cam, CAMDMA_IRQENABLE_L0, 0xffffffff);
	camdma_reg_out(cam, CAMDMA_IRQENABLE_L1, 0);
	camdma_reg_out(cam, CAMDMA_IRQENABLE_L2, 0);
	camdma_reg_out(cam, CAMDMA_IRQENABLE_L3, 0);
}

/* Shutdown the camera DMA driver and controller. */
void
omap24xxcam_dma_exit(struct omap24xxcam_device *cam)
{
	int ch;

	DBG;

	/* Mask all DMA interrupts */
	omap24xxcam_dma_mask_all(cam);

	/* disable all DMA channels */
	for (ch = 0; ch < NUM_CAMDMA_CHANNELS; ch++)
		camdma_reg_out(cam, CAMDMA_CCR(ch), 0);
}

EXPORT_SYMBOL(camdma_init);
EXPORT_SYMBOL(omap24xxcam_dma_ack_all);
EXPORT_SYMBOL(omap24xxcam_dma_ack_ch);
EXPORT_SYMBOL(omap24xxcam_dma_running);
EXPORT_SYMBOL(omap24xxcam_dma_start);
EXPORT_SYMBOL(omap24xxcam_dma_abort_ch);
EXPORT_SYMBOL(omap24xxcam_dma_unmask_irq0);
EXPORT_SYMBOL(omap24xxcam_dma_exit);
