/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <gst/gst.h>

#include "gstdspilbcsink.h"

GstElementDetails gst_dspilbcsink_details = GST_ELEMENT_DETAILS ("DSP ILBC Sink",
    "Sink/Audio",
    "ILBC audio sink",
    "Makoto Sugano <makoto.sugano@nokia.com>");

static gchar *devname = "/dev/dsptask/ilbc_dec";

static GstStaticPadTemplate dspilbcsink_sink_template =
GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ( "audio/x-iLBC, "
                      "rate = (int) 8000, "
                      "channels = (int) 1, "
                      "mode = (int) { 20, 30 }"
                    ) );

// Function prototypes

static void gst_dspilbcsink_class_init (GstDSPILBCSinkClass * klass);
static void gst_dspilbcsink_base_init (gpointer g_class);
static void gst_dspilbcsink_init (GstDSPILBCSink *dspmp3sink, GstDSPILBCSinkClass *g_class);
static void gst_dspilbcsink_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dspilbcsink_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);
static gboolean gst_dspilbcsink_start (GstBaseSink * bsrc);
static gboolean gst_dspilbcsink_stop (GstBaseSink * bsrc);
static GstFlowReturn gst_dspilbcsink_render(GstBaseSink *sink, GstBuffer * buffer);
static gboolean gst_dspilbcsink_setcaps (GstBaseSink * basesink, GstCaps * caps);

static GstStateChangeReturn
    gst_dspilbcsink_change_state (GstElement * element, GstStateChange transition);

// static GstFlowReturn gst_dspilbcsink_preroll(GstBaseSink *sink, GstBuffer * buffer);

/**
 *
 *
 */

void _ilbcsink_do_init()
{
  DBG_PRINT("ILBCSINK DO_INIT\n");
}

GST_BOILERPLATE_FULL (GstDSPILBCSink, gst_dspilbcsink, GstBaseSink,
                      GST_TYPE_BASE_SINK, _ilbcsink_do_init);


/**
 * gst_dspilbcsink_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes ILBC sink element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */

static void
gst_dspilbcsink_dispose (GObject * object)
{
  GstDSPILBCSink *dsp = (GstDSPILBCSink *) object;

  if(dsp->audio) {
    gst_dspaudio_destroy(dsp->audio);
    dsp->audio = NULL;
  }

  g_object_unref(G_OBJECT(dsp->adapter));
  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/**
 * gst_dspilbcsink_base_init:
 * @g_class: ILBC sink class
 *
 * Does the basic initialization of ILBC sink element class. This is
 * called by GStreamer framework.
 */

static void
gst_dspilbcsink_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspilbcsink_sink_template));
  gst_element_class_set_details (gstelement_class, &gst_dspilbcsink_details);

}

/**
 * gst_dspilbcsink_class_init:
 * @klass: ILBC Sink class object
 *
 * Initializes ILBC sink element class. This is called by GStreamer
 * framework.
 */

static void
gst_dspilbcsink_class_init (GstDSPILBCSinkClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSinkClass *gstbase_sink_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbase_sink_class = (GstBaseSinkClass *) klass;

  gobject_class->set_property = gst_dspilbcsink_set_property;
  gobject_class->get_property = gst_dspilbcsink_get_property;
  gobject_class->dispose = gst_dspilbcsink_dispose;

   // Use dspaudio utility to install default properties
  gst_dspaudio_install_properties(G_OBJECT_CLASS (klass));

  gstelement_class->change_state = GST_DEBUG_FUNCPTR (gst_dspilbcsink_change_state);

  gstbase_sink_class->start = GST_DEBUG_FUNCPTR (gst_dspilbcsink_start);
  gstbase_sink_class->stop = GST_DEBUG_FUNCPTR (gst_dspilbcsink_stop);
  gstbase_sink_class->render = GST_DEBUG_FUNCPTR (gst_dspilbcsink_render);
  gstbase_sink_class->set_caps = GST_DEBUG_FUNCPTR (gst_dspilbcsink_setcaps);

//   gstbase_sink_class->preroll = GST_DEBUG_FUNCPTR (gst_dspilbcsink_preroll);
}


/**
 * gst_dspilbcsink_init:
 * @dsp: DSP ILBC sink object
 *
 * Initializes the given ILBC sink element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */

static void
gst_dspilbcsink_init (GstDSPILBCSink * dspilbcsink, GstDSPILBCSinkClass *g_class)
{
  dspilbcsink->audio = gst_dspaudio_new();
  dspilbcsink->adapter = gst_adapter_new();
}


/**
 * gst_dspilbcsink_setcaps:
 * @basesink: GstBaseSink
 * @caps: GstCaps that were offered by the connecting element
 *
 * Returns: TRUE if caps can be accepted, otherwise FALSE
 */

static gboolean
gst_dspilbcsink_setcaps(GstBaseSink * basesink, GstCaps * caps)
{
  DBG_PRINT("gst_dspilbcsink_setcaps\n");
  GstDSPILBCSink *dsp = GST_DSPILBCSINK (basesink);
  SPEECH_PARAMS_DATA init_data;

  GstStructure *structure = gst_caps_get_structure (caps, 0);
  gst_structure_get_int (structure, "mode", &dsp->mode);

  init_data.dsp_cmd = DSP_CMD_SET_SPEECH_PARAMS;
  init_data.audio_fmt = DSP_AFMT_ILBC;
  init_data.sample_rate = SAMPLE_RATE_8KHZ;
  init_data.ds_stream_ID = 0;
  init_data.stream_priority = dsp->audio->priority;
  init_data.frame_size = dsp->mode == 20 ? 160 : 240;
  dsp->framesize = dsp->mode == 20 ? 38 : 50;

  DBG_PRINT("SINK FRAME SIZE: %d bytes\n", dsp->framesize);

  // Read the junk out...
  gst_dspaudio_flush(&dsp->audio->codec);

  if(!gst_dspaudio_setparams(dsp->audio, (char *)&init_data, sizeof(SPEECH_PARAMS_DATA))) {
    dsp->audio->mode = DSP_MODE_ERROR;
    return FALSE;
  }
  dsp->audio->mode = DSP_MODE_INITIALIZED;
  DBG_PRINT("gst_dspilbcsink_setcaps OK\n");
  return TRUE;
}


/**
 * gst_dspilbcsink_render:
 * sink:
 * @buffer:
 *
 */

#ifdef DEBUG
int fd = -1;
#endif

static GstFlowReturn
gst_dspilbcsink_render(GstBaseSink *sink, GstBuffer * buffer)
{
  GstFlowReturn ret = GST_FLOW_OK;
  GstDSPILBCSink *dsp = GST_DSPILBCSINK (sink);
  DSPWaitStatus status;
  const guint8 *data;
  guint avail;

  DBG_PRINT("ILBC RENDER: %d bytes\n", GST_BUFFER_SIZE(buffer));

  gst_buffer_ref(buffer);
  gst_adapter_push (dsp->adapter, buffer);

#ifdef DEBUG
/*  if(fd == -1) {
    fd = open("/home/user/debug.out", O_RDWR | O_CREAT);
    DBG_PRINT("OPENED DEBUG FILE\n");
  }
  write(fd, GST_BUFFER_DATA(buffer), GST_BUFFER_SIZE(buffer));*/
#endif

  if(dsp->audio->mode != DSP_MODE_PLAYING) {
    DBG_PRINT("SETTING DSP TO PLAY\n");
    gst_dspaudio_play(dsp->audio);

    // Force the volume setting
    dsp->audio->volume_changed = TRUE;
    gst_dspaudio_update_dsp_settings(dsp->audio);
  }

  avail = gst_adapter_available(dsp->adapter);
  while(avail >= dsp->framesize) {

    g_mutex_lock(dsp->audio->dsp_mutex);

    status = gst_dspaudio_wait_buffer(dsp->audio);

    if(status == DSP_WAIT_RETRY) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
      DBG_PRINT("ILBC SINK DSP_WAIT_RETRY\n");
      continue;
    }
    if(status == DSP_WAIT_INTERRUPT) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
      DBG_PRINT("ILBC SINK READ INTERRUPT\n");
      break;
    }
    if(status == DSP_WAIT_ERROR) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
      GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE, (NULL),
                         ("loop2: error cmd: %d status: %d",
                          dsp->audio->error_cmd,
                          dsp->audio->error_status));
      ret = GST_FLOW_ERROR;
      break;
    }

    dsp->audio->rw_pending = 1;
    data = gst_adapter_peek(dsp->adapter, dsp->framesize);
    dsp->outbuf.data_size = dsp->framesize / sizeof(short int);
    avail -= dsp->framesize;

    dsp->mmap_ptr[0] = 0;             // Bad frame indication
    dsp->mmap_ptr[1] = 0;             // Last frame indication
    dsp->mmap_ptr[2] = VOICE_FRAME;   // Frame type

    memcpy(&(dsp->mmap_ptr[3]), data, dsp->framesize);
    gst_adapter_flush(dsp->adapter, dsp->framesize);

    if(write(dsp->audio->codec.fd, &(dsp->outbuf), sizeof(ILBC_DATA_WRITE)) < 0) {
      dsp->audio->mode = DSP_MODE_ERROR;
      GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE, (NULL), ("write = -1"));
      g_mutex_unlock(dsp->audio->dsp_mutex);
      ret = GST_FLOW_ERROR;
      break;
    }

//     DBG_PRINT("ILBC: Wrote %d bytes to DSP\n", dsp->outbuf.data_size * sizeof(short int));
    dsp->audio->rw_pending = 0;
    g_mutex_unlock(dsp->audio->dsp_mutex);
  }
  DBG_PRINT("ILBC SINK RENDER EXIT\n");
  return ret;
}


/**
 * gst_dspilbcsink_preroll:
 * @sink:
 * @buffer:
 *
 */

/*
static GstFlowReturn
gst_dspilbcsink_preroll(GstBaseSink *sink, GstBuffer * buffer)
{
  DBG_PRINT("gst_dspilbcsink_preroll\n");
  GstDSPILBCSink *dsp = GST_DSPILBCSINK (sink);
  gst_dspaudio_play(dsp->audio);
  gst_dspaudio_update_dsp_settings(dsp->audio);
  DBG_PRINT("gst_dspilbcsink_preroll OK\n");
  return GST_FLOW_OK;
}
*/

/**
 * gst_dspilbcsink_start:
 * @bsink: GstBaseSink
 *
 */

static gboolean
gst_dspilbcsink_start (GstBaseSink * bsink)
{
  DBG_PRINT("gst_dspilbcsink_start\n");
  GstDSPILBCSink *dsp = GST_DSPILBCSINK (bsink);

  dsp->outbuf.dsp_cmd = DSP_CMD_DATA_WRITE;
  if (gst_dspaudio_open(dsp->audio, devname) == FALSE) {
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }
  dsp->mmap_ptr = (short int *) dsp->audio->codec.mmap_buffer;
  DBG_PRINT("gst_dspilbcsink_start OK\n");
  return TRUE;
}


/**
 * gst_dspilbcsink_stop:
 * @bsink: GstBaseSink
 *
 */

static gboolean
gst_dspilbcsink_stop (GstBaseSink * bsink)
{
  DBG_PRINT("gst_dspilbcsink_stop\n");
  GstDSPILBCSink *dsp = GST_DSPILBCSINK (bsink);

  gst_dspaudio_remove_interrupt(dsp->audio);

  g_mutex_lock(dsp->audio->dsp_mutex);
  DBG_PRINT("gst_dspilbcsink_stop got mutex\n");
  if(gst_dspaudio_stop(dsp->audio)) {
    DBG_PRINT("gst_dspilbcsink_stop send stop. wait reply\n");
    gst_dspaudio_read_cmd(dsp->audio, 500);
    DBG_PRINT("gst_dspilbcsink_stop send stop. got reply\n");
  }
  gst_dspaudio_close(dsp->audio);
  gst_dspaudio_reset(dsp->audio);
  gst_adapter_clear(dsp->adapter);
  g_mutex_unlock(dsp->audio->dsp_mutex);

#ifdef DEBUG
/*  if(fd != -1) {
    close(fd);
    fd = -1;
  }*/
#endif

  DBG_PRINT("gst_dspilbcsink_stop OK\n");
  return TRUE;
}




/**
 * gst_dspilbcsink_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */

static void
gst_dspilbcsink_set_property(GObject * object,
                            guint prop_id,
                            const GValue * value,
                            GParamSpec * pspec)
{
  GstDSPILBCSink *dsp = GST_DSPILBCSINK (object);

  if(prop_id == DSPAUDIO_PROP_MUTE) {
    DBG_PRINT("ILBC_SINK MUTE SETTING\n");
  }

  if(!gst_dspaudio_set_property(dsp->audio, prop_id, value)) {
    g_warning("trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/**
 * gst_dspilbcsink_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */

static void
gst_dspilbcsink_get_property(GObject * object,
                            guint prop_id,
                            GValue * value,
                            GParamSpec * pspec)
{
  GstDSPILBCSink *dsp = GST_DSPILBCSINK (object);

  if(!gst_dspaudio_get_property(dsp->audio, prop_id, value)) {
    g_warning("trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/**
 *
 *
 */

static GstStateChangeReturn
gst_dspilbcsink_change_state (GstElement * element, GstStateChange transition)
{
  GstDSPILBCSink *dsp = GST_DSPILBCSINK (element);
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;

  switch (transition) {
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      DBG_PRINT("ILBC SINK PLAYING TO PAUSED\n");
      gst_dspaudio_interrupt_render(dsp->audio);
      break;

    default:
      break;
  }

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

/*
  switch (transition) {
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      gst_dspaudio_remove_interrupt(dsp->audio);
      break;

    default:
      break;
  }
*/
  return ret;
}

