/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <unistd.h>
#include <string.h>
#include <gst/gst.h>
#include <gst/audio/gstaudioclock.h>

#include "gstdsppcmsrc.h"

GstElementDetails gst_dsppcmsrc_details = GST_ELEMENT_DETAILS ("DSP PCM Src",
    "Source/Audio",
    "PCM audio src",
    "Makoto Sugano <makoto.sugano@nokia.com>");

static gchar *dsp_node_names[2] =
{
  "/dev/dsptask/pcm_rec",
  "/dev/dsptask/g711_enc"
};


static GstStaticPadTemplate dsppcmsrc_src_template =
    GST_STATIC_PAD_TEMPLATE ("src",
                             GST_PAD_SRC,
                             GST_PAD_ALWAYS,
                             GST_STATIC_CAPS ( "audio/x-raw-int, "
                                 "endianness = (int) { " G_STRINGIFY (G_BYTE_ORDER) " }, "
                                 "signed = (boolean) TRUE, "
                                 "width = (int) 16, "
                                 "depth = (int) 16, "
                                 "rate = (int) 8000, "
                                 "channels = (int) 1;"

                                 "audio/x-alaw, "
                                 "rate = (int) 8000, "
                                 "channels = (int) 1; "

                                 "audio/x-mulaw, "
                                 "rate = (int) 8000, "
                                 "channels = (int) 1"
                              ) );

// Function prototypes

static void gst_dsppcmsrc_class_init (GstDSPPCMSrcClass * klass);
static void gst_dsppcmsrc_base_init (gpointer g_class);
static void gst_dsppcmsrc_init (GstDSPPCMSrc *dspmp3src, GstDSPPCMSrcClass *g_class);
static void gst_dsppcmsrc_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dsppcmsrc_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);

static GstClock *gst_dsppcmsrc_provide_clock (GstElement * elem);
static GstClockTime gst_dsppcmsrc_get_time (GstClock * clock, GstDSPPCMSrc * src);


static GstCaps *gst_dsppcmsrc_getcaps (GstBaseSrc * src);
static gboolean gst_dsppcmsrc_setcaps (GstBaseSrc * src, GstCaps *caps);
static GstFlowReturn gst_dsppcmsrc_create (GstPushSrc * psrc, GstBuffer ** buf);
static gboolean gst_dsppcmsrc_start (GstBaseSrc * bsrc);
static gboolean gst_dsppcmsrc_stop (GstBaseSrc * bsrc);

static GstStateChangeReturn
    gst_dsppcmsrc_change_state (GstElement * element, GstStateChange transition);

/**
 *
 *
 */

void _pcmsrc_do_init()
{
  DBG_PRINT("PCM DO_INIT\n");
}

GST_BOILERPLATE_FULL (GstDSPPCMSrc, gst_dsppcmsrc, GstPushSrc,
                      GST_TYPE_PUSH_SRC, _pcmsrc_do_init);



#define GST_TYPE_PCMSRC_DTX_MODE (gst_dsppcmsrc_dtx_mode_get_type())
static GType
gst_dsppcmsrc_dtx_mode_get_type (void)
{
  static GType dsppcmsrc_dtx_mode_type = 0;
  static GEnumValue dsppcmsrc_dtx_modes[] = {
    {GST_PCMSRC_DTX_MODE_OFF, "Don't use DTX", "off"},
    {GST_PCMSRC_DTX_MODE_ON, "Use DTX", "on"},
    {0, NULL, NULL},
  };

  if (!dsppcmsrc_dtx_mode_type) {
    dsppcmsrc_dtx_mode_type = g_enum_register_static ("GstPCMSrcDTXMode",
        dsppcmsrc_dtx_modes);
  }
  return dsppcmsrc_dtx_mode_type;
}


/**
 * gst_dsppcmsrc_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes PCM src element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */

static void
gst_dsppcmsrc_dispose (GObject * object)
{
  GstDSPPCMSrc *dsp = (GstDSPPCMSrc *) object;

  if(dsp->audio) {
    gst_dspaudio_destroy(dsp->audio);
    dsp->audio = NULL;
  }

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/**
 * gst_dsppcmsrc_base_init:
 * @g_class: PCM src class
 *
 * Does the basic initialization of PCM src element class. This is
 * called by GStreamer framework.
 */

static void
gst_dsppcmsrc_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_add_pad_template (gstelement_class,
                                      gst_static_pad_template_get (&dsppcmsrc_src_template));
  gst_element_class_set_details (gstelement_class, &gst_dsppcmsrc_details);

}

/**
 * gst_dsppcmsrc_class_init:
 * @klass: PCM Src class object
 *
 * Initializes PCM src element class. This is called by GStreamer
 * framework.
 */

static void
gst_dsppcmsrc_class_init (GstDSPPCMSrcClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSrcClass *gstbasesrc_class;
  GstPushSrcClass *gstpushsrc_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbasesrc_class = (GstBaseSrcClass *) klass;
  gstpushsrc_class = (GstPushSrcClass *) klass;

  gobject_class->set_property = gst_dsppcmsrc_set_property;
  gobject_class->get_property = gst_dsppcmsrc_get_property;
  gobject_class->dispose = gst_dsppcmsrc_dispose;

  gstelement_class->provide_clock = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_provide_clock);

   // Use dspaudio utility to install default properties
  gst_dspaudio_install_properties(G_OBJECT_CLASS (klass));

  g_object_class_install_property (gobject_class, DSPPCM_PROP_DTX_MODE,
                                   g_param_spec_enum ("dtx", "DTX mode",
                                       "Discontinuous transmission mode (G.711 only)",
                                       GST_TYPE_PCMSRC_DTX_MODE,        /* enum type */
                                       GST_PCMSRC_DTX_MODE_OFF,         /* default value */
                                       G_PARAM_READWRITE));

  gstelement_class->change_state = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_change_state);

  gstbasesrc_class->start = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_start);
  gstbasesrc_class->stop = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_stop);
  gstbasesrc_class->get_caps = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_getcaps);
  gstbasesrc_class->set_caps = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_setcaps);

  gstpushsrc_class->create = GST_DEBUG_FUNCPTR (gst_dsppcmsrc_create);
}


/**
 * gst_dsppcmsrc_init:
 * @dsp: DSP PCM src object
 *
 * Initializes the given PCM src element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */

static void
gst_dsppcmsrc_init (GstDSPPCMSrc * dsppcmsrc, GstDSPPCMSrcClass *g_class)
{
  dsppcmsrc->audio = gst_dspaudio_new();
  GST_BASE_SRC(dsppcmsrc)->blocksize = 80; // Default size = 10ms frame
  dsppcmsrc->dtxmode = GST_PCMSRC_DTX_MODE_ON;

  // FIXME: Do we need to delete this?
  dsppcmsrc->clock = gst_audio_clock_new ("clock",
                                          (GstAudioClockGetTimeFunc) gst_dsppcmsrc_get_time,
                                          dsppcmsrc);
  DBG_PRINT("PCM CLOCK CREATED\n");
}


/**
 *
 *
 */

static GstCaps *
gst_dsppcmsrc_getcaps (GstBaseSrc * src)
{
  DBG_PRINT("gst_dsppcmsrc_getcaps\n");

  GstCaps *caps = gst_caps_copy (
      gst_pad_get_pad_template_caps (GST_BASE_SRC_PAD(src)));

  DBG_PRINT("OUTPUT CAPS: %s\n", gst_caps_to_string(caps));
  return caps;
}


/**
 * gst_dsppcmsrc_round_bsize:
 *
 *
 * Rounds the blocksize to be dividable by 80 (that is 10ms in time)
 * and makes sure that it will be between certain limits
 */

static guint
gst_dsppcmsrc_round_bsize(guint bsize, guint min, guint max)
{
  guint multiplier = bsize / 80;
  guint retval = multiplier * 80;
  if(retval < min) return min;
  if(retval > max) return max;
  return retval;
}


/**
 * gst_dsppcmsink_setcaps:
 * @basesink: GstBaseSrc
 * @caps: GstCaps that were offered by the connecting element
 *
 * Returns: TRUE if caps can be accepted, otherwise FALSE
 */

static gboolean
gst_dsppcmsrc_setcaps(GstBaseSrc *bsrc, GstCaps * caps)
{
  DBG_PRINT("gst_dsppcmsrc_setcaps\n");

  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (bsrc);
  SPEECH_PARAMS_DATA init_data;

  GstStructure *structure = gst_caps_get_structure (caps, 0);
  const gchar *mimetype = gst_structure_get_name (structure);

  if(!strncmp(mimetype, "audio/x-raw-int", 15)) {
    DBG_PRINT("linear\n");
    init_data.audio_fmt = DSP_AFMT_S16_LE;
    init_data.frame_size = gst_dsppcmsrc_round_bsize(bsrc->blocksize, 80, 800);
    dsp->framesize = init_data.frame_size * sizeof(short int);
    dsp->devindex = 0;
  }
  else if(!strncmp(mimetype, "audio/x-alaw", 12)) {
    DBG_PRINT("a-law\n");
    if(dsp->dtxmode == GST_PCMSRC_DTX_MODE_ON) {
      init_data.audio_fmt = DSP_AFMT_ALAW_DTX;
      init_data.frame_size = gst_dsppcmsrc_round_bsize(bsrc->blocksize, 160, 800);
      dsp->dtx_enabled = TRUE;
      DBG_PRINT("DTX ON\n");
    } else {
      init_data.audio_fmt = DSP_AFMT_ALAW;
      init_data.frame_size = gst_dsppcmsrc_round_bsize(bsrc->blocksize, 80, 800);
      dsp->dtx_enabled = FALSE;
      DBG_PRINT("DTX OFF\n");
    }
    dsp->framesize = init_data.frame_size;
    dsp->devindex = 1;
  }
  else if(!strncmp(mimetype, "audio/x-mulaw", 12)) {
    DBG_PRINT("mu-law\n");
    if(dsp->dtxmode == GST_PCMSRC_DTX_MODE_ON) {
      init_data.audio_fmt = DSP_AFMT_ULAW_DTX;
      init_data.frame_size = gst_dsppcmsrc_round_bsize(bsrc->blocksize, 160, 800);
      dsp->dtx_enabled = TRUE;
      DBG_PRINT("DTX ON\n");
    } else {
      init_data.audio_fmt = DSP_AFMT_ULAW;
      init_data.frame_size = gst_dsppcmsrc_round_bsize(bsrc->blocksize, 80, 800);
      dsp->dtx_enabled = FALSE;
      DBG_PRINT("DTX OFF\n");
    }
    dsp->framesize = init_data.frame_size;
    dsp->devindex = 1;
  }
  else {
    DBG_PRINT("Unknown format\n");
    return FALSE;
  }

  dsp->frametimemillis = init_data.frame_size / 8;

  DBG_PRINT("DSP FRAME SIZE: %d bytes (%d millisecs)\n",
            dsp->framesize, dsp->frametimemillis);

  init_data.dsp_cmd = DSP_CMD_SET_SPEECH_PARAMS;
  init_data.sample_rate = SAMPLE_RATE_8KHZ;
  init_data.ds_stream_ID = 0;
  init_data.stream_priority = dsp->audio->priority;

  if (gst_dspaudio_open(dsp->audio, dsp_node_names[dsp->devindex]) == FALSE) {
    gst_dspaudio_ed_microphone(dsp->audio, FALSE);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_open"));
    return FALSE;
  }

  // Read the junk out...
  gst_dspaudio_flush(&dsp->audio->codec);

  if(!gst_dspaudio_setparams(dsp->audio, (char *)&init_data, sizeof(SPEECH_PARAMS_DATA))) {
    dsp->audio->mode = DSP_MODE_ERROR;
    return FALSE;
  }
  dsp->audio->mode = DSP_MODE_INITIALIZED;
  gst_dspaudio_play(dsp->audio);
  gst_dspaudio_update_dsp_settings(dsp->audio);

  return TRUE;
}


/**
 * gst_dsppcmsrc_create:
 * src:
 * @buffer:
 *
 */

static GstFlowReturn
gst_dsppcmsrc_create (GstPushSrc * psrc, GstBuffer ** buf)
{
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC(psrc);
  guint recv_frames = 1;

  if(dsp->audio->mode == DSP_MODE_ERROR) {
    GST_ELEMENT_ERROR (dsp, RESOURCE, READ, (NULL),
                       ("error cmd: %d status: %d",
                        dsp->audio->error_cmd,
                        dsp->audio->error_status));
    return GST_FLOW_ERROR;
  }


  if(dsp->dtx_enabled) {
    *buf = gst_dspaudio_read_dtx_frame(dsp->audio, dsp->framesize);
  }
  else {
    *buf = gst_dspaudio_read_normal_frame(dsp->audio, dsp->framesize, &recv_frames);
  }

  if(*buf) {
    GST_BUFFER_DURATION(*buf) = recv_frames * dsp->frametimemillis * GST_MSECOND;
    GST_BUFFER_TIMESTAMP(*buf) = dsp->framecount * dsp->frametimemillis * GST_MSECOND;
    gst_buffer_set_caps (*buf, GST_PAD_CAPS (GST_BASE_SRC_PAD (psrc)));
    dsp->framecount += recv_frames;
    return GST_FLOW_OK;
  }
  return GST_FLOW_WRONG_STATE;
}


/**
 *
 *
 */
static gboolean
gst_dsppcmsrc_start (GstBaseSrc * bsrc)
{
  DBG_PRINT("gst_dsppcmsrc_start\n");
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC(bsrc);

  if (gst_dspaudio_ed_microphone(dsp->audio, TRUE) == FALSE) {
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_WRITE, (NULL),
                       ("gst_dspaudio_ed_microphone"));
    return FALSE;
  }

  if (gst_dspaudio_aep_initialize(dsp->audio) == FALSE) {
    gst_dspaudio_ed_microphone(dsp->audio, FALSE);
    GST_ELEMENT_ERROR (dsp, RESOURCE,
                       OPEN_READ_WRITE, (NULL),
                       ("gst_dspaudio_aep_initialize"));
    return FALSE;
  }

  // This is used for clocking
  dsp->framecount = 0;
  DBG_PRINT("gst_dsppcmsrc_start OK\n");
  return TRUE;
}



/**
 *
 *
 */

static gboolean
gst_dsppcmsrc_stop (GstBaseSrc * bsrc)
{
  DBG_PRINT("gst_dsppcmsrc_stop\n");
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (bsrc);

  g_mutex_lock(dsp->audio->dsp_mutex);

  if ( dsp->audio->all_frames_read ||
       dsp->audio->read_sent )
  {
    DBG_PRINT("Waiting reply for CMD_DATA_READ\n");
    gst_dspaudio_wait_buffer(dsp->audio);
    DBG_PRINT("Got it\n");
  }

  if(gst_dspaudio_stop(dsp->audio)) {
    gst_dspaudio_read_cmd(dsp->audio, 0);
  }

  gst_dspaudio_ed_microphone(dsp->audio, FALSE);
  gst_dspaudio_close(dsp->audio);
  gst_dspaudio_aep_close(dsp->audio);
  gst_dspaudio_reset(dsp->audio);
  g_mutex_unlock(dsp->audio->dsp_mutex);

  DBG_PRINT("gst_dsppcmsrc_stop OK\n");
  return TRUE;
}


/**
 * gst_dsppcmsrc_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */

static void
gst_dsppcmsrc_set_property(GObject * object,
                           guint prop_id,
                           const GValue * value,
                           GParamSpec * pspec)
{
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (object);

  if(prop_id == DSPPCM_PROP_DTX_MODE) {
    dsp->dtxmode = g_value_get_enum(value);
  }
  else if(!gst_dspaudio_set_property(dsp->audio, prop_id, value)) {
    g_warning("trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/**
 * gst_dsppcmsrc_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */

static void
gst_dsppcmsrc_get_property(GObject * object,
                           guint prop_id,
                           GValue * value,
                           GParamSpec * pspec)
{
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (object);

  if(prop_id == DSPPCM_PROP_DTX_MODE) {
    g_value_set_enum (value, dsp->dtxmode);
  }
  else if(!gst_dspaudio_get_property(dsp->audio, prop_id, value)) {
    g_warning("trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/**
 *
 *
 */

static GstClock *
gst_dsppcmsrc_provide_clock (GstElement * elem)
{
  DBG_PRINT("gst_dsppcmsrc_provide_clock\n");
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (elem);
  return GST_CLOCK (gst_object_ref (GST_OBJECT (dsp->clock)));
}


/**
 *
 *
 */

static GstClockTime
gst_dsppcmsrc_get_time (GstClock * clock, GstDSPPCMSrc * src)
{
  GstClockTime result;
  result = src->framecount * src->frametimemillis * GST_MSECOND;
  return result;
}


/**
 *
 *
 */

static GstStateChangeReturn
gst_dsppcmsrc_change_state (GstElement * element, GstStateChange transition)
{
  GstDSPPCMSrc *dsp = GST_DSPPCMSRC (element);
  GstStateChangeReturn ret = GST_STATE_CHANGE_SUCCESS;

  switch (transition) {
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      DBG_PRINT("PCM SRC PLAYING TO PAUSED\n");
      gst_dspaudio_interrupt_render(dsp->audio);
      break;

    default:
      break;
  }

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

    switch (transition) {
    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      gst_dspaudio_remove_interrupt(dsp->audio);
      break;

    default:
      break;
  }

  return ret;
}

