/************************************************************
*
*	CyberLink for C
*
*	Copyright (C) Satoshi Konno 2005
*
*	File: upnpdump.c
*
*	Revision:
*       05/11/05
*               - first release.
*
*	10/31/05
*		- Added some M-SEARCH sending actions
*
************************************************************/

#include <cybergarage/upnp/cupnp.h>
/* Logging interface */
#include <cybergarage/util/clog.h>
/* Command line parsing */
#include <argp.h>

#if defined(WIN32) && !defined(__CYGWIN__)
#include <conio.h>
#endif

#if defined(TENGINE)
#if defined(PROCESS_BASE) /* Process based */
#include <basic.h>
#define MBEG	int main( W ac, TC *av[] )
#define MEND	0
#define DIR	"/SYS/bin/"
#else /* T-Kernel based */
#include <basic.h>
#include <tk/tkernel.h>
#include <sys/pinfo.h>
#define MBEG	ER main( INT ac, UB *av[] )
#define MEND	E_SYS
#define DIR	"/SYS/bin/"
#endif
#endif

#if defined(WIN32) && !defined(__CYGWIN__)
#define kbgetkey getch
#else
#define kbgetkey getchar
#endif

/* Additional control commands */

#define CMD_NO_ALTERATIONS 1
#define CMD_LOOP_ACTION_CALLS 2

/* Global flag for controlling CP instantiation */
int single_cp=0;

void ControlDeviceAlter(CgUpnpControlPoint *ctrlPoint, int alteration_mask);
static void SSDPNotifyListner(CgUpnpSSDPPacket *ssdpPkt);
static void EventListener(CgUpnpProperty *prop);
static void DeviceListener(char* udn, CgUpnpDeviceStatus status);

const char *argp_program_version = "Multiple Controlpoint (upnpdump fork)";
const char *argp_program_bug_address = "<mikael.saarenpaa@invalid>";

static struct argp_option options[] =
{
	{ "log-defaults", 'l', 0, 0, "Add default outputs: -e stderr -w stderr -i stdout -d stdout" },
	{ "log-error", 'e', "stdout|stderr|FILE", 0, "Add error output" },
	{ "log-warning", 'w', "stdout|stderr|FILE", 0, "Add warning output" },
	{ "log-info", 'i', "stdout|stderr|FILE", 0, "Add info output" },
	{ "log-debug", 'd', "stdout|stderr|FILE", 0, "Add all debug output" },
	{ "log-debug-1", 'f', "stdout|stderr|FILE", 0, "Add level 1 (basic) debug output" },
	{ "log-debug-2", 'g', "stdout|stderr|FILE", 0, "Add level 2 (stack) debug output" },
	{ "log-debug-3", 'h', "stdout|stderr|FILE", 0, "Add level 3 (testing) debug output" },
	{ "log-debug-4", 'j', "stdout|stderr|FILE", 0, "Add level 4 (low level) debug output" },
	{ "log-debug-5", 'k', "stdout|stderr|FILE", 0, "Add level 5 (low level) debug output" },
	{ "single-cp", 's', 0, 0, "Instantiate only single CP for this session"},
	{0}
};

static error_t parse_opt(int key, char *arg, struct argp_state *state)
{
	switch (key)
	{
		case 'l':
			cg_log_add_target("stdout", ( SEV_INFO | SEV_DEBUG_L1 | SEV_DEBUG_L2 | SEV_DEBUG_L3 ) );
			cg_log_add_target("stderr", ( SEV_ERROR | SEV_WARNING ) );
			break;
		case 'e':
			cg_log_add_target(arg, SEV_ERROR );
			break;
		case 'w':
			cg_log_add_target(arg, SEV_WARNING );
			break;
		case 'i':
			cg_log_add_target(arg, SEV_INFO );
			break;
		case 'd':
			cg_log_add_target(arg, ( SEV_DEBUG_L1 | SEV_DEBUG_L2 | SEV_DEBUG_L3 ) );
			break;
		case 'f':
			cg_log_add_target(arg, SEV_DEBUG_L1 );
			break;
		case 'g':
			cg_log_add_target(arg, SEV_DEBUG_L2 );
			break;
		case 'h':
			cg_log_add_target(arg, SEV_DEBUG_L3 );
			break;
		case 'j':
			cg_log_add_target(arg, SEV_DEBUG_L4 );
			break;
		case 'k':
			cg_log_add_target(arg, SEV_DEBUG_L5 );
			break;
		case 's':
			single_cp=1;
			break;
		default:
			return ARGP_ERR_UNKNOWN;
	}

	return 0;
}

static char args_doc[] = "";
static char doc[] = "multiplecp -- A program to test ClinkC Upnp stack.";

static struct argp argp = {options, parse_opt, args_doc, doc};

/////////////////////////////////////////////////////////////////////////////////
// PrintKeyMessage
/////////////////////////////////////////////////////////////////////////////////

void PrintKeyMessage()
{
	printf("'p' : Print\n");
#if !defined(CG_UPNP_NOUSE_ACTIONCTRL)
	printf("'c' : Control\n");
	printf("'o' : Control (multiple actions)\n");
#endif
#if !defined(CG_UPNP_NOUSE_QUERYCTRL)
	printf("'q' : Query\n");
#endif
#if !defined(CG_UPNP_NOUSE_SUBSCRIPTION)
	printf("'s' : Subscribe\n");
	printf("'u' : Unsubscribe\n");
#endif
	printf("'i' : Host IP address changed\n");
	printf("'l' : Add device listeners\n");
	printf("'m' : Change MX value for M-Search\n");
	printf("'r' : M-Search (rootdevice)\n");
	printf("'h' : M-Search (ssdp:all)\n");
	printf("'v' : M-Search (urn:schemas-upnp-org:device:clock:1)\n");
	printf("'t' : M-Search (urn:schemas-upnp-org:service:timer:1)\n");
	printf("'a' : Do all searches four times in a loop.\n");
	printf("'x' : eXit\n");
}

/////////////////////////////////////////////////////////////////////////////////
// Notify Listener
/////////////////////////////////////////////////////////////////////////////////

void SSDPNotifyListener1(CgUpnpSSDPPacket *ssdpPkt)
{
	cg_log_debug("Controlpoint 1 SSDPNotify listener called.\n");
	SSDPNotifyListner(ssdpPkt);
}

void SSDPNotifyListener2(CgUpnpSSDPPacket *ssdpPkt)
{
	cg_log_debug("Controlpoint 2 SSDPNotify listener called.\n");
	SSDPNotifyListner(ssdpPkt);
}

static void SSDPNotifyListner(CgUpnpSSDPPacket *ssdpPkt)
{
	if (cg_upnp_ssdp_packet_isdiscover(ssdpPkt) == TRUE) {
		cg_log_debug_a("ssdp:discover : ST = %s\n",
			cg_upnp_ssdp_packet_getst(ssdpPkt)); 
	}
	else if (cg_upnp_ssdp_packet_isalive(ssdpPkt) == TRUE) {
		cg_log_debug_a("ssdp:alive : uuid = %s, NT = %s, location = %s\n",
			cg_upnp_ssdp_packet_getusn(ssdpPkt), 
			cg_upnp_ssdp_packet_getnt(ssdpPkt), 
			cg_upnp_ssdp_packet_getlocation(ssdpPkt)); 
	}
	else if (cg_upnp_ssdp_packet_isbyebye(ssdpPkt) == TRUE) {
		cg_log_debug_a("ssdp:byebye : uuid = %s, NT = %s\n",
			cg_upnp_ssdp_packet_getusn(ssdpPkt), 
			cg_upnp_ssdp_packet_getnt(ssdpPkt));
	}
	cg_upnp_ssdp_packet_print(ssdpPkt); 
}

/////////////////////////////////////////////////////////////////////////////////
// Print Device
/////////////////////////////////////////////////////////////////////////////////

void PrintDeviceInfo(CgUpnpDevice *dev, int indent)
{
	char indentStr[128];
	int n;
	CgUpnpService *service;
	int serviceCnt;
	CgUpnpAction *action;
	int actionCnt;
	CgUpnpArgumentList *arg;
	int argCnt;
	CgUpnpStateVariable *stateVar;
	int stateVarCnt;
	
	for (n=0; n<indent && n<(sizeof(indentStr)-1); n++)
		indentStr[n] = ' ';
	indentStr[n] = '\0';
	
	serviceCnt = 0;
	for (service = cg_upnp_device_getservices(dev); service != NULL; service = cg_upnp_service_next(service)) {
		printf("%s service[%d] = %s\n", indentStr, ++serviceCnt, cg_upnp_service_getservicetype(service));
		actionCnt = 0;
		for (action = cg_upnp_service_getactions(service); action != NULL; action = cg_upnp_action_next(action)) {
			printf("%s  action[%d] = %s\n", indentStr, ++actionCnt, cg_upnp_action_getname(action));
			argCnt = 0;
			for (arg = cg_upnp_action_getarguments(action); arg != NULL; arg = cg_upnp_argument_next(arg)) {
				printf("%s   arg[%d] = %s\n", indentStr, ++argCnt, cg_upnp_argument_getname(arg));
			}
		}
		stateVarCnt = 0;
		for (stateVar = cg_upnp_service_getstatevariables(service); stateVar != NULL; stateVar = cg_upnp_statevariable_next(stateVar)) {
			printf("%s  stateVar[%d] = %s = %s\n", 
			       indentStr, 
			       ++stateVarCnt, 
			       cg_upnp_statevariable_getname(stateVar),
			       cg_upnp_statevariable_getvalue(stateVar));
		}
	}
}

PrintDevice(CgUpnpDevice *dev, int indent)
{
	CgUpnpDevice *childDev;
	
	PrintDeviceInfo(dev, indent);

	for (childDev = cg_upnp_device_getdevices(dev); childDev != NULL; childDev = cg_upnp_device_next(childDev))
		PrintDevice(childDev, indent+1);
}


void PrintControlPointDevice(CgUpnpControlPoint *ctrlPoint)
{
	CgUpnpDevice *dev;
	int devCnt;

	printf("CP name: %s\n", (char *)ctrlPoint->userData);
		
	printf("Device Num = %d\n", cg_upnp_controlpoint_getndevices(ctrlPoint));
	
	devCnt = 0;
	for (dev = cg_upnp_controlpoint_getdevices(ctrlPoint); dev != NULL; dev = cg_upnp_device_next(dev)) {
		printf("[%d] = %s\n", ++devCnt, cg_upnp_device_getfriendlyname(dev));
		PrintDevice(dev, 1);
	}
}

/////////////////////////////////////////////////////////////////////////////////
// Select*
/////////////////////////////////////////////////////////////////////////////////

CgUpnpDevice *SelectDevice(CgUpnpControlPoint *ctrlPoint)
{
	CgUpnpDevice *dev, *childDev;
	CgUpnpDevice *devArray['z'-'a']; /* Hmm... */
	int n;
	char key;
	int devNum;

	printf("CP name: %s\n", (char *)ctrlPoint->userData);
	
	n = 0;
	for (dev = cg_upnp_controlpoint_getdevices(ctrlPoint); dev != NULL; dev = cg_upnp_device_next(dev)) {
		key = 'a' + n;
		if ('z' < key) break;
		printf(" [%c] = %s\n", key, cg_upnp_device_getfriendlyname(dev));
		devArray[n] = dev;
		
		for (childDev = cg_upnp_device_getdevices(dev); ++n, childDev != NULL; childDev = cg_upnp_device_next(childDev)) {
			key = 'a' + n;
			if ('z' < key) break;
			printf("   [%c] = %s\n", key, cg_upnp_device_getfriendlyname(childDev));
			devArray[n] = childDev;
		}
	}
	if (n == 0)
		return NULL;
	printf("Select Device : ");
	key = kbgetkey();
	key = tolower(key);
	printf("%c\n", key);
	
	if (!isalpha(key))
		return NULL;
	
	devNum = key - 'a';
	if (devNum >= n) return NULL;

	return devArray[devNum];
}

CgUpnpService *SelectService(CgUpnpDevice *dev)
{
	CgUpnpService *service;
	int n;
	char key;
	int serviceNum;

	n = 0;
	for (service = cg_upnp_device_getservices(dev); service != NULL; service = cg_upnp_service_next(service)) {
		key = 'a' + n;
		if ('z' < key)
			break;
		printf(" [%c] = %s\n", key, cg_upnp_service_getservicetype(service));
		n++;
	}
	printf("Select Service : ");
	key = kbgetkey();
	key = tolower(key);
	printf("%c\n", key);
	
	if (!isalpha(key))
		return NULL;
	
	serviceNum = key - 'a';
	service = cg_upnp_device_getservices(dev);
	for (n=0; n<serviceNum; n++)
		service = cg_upnp_service_next(service);

	return service;
}

CgUpnpAction *SelectAction(CgUpnpService *service)
{
	CgUpnpAction *action;
	int n;
	char key;
	int actionNum;

	n = 0;
	for (action = cg_upnp_service_getactions(service); action != NULL; action = cg_upnp_action_next(action)) {
		key = 'a' + n;
		if ('z' < key)
			break;
		printf(" [%c] = %s\n", key, cg_upnp_action_getname(action));
		n++;
	}
	printf("Select Action : ");
	key = kbgetkey();
	key = tolower(key);
	printf("%c\n", key);
	
	if (!isalpha(key))
		return NULL;
	
	actionNum = key - 'a';
	action = cg_upnp_service_getactions(service);
	for (n=0; n<actionNum; n++)
		action = cg_upnp_action_next(action);

	return action;
}

CgUpnpStateVariable *SelectStateVariable(CgUpnpService *service)
{
	CgUpnpStateVariable *stateVar;
	int n;
	char key;
	int serviceNum;

	n = 0;
	for (stateVar = cg_upnp_service_getstatevariables(service); stateVar != NULL; stateVar = cg_upnp_statevariable_next(stateVar)) {
		key = 'a' + n;
		if ('z' < key)
			break;
		printf(" [%c] = %s\n", key, cg_upnp_statevariable_getname(stateVar));
		n++;
	}
	printf("Select StateVariable : ");
	key = kbgetkey();
	key = tolower(key);
	printf("%c\n", key);
	
	if (!isalpha(key))
		return NULL;
	
	serviceNum = key - 'a';
	stateVar = cg_upnp_service_getstatevariables(service);
	for (n=1; n<serviceNum; n++)
		stateVar = cg_upnp_statevariable_next(stateVar);

	return stateVar;
}

/////////////////////////////////////////////////////////////////////////////////
// Action
/////////////////////////////////////////////////////////////////////////////////

#if !defined(CG_UPNP_NOUSE_ACTIONCTRL)

void ControlDevice(CgUpnpControlPoint *ctrlPoint)
{
	printf("CP name: %s\n", (char *)ctrlPoint->userData);
	ControlDeviceAlter(ctrlPoint, CMD_NO_ALTERATIONS);
}

void ControlDeviceAlter(CgUpnpControlPoint *ctrlPoint, int alteration_mask)
{
	CgUpnpDevice *selDev;
	CgUpnpService *selService;
	CgUpnpAction *selAction;
	BOOL actionSuccess;
	CgUpnpArgument *arg;
	char argValue[2048];

	printf("CP name: %s\n", (char *)ctrlPoint->userData);
	
	printf("Control Device\n");
	
	selDev = SelectDevice(ctrlPoint);
	if (selDev == NULL)
		return;
	selService = SelectService(selDev);
	if (selService == NULL)
		return;
	selAction = SelectAction(selService);
	if (selAction == NULL)
		return;
	
	for (arg = cg_upnp_action_getarguments(selAction); arg; arg = cg_upnp_argument_next(arg)) {
		if (cg_upnp_argument_isindirection(arg) == TRUE) {
			printf("%s : ", cg_upnp_argument_getname(arg));
			if (scanf("%s", argValue) == 1)
				cg_upnp_argument_setvalue(arg, argValue);
		}
	}

	/* NOTE: Go through selAction memory management... */
	if (alteration_mask & CMD_LOOP_ACTION_CALLS)
	{
		int loop_count, i;

		printf("\nHow many times action should be sent?");
		
		if ( 1 == scanf("%d", &loop_count) )
		{
			printf("\n");
			for (i=0; i<loop_count; i++)
			{
				actionSuccess = cg_upnp_action_post(selAction);
				printf("Control Result(%d)\n", (int)actionSuccess);

				for (	arg = cg_upnp_action_getarguments(selAction); 
					arg; 
					arg = cg_upnp_argument_next(arg)) 
				{
					if (cg_upnp_argument_isoutdirection(arg) == TRUE)
						printf(" %s = %s\n", 
								cg_upnp_argument_getname(arg), 
								cg_upnp_argument_getvalue(arg));
				}
			}

			sleep(2);
		}
	}

	if ((alteration_mask & CMD_NO_ALTERATIONS) == CMD_NO_ALTERATIONS)
	{	
		actionSuccess = cg_upnp_action_post(selAction);
		
		printf("Control Result(%d)\n", (int)actionSuccess);
		for (arg = cg_upnp_action_getarguments(selAction); arg; arg = cg_upnp_argument_next(arg)) {
			if (cg_upnp_argument_isoutdirection(arg) == TRUE)
				printf(" %s = %s\n", cg_upnp_argument_getname(arg), cg_upnp_argument_getvalue(arg));
		}
	}
}

#endif

////////////////////////////////////////////////////////////////////////////////
// Query
/////////////////////////////////////////////////////////////////////////////////

#if !defined(CG_UPNP_NOUSE_QUERYCTRL)

void QueryDevice(CgUpnpControlPoint *ctrlPoint)
{
	CgUpnpDevice *selDev;
	CgUpnpService *selService;
	CgUpnpStateVariable *selStateVar;
	BOOL querySuccess;
	char *stateValue;

	printf("CP name: %s\n", (char *)ctrlPoint->userData);
	
	printf("Query Device\n");
	
	selDev = SelectDevice(ctrlPoint);
	if (selDev == NULL)
		return;
	selService = SelectService(selDev);
	if (selService == NULL)
		return;
	selStateVar = SelectStateVariable(selService);
	if (selStateVar == NULL)
		return;
	
	querySuccess = cg_upnp_statevariable_post(selStateVar);
	
	stateValue = cg_upnp_statevariable_getvalue(selStateVar);

	printf("Query Result(%d) = %s\n",
		(int)querySuccess, 
		(stateValue != NULL) ? stateValue : "");
}

#endif

////////////////////////////////////////////////////////////////////////////////
// Subscribe
/////////////////////////////////////////////////////////////////////////////////

#if !defined(CG_UPNP_NOUSE_SUBSCRIPTION)

void SubscribeService(CgUpnpControlPoint *ctrlPoint)
{
	CgUpnpDevice *selDev;
	CgUpnpService *selService;
	BOOL subSuccess;

	printf("CP name: %s\n", (char *)ctrlPoint->userData);
	
	printf("Subscribe Device\n");
	
	cg_upnp_controlpoint_lock(ctrlPoint);
	selDev = SelectDevice(ctrlPoint);
	if (selDev == NULL)
	{
		cg_upnp_controlpoint_unlock(ctrlPoint);
		return;
	}
	selService = SelectService(selDev);
	if (selService == NULL)
	{
		cg_upnp_controlpoint_unlock(ctrlPoint);
		return;
	}
	subSuccess = cg_upnp_controlpoint_subscribe(ctrlPoint, selService, 300);
	
	printf("Subscribe Result(%d) = %s\n",
		(int)subSuccess, 
		(subSuccess == TRUE) ? cg_upnp_service_getsubscriptionsid(selService) : "");
	cg_upnp_controlpoint_unlock(ctrlPoint);
	
}

#endif

////////////////////////////////////////////////////////////////////////////////
// Unsubscribe
/////////////////////////////////////////////////////////////////////////////////

#if !defined(CG_UPNP_NOUSE_SUBSCRIPTION)

void UnsubscribeService(CgUpnpControlPoint *ctrlPoint)
{
	CgUpnpDevice *selDev;
	CgUpnpService *selService;
	BOOL subSuccess;

	printf("CP name: %s\n", (char *)ctrlPoint->userData);
	
	printf("Query Device\n");
	
	selDev = SelectDevice(ctrlPoint);
	if (selDev == NULL)
		return;
	selService = SelectService(selDev);
	if (selService == NULL)
		return;

	subSuccess = cg_upnp_controlpoint_unsubscribe(ctrlPoint, selService);

	printf("Unsubscribe Result(%d)\n",
		(int)subSuccess);
}

#endif

/////////////////////////////////////////////////////////////////////////////////
// Set MX value
/////////////////////////////////////////////////////////////////////////////////

void SetMXValue(CgUpnpControlPoint *ctrlPoint)
{
        unsigned int mxValue;

	printf("CP name: %s\n", (char *)ctrlPoint->userData);

        printf("Give new MX value: ");
        if (scanf("%u", &mxValue) == 1) {
                printf( "%d", mxValue );
                cg_upnp_controlpoint_setssdpsearchmx(ctrlPoint, mxValue);
        }
        printf( "\n" );
}

/* Device listeners */

void DeviceListener1(char* udn, CgUpnpDeviceStatus status)
{
	printf("Controlpoint 1 Device listener called.\n");
	DeviceListener(udn, status);
}

void DeviceListener2(char* udn, CgUpnpDeviceStatus status)
{
	printf("Controlpoint 2 Device listener called.\n");
	DeviceListener(udn, status);
}

static void DeviceListener(char* udn, CgUpnpDeviceStatus status)
{
	puts(udn);

	switch (status)
	{
		case CgUpnpDeviceStatusAdded:
			printf("Device was added.\n");
			break;

		case CgUpnpDeviceStatusUpdated:
			printf("Device was updated.\n");
			break;

		case CgUpnpDeviceStatusInvalid:
			printf("Device status is invalid.\n");
			break;

		case CgUpnpDeviceStatusRemoved:
			printf("Device was removed.\n");
			break;

		default:
			printf("Unknown device state!\n");
			break;
	}
}

/////////////////////////////////////////////////////////////////////////////////
// Event
/////////////////////////////////////////////////////////////////////////////////

void EventListener1(CgUpnpProperty *prop)
{
	printf("Controlpoint 1 Event listener called.\n");
	EventListener(prop);
}

void EventListener2(CgUpnpProperty *prop)
{
	printf("Controlpoint 2 Event listener called.\n");
	EventListener(prop);
}

static void EventListener(CgUpnpProperty *prop)
{
	printf("Property Changed (%s) = %s\n",
		cg_upnp_property_getname(prop),
		cg_upnp_property_getvalue(prop));
}

static CgUpnpControlPoint *selectControlPoint(CgUpnpControlPoint *ctrlPoint[], int cp_count)
{
	int i, selected_cp=0;

	if ( single_cp )
		return ctrlPoint[0];
	
	printf("Available control points:\n");

	for ( i=0; i<cp_count; i++)
	{
		printf("%d: %s\n", i, (char *)ctrlPoint[i]->userData);
	}

	do
	{
		printf("Select control point (give first number from the list): ");

		selected_cp = (int)kbgetkey() - 48;
		printf("%d\n", selected_cp);
	}
	while (selected_cp >= cp_count);

	printf("\n");

	return ctrlPoint[selected_cp];
}

/////////////////////////////////////////////////////////////////////////////////
// main
/////////////////////////////////////////////////////////////////////////////////

#if defined(TENGINE)
MBEG
#else
int main( int argc, char* argv[] )
#endif
{
	CgUpnpControlPoint *ctrlPoint[2];
	int key, ctrlPointCount;

	argp_parse(&argp, argc, argv, 0, 0, NULL);

	cg_log_error("Test error\n");
	cg_log_warning("Test warning\n");
	cg_log_info("Test info\n");
	cg_log_debug("Test debug\n");
	cg_log_debug_a("Low level debug\n");
	
	ctrlPoint[0] = cg_upnp_controlpoint_new();
	ctrlPoint[0]->userData = (void *)"Control point 1";
	
	ctrlPoint[1] = cg_upnp_controlpoint_new();
	ctrlPoint[1]->userData = (void *)"Control point 2";

	ctrlPointCount = 2;
	
	cg_upnp_controlpoint_setssdplistener(ctrlPoint[0], SSDPNotifyListener1);
	cg_upnp_controlpoint_setssdplistener(ctrlPoint[1], SSDPNotifyListener2);
	cg_upnp_controlpoint_seteventlistener(ctrlPoint[0], EventListener1);
	cg_upnp_controlpoint_seteventlistener(ctrlPoint[1], EventListener2);

	if (cg_upnp_controlpoint_start(ctrlPoint[0]) == FALSE) {
		cg_log_error("Couldn't start control point 1 !!");
		exit(1);
	}

	if (cg_upnp_controlpoint_start(ctrlPoint[1]) == FALSE) {
		cg_log_error("Couldn't start control point 2 !!");
		exit(1);
	}
	
	PrintKeyMessage();

#if (!defined(WIN32) || defined(__CYGWIN__)) && !defined(TENGINE)
	kbinit();
#endif
	
	key = 0;
	do
	{
#if !defined(TENGINE)
		key = kbgetkey();
		key = toupper(key);
		switch (key) {
		case 'P':
			PrintControlPointDevice(selectControlPoint(ctrlPoint, ctrlPointCount));
			break;
#if !defined(CG_UPNP_NOUSE_ACTIONCTRL)
		case 'C':
			ControlDeviceAlter(selectControlPoint(ctrlPoint, ctrlPointCount), CMD_NO_ALTERATIONS);
			break;

		case 'O':
			ControlDeviceAlter(selectControlPoint(ctrlPoint, ctrlPointCount), CMD_LOOP_ACTION_CALLS);
			break;
#endif
#if !defined(CG_UPNP_NOUSE_QUERYCTRL)
		case 'Q':
			QueryDevice(selectControlPoint(ctrlPoint, ctrlPointCount));
			break;
#endif
#if !defined(CG_UPNP_NOUSE_SUBSCRIPTION)
		case 'S':
			SubscribeService(selectControlPoint(ctrlPoint, ctrlPointCount));
			break;
		case 'U':
			UnsubscribeService(selectControlPoint(ctrlPoint, ctrlPointCount));
			break;
#endif
		case 'I':
			printf("Indicating IP address change to the stack\n");
		        cg_upnp_controlpoint_ipchanged(selectControlPoint(ctrlPoint, ctrlPointCount));
			printf("Done (Indicating IP address change to the stack)\n");
			break;
		case 'L':
			printf("Adding device listeners... ");
			cg_upnp_controlpoint_setdevicelistener(ctrlPoint[0], DeviceListener1);
			cg_upnp_controlpoint_setdevicelistener(ctrlPoint[1], DeviceListener2);
			printf("Done (Adding device listeners).\n");
			break;
			
                case 'M':
                        SetMXValue(selectControlPoint(ctrlPoint, ctrlPointCount));
                        break;
		case 'R':
	          printf("M-Search upnp::rootdevice\n");
		  cg_upnp_controlpoint_search(selectControlPoint(ctrlPoint, ctrlPointCount), CG_UPNP_ST_ROOT_DEVICE);
		  break;
		case 'H':
	          printf("M-Search ssdp:all\n");
		  cg_upnp_controlpoint_search(selectControlPoint(ctrlPoint, ctrlPointCount), "ssdp:all");
		  break;
		case 'V':
	          printf("M-Search device type\n");
		  cg_upnp_controlpoint_search(selectControlPoint(ctrlPoint, ctrlPointCount), "urn:schemas-upnp-org:device:clock:1");
		  break;
		case 'T':
	          printf("M-Search service type\n");
		  cg_upnp_controlpoint_search(selectControlPoint(ctrlPoint, ctrlPointCount), "urn:schemas-upnp-org:service:timer:1");
		  break;
		case 'A':
		  {
			  int i;
			  CgUpnpControlPoint *ctrlTemp = selectControlPoint(ctrlPoint, ctrlPointCount);

			  printf("Doing all searches four times...\n");
			  
			  for (i=0; i<4; i++)
			  {
				  cg_upnp_controlpoint_search(ctrlTemp, CG_UPNP_ST_ROOT_DEVICE);
				  cg_upnp_controlpoint_search(ctrlTemp, "ssdp:all");
				  cg_upnp_controlpoint_search(ctrlTemp, "urn:schemas-upnp-org:device:clock:1");
				  cg_upnp_controlpoint_search(ctrlTemp, "urn:schemas-upnp-org:service:timer:1");
			  }
			  
			  printf("Done (doing all searches four times).\n");
		  }
		  break;
		case 'X':
			break;
		default:
			PrintKeyMessage();
		}
#else
		cg_wait(1000);
#endif
	} while( key != 'X');

#if (!defined(WIN32) || defined(__CYGWIN__)) && !defined(TENGINE)
	kbexit();
#endif
	
	cg_upnp_controlpoint_stop(ctrlPoint[0]);
	cg_upnp_controlpoint_delete(ctrlPoint[0]);

	cg_upnp_controlpoint_stop(ctrlPoint[1]);
	cg_upnp_controlpoint_delete(ctrlPoint[1]);
	
	return(0);
}

