/******************************************************************
*
*	CyberNet for C
*
*	Copyright (C) Satoshi Konno 2005
*
*       Copyright (C) 2006 Nokia Corporation. All rights reserved.
*
*       This is licensed under BSD-style license with patent exclusion,
*       see file COPYING.
*
*	File: chttp_packet.cpp
*
*	Revision:
*
*	01/25/05
*		- first revision
*	08/21/05
*		- Thanks for Theo Beisch <theo.beisch@gmx.de>
*		- Changed cg_http_packet_read() to store length in HTTPPacket.
*	10/25/05
*		- The following functions move to define macros.
*		  cg_http_packet_setcontent()
*		  cg_http_packet_setcontentpointer()
*		  cg_http_packet_getcontent()
*		- cg_http_packet_setncontent() is added.
*	10/31/05
*		- cg_http_packet_sethost:
*		  port was an excess parameter for s(n)printf when port <= 0
*	11/11/05
*		- Added cg_http_packet_setheaderlonglong() and cg_http_packet_getheaderlonglong().
*		- Extended cg_http_packet_setcontentlength() and cg_http_packet_getcontentlength() to 64bit
*		  when the compiler is supported C99 or the platform is WIN32.
*
******************************************************************/

#include <cybergarage/http/chttp.h>
#include <cybergarage/util/clog.h>

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <cybergarage/net/cinterface.h>

#include <limits.h>

#define READBUF_LENGTH 1024

/****************************************
* cg_http_packet_new
****************************************/

CgHttpPacket *cg_http_packet_new()
{
	cg_log_debug_l4("Entering...\n");

	CgHttpPacket *httpPkt= (CgHttpPacket *)malloc(sizeof(CgHttpPacket));

	if ( NULL != httpPkt )
	{
		cg_http_packet_init(httpPkt);
	}

	return httpPkt;

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_delete
****************************************/

void cg_http_packet_delete(CgHttpPacket *httpPkt)
{
	cg_log_debug_l4("Entering...\n");

	cg_http_packet_clean(httpPkt);
	free(httpPkt);

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_init
****************************************/

void cg_http_packet_init(CgHttpPacket *httpPkt)
{
	cg_log_debug_l4("Entering...\n");

	httpPkt->headerList = cg_http_headerlist_new();
	httpPkt->content = cg_string_new();

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_clean
****************************************/

void cg_http_packet_clean(CgHttpPacket *httpPkt)
{
	cg_log_debug_l4("Entering...\n");

	cg_http_headerlist_delete(httpPkt->headerList);
	cg_string_delete(httpPkt->content);

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_clear
****************************************/

void cg_http_packet_clear(CgHttpPacket *httpPkt)
{
	cg_log_debug_l4("Entering...\n");

	cg_http_headerlist_clear(httpPkt->headerList);
	cg_string_setvalue(httpPkt->content, NULL);

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_setheader
****************************************/

void cg_http_packet_setheadervalue(CgHttpPacket *httpPkt, char* name, char *value)
{
	cg_log_debug_l4("Entering...\n");

	cg_http_headerlist_set(httpPkt->headerList, name, value);

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_setheaderinteger
****************************************/

void cg_http_packet_setheaderinteger(CgHttpPacket *httpPkt, char* name, int value)
{
	cg_log_debug_l4("Entering...\n");

	char svalue[CG_STRING_INTEGER_BUFLEN];
	cg_http_packet_setheadervalue(httpPkt, name, cg_int2str(value, svalue, sizeof(svalue)));

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_setheaderlong
****************************************/

void cg_http_packet_setheaderlong(CgHttpPacket *httpPkt, char* name, long value)
{
	cg_log_debug_l4("Entering...\n");

	char svalue[CG_STRING_LONG_BUFLEN];
	cg_http_packet_setheadervalue(httpPkt, name, cg_long2str(value, svalue, sizeof(svalue)));

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_setheaderlonglong
****************************************/

#if defined(__USE_ISOC99)
void cg_http_packet_setheaderlonglong(CgHttpPacket *httpPkt, char* name, long long value)
{
	cg_log_debug_l4("Entering...\n");

	char svalue[CG_STRING_LONGLONG_BUFLEN];
	cg_http_packet_setheadervalue(httpPkt, name, cg_longlong2str(value, svalue, sizeof(svalue)));

	cg_log_debug_l4("Leaving...\n");
}
#elif defined(WIN32)
void cg_http_packet_setheaderlonglong(CgHttpPacket *httpPkt, char* name, __int64 value)
{
	cg_log_debug_l4("Entering...\n");

	char svalue[CG_STRING_LONGLONG_BUFLEN];
	cg_http_packet_setheadervalue(httpPkt, name, cg_longlong2str(value, svalue, sizeof(svalue)));

	cg_log_debug_l4("Leaving...\n");
}
#else
void cg_http_packet_setheaderlonglong(CgHttpPacket *httpPkt, char* name, long long value)
{
	cg_log_debug_l4("Entering...\n");


	cg_log_debug_l4("Leaving...\n");
}
#endif


/****************************************
* cg_http_packet_getheadervalue
****************************************/

char *cg_http_packet_getheadervalue(CgHttpPacket *httpPkt, char* name)
{
	cg_log_debug_l4("Entering...\n");

	return cg_http_headerlist_getvalue(httpPkt->headerList, name);

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_getheadervalue
****************************************/

int cg_http_packet_getheaderinteger(CgHttpPacket *httpPkt, char* name)
{
	cg_log_debug_l4("Entering...\n");

	char *value = cg_http_packet_getheadervalue(httpPkt, name); 
	return (value != NULL) ? atoi(value) : 0;

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_getheadervalue
****************************************/

long cg_http_packet_getheaderlong(CgHttpPacket *httpPkt, char* name)
{
	cg_log_debug_l4("Entering...\n");

	char *value = cg_http_packet_getheadervalue(httpPkt, name); 
	return (value != NULL) ? atol(value) : 0;

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_getheaderlonglong
****************************************/

#if defined(__USE_ISOC99) || defined(WIN32)
#if defined(__USE_ISOC99)
long long cg_http_packet_getheaderlonglong(CgHttpPacket *httpPkt, char* name)
#elif defined(WIN32)
__int64 cg_http_packet_getheaderlonglong(CgHttpPacket *httpPkt, char* name)
#endif
{
	cg_log_debug_l4("Entering...\n");

	char *value = cg_http_packet_getheadervalue(httpPkt, name); 
	return (value != NULL) ? cg_str2longlong(value) : 0;

	cg_log_debug_l4("Leaving...\n");
}
#endif

/****************************************
* cg_http_packet_getheadervalue
****************************************/

void cg_http_packet_sethost(CgHttpPacket *httpPkt, char *addr, int port)
{
	cg_log_debug_l4("Entering...\n");

	char host[CG_NET_IPV4_ADDRSTRING_MAXSIZE + CG_STRING_INTEGER_BUFLEN];

#if defined(HAVE_SNPRINTF)
	if (0 < port) {
		if (cg_net_isipv6address(addr) == TRUE)
			snprintf(host, sizeof(host), "[%s]:%d", addr, port);
		else
			snprintf(host, sizeof(host), "%s:%d", addr, port);
	}
	else {
		if (cg_net_isipv6address(addr) == TRUE)
			snprintf(host, sizeof(host), "[%s]", addr);
		else
			snprintf(host, sizeof(host), "%s", addr);
	}
#else	
	if (0 < port) {
		if (cg_net_isipv6address(addr) == TRUE)
			sprintf(host, "[%s]:%d", addr, port);
		else
			sprintf(host, "%s:%d", addr, port);
	}
	else {
		if (cg_net_isipv6address(addr) == TRUE)
			sprintf(host, "[%s]", addr);
		else
			sprintf(host, "%s", addr);
	}
#endif

	cg_http_packet_setheadervalue(httpPkt, CG_HTTP_HOST, host);

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_post
****************************************/

void cg_http_packet_post(CgHttpPacket *httpPkt, CgSocket *sock)
{
	cg_log_debug_l4("Entering...\n");

	CgHttpHeader *header;
	char *name, *value;
	char *content;
	long contentLen;
	
	/**** send headers ****/
	for (header = cg_http_packet_getheaders(httpPkt); header != NULL; header = cg_http_header_next(header)) {
		name = cg_http_header_getname(header);
		if (name == NULL)
			continue;
		cg_socket_write(sock, name, cg_strlen(name));
		cg_socket_write(sock, CG_HTTP_COLON, sizeof(CG_HTTP_COLON)-1);
		cg_socket_write(sock, CG_HTTP_SP, sizeof(CG_HTTP_SP)-1);
		value = cg_http_header_getvalue(header);
		if (value != NULL)
			cg_socket_write(sock, value, cg_strlen(value));
		cg_socket_write(sock, CG_HTTP_CRLF, sizeof(CG_HTTP_CRLF)-1
		);
	}
	cg_socket_write(sock, CG_HTTP_CRLF, sizeof(CG_HTTP_CRLF)-1);
	
	/**** send content ****/
	content = cg_http_packet_getcontent(httpPkt);
	contentLen = cg_http_packet_getcontentlength(httpPkt);
	if (content != NULL && 0 < contentLen)
		cg_socket_write(sock, content, contentLen);

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_read_headers
****************************************/

void cg_http_packet_read_headers(CgHttpPacket *httpPkt, CgSocket *sock, char *lineBuf, int lineBufSize)
{
	cg_log_debug_l4("Entering...\n");

	CgStringTokenizer *strTok;
	CgHttpHeader *header;
	int readLen;
	char *name, *value;
	
	while (1) {
		readLen = cg_socket_readline(sock, lineBuf, lineBufSize);
		if (readLen <= 2)
			break;
		name = NULL;
		value = NULL;
		strTok = cg_string_tokenizer_new(lineBuf, CG_HTTP_HEADERLINE_DELIM);
		if (cg_string_tokenizer_hasmoretoken(strTok) == TRUE)
			name = cg_string_tokenizer_nexttoken(strTok);
		if (cg_string_tokenizer_hasmoretoken(strTok) == TRUE) {
			value = cg_string_tokenizer_nextalltoken(strTok);
			cg_strrtrim(value, CG_HTTP_HEADERLINE_DELIM, cg_strlen(CG_HTTP_HEADERLINE_DELIM));
		}
		if (name != NULL) {
			if (value == NULL)
				value = "";
			header = cg_http_header_new();
			cg_http_header_setname(header, name);
			cg_http_header_setvalue(header, value);
			cg_http_packet_addheader(httpPkt, header);
		}
		cg_string_tokenizer_delete(strTok);
	}

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_read_chunk
****************************************/

long cg_http_packet_read_chunk(CgHttpPacket *httpPkt, CgSocket *sock, char *lineBuf, int lineBufSize)
{
	cg_log_debug_l4("Entering...\n");

	long readLen = 0;
	long conLen = 0;
	int tries = 0;
	char *content = NULL;
	
	/* Read chunk header */
	readLen = cg_socket_readline(sock, lineBuf, lineBufSize);
	
	conLen = cg_strhex2long(lineBuf);
	if (conLen < 1) return 0;
	
	content = (char *)malloc(conLen+1);

	if (content == NULL)
	{
		cg_log_debug_s("Memory allocation problem!\n");
		return 0;
	}
		
	content[conLen] = 0;
	
	readLen = 0;
	/* Read content until conLen is reached, or tired of trying */
	while (readLen < conLen && tries < 20)
	{
		readLen += cg_socket_read(sock, (content+readLen), (conLen-readLen));
		tries++;
	}
	
	/* Append content to packet */
	cg_http_packet_appendncontent(httpPkt, content, readLen);
	free(content); content = NULL;
	
	if (readLen == conLen)
	{
		/* Read CRLF bytes */
		cg_socket_readline(sock, lineBuf, lineBufSize);
	}
	
	return readLen;

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_read_body
****************************************/

BOOL cg_http_packet_read_body(CgHttpPacket *httpPkt, CgSocket *sock, char *lineBuf, int lineBufSize)
{
	cg_log_debug_l4("Entering...\n");

	long readLen;
	long conLen;
	char *content;
	char readBuf[READBUF_LENGTH + 1];
	int tries = 0;

	conLen = cg_http_packet_getcontentlength(httpPkt);
	content = NULL;
	if (0 < conLen) {
		content = (char *)malloc(conLen+1);
		if (content == NULL)
		{
			cg_log_debug_s("Memory allocation problem!\n");
			return FALSE;
		}
		
		content[0] = '\0';
		readLen = 0;
		
		/* Read content until conLen is reached, or tired of trying */
		while (readLen < conLen && tries < 20)
		{
			readLen += cg_socket_read(sock, (content+readLen), (conLen-readLen));
			tries++;
		}
		
		if (readLen <= 0)
			return TRUE;
		content[readLen] = '\0';
		cg_http_packet_setcontentpointer(httpPkt, content);
	}
	else if (cg_http_packet_getheadervalue(httpPkt, 
					CG_HTTP_CONTENT_LENGTH) == NULL)
	{
		/* header existance must be checked! otherwise packets which
		   rightly report 0 as content length, will jam the http */
		
		/* Check if we read chunked encoding */
		if (cg_http_packet_ischunked(httpPkt) == TRUE)
		{
			conLen = 0;
			do {
				readLen = cg_http_packet_read_chunk(httpPkt, sock, lineBuf, lineBufSize);
				conLen += readLen;
			} while (readLen > 0);
			
			cg_http_packet_setcontentlength(httpPkt,conLen);
		} else {
			readLen = 0;
			conLen = 0;
			while ((readLen = cg_socket_read(sock, readBuf, READBUF_LENGTH)) > 0)
			{
				cg_http_packet_appendncontent(httpPkt, readBuf, readLen);
				conLen += readLen;
			}

			cg_http_packet_setcontentlength(httpPkt, conLen);
		}
	}

	return TRUE;

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_read
****************************************/

BOOL cg_http_packet_read(CgHttpPacket *httpPkt, CgSocket *sock, char *lineBuf, int lineBufSize)
{
	cg_log_debug_l4("Entering...\n");

	cg_http_packet_clear(httpPkt);
	cg_http_packet_read_headers(httpPkt, sock, lineBuf, lineBufSize);
	
	return cg_http_packet_read_body(httpPkt, sock, lineBuf, lineBufSize);

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_getheadersize
****************************************/

int cg_http_packet_getheadersize(CgHttpPacket *httpPkt)
{
	cg_log_debug_l4("Entering...\n");

	CgHttpHeader *header;
	int headerSize;
	char *name;
	char *value;
	
	headerSize = 0;
	for (header = cg_http_packet_getheaders(httpPkt); header != NULL; header = cg_http_header_next(header)) {
		name = cg_http_header_getname(header);
		value = cg_http_header_getvalue(header);
		headerSize += cg_strlen(name); 
		headerSize += sizeof(CG_HTTP_COLON)-1; 
		headerSize += sizeof(CG_HTTP_SP)-1; 
		headerSize += cg_strlen(value); 
		headerSize += sizeof(CG_HTTP_CRLF)-1; 
	}
	headerSize += sizeof(CG_HTTP_CRLF)-1; 

	return headerSize;

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_copy
****************************************/

void cg_http_packet_copy(CgHttpPacket *destHttpPkt, CgHttpPacket *srcHttpPkt)
{
	cg_log_debug_l4("Entering...\n");

	CgHttpHeader *srcHeader;
	CgHttpHeader *destHeader;
	
	cg_http_packet_clear(destHttpPkt);
	
	/**** copy headers ****/
	for (srcHeader = cg_http_packet_getheaders(srcHttpPkt); srcHeader != NULL; srcHeader = cg_http_header_next(srcHeader)) {
		destHeader = cg_http_header_new();
		cg_http_header_setname(destHeader, cg_http_header_getname(srcHeader));
		cg_http_header_setvalue(destHeader, cg_http_header_getvalue(srcHeader));
		cg_http_packet_addheader(destHttpPkt, destHeader);
	}
	
	/**** copy content ****/
	cg_http_packet_setcontent(destHttpPkt, cg_http_packet_getcontent(srcHttpPkt));

	cg_log_debug_l4("Leaving...\n");
}

/****************************************
* cg_http_packet_print
****************************************/

void cg_http_packet_print(CgHttpPacket *httpPkt)
{
	cg_log_debug_l4("Entering...\n");
#ifdef CLOG_DEBUG_STACK
	CgHttpHeader *header;
	char *content;
	long contentLen;
	
	/**** print headers ****/
	for (header = cg_http_packet_getheaders(httpPkt); header != NULL; header = cg_http_header_next(header)) {
	printf("%s: %s\n",
			cg_http_header_getname(header),
			cg_http_header_getvalue(header));
	}
	printf("\n");
	
	/**** print content ****/
	content = cg_http_packet_getcontent(httpPkt);
	contentLen = cg_http_packet_getcontentlength(httpPkt);
	
	if (content != NULL && 0 < contentLen)
		printf("%s\n", content);
#endif
	cg_log_debug_l4("Leaving...\n");
}
