/*
 * TSLIB based touchscreen driver for KDrive
 * Porting to new input API and event queueing by Daniel Stone.
 * Derived from ts.c by Keith Packard
 * Derived from ps2.c by Jim Gettys
 *
 * Copyright  1999 Keith Packard
 * Copyright  2000 Compaq Computer Corporation
 * Copyright  2002 MontaVista Software Inc.
 * Copyright  2006 Nokia Corporation
 * 
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation, and that the name of the authors and/or copyright holders
 * not be used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  The authors and/or
 * copyright holders make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *
 * THE AUTHORS AND/OR COPYRIGHT HOLDERS DISCLAIM ALL WARRANTIES WITH REGARD
 * TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS, IN NO EVENT SHALL THE AUTHORS AND/OR COPYRIGHT HOLDERS BE
 * LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#ifdef HAVE_KDRIVE_CONFIG_H
#include <kdrive-config.h>
#endif

#define NEED_EVENTS
#include <X11/X.h>
#include <X11/Xproto.h>
#include <X11/Xpoll.h>
#include "inputstr.h"
#include "scrnintstr.h"
#include "kdrive.h"
#include <sys/ioctl.h>
#include <tslib.h>
#include <dirent.h>
#include <linux/input.h>

#define TSLIB_QUEUE_SIZE 3

struct TslibPrivate {
    int fd;
    int lastx, lasty;
    struct tsdev *tsDev;
    int numHeldEvents, holdEvents;
    struct {
        int x;
        int y;
        int z;
        int flags;
    } heldEvent[TSLIB_QUEUE_SIZE];
    int lastFlags;
    void (*raw_event_hook)(int x, int y, int pressure, void *closure);
    void *raw_event_closure;
    int phys_screen;
};


static void
TsRead (int fd, void *closure)
{
    KdPointerInfo       *pi = closure;
    struct TslibPrivate *private = pi->driverPrivate;
    struct ts_sample    event;
    long                x = 0, y = 0;
    int                 i = 0;
    unsigned long       flags = 0;
    int                 discard = 0;

    if (!private->tsDev) {
        DebugF("[tslib] EXTREME BADNESS: TsRead called while tsDev is null!\n");
        return;
    }

    if (private->raw_event_hook) {
        DebugF("[tslib] raw event stolen\n");
        while (ts_read_raw(private->tsDev, &event, 1) == 1)
            (*private->raw_event_hook)(event.x, event.y, event.pressure,
                                       private->raw_event_closure);
        return;
    }

    while (ts_read(private->tsDev, &event, 1) == 1) {
        DebugF("[tslib] originally from (%d, %d, %d)\n", event.x, event.y,
               event.pressure);
        discard = 0;
        if (event.pressure) {
            if (event.pressure > pi->dixdev->touchscreen->button_threshold) 
                flags = KD_BUTTON_8;
            else
                flags = KD_BUTTON_1;

            /* 
             * Here we test for the touch screen driver actually being on the
             * touch screen, if it is we send absolute coordinates. If not,
             * then we send delta's so that we can track the entire vga screen.
             */
            if (KdCurScreen == private->phys_screen) {
                x = event.x;
                y = event.y;
            } else {
                if (private->lastx == 0 || private->lasty == 0) {
                    x = event.x;
                    y = event.y;
                } else {
                    x = private->lastx + event.x;
                    y = private->lasty + event.y;
	    	}
            }
        } else {
            flags = 0;
            x = private->lastx;
            y = private->lasty;
        }

        /* this ... is ugly.
         * basically, we need two things to support thumb cleanly:
         *  - pressure filtering.  because it's pressure-based, often we get
         *    the following sequence: pen down, thumb down, thumb up, pen down,
         *    thumb up.  so we need to filter the first n events to see whether
         *    we have a thumb event coming, and discard the first pen events if
         *    so.  we also need to check if we're trying to post a pen down
         *    after a thumb down (without an intervening thumb up, and discard
         *    it if so).
         *  - accuracy filtering.  the first n events just aren't going to be
         *    accurate.  so we need to discard them, which negates the call for
         *    keeping the initial thumb events above.
         *
         * in addition to this, we need general stability filtering, so we take
         * a three-sample filtering, and discard the result if the total movement
         * is less than 4px in either direction.
         *
         * i'm really, really sorry. -daniels
         */
        if (flags & (KD_BUTTON_1 | KD_BUTTON_8) && private->holdEvents) {
            if (private->numHeldEvents >= TSLIB_QUEUE_SIZE) {
                if (abs(pi->dixdev->valuator->lastx -
                        private->heldEvent[private->numHeldEvents - 1].x) < 4 &&
                    abs(pi->dixdev->valuator->lasty -
                        private->heldEvent[private->numHeldEvents - 1].y) < 4 &&
                    private->lastFlags & (KD_BUTTON_1 | KD_BUTTON_8)) {
                    DebugF("discarding insignificant movement -- (%d, %d) to "
                           "(%d, %d) from %x\n", pi->dixdev->valuator->lastx,
                           pi->dixdev->valuator->lasty,
                           private->heldEvent[private->numHeldEvents - 1].x,
                           private->heldEvent[private->numHeldEvents - 1].y,
                           private->lastFlags);
                    discard = 1;
                }
                else if (flags & KD_BUTTON_8) {
                    if (private->lastFlags & KD_BUTTON_8) {
                        DebugF("replaying subsequent thumb events from queue\n");
                        for (i = 0; i < private->numHeldEvents; i++) {
                            if (private->heldEvent[i].flags & KD_BUTTON_8) {
                                KdEnqueuePointerEvent(pi,
                                                      private->heldEvent[i].flags,
                                                      private->heldEvent[i].x,
                                                      private->heldEvent[i].y,
                                                      private->heldEvent[i].z);
                                private->lastx = private->heldEvent[i].x;
                                private->lasty = private->heldEvent[i].y;
                                private->lastFlags = private->heldEvent[i].flags;
                            }
                        }
                    }
                    else {
                        DebugF("discarding previously queued thumb events\n");
                        private->lastFlags = KD_BUTTON_8;
                    }
                }
                else if (flags & KD_BUTTON_1) {
                    if (private->lastFlags & KD_BUTTON_8) {
                        DebugF("discarding pen down after thumb down\n");
                        discard = 1;

                        for (i = 0; i < private->numHeldEvents; i++) {
                            if (private->heldEvent[i].flags & KD_BUTTON_8) {
                                KdEnqueuePointerEvent(pi,
                                                      private->heldEvent[i].flags,
                                                      private->heldEvent[i].x,
                                                      private->heldEvent[i].y,
                                                      private->heldEvent[i].z);
                                private->lastx = private->heldEvent[i].x;
                                private->lasty = private->heldEvent[i].y;
                                private->lastFlags = private->heldEvent[i].flags;
                            }
                            else {
                                DebugF("discarding pen down after thumb down "
                                       "(queue)\n");
                            }
                        }
                    }
                    else {
                        DebugF("replaying %d pen events from queue\n", private->numHeldEvents);
                        for (i = 0; i < private->numHeldEvents; i++) {
                            if (private->heldEvent[i].flags & KD_BUTTON_1) {
                                DebugF("    (from queue) event at (%u, %u), "
                                       "pressure is %d, sending flags %x\n",
                                       private->heldEvent[i].x,
                                       private->heldEvent[i].y,
                                       private->heldEvent[i].z,
                                       private->heldEvent[i].flags);
                                KdEnqueuePointerEvent(pi,
                                                      private->heldEvent[i].flags,
                                                      private->heldEvent[i].x,
                                                      private->heldEvent[i].y,
                                                      private->heldEvent[i].z);
                                private->lastx = private->heldEvent[i].x;
                                private->lasty = private->heldEvent[i].y;
                                private->lastFlags = private->heldEvent[i].flags;
                            }
                            else {
                                DebugF("discarding thumb event in pen stream?\n");
                            }
                        }
                    }
                }

                private->numHeldEvents = 0;
            }

            if (!discard) {
                DebugF("capturing and queuing event (%lu, %lu, %d)\n", x, y,
                       event.pressure);
                private->heldEvent[private->numHeldEvents].x = x;
                private->heldEvent[private->numHeldEvents].y = y;
                private->heldEvent[private->numHeldEvents].z = event.pressure;
                private->heldEvent[private->numHeldEvents].flags = flags;
                private->numHeldEvents++;
            } else {
                DebugF("discarding current event\n");
            }

            continue;
        }
        else if (private->holdEvents) {
            if (private->numHeldEvents) {
                if (private->lastFlags & KD_BUTTON_1) {
                    DebugF("pen up received while holding events; replaying "
                           "%d last queued events\n", i);
                    for (i = 0; i < private->numHeldEvents; i++) {
                        if (private->heldEvent[i].flags & KD_BUTTON_1) {
                            DebugF("    (from queue) event at (%u, %u), "
                                   "pressure is %d, sending flags %x\n",
                                   private->heldEvent[i].x,
                                   private->heldEvent[i].y,
                                   private->heldEvent[i].z,
                                   private->heldEvent[i].flags);
                            KdEnqueuePointerEvent(pi, private->heldEvent[i].flags,
                                                  private->heldEvent[i].x,
                                                  private->heldEvent[i].y,
                                                  private->heldEvent[i].z);
                            private->lastx = private->heldEvent[i].x;
                            private->lasty = private->heldEvent[i].y;
                            private->lastFlags = private->heldEvent[i].flags;
                        }
                        else {
                            DebugF("dropping thumb event in pen stream\n");
                        }
                    }
                }
                else if (private->lastFlags & KD_BUTTON_8) {
                    DebugF("thumb up received while holding events; replaying "
                           "last queued event, discarding %d\n", i);
                    DebugF("    (from queue) event at (%u, %u), pressure is "
                           "%d, sending flags %x\n", private->heldEvent[i].x,
                           private->heldEvent[i].y, private->heldEvent[i].z,
                           private->heldEvent[i].flags);
                    for (i = 0; i < private->numHeldEvents; i++) {
                        if (private->heldEvent[i].flags & KD_BUTTON_8) {
                            KdEnqueuePointerEvent(pi,
                                                  private->heldEvent[i].flags,
                                                  private->heldEvent[i].x,
                                                  private->heldEvent[i].y,
                                                  private->heldEvent[i].z);
                            private->lastx = private->heldEvent[i].x;
                            private->lasty = private->heldEvent[i].y;
                            private->lastFlags = private->heldEvent[i].flags;
                        }
                        else {
                            DebugF("dropping pen event in thumb stream\n");
                        }
                    }
                }
                else {
                    i = private->numHeldEvents - 1;
                    DebugF("less than three events total in queue, sending last\n");
                    DebugF("    (from queue) event at (%u, %u), pressure is "
                           "%d, sending flags %x\n", private->heldEvent[i].x,
                           private->heldEvent[i].y, private->heldEvent[i].z,
                           private->heldEvent[i].flags);
                    KdEnqueuePointerEvent(pi,
                                          private->heldEvent[i].flags,
                                          private->heldEvent[i].x,
                                          private->heldEvent[i].y,
                                          private->heldEvent[i].z);
                    private->lastx = private->heldEvent[i].x;
                    private->lasty = private->heldEvent[i].y;
                    private->lastFlags = private->heldEvent[i].flags;
                }
            }
            private->numHeldEvents = 0;
            x = private->lastx;
            y = private->lasty;
        }

        private->lastFlags = flags & (KD_BUTTON_1 | KD_BUTTON_8);
        DebugF("direct event at (%lu, %lu), pressure is %d, sending flags %lu\n",
               x, y, event.pressure, flags);
        KdEnqueuePointerEvent (pi, flags, x, y, event.pressure);
        private->lastx = x;
        private->lasty = y;
    }
}

/* must always be NULL-terminated. */
char *valid_ts_names[] = {
    "ADS784x Touchscreen",
    "omap-ts",
    "TSC2301 touchscreen",
    NULL
};

#define TS_NAME_SIZE 32

static Status
TslibEnable (KdPointerInfo *pi)
{
    struct TslibPrivate *private = pi->driverPrivate;
    AxisInfoPtr axis = pi->dixdev->valuator->axes + 2;

    /* theoretically belongs in TslibInit, but InitPointerDeviceStruct
     * will trash this.  le sigh. */
    axis->min_value = 0;
    axis->max_value = 255;
    axis->min_resolution = 1;
    axis->max_resolution = 1;

    private->holdEvents = 1;
    private->numHeldEvents = 0;
    private->lastFlags = 0;
    private->tsDev = ts_open(pi->path, 0);
    private->fd = ts_fd(private->tsDev);
    if (!private->tsDev || ts_config(private->tsDev) || private->fd < 0) {
        ErrorF("[tslib/TslibEnable] failed to open %s\n", pi->path);
        if (private->fd > 0);
            close(private->fd);
        return BadAlloc;
    }
    if (pi->dixdev && pi->dixdev->touchscreen &&
        pi->dixdev->touchscreen->button_threshold == 0)
        pi->dixdev->touchscreen->button_threshold = 115;

    DebugF("[tslib/TslibEnable] successfully enabled %s\n", pi->path);
    KdRegisterFd(private->fd, TsRead, pi);
  
    return Success;
}


static void
TslibDisable (KdPointerInfo *pi)
{
    struct TslibPrivate *private = pi->driverPrivate;

    if (private->fd) {
        KdUnregisterFd(pi, private->fd, TRUE);
        close(private->fd);
    }
    if (private->tsDev)
        ts_close(private->tsDev);

    private->fd = 0;
    private->tsDev = NULL;
}


static Status
TslibInit (KdPointerInfo *pi)
{
    int		        fd = 0, i = 0;
    char                devpath[PATH_MAX], devname[TS_NAME_SIZE];
    DIR                 *inputdir = NULL;
    struct dirent       *inputent = NULL;
    struct TslibPrivate *private = NULL;

    if (!pi || !pi->dixdev)
        return !Success;
    
    if (!pi->path || strcmp(pi->path, "auto") == 0) {
        if (!(inputdir = opendir("/dev/input"))) {
            ErrorF("[tslib/TslibInit]: couldn't open /dev/input!\n");
            return BadMatch;
        }

        while ((inputent = readdir(inputdir))) {
            if (strncmp(inputent->d_name, "event", 5) != 0)
                continue;

            snprintf(devpath, PATH_MAX, "/dev/input/%s", inputent->d_name);
            fd = open(devpath, O_RDWR);

            if (!ioctl(fd, EVIOCGNAME(sizeof(devname)), devname)) {
                close(fd);
                continue;
            }
            close(fd);

            for (i = 0; valid_ts_names[i]; i++) {
                if (strcmp(devname, valid_ts_names[i]) == 0) {
                    pi->path = KdSaveString(devpath);
                    break;
                }
            }
        }
                
        closedir(inputdir);
    }

    if (!pi->path || strcmp(pi->path, "auto") == 0) {
        ErrorF("[tslib/TslibInit]: couldn't find device!\n");
        return BadMatch;
    }

    pi->driverPrivate = (struct TslibPrivate *)
                        xcalloc(sizeof(struct TslibPrivate), 1);
    if (!pi->driverPrivate)
        return !Success;

    private = pi->driverPrivate;
    /* hacktastic */
    private->phys_screen = 0;
    private->raw_event_hook = NULL;
    private->raw_event_closure = NULL;
    pi->nAxes = 3;
    pi->nButtons = 8;
    pi->name = KdSaveString("Touchscreen");
    pi->inputClass = KD_TOUCHSCREEN;
    DebugF("[tslib/TslibInit] successfully inited for device %s\n", pi->path);

    return Success;
}


static void
TslibFini (KdPointerInfo *pi)
{
    if (pi->driverPrivate) {
        xfree(pi->driverPrivate);
        pi->driverPrivate = NULL;
    }
}

static int
TslibCtrl (KdPointerInfo *pi, int control, void *data)
{
    struct TslibPrivate *private = NULL;
    TSRawEvent *rawevent = data;

    if (!pi || !pi->driverPrivate)
        return BadImplementation;

    private = pi->driverPrivate;

    if (control == DEVICE_RAWEVENT) {
        if (rawevent) {
            private->raw_event_hook = rawevent->hook;
            private->raw_event_closure = rawevent->closure;
        }
        else {
            private->raw_event_hook = NULL;
            private->raw_event_closure = NULL;
        }
        return Success;
    }

    return BadImplementation;
}


KdPointerDriver TsDriver = {
    "tslib",
    TslibInit,
    TslibEnable,
    TslibDisable,
    TslibFini,
    TslibCtrl,
    NULL,
};
