/*
 * Copyright (c) 2003 Nokia
 * Author: tsavola@movial.fi
 *
 * This program is licensed under GPL (see COPYING for details)
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include "types.h"
#include "config.h"
#include "common.h"
#include "client.h"

/**
 * Compares the beginning of BUF to TGT. Comparison ends at first whitespace.
 */
static char *is_target(char *tgt,
		       char *buf)
{
	int a, b;

	while (1) {
		a = *tgt++;
		b = *buf++;

		if (!a || !b || isspace(b)) {
			return (!a && isspace(b)) ? buf : NULL;
		}

		if (a != b) {
			return NULL;
		}
	}
}

/**
 * Parses BUF into CFG.
 */
static bool_t read_params(char *buf,
			  config_t *cfg)
{
	char *user, *host, *port, *pwd;

	buf = skip_spaces(buf);
	if (!buf) {
		error_noerr("Invalid parameter string");
		return FALSE;
	}

	host = strchr(buf, '@');
	if (host) {
		user = buf;
		*host++ = '\0';
	} else {
		user = NULL;
		host = buf;
	}

	port = strchr(host, ':');
	if (port) {
		*port++ = '\0';
		buf = port;
	} else {
		port = NULL;
		buf = host;
	}

	buf = find_space(buf);
	if (!buf) {
		error_noerr("Invalid parameter string");
		return FALSE;
	}
	*buf++ = '\0';

	pwd = skip_spaces(buf);
	if (!pwd) {
		error_noerr("Invalid parameter string");
		return FALSE;
	}

	cfg->user = user ? strdup(user) : NULL;
	cfg->host = strdup(host);
	cfg->port = port ? atoi(port) : 0;
	cfg->pwd = strdup(pwd);

	return TRUE;
}

static bool_t read_opts(FILE *file,
			char *buf,
			size_t size,
			config_t *cfg)
{
	long pos;
	size_t cnt = 0;
	char **p;

	pos = ftell(file);

	while (read_line(file, buf, size) > 0 &&
	       strlen(skip_spaces(buf)) > 0) {
		cnt++;
	}

	fseek(file, pos, SEEK_SET);

	cfg->opts = calloc(cnt + 1, sizeof (char *));
	if (!cfg->opts) {
		error_noerr(oom);
		return FALSE;
	}

	for (p = cfg->opts; cnt-- > 0; ) {
		read_line(file, buf, size);
		buf = skip_spaces(buf);

		/* Let's be careful. */
		if (strlen(buf) == 0) {
			break;
		}

		*p++ = strdup(buf);
	}

	return TRUE;
}

bool_t read_config(const char *filename,
		   const char *target,
		   config_t *cfg)
{
	FILE *file;
	char buf[1024], *bufp;
	ssize_t len;
	bool_t ok = FALSE;

	file = fopen(filename, "r");
	if (!file) {
		error("Can't open %s", filename);
		return FALSE;
	}

	while (1) {
		len = read_line(file, buf, sizeof (buf));
		if (len < 0) {
			error_noerr("Target %s is not listed in %s",
				    target, filename);
			break;
		}
		if (len == 0 || isspace(buf[0])) {
			continue;
		}

		bufp = is_target((char *) target, buf);
		if (bufp) {
			ok = read_params(bufp, cfg) &&
			     read_opts(file, buf, sizeof (buf), cfg);
			break;
		}
	}

	fclose(file);

	return ok;
}

void init_config(config_t *cfg)
{
	memset(cfg, 0, sizeof (config_t));
}

void free_config(config_t *cfg)
{
	if (cfg->user) {
		free(cfg->user);
	}

	if (cfg->host) {
		free(cfg->host);
	}

	if (cfg->pwd) {
		free(cfg->pwd);
	}

	if (cfg->opts) {
		free_vec((void **) cfg->opts, NULL);
	}
}

char **get_targets(const char *filename)
{
	FILE *file;
	char buf[1024];
	ssize_t len;
	size_t cnt;
	char **targets, **p;

	file = fopen(filename, "r");
	if (!file) {
		return NULL;
	}

	for (cnt = 0; TRUE; ) {
		len = read_line(file, buf, sizeof (buf));
		if (len < 0) {
			break;
		}

		if (len > 0 && !isspace(*buf)) {
			cnt++;
		}
	}

	rewind(file);

	targets = calloc(cnt + 1, sizeof (char *));
	if (!targets) {
		error_noerr(oom);
		return NULL;
	}

	for (p = targets; TRUE; ) {
		len = read_line(file, buf, sizeof (buf));
		if (len < 0) {
			break;
		}

		if (len > 0 && !isspace(*buf)) {
			*find_space(buf) = '\0';
			*p++ = strdup(buf);
		}
	}

	return targets;
}

