/**
  @file obc-main.c

  @author Johan Hedberg <johan.hedberg@nokia.com>

  Copyright (C) 2004 Nokia. All rights reserved.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 
*/
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdint.h>
#include <string.h>
#include <strings.h>
#include <getopt.h>
#include <ctype.h>
#include <time.h>
#include <signal.h>
#include <glib.h>

#include <openobex/obex.h>

#include <gconf/gconf-client.h>

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#ifdef HAVE_LIBREADLINE
# include <readline/readline.h>
# include <readline/history.h>
#endif

#ifdef USE_BTCOND
# include <bt-gconf.h>
#endif

#include <gw-obex.h>

#include "dbus.h"
#include "bt.h"
#include "fl.h"
#include "get.h"
#include "put.h"
#include "async.h"
#include "completion.h"
#include "obc-main.h"

#define OBC_DEFAULT_MTU 10000

#define g_str_equal(a, b) (strcasecmp((a), (b)) == 0)

extern GMainLoop *event_loop;

static GConfClient *gcclient = NULL;

static ObcContext *obc_ctx = NULL;

volatile gboolean do_abort = FALSE;

static struct {
    int         code;
    const char *str;
} response_codes[] = {
    { 0x10,     "Continue" },
    { 0x20,     "OK, Success" },
    { 0x21,     "Created" },
    { 0x22,     "Accepted" },
    { 0x23,     "Non-Authorative Information" },
    { 0x24,     "No Content" },
    { 0x25,     "Reset Content" },
    { 0x26,     "Partial Content" },
    { 0x30,     "Multiple Choices" },
    { 0x31,     "Moved Permanently" },
    { 0x32,     "Moved temporarely" },
    { 0x33,     "See Other" },
    { 0x34,     "Not modified" },
    { 0x35,     "Use Proxy" },

    { 0x40,     "Bad Request - server couldn't understand request" },
    { 0x41,     "Unauthorized" },
    { 0x42,     "Payment required" },
    { 0x43,     "Forbidden - operation is understood but refused" },
    { 0x44,     "Not Found" },
    { 0x45,     "Method not allowed" },
    { 0x46,     "Not Acceptable" },
    { 0x47,     "Proxy Authentication required" },
    { 0x48,     "Request Time Out" },
    { 0x49,     "Conflict" },
    { 0x4A,     "Gone" },
    { 0x4B,     "Length Required" },
    { 0x4C,     "Precondition failed" },
    { 0x4D,     "Requested entity too large" },
    { 0x4E,     "Request URL too large" },
    { 0x4F,     "Unsupported media type" },

    { 0x50,     "Internal Server Error" },
    { 0x51,     "Not Implemented" },
    { 0x52,     "Bad Gateway" },
    { 0x53,     "Service Unavailable" },
    { 0x54,     "Gateway Timeout" },
    { 0x55,     "HTTP version not supported" },

    { 0x60,     "Database Full" },
    { 0x61,     "Database Locked" },

    { 256,      "Disconnected" },
    { 257,      "Aborted" },
    { 258,      "Internal Error" },
    { 259,      "Service not available" },
    { 260,      "Connecting failed" },
    { 261,      "Operation timed out" },
    { 262,      "Invalid/corrupt data received" },
    { 263,      "Invalid parameters" },
    { 264,      "Access denied (local)" },
    { 265,      "Busy performing another request" },
    { 266,      "No data available" },

    { 0, NULL }
};

static void obc_reset(ObcContext *ctx, gboolean connect);
static void process_input(ObcContext *ctx, char *input);

static char *response_to_string(int rsp)
{
    const char *str;
    char *ret;
    int i;

    /* Clear final bit */
    if (rsp < 256)
        rsp &= 0x7F;

    for (i = 0, str = NULL; response_codes[i].str != NULL; i++) {
        if (response_codes[i].code == rsp) {
            str = response_codes[i].str;
            break;
        }
    }

    if (rsp > 255) {
        if (str)
            ret = g_strdup_printf("%s (err: %d)", str, rsp);
        else
            ret = g_strdup_printf("(err: %d)", rsp);
    }
    else {
        if (str)
            ret = g_strdup_printf("%s (err: 0x%02x)", str, (uint8_t)rsp);
        else
            ret = g_strdup_printf("(err: 0x%02x)", (uint8_t)rsp);
    }

    return ret;
}

static gboolean is_bda(const char *bda)
{
   int i;
    if (strlen(bda) != 17)
        return FALSE;
    for (i = 0; i < 17; i++) {
        if (!(isxdigit(bda[i]) || bda[i] == ':'))
            return FALSE;
    }
    return TRUE;
}

static char *get_device_name(ObcContext *ctx)
{
    if (ctx->cwd[0] == '\0')
        return g_strdup("disconnected");
    if (ctx->user_dev)
        return g_strdup(ctx->rfcomm_dev ? ctx->rfcomm_dev : "");
    if (ctx->resolved_name)
        return g_strdup(ctx->resolved_name);

    if (ctx->user_bda) {
        if (gcclient == NULL)
            gcclient = gconf_client_get_default();

        if (gcclient) {
            char *path;

#ifdef USE_BTCOND
            path = GNOME_BT_DEV_NAME(ctx->user_bda);
#else
            path = g_strdup_printf("/system/bluetooth/device/%s/name",
                                   ctx->user_bda);
#endif
            ctx->resolved_name = gconf_client_get_string(gcclient, path, NULL);
            g_free(path);

            if (ctx->resolved_name)
                return g_strdup(ctx->resolved_name);
        }

        if (ctx->try_remote_name) {
            ctx->resolved_name = bt_get_remote_name(ctx->user_bda);
            if (ctx->resolved_name)
                return g_strdup(ctx->resolved_name);
            else
                ctx->try_remote_name = FALSE;
        }

        return g_strdup(ctx->user_bda);
    }

    return g_strdup("");
}

static void obc_main(ObcContext *ctx)
{
    while (TRUE) {
        char *name, *prompt, *cmd, *lf;
#ifndef HAVE_LIBREADLINE
        char tmp[256];
#endif

        name = get_device_name(ctx);
        prompt = g_strdup_printf("%s %s> ", name ? name : "", ctx->cwd);
        g_free(name);

#ifdef HAVE_LIBREADLINE
        cmd = readline(prompt);
#else
        printf("%s", prompt);
        fflush(stdout);
        if (fgets(tmp, sizeof(tmp), stdin))
            cmd = g_strdup(tmp);
        else
            cmd = NULL;
#endif

        if (cmd == NULL)
            cmd = g_strdup("exit");

        lf = strchr(cmd, '\n');
        if (lf)
            *lf = '\0';

        if (strlen(cmd) > 0) {
#ifdef HAVE_LIBREADLINE
            add_history(cmd);
#endif
            process_input(ctx, cmd);
        }

        g_free(cmd);
        g_free(prompt);
    }
}

static gchar *get_preferred_bda(void)
{
#ifdef USE_BTCOND
    if (gcclient == NULL) {
        gcclient = gconf_client_get_default();
        if (gcclient == NULL) {
            fprintf(stderr, "gconf_client_get_default() failed.\n");
            return NULL;
        }
    }

    return gconf_client_get_string(gcclient,
            BTCOND_GCONF_PREFERRED,
            NULL);
#else
    return NULL;
#endif
}

void progress_cb(GwObex *obex, gint cmd, gint current, gint target,
                 ObcContext *ctx)
{
    double speed;
    double tdelta;
    const char *unit = "bytes/s", *nl = "";

    if (ctx->xfer_complete)
        return;

    /* Make sure we don't print \n twice of the callback
     * is called twice with 100% progress */
    if (!ctx->xfer_complete && current == target) {
        ctx->xfer_complete = TRUE;
        nl = "\n";
    }

    if (target > 0)
        printf("%s %15s: %3d%% (%d/%d bytes), %ds",
               cmd == OBEX_CMD_PUT ? "put" : "get",
               ctx->object ? ctx->object : "(unknown)",
               (int)(((float)current/target)*100),
               current, target,
               (unsigned int)(time(NULL) - ctx->start));
    else
        printf("%s %15s: %d bytes, %ds",
               cmd == OBEX_CMD_PUT ? "put" : "get",
               ctx->object ? ctx->object : "(unknown)",
               current,
               (unsigned int)(time(NULL) - ctx->start));


    tdelta = (double)(time(NULL) - ctx->start);
    if (tdelta > 0)
        speed = (double)current / tdelta;
    else {
        if (target > 0)
            printf("%30s\r%s", " ", nl);
        else
            printf("%40s\r%s", " ", nl);
        return;
    }
    
    if (speed > 1024) {
        speed /= 1024;
        unit = "kB/s";
    }

    printf(", %.02f %s%20s\r%s", speed, unit, " ", nl);

    fflush(stdout);
}

static void disconnect_cb(ObcContext *ctx, gpointer data)
{
    printf("\nDisconnected.\n");
    obc_reset(ctx, FALSE);
}

static gboolean cancel_cb(gpointer data)
{
    if (do_abort) {
        do_abort = FALSE;
        return TRUE;
    }
    return FALSE;
}

static gboolean cmd_exit(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    if (ctx->obex == NULL) {
        if (ctx->sk >= 0) {
            close(ctx->sk);
            ctx->sk = -1;
        }
        else
            disconnect_obex_dev();
    }
    else
        gw_obex_close(ctx->obex);

    exit(EXIT_SUCCESS);

    return TRUE;
}


static void abort_sig(int sig)
{
    if (do_abort)
        cmd_exit(obc_ctx, 0, NULL, NULL);
    else {
        printf("Aborting!\n");
        do_abort = TRUE;
    }
}

static gboolean cmd_cap(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    gchar *cap, *fmt;
    gint cap_len;
    gboolean ret, prog, show_prog = FALSE;

    prog = ctx->show_progress;
    obc_show_progress(ctx, show_prog);

    ctx->object = g_strdup("x-obex/capability");
    ret = gw_obex_get_capability(ctx->obex, &cap, &cap_len, err);
    g_free(ctx->object);
    ctx->object = NULL;

    obc_show_progress(ctx, prog);

    if (ret == FALSE) {
        printf("Getting capability failed\n");
        return FALSE;
    }

    fmt = g_strdup_printf("%%%ds", cap_len);
    printf(fmt, cap);
    g_free(fmt);

    g_free(cap);
    return TRUE;
}

void obc_show_progress(ObcContext *ctx, gboolean show)
{
    ctx->show_progress = show;
    if (ctx->obex == NULL)
        return;

    if (show)
        gw_obex_set_progress_callback(ctx->obex,
                (gw_obex_progress_cb_t)progress_cb,
                ctx);
    else
        gw_obex_set_progress_callback(ctx->obex, NULL, NULL);

}

static gboolean cmd_prog(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    ctx->show_progress = !ctx->show_progress;
    obc_show_progress(ctx, ctx->show_progress);
    if (ctx->show_progress)
        printf("Progress info enabled\n");
    else
        printf("Progress info disabled\n");
    return TRUE;
}

static void cd_help(ObcContext *ctx, const char *name)
{
    printf("%s [directory]\n"
           "Change remote directory.\n"
           "No parameters change to root directory.\n"
           "\"..\" will move one level up.\n",
           name);
}

static gboolean cmd_cd(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    gboolean ret;

    ret = gw_obex_chdir(ctx->obex, argc < 2 ? NULL : argv[1], err);

    if (ret) {
        fl_list_free(ctx->fl_cache);
        ctx->fl_cache = FALSE;

        if (argc < 2)
            strcpy(ctx->cwd, "/");
        else if (strncmp(argv[1], "..", 2) == 0) {
            char *ptr = strrchr(ctx->cwd, '/');
            if (ptr) {
                if (ptr == ctx->cwd)
                    ptr++;
                *ptr = '\0';
            }
        }
        else if (strlen(argv[1]) + strlen(ctx->cwd) + 1 < sizeof(ctx->cwd)) {
            if (strlen(ctx->cwd) > 1)
                strcpy(&ctx->cwd[strlen(ctx->cwd)], "/");
            strcpy(&ctx->cwd[strlen(ctx->cwd)], argv[1]);
        }
        else
            printf("\nYour directory hierarchy is too deep!\n");
    }

    return ret;
}

static gboolean cmd_mkdir(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    if (argc < 2)
        return FALSE;
    return gw_obex_mkdir(ctx->obex, argv[1], err);
}

static gboolean cmd_del(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    int i;

    if (argc < 2)
        return FALSE;

    for (i = 1; i < argc; i++) {
        if (argc > 2) {
            printf("Deleting %s...", argv[i]);
            fflush(stdout);
        }
        if (!gw_obex_delete(ctx->obex, argv[i], err)) {
            if (argc > 2)
                printf("failed.\n");
            return FALSE;
        }
        if (argc > 2)
            printf("done.\n");
    }

    return TRUE;
}

static gboolean cmd_exec(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    if (argc < 2)
        return FALSE;

    if (system(argv[1]) < 0)
        return FALSE;

    return TRUE;
}

static gboolean cmd_lcd(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    if (argc < 2) {
        char *home = getenv("HOME");
        if (!home)
            return FALSE;
        if (chdir(home) < 0)
            return FALSE;
    }
    else if (chdir(argv[1]) < 0)
            return FALSE;

    return TRUE;
}

static gboolean cmd_lls(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    char *av[] = { "|", "ls", NULL, NULL };

    if (argc > 1)
        av[2] =  argv[1];

    return cmd_exec(ctx, argc > 1 ? 3 : 2, av, err);
}

static gboolean cmd_lpwd(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    char lpwd[PATH_MAX];

    if (getcwd(lpwd, sizeof(lpwd))) {
        printf("%s\n", lpwd);
        return TRUE;
    }
    
    return FALSE;
}

static gboolean cmd_pwd(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    if (ctx->cwd[0] == '\0') {
        printf("not connected\n");
        return FALSE;
    }
    else {
        printf("%s\n", ctx->cwd);
        return TRUE;
    }
}

static gboolean cmd_copy(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    if (argc < 3)
        return FALSE;

    return gw_obex_copy(ctx->obex, argv[1], argv[2], err);
}

static gboolean cmd_move(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    if (argc < 3)
        return FALSE;

    return gw_obex_move(ctx->obex, argv[1], argv[2], err);
}

static void conn_help(ObcContext *ctx, const char *name)
{
    printf("%s [-f | -p] [BT-name | BT-address]\n"
           "Connect to remote device\n"
           "If no device identifier was given the connection will be made to the\n"
           "last device obc was connected to. The identifier may be a BT-name,\n"
           "BT-address or the numeric id from the last scan list (e.g. \"#1\")\n"
           "Parameters:\n"
           "-f\tUse OBEX File Transfer\n"
           "-p\tUse OBEX Object Push\n"
           "-C <ch>\tUse specific RFCOMM channel\n",
           name);
}

static gboolean cmd_conn(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    int c;

    while ((c = getopt(argc, argv, "PpfC:")) != -1) {
        switch (c) {
            case 'P':
                g_free(ctx->target);
                ctx->target = g_malloc(OBEX_PBAP_UUID_LEN);
                memcpy(ctx->target, OBEX_PBAP_UUID, OBEX_PBAP_UUID_LEN);
                ctx->target_len = OBEX_PBAP_UUID_LEN;
                break;
            case 'f':
                g_free(ctx->target);
                ctx->target = g_malloc(OBEX_FTP_UUID_LEN);
                memcpy(ctx->target, OBEX_FTP_UUID, OBEX_FTP_UUID_LEN);
                ctx->target_len = OBEX_FTP_UUID_LEN;
                break;
            case 'p':
                g_free(ctx->target);
                ctx->target = NULL;
                ctx->target_len = 0;
                break;
            case 'C':
                ctx->channel = atoi(optarg);
                break;
            default:
                printf("Unhandled option character: '%c'\n", c);
                break;
        }
    }

    c = optind;
    optind = 0;

    if (c < argc) {
        BtDev *dev;
        char *bda;
        int len, id;

        bda = NULL;
        len = strlen(argv[c]);

        if (len > 1 && len < 4 && argv[c][0] == '#' &&
                sscanf(argv[c], "#%d", &id) == 1) {
            dev = g_slist_nth_data(ctx->dev_cache, id - 1);
            if (dev)
                bda = dev->bda;
        }
        else if (is_bda(argv[c]))
            bda = argv[c];
        else {
            dev = bt_scan_find_name(ctx->dev_cache, argv[c]);
            if (dev)
                bda = dev->bda;
        }

        if (bda == NULL) {
            printf("Unable to determine Bluetooth address for \"%s\"\n", argv[c]);
            return FALSE;
        }
        
        g_free(ctx->user_bda);
        ctx->user_bda = g_strdup(bda);
    }

    obc_reset(ctx, TRUE);
    return TRUE;
}

static gboolean cmd_mtu(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    gint new_mtu;

    if (argc < 2)
        return FALSE;

    new_mtu = strtol(argv[1], NULL, 0);
    if (new_mtu > 0) {
        ctx->mtu = new_mtu;
        printf("MTU set to %d\n", new_mtu);
        return TRUE;
    }

    printf("Invalid MTU\n");

    return FALSE;
}

static gboolean cmd_sync(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    ctx->async = FALSE;
    return TRUE;
}

static gboolean cmd_async(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    ctx->async = TRUE;
    return TRUE;
}

static gboolean cmd_iter(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    g_main_context_iteration(NULL, FALSE);
    return TRUE;
}

static gboolean cmd_dc(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    if (ctx->sk >= 0) {
        close(ctx->sk);
        ctx->sk = -1;
    }
    else
        disconnect_obex_dev();
    obc_reset(ctx, FALSE);
    return TRUE;
}

static void scan_help(ObcContext *ctx, const char *name)
{
    printf("%s [-s]\n"
           "Scan for remote devices\n"
           "Parameters:\n"
           "-s\tShow results from last scan\n", name);
}

static gboolean cmd_scan(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    GSList *l;
    int i, c;
    gboolean show;

    show = FALSE;

    while ((c = getopt(argc, argv, "s")) != -1) {
        switch (c) {
            case 's':
                show = TRUE;
                break;
            default:
                printf("Unhandled option character: '%c'\n", c);
                break;
        }
    }

    c = optind;
    optind = 0;

    if (!show) {
        if (ctx->dev_cache)
            bt_scan_free(ctx->dev_cache);
        ctx->dev_cache = bt_scan();
    }

    for (i = 1, l = ctx->dev_cache; l != NULL; l = l->next, i++) {
        BtDev *dev = l->data;
        printf("#%-2d: %s  %s\n", i, dev->bda, dev->name ? dev->name : "");
    }

    return TRUE;
}

static gboolean cmd_cat(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    int i, c;
    char *type = NULL;
    gboolean prog, empty_name = FALSE;

    if (argc < 2)
        return FALSE;

    while ((c = getopt(argc, argv, "t:n")) != -1) {
        switch (c) {
            case 't':
                type = optarg;
                break;
            case 'n':
                empty_name = TRUE;
                break;
        }
    }

    c = optind;
    optind = 0;

    if (argc == c && type == NULL)
        return FALSE;

    prog = ctx->show_progress;
    obc_show_progress(ctx, FALSE);

    if (argc == c && type) {
        char *name;

        if (empty_name)
            name = "";
        else
            name = NULL;

        if (!gw_obex_get_fd(ctx->obex, STDOUT_FILENO, name, type, err)) {
            printf("Getting %s failed\n", type);
            obc_show_progress(ctx, prog);
            return FALSE;
        }
        return TRUE;
    }

    for (i = c; i < argc; i++) {
        if (!gw_obex_get_fd(ctx->obex, STDOUT_FILENO, argv[i], type, err)) {
            printf("Getting %s failed\n", argv[i]);
            obc_show_progress(ctx, prog);
            return FALSE;
        }
    }

    obc_show_progress(ctx, prog);

    return TRUE;
}

Command cmd_table[];

static gboolean cmd_help(ObcContext *ctx, gint argc, gchar *argv[], gint *err)
{
    Command *cmd;

    if (argc < 2) {
        printf("Available commands: (type \"help <command>\" for detailed info)\n");
        for (cmd = cmd_table; cmd->name != NULL; cmd++) {
            if (cmd->help != NULL)
                printf(" %-10s %-70s\n", cmd->name, cmd->help);
        }
    }
    else {
        Command *match = NULL;
        for (cmd = cmd_table; cmd->name != NULL; cmd++) {
            if (g_str_equal(cmd->name, argv[1])) {
                match = cmd;
                break;
            }
        }

        if (match == NULL)
            printf("Unknown command\n");
        else if (match->long_help == NULL)
            printf("No help available for %s\n", argv[1]);
        else
            match->long_help(ctx, argv[1]);
    }

    return TRUE;
}

Command cmd_table[] = {
    { "exit",       cmd_exit,  CPL_NONE, TRUE,  NULL,      NULL },
    { "quit",       cmd_exit,  CPL_NONE, TRUE,  NULL,      "Exit the program" },
    { "help",       cmd_help,  CPL_CMD,  TRUE,  NULL,      "Show available commands" },
    { "ls",         cmd_ls,    CPL_REM,  FALSE, ls_help,   "Show files in remote directory" },
    { "dir",        cmd_ls,    CPL_REM,  FALSE, ls_help,   NULL },
    { "put",        cmd_put,   CPL_LOC,  FALSE, put_help,  "Send file to remote device" },
    { "get",        cmd_get,   CPL_REM,  FALSE, get_help,  "Get file from remote device" },
    { "open",       cmd_open,  CPL_REM,  FALSE, open_help, "Open file for reading" },
    { "close",      cmd_close, CPL_NONE, FALSE, close_help,"Close file" },
    { "read",       cmd_read,  CPL_NONE, FALSE, read_help, "Read data from file" },
    { "abort",      cmd_abort, CPL_NONE, FALSE, abort_help,"Abort a transfer" },
    { "cd",         cmd_cd,    CPL_REM,  FALSE, cd_help,   "Change directory" },
    { "delete",     cmd_del,   CPL_REM,  FALSE, NULL,      NULL },
    { "del",        cmd_del,   CPL_REM,  FALSE, NULL,      NULL },
    { "rm",         cmd_del,   CPL_REM,  FALSE, NULL,      "Remove remote file" },
    { "mkdir",      cmd_mkdir, CPL_NONE, FALSE, NULL,      "Create a new directory" },
    { "move",       cmd_move,  CPL_REM,  FALSE, NULL,      NULL },
    { "mv",         cmd_move,  CPL_REM,  FALSE, NULL,      "Move/Rename file on remote device" },
    { "copy",       cmd_copy,  CPL_REM,  FALSE, NULL,      NULL },
    { "cp",         cmd_copy,  CPL_REM,  FALSE, NULL,      "Copy file on remote device" },
    { "progress",   cmd_prog,  CPL_NONE, FALSE, NULL,      "Toggle progress indication" },
    { "sync",       cmd_sync,  CPL_NONE, TRUE,  NULL,      "Use gwobex sync functions" },
    { "async",      cmd_async, CPL_NONE, TRUE,  NULL,      "Use gwobex async functions" },
    { "iter",       cmd_iter,  CPL_NONE, TRUE,  NULL,      "Call g_main_context_iteration (for testing purposes)" },
    { "mtu",        cmd_mtu,   CPL_NONE, TRUE,  NULL,      "Set the MTU (max bytes written/read at a time)" },
    { "cap",        cmd_cap,   CPL_NONE, FALSE, NULL,      "Get capability object" },
    { "lcd",        cmd_lcd,   CPL_LOC,  TRUE,  NULL,      "Change local working directory" },
    { "pwd",        cmd_pwd,   CPL_NONE, FALSE, NULL,      "Print remote working directory" },
    { "lpwd",       cmd_lpwd,  CPL_NONE, TRUE,  NULL,      "Print local working directory" },
    { "lls",        cmd_lls,   CPL_LOC,  TRUE,  NULL,      "Show files in local directory" },
    { "|",          cmd_exec,  CPL_LOC,  TRUE,  NULL,      "Execute local command" },
    { "cat",        cmd_cat,   CPL_REM,  FALSE, NULL,      "Show contents of remote file" },
    { "connect",    cmd_conn,  CPL_BT,   TRUE,  conn_help, "Connect to remote device" },
    { "disconnect", cmd_dc,    CPL_NONE, FALSE, NULL,      "Disconnect from remote device" },
    { "dc",         cmd_dc,    CPL_NONE, FALSE, NULL,      NULL },
    { "scan",       cmd_scan,  CPL_NONE, TRUE,  scan_help, "Scan for remote devices" },
    { NULL }
};

static gboolean input_command(ObcContext *ctx, char *line, gint *err)
{
    Command *cmd, *match;
    gboolean ret;
    GError *gerr = NULL;
    gint argc;
    gchar **argv;

    g_shell_parse_argv(line, &argc, &argv, &gerr);
    if (gerr != NULL) {
        fprintf(stderr, "Parsing failed: %s", gerr->message);
        g_error_free(gerr);
        return FALSE;
    }

    for (match = NULL, cmd = cmd_table; cmd->name != NULL; cmd++) {
        if (g_str_equal(argv[0], cmd->name)) {
            match = cmd;
            break;
        }
    }

    ctx->xfer_complete = FALSE;
    ctx->start         = time(NULL);

    if (match) {
        if (ctx->obex == NULL && !match->dc_allow) {
            printf("** Not connected **\n");
            ret = TRUE;
        }
        else
            ret = match->func(ctx, argc, argv, err);
    }
    else {
        printf("** Unknown command! **\n");
        ret = TRUE;
    }

    g_strfreev(argv);

    return ret;
}

static void process_input(ObcContext *ctx, char *input)
{
    gint err;
    if (!input_command(ctx, input, &err)) {
        char *str = response_to_string(err);
        printf("** Command failed: %s **\n", str);
        g_free(str);
        if (err == GW_OBEX_ERROR_DISCONNECT)
            obc_reset(ctx, FALSE);
    }
}

static void obc_reset(ObcContext *ctx, gboolean connect)
{
    if (connect && ctx->user_bda == NULL) {
        printf("No BDA to connect to!\n");
        return;
    }

    if (ctx->obex) {
        gw_obex_close(ctx->obex);
        ctx->obex = NULL;
    }

    if (connect) {
        gint err;

        if (ctx->use_btcond) {
            if (!ctx->user_dev) {
                printf("Getting rfcomm device from btcond via D-BUS\n");
                g_free(ctx->rfcomm_dev);
                ctx->rfcomm_dev = get_obex_dev(ctx->user_bda, ctx->target != NULL,
                                               ctx->use_auth, ctx->use_encr,
                                               ctx->set_role);
                if (ctx->rfcomm_dev == NULL)
                    printf("Unable to get device from btcond\n");
                else
                    printf("Using device: %s\n", ctx->rfcomm_dev);
            }

            if (ctx->rfcomm_dev) {
                ctx->obex = gw_obex_setup_dev(ctx->rfcomm_dev,
                                              ctx->target, ctx->target_len,
                                              NULL, &err);

                if (ctx->obex == NULL)
                    printf("OBEX setup failed: %s\n", response_to_string(err));
            }
        }
        else {
            if (ctx->sk >= 0)
                close(ctx->sk);
            ctx->sk = bt_connect(ctx->user_bda, ctx->channel, ctx->target != NULL,
                                 ctx->use_auth, ctx->use_encr, ctx->set_role);
            if (ctx->sk < 0)
                fprintf(stderr, "Unable to connect to %s\n", ctx->user_bda);
            else {
                ctx->obex = gw_obex_setup_fd(ctx->sk,
                                             ctx->target, ctx->target_len,
                                             NULL, &err);

                if (ctx->obex == NULL)
                    printf("OBEX setup failed: %s\n", response_to_string(err));
            }
        }

        if (ctx->obex) {
            ctx->try_remote_name = TRUE;
            printf("Connected.\n");
        }
        else {
            if (ctx->sk >= 0) {
                close(ctx->sk);
                ctx->sk = -1;
            }
            else
                disconnect_obex_dev();
            obc_reset(ctx, FALSE);
            return;
        }

    }

    if (ctx->obex) {
        strcpy(ctx->cwd, "/");
        obc_show_progress(ctx, ctx->show_progress);
        gw_obex_set_disconnect_callback(ctx->obex,
                                        (gw_obex_disconnect_cb_t)disconnect_cb,
                                        ctx);
        gw_obex_set_cancel_callback(ctx->obex,
                                    (gw_obex_cancel_cb_t)cancel_cb,
                                    ctx);
    }
    else
       ctx->cwd[0] = '\0';
} 

void obc_init(ObcCfg *cfg)
{
    ObcContext *ctx;

    ctx = g_new0(ObcContext, 1);
    ctx->sk            = -1;
    ctx->start         = -1;
    ctx->mtu           = OBC_DEFAULT_MTU;
    ctx->show_progress = TRUE;

    if (cfg->dst && cfg->is_dev) {
        ctx->rfcomm_dev = g_strdup(cfg->dst);
	ctx->user_dev = TRUE;
    }
    else {
	ctx->user_dev = FALSE;
        if (cfg->dst)
            ctx->user_bda = g_strdup(cfg->dst);
        else
            ctx->user_bda = get_preferred_bda();
    }

    signal(SIGINT, abort_sig);
    signal(SIGUSR1, abort_sig);

    if (cfg->target) {
        ctx->target = g_malloc(cfg->target_len);
        memcpy(ctx->target, cfg->target, cfg->target_len);
        ctx->target_len = cfg->target_len;
    }

    ctx->use_auth = cfg->auth;
    ctx->use_encr = cfg->encrypt;
    ctx->set_role = cfg->role;
    ctx->channel  = cfg->chan;

    ctx->use_btcond = (cfg->use_btcond && btcond_is_active());

    if (ctx->user_bda)
        obc_reset(ctx, TRUE);
    else
        obc_reset(ctx, FALSE);

    if (cfg->cmd) {
        gchar **strv, **i;
        int err;

        if (ctx->obex == NULL) {
            printf("Unable to send commands: not connected\n");
            exit(EXIT_FAILURE);
        }

        strv = g_strsplit(cfg->cmd, ";", 0);

        for (i = strv; *i != NULL; i++) {
            char *c = g_strstrip(*i);
            printf("Sending \"%s\"\n", c);
            if (!input_command(ctx, c, &err)) {
                char *str = response_to_string(err);
                printf("Command failed: %s\n", str);
                g_free(str);
                exit(EXIT_FAILURE);
            }
        }

        g_strfreev(strv);

        exit(EXIT_SUCCESS);
    }

    printf("\nType \"help\" to see available commands\n");

    obc_completion_init(ctx);
    obc_ctx = ctx;

    obc_main(ctx);
}

