/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <unistd.h>
#include <string.h>
#include <glib.h>

#include "ob-backend.h"
#include "test-engine.h"

#define d(x) x

extern gboolean engine_cancel;


/*
 * Test engine implementation.
 */

typedef enum {
	TEST_ENGINE_STATE_NORMAL,
	TEST_ENGINE_STATE_ERROR,
	TEST_ENGINE_STATE_CONFLICT,
	TEST_ENGINE_STATE_FINISHED
} TestEngineState;

typedef struct {
	/* The state is an interal detail to make it easier to debug the test
	 * engine.
	 */
	TestEngineState  state;

	ObBackend   *backend;

	/* The engine runs in its own thread. We use a mainloop to simulate
	 * events by adding timeouts.
	 */
	GMainContext    *main_context;
	GMainLoop       *main_loop;

	GSource         *event_source;

	int              files;
	int              total;
	
	gboolean         simulate_errors;
	gboolean         simulate_conflicts;
} TestEngine;


static gboolean test_engine_simulate_event_func  (TestEngine         *engine);
static void     test_engine_push_progress_event  (TestEngine         *engine,
						  int                 files,
						  int                 total,
						  int                 complete);
static void     test_engine_push_error_event     (TestEngine         *engine,
						  int                 code,
						  const char         *message);
static void     test_engine_push_conflict_event  (TestEngine         *engine,
						  ObConflictType  type,
						  const char         *name,
						  time_t              old_time,
						  time_t              new_time);
static void     test_engine_push_finished_event  (TestEngine         *engine);
static void     test_engine_push_cancelled_event (TestEngine         *engine);


static void
test_engine_free (TestEngine *engine)
{
	g_object_unref (engine->backend);

	g_source_destroy (engine->event_source);
	g_source_unref (engine->event_source);

	g_free (engine);
}

static TestEngine *
test_engine_new (ObBackend *backend,
		 gboolean       simulate_errors,
		 gboolean       simulate_conflicts)
{
	TestEngine *engine;
	
	engine = g_new0 (TestEngine, 1);
	engine->backend = g_object_ref (backend);
	engine->state = TEST_ENGINE_STATE_NORMAL;

	engine->main_context = g_main_context_new ();
	engine->main_loop = g_main_loop_new (engine->main_context, FALSE);

	/* We simulate events through a timeout. */
	engine->event_source = g_timeout_source_new (150);
	g_source_set_callback (engine->event_source,
			       (GSourceFunc) test_engine_simulate_event_func,
			       engine, NULL);
	g_source_attach (engine->event_source, engine->main_context);
	
	engine->simulate_errors = simulate_errors;
	engine->simulate_conflicts = simulate_conflicts;

	engine->files = 0;
	engine->total = 20;

	return engine;
}


/* This is called from the engine thread, since the timeout is run in our own
 * thread-specific main context.
 */
static gboolean
test_engine_simulate_event_func (TestEngine *engine)
{
	int complete;
	
	/* Note: Just test cancelling. We need to make it thread-safe. */
	if (engine_cancel) {
		/* Simulate a delay before exiting. */
		usleep (1 * 1000*1000);
		
		ob_backend_confirm_cancel (engine->backend);

		/* Quit the thread when it's cancelled. */
		g_main_loop_quit (engine->main_loop);
		return FALSE;
	}
	
	engine->files++;

	complete = (100 * engine->files) / engine->total;
	
	if (complete > 40 && engine->simulate_errors) {
		g_assert (engine->state == TEST_ENGINE_STATE_NORMAL);
		
		engine->state = TEST_ENGINE_STATE_ERROR;
		test_engine_push_error_event (engine,
					      12,
					      "Simulated error condition");

		/* Quit the thread after an error. */
		g_main_loop_quit (engine->main_loop);
		return FALSE;
	}

	if (engine->files ==  5 && engine->simulate_conflicts) {
		ObConflictResponse response;

		test_engine_push_conflict_event (engine,
						 OB_CONFLICT_TYPE_FILE_FILE,
						 "/test/path/document.txt",
						 123, 456);

		g_print ("wait for response\n");
		response = ob_backend_wait_for_conflict_response (engine->backend);
		g_print ("got response\n");

		if (response == OB_CONFLICT_RESPONSE_CANCEL) {
			g_print ("Got cancel as conflict response\n");

			engine->state = TEST_ENGINE_STATE_FINISHED;
			test_engine_push_cancelled_event (engine);

			/* Quit the thread since we've been cancelled. */
			g_main_loop_quit (engine->main_loop);
			return FALSE;
		}
	}
	
	test_engine_push_progress_event (engine,
					 engine->files, engine->total,
					 complete);

	if (engine->files >= engine->total) {
		g_assert (engine->state == TEST_ENGINE_STATE_NORMAL);
		
		engine->state = TEST_ENGINE_STATE_FINISHED;
		test_engine_push_finished_event (engine);

		/* Quit the thread when its finished. */
		g_main_loop_quit (engine->main_loop);
		return FALSE;
	}

	return TRUE;
}

static void
test_engine_push_progress_event (TestEngine *engine,
				 int         files,
				 int         total,
				 int         complete)
{
	ObEvent *event;

	g_assert (engine->state == TEST_ENGINE_STATE_NORMAL);

	event = ob_event_new_progress (files, total, complete);

	ob_backend_push_event (engine->backend, event);
}

static void
test_engine_push_error_event (TestEngine *engine,
			      int         code,
			      const char *message)
{
	ObEvent *event;

	g_assert (engine->state == TEST_ENGINE_STATE_ERROR);

	event = ob_event_new_error (code, message);
			
	ob_backend_push_event (engine->backend, event);
}

static void
test_engine_push_conflict_event (TestEngine         *engine,
				 ObConflictType  type,
				 const char         *name,
				 time_t              old_time,
				 time_t              new_time)
{
	ObEvent *event;

	g_assert (engine->state == TEST_ENGINE_STATE_NORMAL);
	
	event = ob_event_new_conflict (type, name, old_time, new_time);

	ob_backend_push_event (engine->backend, event);
}

static void
test_engine_push_finished_event (TestEngine *engine)
{
	ObEvent *event;

	g_assert (engine->state == TEST_ENGINE_STATE_FINISHED);

	event = ob_event_new_finished (2*60+14);

	ob_backend_push_event (engine->backend, event);
}

static void
test_engine_push_cancelled_event (TestEngine *engine)
{
	ObEvent *event;

	g_assert (engine->state == TEST_ENGINE_STATE_FINISHED);

	event = ob_event_new_cancelled ();

	ob_backend_push_event (engine->backend, event);
}

static gpointer
test_engine_backup_thread_func (TestEngine *engine)
{
	/* Run the mainloop to get the simulated events fired off. */
	g_main_loop_run (engine->main_loop);

	d(g_print ("TE: Test thread done.\n"));

	test_engine_free (engine);
	
	return NULL;
}

gboolean
test_engine_start_backup (ObBackend *backend,
			  gboolean   simulate_errors)
{
	TestEngine *engine;	
	GThread    *thread;

	engine = test_engine_new (backend, simulate_errors, FALSE);
	
	thread = g_thread_create ((GThreadFunc) test_engine_backup_thread_func,
				  engine, TRUE, NULL);
	
	if (!thread) {
		test_engine_free (engine);
		return FALSE;
	}
	
	return TRUE;
}

gboolean
test_engine_start_restore (ObBackend *backend,
			   gboolean   simulate_errors,
			   gboolean   simulate_conflicts)
{
	TestEngine *engine;
	GThread    *thread;

	engine = test_engine_new (backend, simulate_errors, simulate_conflicts);

	/* Note: It might make sense to implement a separate test for restore.
	 */
	thread = g_thread_create ((GThreadFunc) test_engine_backup_thread_func,
				  engine, FALSE, NULL);
	
	if (!thread) {
		test_engine_free (engine);
		return FALSE;
	}
	
	return TRUE;
}
