/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <libgnomevfs/gnome-vfs.h>
#include <gsf/gsf-utils.h>
#include <gsf/gsf-input.h>
#include <gsf/gsf-input-stdio.h>
#include <gsf/gsf-infile-zip.h>
#include <gsf/gsf-infile.h>
#include <gsf/gsf-output-stdio.h>
#include <gsf/gsf-outfile.h>
#include <gsf/gsf-outfile-zip.h>

#include "ob-archiver.h"
#include "ob-archiver-test.h"

#define d(x)


/*
 * Test archiver implementation.
 */

typedef enum {
	TEST_STATE_NORMAL,
	TEST_STATE_ERROR,
	TEST_STATE_CONFLICT,
	TEST_STATE_FINISHED
} TestState;

typedef struct {
	/* The state is an interal detail to make it easier to debug the test
	 * archiver.
	 */
	TestState        state;

	/* We use a mainloop to simulate events. */
	GMainContext    *main_context;
	GMainLoop       *main_loop;

	GSource         *event_source;

	gboolean         simulate_errors;
	gboolean         simulate_conflicts;
} ObArchiverTestPriv;


static void     archiver_test_finalize           (GObject    *object);
static gpointer archiver_test_pack_thread_func   (ObArchiver *archiver);
static gpointer archiver_test_unpack_thread_func (ObArchiver *archiver);

static gboolean archiver_test_simulate_event_func (ObArchiver *archiver);


#define OB_ARCHIVER_TEST_GET_PRIV(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), OB_TYPE_ARCHIVER_TEST, ObArchiverTestPriv))

G_DEFINE_TYPE (ObArchiverTest, ob_archiver_test, OB_TYPE_ARCHIVER);


static void
ob_archiver_test_class_init (ObArchiverTestClass *klass)
{
        GObjectClass    *object_class;
	ObArchiverClass *archiver_class;

        object_class = G_OBJECT_CLASS (klass);
        object_class->finalize = archiver_test_finalize;

	archiver_class = OB_ARCHIVER_CLASS (klass);
	archiver_class->pack_thread_func = archiver_test_pack_thread_func;
	archiver_class->unpack_thread_func = archiver_test_unpack_thread_func;

	g_type_class_add_private (object_class, sizeof (ObArchiverTestPriv));
}

static void
ob_archiver_test_init (ObArchiverTest *archiver)
{
	ObArchiverTestPriv *priv;

	priv = OB_ARCHIVER_TEST_GET_PRIV (archiver);
	
	priv->state = TEST_STATE_NORMAL;

	priv->main_context = g_main_context_new ();
	priv->main_loop = g_main_loop_new (priv->main_context, FALSE);

	/* We simulate events through a timeout. */
	priv->event_source = g_timeout_source_new (150);
	g_source_set_callback (priv->event_source,
			       (GSourceFunc) archiver_test_simulate_event_func,
			       archiver, NULL);
	g_source_attach (priv->event_source, priv->main_context);
	
	ob_archiver_set_total_files (OB_ARCHIVER (archiver), 20);
	ob_archiver_set_total_size (OB_ARCHIVER (archiver), 10*1024);
}

static void
archiver_test_finalize (GObject *object)
{
	ObArchiverTestPriv *priv;

	priv = OB_ARCHIVER_TEST_GET_PRIV (object);
	
	g_source_destroy (priv->event_source);
	g_source_unref (priv->event_source);

	G_OBJECT_CLASS (ob_archiver_test_parent_class)->finalize (object);
}

ObArchiver *
ob_archiver_test_new (gboolean simulate_errors,
		      gboolean simulate_conflicts)
{
	ObArchiver         *archiver;
	ObArchiverTestPriv *priv;

	archiver = g_object_new (OB_TYPE_ARCHIVER_TEST, NULL);

	priv = OB_ARCHIVER_TEST_GET_PRIV (archiver);
	
	priv->simulate_errors = simulate_errors;
	priv->simulate_conflicts = simulate_conflicts;
	
	return archiver;
}


/* This is called from the archiver thread, since the timeout is run in our own
 * thread-specific main context.
 */
static gboolean
archiver_test_simulate_event_func (ObArchiver *archiver)
{
	ObArchiverTestPriv *priv;
	int                 progress;
	int                 files;

	priv = OB_ARCHIVER_TEST_GET_PRIV (archiver);
	
	if (ob_archiver_is_cancelled (archiver)) {
		/* Simulate a delay before exiting. */
		usleep (1 * 1000*1000);
		
		ob_backend_confirm_cancel (ob_archiver_get_backend (archiver));

		/* Quit the thread when it's cancelled. */
		g_main_loop_quit (priv->main_loop);
		return FALSE;
	}
	
	ob_archiver_add_processed_files (archiver, 1);
	ob_archiver_add_processed_size (archiver, 5*1024);

	files = ob_archiver_get_processed_files (archiver);

	progress = ob_archiver_get_progress (archiver);
	
	if (progress > 40 && priv->simulate_errors) {
		g_assert (priv->state == TEST_STATE_NORMAL);
		
		priv->state = TEST_STATE_ERROR;
		ob_archiver_push_error_event (archiver,
					      12,
					      "Simulated error condition");

		/* Quit the thread after an error. */
		g_main_loop_quit (priv->main_loop);
		return FALSE;
	}

	if (files ==  5 && priv->simulate_conflicts) {
		ObConflictResponse response;
		
		ob_archiver_push_conflict_event (archiver,
						 OB_CONFLICT_TYPE_FILE_FILE,
						 "/test/path/document.txt",
						 123, 456);

		g_print ("wait for response\n");
		response = ob_backend_wait_for_conflict_response (ob_archiver_get_backend (archiver));
		g_print ("got response\n");

		if (response == OB_CONFLICT_RESPONSE_CANCEL) {
			g_print ("Got cancel as conflict response\n");

			priv->state = TEST_STATE_FINISHED;
			ob_archiver_push_cancelled_event (archiver);

			/* Quit the thread since we've been cancelled. */
			g_main_loop_quit (priv->main_loop);
			return FALSE;
		}
	}
	
	ob_archiver_push_progress_event (archiver);

	if (files >= ob_archiver_get_total_files (archiver)) {
		g_assert (priv->state == TEST_STATE_NORMAL);
		
		priv->state = TEST_STATE_FINISHED;
		ob_archiver_push_finished_event (archiver);

		/* Quit the thread when its finished. */
		g_main_loop_quit (priv->main_loop);
		return FALSE;
	}

	return TRUE;
}

static gpointer
archiver_test_pack_thread_func (ObArchiver *archiver)
{
	ObArchiverTestPriv *priv;
	
	priv = OB_ARCHIVER_TEST_GET_PRIV (archiver);

	ob_archiver_set_total_size (archiver, 100*1024);
	ob_archiver_set_total_files (archiver, 20);
	
	g_print ("ArchiverTest pack test thread starting.\n");

	/* Run the mainloop to get the simulated events fired off. */
	g_main_loop_run (priv->main_loop);
	
	g_print ("Thread done.\n");
	
	return NULL;
}

static gpointer
archiver_test_unpack_thread_func (ObArchiver *archiver)
{
	ObArchiverTestPriv *priv;

	priv = OB_ARCHIVER_TEST_GET_PRIV (archiver);

	ob_archiver_set_total_size (archiver, 100*1024);
	ob_archiver_set_total_files (archiver, 20);
	
	g_print ("ArchiverTest unpack test thread starting.\n");
	
	/* Run the mainloop to get the simulated events fired off. */
	g_main_loop_run (priv->main_loop);
	
	g_print ("Thread done.\n");
	
	return NULL;
}

