/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <glib.h>
#include <string.h>

#include "ob-utils.h"
#include "ob-types.h"
#include "ob-category.h"
#include "ob-log.h"
#include "ob-backup-info.h"
#include "ob-restore-transform.h"

#define d(x) 

static char * restore_transform_create_file_list (ObRestoreTransform *transform);

struct _ObRestoreTransform {
	gint          refcount;
	GSList       *files[OB_NUM_CATEGORIES];
	ObBackupInfo *backup_info;
};

ObRestoreTransform *
ob_restore_transform_new (ObBackupInfo *info)
{
	ObRestoreTransform *transform;

	transform = g_new0 (ObRestoreTransform, 1);
	transform->refcount = 1;
	transform->backup_info = ob_backup_info_ref (info);

	return transform;
}

static void
restore_transform_free (ObRestoreTransform *transform)
{
	int i;
       
	for (i = 0; i < OB_NUM_CATEGORIES; i++) {
		g_slist_foreach (transform->files[i], (GFunc) g_free, NULL);
		g_slist_free (transform->files[i]);
	}

	ob_backup_info_ref (transform->backup_info);
	
	g_free (transform);
}

ObRestoreTransform *
ob_restore_transform_ref (ObRestoreTransform *transform)
{
	transform->refcount++;

	return transform;
}

void
ob_restore_transform_unref (ObRestoreTransform *transform)
{
	transform->refcount--;

	if (transform->refcount == 0) {
		restore_transform_free (transform);
		return;
	}
}

void
ob_restore_transform_add_file (ObRestoreTransform *transform,
			       ObCategory          category,
			       const gchar        *path)
{
	int i;
	
	g_return_if_fail (category >= OB_CATEGORY_EMAILS &&
			  category <= OB_CATEGORY_OTHER);

	i = ob_category_to_index (category);
	transform->files[i] = g_slist_prepend (transform->files[i], g_strdup (path));
}

typedef enum {
	OB_SOFTWARE_VERSION_INVALID = 0,
	OB_SOFTWARE_VERSION_IT2005,
	OB_SOFTWARE_VERSION_IT2006,
	OB_SOFTWARE_VERSION_IT2007
} ObSoftwareVersion;

typedef struct {
	ObSoftwareVersion  version;
	const gchar       *restore_dir;
} ObVersionDir;

static const ObVersionDir version_dirs[] = {
	{ OB_SOFTWARE_VERSION_IT2005, "it2005" },
	{ OB_SOFTWARE_VERSION_IT2006, "it2006" },
	{ OB_SOFTWARE_VERSION_IT2007, "it2007" }
};
	
	
static ObSoftwareVersion
restore_transform_parse_software_version (const gchar *str)
{
	ObSoftwareVersion version;

	if ((strstr (str, "2005.") || strstr (str, "2006.")) &&
	    !strstr (str, "SE_")) {
		version = OB_SOFTWARE_VERSION_IT2005;
	}
	else if  (strstr (str, "2006SE")) {
		version = OB_SOFTWARE_VERSION_IT2006;
	}
	else if  (strstr (str, "2007SE")) {
		version = OB_SOFTWARE_VERSION_IT2007;
	} else {
		ob_log_warning ("Unrecognized version string: %s", str);
		version = OB_SOFTWARE_VERSION_INVALID;
	}

	return version;
}

static gboolean
restore_transform_check_from_and_to (ObRestoreTransform *transform,
				     ObSoftwareVersion  *from,
				     ObSoftwareVersion  *to)
{
	const char *device_version;
	const char *backup_version;

	device_version = ob_utils_get_device_version ();
	*to = restore_transform_parse_software_version (device_version);

	backup_version = ob_backup_info_get_device_version (transform->backup_info);
	*from = restore_transform_parse_software_version (backup_version);

	if (*to == OB_SOFTWARE_VERSION_INVALID ||
	    *from == OB_SOFTWARE_VERSION_INVALID) {
		return FALSE;
	}
	
	return TRUE;
}

/* Creates the file list with categories and files, the return value is the full
 * path to the created file.
 */
static char *
restore_transform_create_file_list (ObRestoreTransform *transform)
{
	int   fd;
	char *path;
	FILE *file;
	int   i;

	fd = g_file_open_tmp ("ob-files-XXXXXX", &path, NULL);
	if (fd == -1) {
		return NULL;
	}

	file = fdopen (fd, "w");
	if (!file) {
		close (fd);
		g_free (path);
		return NULL;
	}

	for (i = 0; i < OB_NUM_CATEGORIES; i++) {
		ObCategory  category;
		GSList     *l;
		
		category = ob_category_from_index (i);

		fprintf (file, "[%s]\n", ob_category_to_non_translated_string (category));

		for (l = transform->files[i]; l; l = l->next) {
			fprintf (file, "%s\n", (char *) l->data);
		}
	}

	fclose (file);
	
	return path;
}

static void
restore_transform_execute_scripts (ObRestoreTransform *transform,
				   const char         *base_dir,
				   const char         *filelist_path)
{
	GDir       *dir;
	const char *name;

	ob_log_info ("Run script %s", base_dir);

	dir = g_dir_open (base_dir, 0, NULL);
	if (!dir) {
		return;
	}
	
	while ((name = g_dir_read_name (dir))) {
		char     *path;
		char     *cmdline;
		int       exit_status;
		gboolean  script_retval;

		/* Skip backup files and hidden files. */
		if (g_str_has_suffix (name, ".bak") ||
		    g_str_has_suffix (name, "~") ||
		    g_str_has_prefix (name, ".")) {
			ob_log_info ("Skipping: %s", name);
			continue;
		}
		
		path = g_build_filename (base_dir, name, NULL);
			
		if (g_file_test (path, G_FILE_TEST_IS_DIR)) {
			g_free (path);
			ob_log_info ("Skipping dir: %s", name);
			continue;
		}
		
		if (!g_file_test (path, G_FILE_TEST_IS_EXECUTABLE)) {
			g_free (path);
			ob_log_info ("Skipping non-exec: %s", name);
			continue;
		}
		
		ob_log_info ("Found script: %s", path);

		cmdline = g_strdup_printf ("%s %s", path, filelist_path); 
		script_retval = g_spawn_command_line_sync (
			cmdline, 
			NULL, NULL,
			&exit_status,
			NULL);
		g_free (cmdline);
		
		if (!script_retval || exit_status != 0) {
			ob_log_warning ("Could not run transformation script:\n%s (exit status %d)",
					path, exit_status);
		}
		
		g_free (path);
	}

	g_dir_close (dir);
}

static void
restore_transform_execute_scripts_for_version (ObRestoreTransform *transform,
					       const gchar        *version,
					       const gchar        *filelist_path)
{
	gchar *dir;
	gchar *tmp;
	
	dir = g_build_filename (SYSCONFDIR, "osso-backup", "restore.d", version, NULL);
	restore_transform_execute_scripts (transform, dir, filelist_path);
	g_free (dir);
		
	tmp = ob_utils_get_install_home ();
	dir = g_build_filename (tmp, "etc", "osso-backup", "restore.d", version, NULL);
	restore_transform_execute_scripts (transform, dir, filelist_path);
	g_free (tmp);
	g_free (dir);
}

void
ob_restore_transform_execute (ObRestoreTransform *transform)
{
	gboolean           has_files;
	int                i;
	char              *filelist_path;
	ObSoftwareVersion  from;
	ObSoftwareVersion  to;

	has_files = FALSE;
	for (i = 0; i < OB_NUM_CATEGORIES; i++) {
		if (transform->files[i]) {
			has_files = TRUE;
			break;
		}
	}
	
	if (!has_files) {
		ob_log_info ("No files to transform.");
	}
	
	filelist_path = restore_transform_create_file_list (transform);

	d(g_printerr ("Created temp file: %s\n", filelist_path));

	if (restore_transform_check_from_and_to (transform, &from, &to)) {
		int from_index, to_index;
		
		from_index = 0;
		to_index = 0;
		for (i = 0; i < G_N_ELEMENTS (version_dirs); i++) {
			if (version_dirs[i].version == from) {
				from_index = i + 1;
			}
			else if (version_dirs[i].version == to) {
				to_index = i;
			}
		}

		for (i = from_index; i <= to_index; i++) {
			restore_transform_execute_scripts_for_version (
				transform, version_dirs[i].restore_dir, filelist_path);
		}
	}
	
	/* And scripts that are ran every time. */
	restore_transform_execute_scripts_for_version (transform, "always", filelist_path);
	
	unlink (filelist_path);
	g_free (filelist_path);
}

