/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <string.h>
#include <glib.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libxml/parser.h>
#include <libxml/tree.h>

#include "ob-config.h"
#include "ob-utils.h"
#include "ob-log.h"

#define MEMORY_CARD_ENV          "MMC_MOUNTPOINT"
#define INTERNAL_MEMORY_CARD_ENV "INTERNAL_MMC_MOUNTPOINT"
#define DOCUMENTS_PATH_ENV       "MYDOCSDIR"

#define DEFAULT_APP_KILLER_DELAY 5
#define DEFAULT_DOCUMENTS_PATH   "$HOME/MyDocs"
#define DEFAULT_MEMORY_CARD_PATH "/media/mmc1"

struct _ObConfig {
	GnomeVFSURI *memory_card_uri;
	GnomeVFSURI *internal_memory_card_uri;
	GnomeVFSURI *gconf_uri;
	GnomeVFSURI *documents_uri;

	int          app_killer_delay;
};


static gboolean config_parse_conf (ObConfig *config, const char *path);

static ObConfig *the_config;


ObConfig *
ob_config_get (void)
{
	return the_config;
}

/* Only exposed for testing. */
gboolean
_ob_config_init_empty (void)
{
	if (the_config) {
		ob_log_warning ("Config already inited.");
		return FALSE;
	}
	
	the_config = g_new0 (ObConfig, 1);

	the_config->app_killer_delay = DEFAULT_APP_KILLER_DELAY;
	
	return TRUE;
}

void
ob_config_init (const char *dir)
{
	char       *path;
	const char *env_mmc;
	const char *mydocs;

	if (!_ob_config_init_empty ()) {
		return;
	}

	if (!dir) {
		dir = SYSCONFDIR "/osso-backup";
	}
	
	path = g_build_filename (dir, "osso-backup.conf", NULL);
	if (!config_parse_conf (the_config, path)) {
		ob_log_warning ("No valid backup configuration found.");
	}
	g_free (path);

	/* Memory card specifed in the environment variable overrides the config
	 * file.
	 */
	env_mmc = g_getenv (MEMORY_CARD_ENV);
	if (env_mmc && env_mmc[0]) {
		if (the_config->memory_card_uri) {
			gnome_vfs_uri_unref (the_config->memory_card_uri);
		}

		the_config->memory_card_uri = gnome_vfs_uri_new (env_mmc);
	}

	if (!the_config->memory_card_uri) {
		ob_log_warning ("No valid memory card mountpoint specified.");
	}

	/* And the internal MMC. */
	env_mmc = g_getenv (INTERNAL_MEMORY_CARD_ENV);
	if (env_mmc && env_mmc[0]) {
		if (the_config->internal_memory_card_uri) {
			gnome_vfs_uri_unref (the_config->internal_memory_card_uri);
		}

		the_config->internal_memory_card_uri = gnome_vfs_uri_new (env_mmc);
	}

	if (!the_config->internal_memory_card_uri) {
		ob_log_warning ("No valid internal memory card mountpoint specified.");
	}
	
	/* Documents path specifed in the environment variable overrides the
	 * config file.
	 */
	mydocs = g_getenv (DOCUMENTS_PATH_ENV);
	if (mydocs && mydocs[0]) {
		if (the_config->documents_uri) {
			gnome_vfs_uri_unref (the_config->documents_uri);
		}

		the_config->documents_uri = gnome_vfs_uri_new (mydocs);
	}
}

void
ob_config_shutdown (void)
{
	if (!the_config) {
		return;
	}
	
	if (the_config->memory_card_uri) {
		gnome_vfs_uri_unref (the_config->memory_card_uri);
	}
	if (the_config->internal_memory_card_uri) {
		gnome_vfs_uri_unref (the_config->internal_memory_card_uri);
	}
	if (the_config->gconf_uri) {
		gnome_vfs_uri_unref (the_config->gconf_uri);
	}
	if (the_config->documents_uri) {
		gnome_vfs_uri_unref (the_config->documents_uri);
	}
	
	g_free (the_config);
	the_config = NULL;
}

GnomeVFSURI *
ob_config_get_memory_card_mountpoint (ObConfig         *config,
				      ObMemoryCardType  type)
{
	g_return_val_if_fail (config != NULL, NULL);

	if (type == OB_MEMORY_CARD_INTERNAL) {
		return config->internal_memory_card_uri;
	} else {
		return config->memory_card_uri;
	}
}

void
_ob_config_set_memory_card_mountpoint (ObConfig         *config,
				       ObMemoryCardType  type,
				       GnomeVFSURI      *uri)
{
	g_return_if_fail (config != NULL);
	g_return_if_fail (uri != NULL);

	gnome_vfs_uri_ref (uri);
	
	if (type == OB_MEMORY_CARD_INTERNAL) {
		if (config->internal_memory_card_uri) {
			gnome_vfs_uri_unref (config->internal_memory_card_uri);
		}
		
		config->internal_memory_card_uri = uri;
	} else {
		if (config->memory_card_uri) {
			gnome_vfs_uri_unref (config->memory_card_uri);
		}
		
		config->memory_card_uri = uri;
	}
}

GnomeVFSURI *
ob_config_get_gconf_dir (ObConfig *config)
{
	g_return_val_if_fail (config != NULL, NULL);

	return config->gconf_uri;
}

void
_ob_config_set_gconf_dir (ObConfig    *config,
			  GnomeVFSURI *uri)
{
	g_return_if_fail (config != NULL);
	g_return_if_fail (uri != NULL);

	gnome_vfs_uri_ref (uri);
	
	if (config->gconf_uri) {
		gnome_vfs_uri_unref (config->gconf_uri);
	}

	config->gconf_uri = uri;
}

GnomeVFSURI *
ob_config_get_documents_dir (ObConfig *config)
{
	g_return_val_if_fail (config != NULL, NULL);

	return config->documents_uri;
}

void
_ob_config_set_documents_dir (ObConfig    *config,
			      GnomeVFSURI *uri)
{
	g_return_if_fail (config != NULL);
	g_return_if_fail (uri != NULL);

	gnome_vfs_uri_ref (uri);
	
	if (config->documents_uri) {
		gnome_vfs_uri_unref (config->documents_uri);
	}

	config->documents_uri = uri;
}

int
ob_config_get_app_killer_delay (ObConfig *config)
{
	g_return_val_if_fail (config != NULL, DEFAULT_APP_KILLER_DELAY);

	return config->app_killer_delay;
}


/*
 * The XML parser.
 */
static GnomeVFSURI *
parse_path (const char *str)
{
	char *path;

        path = ob_utils_parse_path (str);
        if (!path) {
                return NULL;
        }

	return gnome_vfs_uri_new (path);
}

static gboolean
config_parse_memory_card (ObConfig         *config,
			  ObMemoryCardType  type,
			  xmlNode          *node)
{
	char *path;

	if ((type == OB_MEMORY_CARD_INTERNAL && config->internal_memory_card_uri) ||
	    (type == OB_MEMORY_CARD_EXTERNAL && config->memory_card_uri)) {
		ob_log_warning ("Already got memory card mountpoint in config.");
		return FALSE;
	}

	path = NULL;
	for (node = node->children; node; node = node->next) {
		if (strcmp (node->name, "mountpoint") == 0) {
			path = xmlNodeGetContent (node);
			break;
		}
	}

	if (!path) {
		return FALSE;
	}
	
	if (type == OB_MEMORY_CARD_INTERNAL) {
		config->internal_memory_card_uri = parse_path (path);
	} else {
		config->memory_card_uri = parse_path (path);
	}

	xmlFree (path);
	
	return TRUE;
}

static gboolean
config_parse_gconf_dir (ObConfig *config,
			xmlNode  *node)
{
	char *path;

	if (config->gconf_uri) {
		ob_log_warning ("Already got gconf dir in config.");
		return FALSE;
	}
	
	path = xmlNodeGetContent (node);

	config->gconf_uri = parse_path (path);
	
	xmlFree (path);

	if (!config->gconf_uri) {
		ob_log_warning ("Invalid gconf dir found in config.");
		return FALSE;
	}
	
	return TRUE;
}

static gboolean
config_parse_documents_dir (ObConfig *config,
			    xmlNode  *node)
{
	char *path;

	if (config->documents_uri) {
		ob_log_warning ("Already got documents dir in config.");
		return FALSE;
	}
	
	path = xmlNodeGetContent (node);

	config->documents_uri = parse_path (path);
	
	xmlFree (path);

	if (!config->documents_uri) {
		ob_log_warning ("Invalid documents dir found in config.");
		return FALSE;
	}
	
	return TRUE;
}

static gboolean
config_parse_app_killer_delay (ObConfig *config,
			       xmlNode  *node)
{
	char *str;

	str = xmlNodeGetContent (node);
	if (str) {
		config->app_killer_delay = atoi (str);
	} else {
		config->app_killer_delay = DEFAULT_APP_KILLER_DELAY;
	}
	
	xmlFree (str);
	
	return TRUE;
}

static gboolean
config_parse_conf (ObConfig *config, const char *path)
{
	xmlDoc  *doc;
	xmlNode *root, *node;

	if (!g_file_test (path, G_FILE_TEST_IS_REGULAR)) {
		return FALSE;
	}
	
	doc = xmlParseFile (path);
	if (!doc) {
		return FALSE;
	}

	root = xmlDocGetRootElement (doc);

	if (!root || strcmp (root->name, "backup-configuration") != 0) {
		xmlFreeDoc (doc);

		return FALSE;
	}

	for (node = root->children; node; node = node->next) {
		if (strcmp (node->name, "internal-memory-card") == 0) {
			config_parse_memory_card (config, OB_MEMORY_CARD_INTERNAL, node);
		}
		if (strcmp (node->name, "memory-card") == 0) {
			config_parse_memory_card (config, OB_MEMORY_CARD_EXTERNAL, node);
		}
		else if (strcmp (node->name, "gconf-dir") == 0) {
			config_parse_gconf_dir (config, node);
		}
		else if (strcmp (node->name, "documents-path") == 0) {
			config_parse_documents_dir (config, node);
		}
		else if (strcmp (node->name, "app-killer-delay") == 0) {
			config_parse_app_killer_delay (config, node);
		}
	}

	/* Set up mandatory defaults in case any config is missing. */
	if (!config->documents_uri) {
		config->documents_uri = parse_path (DEFAULT_DOCUMENTS_PATH);
	}
	if (!config->memory_card_uri) {
		config->memory_card_uri = gnome_vfs_uri_new (DEFAULT_MEMORY_CARD_PATH);
	}

	xmlFreeDoc (doc);

	return TRUE;
}

