/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <libogs/ogs-file-system.h>

#include "ob-config.h"
#include "ob-utils.h"
#include "ob-category.h"

#define d(x) 

ObFileEntry *
ob_file_entry_new (GnomeVFSURI      *uri,
		   GnomeVFSFileInfo *file_info)
{
	ObFileEntry *entry;

	g_return_val_if_fail (uri != NULL, NULL);
	g_return_val_if_fail (file_info != NULL, NULL);

	entry = g_new0 (ObFileEntry, 1);
	entry->uri = gnome_vfs_uri_ref (uri);

	gnome_vfs_file_info_ref (file_info);
	entry->file_info = file_info;

	return entry;
}

ObFileEntry *
ob_file_entry_dup (ObFileEntry *entry)
{
	ObFileEntry *new_entry;

	g_return_val_if_fail (entry != NULL, NULL);

	new_entry = g_new0 (ObFileEntry, 1);
	new_entry->uri = gnome_vfs_uri_ref (entry->uri);

	gnome_vfs_file_info_ref (entry->file_info);
	new_entry->file_info = entry->file_info;

	return new_entry;
}

void
ob_file_entry_free (ObFileEntry *entry)
{
	g_return_if_fail (entry != NULL);
	
	gnome_vfs_uri_unref (entry->uri);
	gnome_vfs_file_info_unref (entry->file_info);

	g_free (entry);
}

void
ob_file_entry_free_list (GSList *entries)
{
	g_slist_foreach (entries, (GFunc) ob_file_entry_free, NULL);
	g_slist_free (entries);
}

/* For debugging. */
void
ob_file_entry_print (ObFileEntry *entry)
{
	const gchar *path;
	gchar       *dir, *base;
	
	path = gnome_vfs_uri_get_path (entry->uri);
	dir = g_path_get_dirname (path);
	base = g_path_get_basename (dir);
	
	g_print (" %s/%s (%s)\n", base, entry->file_info->name, entry->file_info->mime_type);

	g_free (dir);
	g_free (base);
}

ObCategoryFiles *
ob_category_files_new (void)
{
	ObCategoryFiles *files;

	files = g_new0 (ObCategoryFiles, 1);

	return files;
}

void
ob_category_files_free (ObCategoryFiles *files)
{
	int i;
	
	g_return_if_fail (files != NULL);

	for (i = 0; i < OB_NUM_CATEGORIES; i++) {
		g_slist_foreach (files->file_entries[i],
				 (GFunc) ob_file_entry_free, NULL);
	}

	g_free (files);
}

/*
 * Code for getting filling the ObCategoryFiles struct with the files to backup.
 */

typedef struct {
	ObCategoryFiles *files;
	ObCategory       category;
} FilesAndCategory;


/* Used to get a specific category. */
static gboolean
specific_file_callback (OgsFileSystem    *file_system,
			GnomeVFSURI      *uri,
			GnomeVFSFileInfo *info,
			OgsSearchContext *context,
			gpointer          user_data)
{
	FilesAndCategory *fac;
	ObCategoryFiles  *files;
	ObFileEntry      *entry;
	int               index;

	fac = (FilesAndCategory *) user_data;
	files = fac->files;
	
	entry = ob_file_entry_new (uri, info);

	index = ob_category_to_index (fac->category);

	files->file_entries[index] =
		g_slist_prepend (files->file_entries[index], entry);

	files->file_size[index] += info->size;
	files->num_files[index]++;

	return TRUE;
}

/* Used to get files in the "other" category, i.e. files that don't fit in
 * anywhere else.
 */
static gboolean
all_file_callback (OgsFileSystem    *file_system,
		   GnomeVFSURI      *uri,
		   GnomeVFSFileInfo *info,
		   OgsSearchContext *context,
		   gpointer          user_data)
{
	ObCategoryFiles *files;
	ObFileEntry     *entry;
	ObCategory       category;
	int              index;

	/* For testing. */
	/*g_usleep (1000*100);*/
	
	files = (ObCategoryFiles *) user_data;

	entry = ob_file_entry_new (uri, info);

	category = ob_category_from_mime_type (info->mime_type);
	index = ob_category_to_index (category);
	
	files->file_entries[index] = g_slist_prepend (files->file_entries[index],
						      entry);
	
	files->file_size[index] += info->size;
	files->num_files[index]++;
	
	return TRUE;
}

static void
ob_category_files_print_categories (ObCategoryFiles *files,
				    int              categories)
{
	int         i;
	ObCategory  category;
	GSList     *l;

	d(goto print_debug);
	return;
	d(print_debug:)
	
	if (categories == OB_CATEGORY_ALL) {
		for (i = 0; i < OB_NUM_CATEGORIES ; i++) {
			category = ob_category_from_index (i);
			g_print ("%s (%d)\n",
				 ob_category_to_non_translated_string (category),
				 files->num_files[i]);
			for (l = files->file_entries[i]; l; l = l->next) {
				ob_file_entry_print (l->data);
			}
		}
		g_print ("\n");
		return;
	}

	for (i = 0; i < OB_NUM_CATEGORIES ; i++) {
		category = ob_category_from_index (i);
		if (categories & category) {
			g_print ("%s (%d)\n",
				 ob_category_to_non_translated_string (category),
				 files->num_files[i]);
			for (l = files->file_entries[i]; l; l = l->next) {
				ob_file_entry_print (l->data);
			}
		}
	}
	
	g_print ("\n");
}

#if 0
typedef gboolean (*ObCategoryCancellationFunc) (gpointer user_data);

typedef struct {
	OgsFileSystemCancellationCallback func;
	gpointer                          data;
} CancelFuncData;

static gboolean
category_cancellation_cb (gpointer user_data)
{
	CancelFuncData *cancel_func_data;

	cancel_func_data = user_data;

	if (cancel_func_data->func) {
		return cancel_func_data->func (cancel_func_data->data);
	}
	
	return FALSE;
}
#endif

void
ob_category_files_read (ObCategoryFiles *files,
			int              categories)
{
	OgsFileSystem    *fs;
	FilesAndCategory  fac;
	/*CancelFuncData    cancel_func_data;*/
	GnomeVFSURI      *uri;
	
	fac.files = files;

	uri = ob_config_get_documents_dir (ob_config_get ());
	
	fs = ob_utils_get_file_system ();

#if 0
	  cancel_func_data.func = cancellation_func;
	  cancel_func_data.data = user_data;

	  ogs_file_system_set_cancellation_callback (fs,
						   category_cancellation_cb,
						   &cancel_func_data);
#endif
	
	/* Special-case the 'all' category. */
	if (categories == OB_CATEGORY_ALL) {
		d(g_print ("Getting 'all' files:\n"));
		ogs_file_system_get_files_with_category (fs, OGS_FILE_SYSTEM_DOCUMENTS, 
							 OSSO_MIME_CATEGORY_ALL, NULL,
							 all_file_callback,
							 files);
		ob_category_files_print_categories (files, OB_CATEGORY_ALL);
		return;
	}
	
	if (categories & OB_CATEGORY_EMAILS) {
		d(g_print ("Getting 'emails' files:\n"));
		fac.category = OB_CATEGORY_EMAILS;
		ogs_file_system_get_files_with_category (fs, OGS_FILE_SYSTEM_DOCUMENTS, 
							 OSSO_MIME_CATEGORY_EMAILS, NULL,
							 specific_file_callback,
							 &fac);
		ob_category_files_print_categories (files, fac.category);
	}
	
	if (categories & OB_CATEGORY_CONTACTS) {
		d(g_print ("Getting 'contacts' files:\n"));
		fac.category = OB_CATEGORY_CONTACTS;
		ogs_file_system_get_files_with_category (fs, OGS_FILE_SYSTEM_DOCUMENTS, 
							 OSSO_MIME_CATEGORY_CONTACTS, NULL,
							 specific_file_callback,
							 &fac);
		ob_category_files_print_categories (files, fac.category);
	}
	
	if (categories & OB_CATEGORY_DOCUMENTS) {
		d(g_print ("Getting 'documents' files:\n"));
		fac.category = OB_CATEGORY_DOCUMENTS;
		ogs_file_system_get_files_with_category (fs, OGS_FILE_SYSTEM_DOCUMENTS, 
							 OSSO_MIME_CATEGORY_DOCUMENTS, NULL,
							 specific_file_callback,
							 &fac);
		ob_category_files_print_categories (files, fac.category);
	}

	if (categories & OB_CATEGORY_MEDIA) {
		d(g_print ("Getting 'media' files:\n"));
		fac.category = OB_CATEGORY_MEDIA;
		ogs_file_system_get_files_with_category (fs, OGS_FILE_SYSTEM_DOCUMENTS, 
							 OSSO_MIME_CATEGORY_AUDIO, NULL,
							 specific_file_callback,
							 &fac);
		ogs_file_system_get_files_with_category (fs, OGS_FILE_SYSTEM_DOCUMENTS, 
							 OSSO_MIME_CATEGORY_VIDEO, NULL,
							 specific_file_callback,
							 &fac);
		ogs_file_system_get_files_with_category (fs, OGS_FILE_SYSTEM_DOCUMENTS, 
							 OSSO_MIME_CATEGORY_IMAGES, NULL,
							 specific_file_callback,
							 &fac);
		ob_category_files_print_categories (files, fac.category);
	}
	
	if (categories & OB_CATEGORY_BOOKMARKS) {
		d(g_print ("Getting 'bookmarks' files:\n"));
		fac.category = OB_CATEGORY_BOOKMARKS;
		ogs_file_system_get_files_with_category (fs, OGS_FILE_SYSTEM_DOCUMENTS, 
							 OSSO_MIME_CATEGORY_BOOKMARKS, NULL,
							 specific_file_callback,
							 &fac);
		ob_category_files_print_categories (files, fac.category);
	}

	/* Note: Settings are not part of the mime/category mapping. */
#if 0
	/* Doing this for later when we have one fs during the session. */
	ogs_file_system_set_cancellation_callback (fs, NULL, NULL);
#endif
}

ObCategory
ob_category_from_index (int index)
{
	switch (index) {
	case OB_CATEGORY_INDEX_EMAILS:
		return OB_CATEGORY_EMAILS;
	case OB_CATEGORY_INDEX_CONTACTS:
		return OB_CATEGORY_CONTACTS;
	case OB_CATEGORY_INDEX_DOCUMENTS:
		return OB_CATEGORY_DOCUMENTS;
	case OB_CATEGORY_INDEX_MEDIA:
		return OB_CATEGORY_MEDIA;
	case OB_CATEGORY_INDEX_BOOKMARKS:
		return OB_CATEGORY_BOOKMARKS;
	case OB_CATEGORY_INDEX_SETTINGS:
		return OB_CATEGORY_SETTINGS;
	case OB_CATEGORY_INDEX_OTHER:
		return OB_CATEGORY_OTHER;
	}

	g_assert_not_reached ();
	return OB_CATEGORY_OTHER;
}

int
ob_category_to_index (ObCategory category)
{
	switch (category) {
		
	case OB_CATEGORY_EMAILS:
		return OB_CATEGORY_INDEX_EMAILS;
	case OB_CATEGORY_CONTACTS:
		return OB_CATEGORY_INDEX_CONTACTS;
	case OB_CATEGORY_DOCUMENTS:
		return OB_CATEGORY_INDEX_DOCUMENTS;
	case OB_CATEGORY_MEDIA:
		return OB_CATEGORY_INDEX_MEDIA;
	case OB_CATEGORY_BOOKMARKS:
		return OB_CATEGORY_INDEX_BOOKMARKS;
	case OB_CATEGORY_SETTINGS:
		return OB_CATEGORY_INDEX_SETTINGS;
	case OB_CATEGORY_OTHER:
		return OB_CATEGORY_INDEX_OTHER;
	case OB_CATEGORY_ALL:
		/* shouldn't happen */
		break;
	}

	g_assert_not_reached ();
	return OB_CATEGORY_INDEX_OTHER;
}

ObCategory
ob_category_from_mime_type (const gchar *mime_type)
{
	OssoMimeCategory category;

	category = osso_mime_get_category_for_mime_type (mime_type);
	switch (category) {
	case OSSO_MIME_CATEGORY_BOOKMARKS:
                return OB_CATEGORY_BOOKMARKS;
        case OSSO_MIME_CATEGORY_CONTACTS:
		return OB_CATEGORY_CONTACTS;
        case OSSO_MIME_CATEGORY_DOCUMENTS:
                return OB_CATEGORY_DOCUMENTS;
        case OSSO_MIME_CATEGORY_EMAILS:
		return OB_CATEGORY_EMAILS;
        case OSSO_MIME_CATEGORY_IMAGES:
        case OSSO_MIME_CATEGORY_AUDIO:
        case OSSO_MIME_CATEGORY_VIDEO:
                return OB_CATEGORY_MEDIA;
        case OSSO_MIME_CATEGORY_OTHER:
		return OB_CATEGORY_OTHER;
	case OSSO_MIME_CATEGORY_ALL:
		/* silence compiler. */
		g_assert_not_reached ();
	}

	return OB_CATEGORY_OTHER;
}

const gchar *
ob_category_to_string (ObCategory category)
{
	switch (category) {
	case OB_CATEGORY_EMAILS:
		return _("back_fi_messages");
		break;
	case OB_CATEGORY_CONTACTS:
		return _("back_fi_contacts");
		break;
	case OB_CATEGORY_DOCUMENTS:
		return _("back_fi_documents");
		break;
	case OB_CATEGORY_MEDIA:
		return _("back_fi_media");
		break;
	case OB_CATEGORY_BOOKMARKS:
		return _("back_fi_bookmarks");
		break;
	case OB_CATEGORY_SETTINGS:
		return _("back_fi_settings");
		break;
	case OB_CATEGORY_OTHER:
		return _("back_fi_other");
		break;
	default:
		return "Error";
		break;
	}
}

const gchar *
ob_category_to_non_translated_string (ObCategory category)
{
	const char *name;

	switch (category) {
	case OB_CATEGORY_EMAILS:
		name = "emails";
		break;
	case OB_CATEGORY_CONTACTS:
		name = "contacts";
		break;
	case OB_CATEGORY_DOCUMENTS:
		name = "documents";
		break;
	case OB_CATEGORY_MEDIA:
		name = "media";
		break;
	case OB_CATEGORY_BOOKMARKS:
		name = "bookmarks";
		break;
	case OB_CATEGORY_SETTINGS:
		name = "settings";
		break;
	case OB_CATEGORY_OTHER:
		name = "other";
		break;

	default:
	case OB_CATEGORY_ALL:
		/* Silence the compiler. */
		name = NULL;
		break;
	}

	return name;
}

GnomeVFSFileSize
ob_category_files_get_size (ObCategoryFiles *files,
			    int              categories)
{
	GnomeVFSFileSize size;
	int              i;
	ObCategory       category;

	size = 0;
	
	for (i = 0; i < OB_NUM_CATEGORIES; i++) {
		category = ob_category_from_index (i);
		if (categories & category) {
			size += files->file_size[i];
		}
	}
			
	return size;
}

int
ob_category_files_get_num_files (ObCategoryFiles *files,
				 int              categories)

{
	int        num_files;
	int        i;
	ObCategory category;
	
	num_files = 0;
	
	for (i = 0; i < OB_NUM_CATEGORIES; i++) {
		category = ob_category_from_index (i);
		if (categories & category) {
			num_files += files->num_files[i];
		}
	}
			
	return num_files;
}

/* Returns a newly allocated list, needs to be freed with
 * ob_category_files_free_entries().
 */
GSList *
ob_category_files_get_file_entries (ObCategoryFiles *files,
				    ObCategory       category)
{
	GSList      *list, *l;
	int          index;
	ObFileEntry *entry;

	g_return_val_if_fail (files != NULL, NULL);
	g_return_val_if_fail (category != OB_CATEGORY_ALL, NULL);
	
	index = ob_category_to_index (category);

	list = NULL;
	for (l = files->file_entries[index]; l; l = l->next) {
		entry = ob_file_entry_dup (l->data);
		list = g_slist_prepend (list, entry);
	}
	
	return g_slist_reverse (list);
}

