/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <unistd.h>
#include <libgnomevfs/gnome-vfs.h>

#include "ob-archiver.h"

/* Used for debugging messages. */
#define d(x)

/* Used for debugging progress events, very verbose. */
#define p(x)

/* Define this to add sleeping every now and then. */
/*#define SIMULATE_SLOW*/


struct _ObArchiverPriv {
	ObBackend   *backend;

	GnomeVFSURI *archive_uri;     /* location of the archive */
	GnomeVFSURI *data_uri;        /* location of the user data */

	gboolean     replace_all;     /* replace all files on conflict */

	int          progress;        /* 0-100, estimated time */
	int          total_files;     /* total amount of files */
	int          processed_files; /* amount of processed files*/

	gsize        total_size;      /* total size of files */
	gsize        processed_size;  /* size of processed files*/

	int	     last_sync;	      /* progress mark from last sync(2) */
	GTimeVal     last_progress;   /* time of last progress update */

	ObCategory   current_category;
};


#define GET_PRIV(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), OB_TYPE_ARCHIVER, ObArchiverPriv))

static void archiver_finalize (GObject *object);

G_DEFINE_TYPE (ObArchiver, ob_archiver, G_TYPE_OBJECT)


static void
ob_archiver_class_init (ObArchiverClass *klass)
{
        GObjectClass *object_class;

        object_class = G_OBJECT_CLASS (klass);

        object_class->finalize = archiver_finalize;

        g_type_class_add_private (object_class, sizeof (ObArchiverPriv));
}

static void
ob_archiver_init (ObArchiver *archiver)
{
	archiver->priv = GET_PRIV (archiver);
}

static void
archiver_finalize (GObject *object)
{
	ObArchiverPriv *priv;

	priv = OB_ARCHIVER (object)->priv;

	if (priv->backend) {
		g_object_unref (priv->backend);
	}

	if (priv->archive_uri) {
		gnome_vfs_uri_unref (priv->archive_uri);
	}
	if (priv->data_uri) {
		gnome_vfs_uri_unref (priv->data_uri);
	}

	G_OBJECT_CLASS (ob_archiver_parent_class)->finalize (object);
}

int
ob_archiver_get_processed_files (ObArchiver *archiver)
{
	return archiver->priv->processed_files;
}

void
ob_archiver_set_total_files (ObArchiver *archiver,
			     int         files)
{
	archiver->priv->total_files = files;
}

int
ob_archiver_get_total_files (ObArchiver *archiver)
{
	return archiver->priv->total_files;
}

gsize
ob_archiver_get_processed_size (ObArchiver *archiver)
{
	return archiver->priv->processed_size;
}

void
ob_archiver_set_total_size (ObArchiver *archiver,
			    gsize       size)
{
	archiver->priv->total_size = size;
}

gsize
ob_archiver_get_total_size (ObArchiver *archiver)
{
	return archiver->priv->total_size;
}

void
ob_archiver_add_processed_size (ObArchiver *archiver,
				gsize       size)
{
	archiver->priv->processed_size += size;
}

void
ob_archiver_add_processed_files  (ObArchiver *archiver,
				  int         files)
{
	archiver->priv->processed_files += files;
}

void
ob_archiver_set_processed_size (ObArchiver *archiver,
				gsize       size)
{
	archiver->priv->processed_size = size;
}

void
ob_archiver_set_processed_files  (ObArchiver *archiver,
				  int         files)
{
	archiver->priv->processed_files = files;
}

void
ob_archiver_set_backend (ObArchiver *archiver,
			 ObBackend  *backend)
{
	if (backend) {
		g_object_ref (backend);
	}

	if (archiver->priv->backend) {
		g_object_unref (archiver->priv->backend);
	}

	archiver->priv->backend = backend;
}

ObBackend *
ob_archiver_get_backend (ObArchiver  *archiver)
{
	return archiver->priv->backend;
}

void
ob_archiver_set_replace_all (ObArchiver *archiver,
			     gboolean    replace_all)
{
	archiver->priv->replace_all = replace_all;
}

gboolean
ob_archiver_get_replace_all (ObArchiver *archiver)
{
	return archiver->priv->replace_all;
}

void
ob_archiver_set_current_category (ObArchiver *archiver,
				  ObCategory  category)
{
	archiver->priv->current_category = category;
}

ObCategory
ob_archiver_get_current_category (ObArchiver *archiver)
{
	return archiver->priv->current_category;
}

void
ob_archiver_push_progress_event (ObArchiver *archiver)
{
	ObArchiverPriv *priv;
	GTimeVal        tv;
	ObEvent        *event;
	int             progress;

	priv = archiver->priv;

#ifdef SIMULATE_SLOW
	g_usleep (G_USEC_PER_SEC / 50);
#endif
	
	progress = archiver->priv->processed_size /
		MAX (1, (archiver->priv->total_size / 100));

	if (priv->processed_files < priv->total_files) {
		int sync_diff;

		/* Sync disk every once in a while. */
		sync_diff = ABS (priv->last_sync - progress);
		if (sync_diff >= 10) {
			priv->last_sync = progress;

			d(g_print ("Syncing... "));
			sync();
			d(g_print ("done\n"));
		}

		/* Ensure we don't reach 100% too early. */
		progress = MIN (progress, 99);
	} else {
		/* Force sync at >= 100%, so all buffers are flushed. */
		priv->last_sync = 0;

		/* See noflushd(8) on why three syncs are better than one. */
		d(g_print ("Final sync... "));
		sync(); sync(); sync();
		d(g_print ("done\n"));
	}

	/* Throttle progress events to save CPU cycles, update at roughly 4Hz. */
	g_get_current_time (&tv);
	if (tv.tv_sec > priv->last_progress.tv_sec ||
	    tv.tv_usec - priv->last_progress.tv_usec > 250*1000) {
		priv->last_progress = tv;

		/* Report progress _after_ syncing data. */
		event =	ob_event_new_progress (OB_PROGRESS_TYPE_NORMAL,
					       priv->processed_files,
					       priv->total_files,
					       progress,
					       priv->current_category);
		ob_backend_push_event (priv->backend, event);
		p(g_print ("Progress: %d/%d processed files, %d/%d processed bytes, category:%d->'%s'\n",
			   priv->processed_files, 
			   priv->total_files,
			   archiver->priv->processed_size,
			   archiver->priv->total_size, 
			   priv->current_category,
			   ob_category_to_string (priv->current_category)));
	}
}

void
ob_archiver_push_progress_finalizing_event (ObArchiver *archiver)
{
	ObArchiverPriv *priv;
	ObEvent        *event;

	priv = archiver->priv;

	event =	ob_event_new_progress (OB_PROGRESS_TYPE_FINALIZING, 0, 0, 0, 0);
	ob_backend_push_event (priv->backend, event);
}

void
ob_archiver_push_error_event (ObArchiver *archiver,
			      GError     *error)
{
	ob_backend_push_event (archiver->priv->backend,
			       ob_event_new_error (error));
}

void
ob_archiver_push_conflict_event (ObArchiver     *archiver,
				 ObConflictType  type,
				 GnomeVFSURI    *uri,
				 time_t          existing_time,
				 time_t          backup_time)
{
	ob_backend_push_event (archiver->priv->backend,
			       ob_event_new_conflict (type, uri,
						      existing_time,
						      backup_time));
}

void
ob_archiver_push_finished_event (ObArchiver *archiver)
{
	ObArchiverPriv *priv;

	priv = archiver->priv;

	ob_backend_push_event (priv->backend,
			       ob_event_new_finished (priv->processed_size,
						      priv->processed_files));
}

void
ob_archiver_push_cancelled_event (ObArchiver *archiver)
{
	ob_backend_push_event (archiver->priv->backend,
			       ob_event_new_cancelled ());
}

gboolean
ob_archiver_pack_start_thread (ObArchiver *archiver)
{
	GThread *thread;

	if (OB_ARCHIVER_GET_CLASS (archiver)->pack_thread_func) {
		thread = g_thread_create (
			(GThreadFunc) OB_ARCHIVER_GET_CLASS (archiver)->pack_thread_func,
			archiver,
			FALSE, NULL);

		return thread != NULL;
	}

	g_warning ("Pack thread func not implemented.");
	return FALSE;
}

gboolean
ob_archiver_unpack_start_thread (ObArchiver *archiver)
{
	GThread *thread;

	if (OB_ARCHIVER_GET_CLASS (archiver)->unpack_thread_func) {
		thread = g_thread_create (
			(GThreadFunc) OB_ARCHIVER_GET_CLASS (archiver)->unpack_thread_func,
			archiver,
			FALSE, NULL);

		return thread != NULL;
	}

	g_warning ("Unpack thread func not implemented.");
	return FALSE;
}
