/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005-2006 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <glib/gi18n.h>
#include <gsf/gsf-input-stdio.h>
#include <gsf/gsf-infile-zip.h>
#include <gsf/gsf-infile.h>
#include <gsf/gsf-output-stdio.h>
#include <gsf/gsf-outfile.h>
#include <gsf/gsf-outfile-zip.h>
#include <libgnomevfs/gnome-vfs.h>

#include "gsf-output-crypt.h"
#include "gsf-input-crypt.h"
#include "ob-archiver.h"
#include "ob-archiver-zip.h"
#include "ob-vfs-utils.h"
#include "ob-utils.h"
#include "ob-error.h"
#include "ob-log.h"
#include "ob-category.h"
#include "ob-restore-transform.h"

#define d(x)

#define IS_GSF_DIR(x) (GSF_IS_INFILE(x) && gsf_infile_num_children(GSF_INFILE(x)) != -1)


/* After this number of kilobytes, a progress event is emitted. */
#define PROGRESS_CHUNK 32*1024

/* Buffer size used when reading/writing. */
#define BUF_SIZE 4*1024

/* The margin we are using when checking if there is space enough when restoring. */
#define RESTORE_SPACE_MARGIN 512*1024

/* Where files are temporarily unpacked before moved into place. */
#define TMP_URI "file:///var/tmp"
#define TMP_ENV "BACKUP_TMP"

/*#define SHUTDOWN_GCONF*/

typedef struct {
	GnomeVFSURI *temp_uri;
	GnomeVFSURI *target_uri;
} TempFileEntry;

typedef enum {
	CONFLICT_MODE_ASK,
	CONFLICT_MODE_DONT_ASK
} ConflictMode;

static void           archiver_zip_finalize (GObject          *object);

/*
 * Packing
 */
static ArchiverResult archiver_pack_file    (ObArchiver        *archiver,
					     GsfOutput         *dir,
					     GnomeVFSURI       *uri,
					     GnomeVFSFileInfo  *file_info,
					     GError           **error);
static GsfOutput *    archiver_pack_subdir  (ObArchiver        *archiver,
					     GsfOutput         *dir,
					     const char        *subdirname,
					     GError           **error);
static ArchiverResult archiver_zip_pack     (ObArchiver        *archiver);


/*
 * Unpacking
 */
static ObConflictResponse
archiver_handle_conflict_file_file            (ObArchiver   *archiver,
					       GnomeVFSURI *target_uri,
					       time_t       existing_timestamp,
					       time_t       backup_timestamp);
static ObConflictResponse
archiver_handle_conflict_dir_file             (ObArchiver  *archiver,
					       GnomeVFSURI *target_uri,
					       time_t       existing_timestamp,
					       time_t       backup_timestamp);
static ObConflictResponse
archiver_handle_conflict_file_dir             (ObArchiver  *archiver,
					       GsfInput    *input_dir,
					       GnomeVFSURI *uri_file);
static ArchiverResult archiver_unpack_file    (ObArchiver   *archiver,
					       GsfInput     *input,
					       GnomeVFSURI  *target_uri,
					       ConflictMode  conflict_mode,
					       ObCategory    category,
					       GError      **error);
static ArchiverResult archiver_unpack_dir     (ObArchiver   *archiver,
					       GsfInput     *input,
					       GnomeVFSURI  *uri,
					       mode_t        mode,
					       ConflictMode  conflict_mode,
					       GnomeVFSURI **renamed_uri,
					       gboolean     *stop_traversing,
					       GError      **error);
static gint           
archiver_unpack_find_foreach                  (const gchar  *uri_str,
					       const gchar  *to_find);
static ArchiverResult archiver_unpack_recurse (ObArchiver   *archiver,
					       ObCategory    category,
					       GsfInput     *archive_parent,
					       GnomeVFSURI  *uri,
					       GList        *excluded_uris,
					       GSList      **temp_file_list,
					       ConflictMode  conflict_mode,
					       GError      **error);
static ArchiverResult archiver_zip_unpack     (ObArchiver   *archiver);


/*
 * VFuncs
 */
static gpointer archiver_zip_pack_thread_func   (ObArchiver *archiver);
static gpointer archiver_zip_unpack_thread_func (ObArchiver *archiver);


struct _ObArchiverZipPriv {
	ObBackupInfo       *backup_info;
	ObBackupLocations  *locations;
	ObCategoryFiles    *category_files;
	GnomeVFSURI        *gconf_dir;

	guchar             *buf;

	/* The categories to save. */
	int                 categories;

	/* The categories that were successfully written. */
	int                 written_categories;

	/* The categories any previous backup with this name has. */
	int                 old_categories;

	char               *password;

	/* For sanity checking. */
	gboolean            is_restore;

	/* For transforming the restored files from an older platform. */
	ObRestoreTransform *restore_transform;
};

G_DEFINE_TYPE (ObArchiverZip, ob_archiver_zip, OB_TYPE_ARCHIVER)

#define GET_PRIV(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), OB_TYPE_ARCHIVER_ZIP, ObArchiverZipPriv))

static void
ob_archiver_zip_class_init (ObArchiverZipClass *klass)
{
        GObjectClass    *object_class;
	ObArchiverClass *archiver_class;

        object_class = G_OBJECT_CLASS (klass);
        object_class->finalize = archiver_zip_finalize;

	archiver_class = OB_ARCHIVER_CLASS (klass);
	archiver_class->pack_thread_func = archiver_zip_pack_thread_func;
	archiver_class->unpack_thread_func = archiver_zip_unpack_thread_func;

	g_type_class_add_private (object_class, sizeof (ObArchiverZipPriv));
}

static void
ob_archiver_zip_init (ObArchiverZip *archiver)
{
	ObArchiverZipPriv *priv;

	priv = GET_PRIV (archiver);
	archiver->priv = priv;

	priv->buf = g_malloc0 (BUF_SIZE);
}

static void
archiver_zip_finalize (GObject *object)
{
	ObArchiverZipPriv *priv;

	priv = OB_ARCHIVER_ZIP (object)->priv;

	if (priv->backup_info) {
		ob_backup_info_unref (priv->backup_info);
	}

	g_free (priv->buf);
	g_free (priv->password);

	if (priv->restore_transform) {
		ob_restore_transform_unref (priv->restore_transform);
	}
	
	G_OBJECT_CLASS (ob_archiver_zip_parent_class)->finalize (object);
}

/* Note: This could be done a bit nicer, we don't need to pass in the gconf dir,
 * it's available in the config singleton now.
 */
ObArchiver *
ob_archiver_zip_new_backup (ObBackupInfo      *backup_info,
			    ObCategoryFiles   *category_files,
			    int                old_categories,
			    int                total_files,
			    GnomeVFSFileSize   total_size,
			    const char        *password)
{
	ObArchiver        *archiver;
	ObArchiverZipPriv *priv;

	g_return_val_if_fail (backup_info != NULL, NULL);
	g_return_val_if_fail (category_files != NULL, NULL);

	archiver = g_object_new (OB_TYPE_ARCHIVER_ZIP, NULL);

	priv = OB_ARCHIVER_ZIP (archiver)->priv;

	priv->backup_info = ob_backup_info_ref (backup_info);
	priv->category_files = category_files;
	priv->locations = ob_backup_locations_get ();

	priv->gconf_dir = ob_config_get_gconf_dir (ob_config_get ());

	/* Keep the old categories so that we can restore them when done. This
	 * way we can write out the metadata for the old, unchanged
	 * categories. Mask out the new categories, so we don't set them if they
	 * are not successfully written.
	 */
	priv->categories = ob_backup_info_get_categories (backup_info);
	priv->written_categories = 0;

	/* When replacing all categories, we replace the whole directory, so
	 * don't consider the previous categories.
	 */
	if (priv->categories == OB_CATEGORY_ALL) {
		priv->old_categories = 0;
	} else {
		priv->old_categories = old_categories;
	}

	if (password && strlen (password) > 0) {
		priv->password = g_strdup (password);
	} else {
		priv->password = NULL;
	}

	ob_archiver_set_total_files (archiver, total_files);
	ob_archiver_set_total_size (archiver, total_size);

	return archiver;
}

/* "categories" are the categories that should be tried, the ones in backup_info
 * are the ones available.
 */
ObArchiver *
ob_archiver_zip_new_restore (ObBackupInfo *backup_info,
			     int           categories,
			     const char   *password)
{
	ObArchiver        *archiver;
	ObArchiverZipPriv *priv;

	g_return_val_if_fail (backup_info != NULL, NULL);

	archiver = g_object_new (OB_TYPE_ARCHIVER_ZIP, NULL);

	priv = OB_ARCHIVER_ZIP (archiver)->priv;

	priv->backup_info = ob_backup_info_ref (backup_info);
	priv->locations = ob_backup_locations_get ();

	priv->categories = categories;

	priv->is_restore = TRUE;

	priv->gconf_dir = ob_config_get_gconf_dir (ob_config_get ());

	if (password && strlen (password) > 0) {
		priv->password = g_strdup (password);
	} else {
		priv->password = NULL;
	}

	priv->restore_transform = ob_restore_transform_new (backup_info);

	return archiver;
}


/*
 * Utilities
 */

static int
archiver_zip_lock_card (ObArchiverZip *archiver)
{
	ObArchiverZipPriv *priv;
	GnomeVFSURI       *uri;
	char              *path, *full_path;
	int                fd;

	priv = archiver->priv;
	
	ob_backup_info_create_dir (priv->backup_info);
	
	uri = ob_backup_info_get_uri (priv->backup_info);
	path = gnome_vfs_unescape_string (gnome_vfs_uri_get_path (uri), NULL);
	full_path = g_build_filename (path, ".ob-XXXXXX", NULL);
	g_free (path);

	fd = mkstemp (full_path);
	if (fd == -1) {
		ob_log_warning ("Couldn't create lock file.");
		g_free (full_path);
		return fd;
	}

	/* Unlink, so we don't need to care about that when done or if we
	 * die.
	 */
	unlink (full_path);

	g_free (full_path);

	return fd;
}

#ifdef SHUTDOWN_GCONF
static void
shutdown_gconfd (void)
{
#ifdef USE_SUDO
	int      exit_status;
	gboolean script_retval;

	script_retval = g_spawn_command_line_sync (
		"sudo /etc/osso-af-init/gconf-daemon.sh stop",
		NULL, NULL,
		&exit_status,
		NULL);

	if (!script_retval || exit_status != 0) {
		ob_log_warning ("Could not shutdown gconfd");
	}
#endif
}
#endif

static char *
get_without_leading_slash (const char *str)
{
	if (!str) {
		return NULL;
	}

	if (str[0] == '/') {
		return g_strdup (str + 1);
	} else {
		return g_strdup (str);
	}
}

static char *
get_without_trailing_slash (const char *str)
{
	size_t len;

	if (!str) {
		return NULL;
	}

	len = strlen (str);
	if (len == 0) {
		return g_strdup (str);
	}

	if (str[len - 1] == '/') {
		return g_strndup (str, len - 1);
	} else {
		return g_strdup (str);
	}
}

static gchar *
get_path_without_leading_slash_from_uri (GnomeVFSURI *uri)
{
	const char *path;
	char       *tmp;
	char       *without;

	path = gnome_vfs_uri_get_path (uri);
	tmp = gnome_vfs_unescape_string (path, NULL);

	without = get_without_leading_slash (tmp);

	g_free (tmp);

	return without;
}

static char *
get_path_without_trailing_slash_from_uri (GnomeVFSURI *uri)
{
	const char *path;
	char       *tmp;
	char       *without;

	path = gnome_vfs_uri_get_path (uri);
	tmp = gnome_vfs_unescape_string (path, NULL);

	without = get_without_trailing_slash (tmp);

	g_free (tmp);

	return without;
}

/* Compares without trailing slashes in both the URI and path. */
static gboolean
uri_is_equal_to_path (GnomeVFSURI *uri, const char *path)
{
	char     *uri_path;
	char     *tmp_path;
	gboolean  ret;

	uri_path = get_path_without_trailing_slash_from_uri (uri);
	tmp_path = get_without_trailing_slash (path);

	if (strcmp (uri_path, path) == 0) {
		ret = TRUE;
	} else {
		ret = FALSE;
	}

	g_free (uri_path);
	g_free (tmp_path);

	return ret;
}

/* Checks if the uri is inside the dir that path points to. */
static gboolean
uri_is_under_dir (GnomeVFSURI *uri, const char *dir)
{
	GnomeVFSURI *dir_uri;
	gboolean  ret;

	dir_uri = gnome_vfs_uri_new (dir);
	if (!dir_uri) {
		return FALSE;
	}

	ret = gnome_vfs_uri_is_parent (dir_uri, uri, TRUE);

	gnome_vfs_uri_unref (dir_uri);

	return ret;
}


/*
 * Packing.
 */

/* Adds a file to the specified dir in the archive. */
static ArchiverResult
archiver_pack_file (ObArchiver        *archiver,
		    GsfOutput         *dir,
		    GnomeVFSURI       *uri,
		    GnomeVFSFileInfo  *file_info,
		    GError           **error)
{
	ObArchiverZipPriv *priv;
	char              *filename;
	GsfOutput         *file;
	GnomeVFSResult     result;
	GnomeVFSHandle    *handle;
	GnomeVFSFileSize   bytes_read;
	GnomeVFSFileSize   total_bytes_read;
	ArchiverResult     retval;

	priv = OB_ARCHIVER_ZIP (archiver)->priv;

	if (ob_archiver_is_cancelled (archiver)) {
		return ARCHIVER_RESULT_CANCELLED;
	}

	d(g_printerr ("Packing file: %s\n", gnome_vfs_uri_get_path (uri)));

	result = gnome_vfs_open_uri (&handle, uri, GNOME_VFS_OPEN_READ);
	if (result != GNOME_VFS_OK) {
		/* The file can't be read, due to permission for example. This
		 * is not fatal, just skip the file.
		 */
		ob_log_warning ("Couldn't read file to save");
		return ARCHIVER_RESULT_OK;
	}

	if (ob_archiver_is_cancelled (archiver)) {
		return ARCHIVER_RESULT_CANCELLED;
	}

	filename = gnome_vfs_uri_extract_short_name (uri);
	file = gsf_outfile_new_child  (GSF_OUTFILE (dir), filename, FALSE);
	g_free (filename);

	if (!file) {
		/* Couldn't create the file in the archive. This is a fatal
		 * error.
		 */
		gnome_vfs_close (handle);

		g_set_error (error, OB_ERROR, OB_ERROR_BACKUP_GENERIC,
			     "Could not create file in archive");
		return ARCHIVER_RESULT_ERROR;
	}
	
	/* Note: We must set the mode and timestamp before starting to write,
	 * otherwise the direntry is created already and the values are not
	 * used.
	 */
	/* Only get the rwx flags. */
	gsf_outfile_zip_set_mode (GSF_OUTFILE_ZIP (file),
				  file_info->permissions &
				  (S_IRWXU | S_IRWXG | S_IRWXO));
	gsf_outfile_zip_set_time (GSF_OUTFILE_ZIP (file), file_info->mtime);

	retval = ARCHIVER_RESULT_OK;
	total_bytes_read = 0;
	while (1) {
		if (ob_archiver_is_cancelled (archiver)) {
			retval = ARCHIVER_RESULT_CANCELLED;
			break;
		}

		result = gnome_vfs_read (handle,
					 priv->buf, BUF_SIZE,
					 &bytes_read);

		if (result == GNOME_VFS_ERROR_EOF) {
			break;
		}
		else if (result != GNOME_VFS_OK) {
			/* We've created the entry in the archive already and
			 * there is no way to revert that. Therefore this is a
			 * fatal error.
			 */
			g_set_error (error, OB_ERROR, OB_ERROR_BACKUP_GENERIC,
				     "Error reading vfs file");
			retval = ARCHIVER_RESULT_ERROR;
			break;
		}

		if (ob_archiver_is_cancelled (archiver)) {
			retval = ARCHIVER_RESULT_CANCELLED;
			break;
		}

		if (!gsf_output_write (file, bytes_read, priv->buf)) {
			/* Could not write to zip file, a fatal error. */

			/* Note: We might have run out of space here, but the
			 * gsf output won't let us know. Therefore we check for
			 * free space here, if it's less than 10k (we write in
			 * 4k chunks), we will assume that the problem is lack
			 * of space.
			 */
			GnomeVFSFileSize size;

			result = gnome_vfs_get_volume_free_space (
				ob_backup_info_get_uri (priv->backup_info),
				&size);

			if (result == GNOME_VFS_OK && size < 1024*10) {
				g_set_error (error, OB_ERROR, OB_ERROR_BACKUP_NO_SPACE,
					     "Backup, no space on memorycard");
			} else {
				g_set_error (error, OB_ERROR, OB_ERROR_BACKUP_GENERIC, 
					     "Backup, could not write to archive");
			}

			retval = ARCHIVER_RESULT_ERROR;
			break;
		}

		ob_archiver_add_processed_size (archiver, bytes_read);

		/* Emit progress events every chunk. */
		total_bytes_read += bytes_read;
		if (total_bytes_read % PROGRESS_CHUNK == 0) {
			ob_archiver_push_progress_event (archiver);
		}

		if (ob_archiver_is_cancelled (archiver)) {
			retval = ARCHIVER_RESULT_CANCELLED;
			break;
		}

		/* stop thread during backup (simulates device crash) */
		/* kill (getpid(), 19); */
	}

	gnome_vfs_close (handle);

	if (!gsf_output_close (file)) {
		/* Fatal error. */
		if (retval == ARCHIVER_RESULT_OK) {
			g_set_error (error, OB_ERROR, OB_ERROR_BACKUP_GENERIC,
				     "Backup, could not close archive file");
			retval = ARCHIVER_RESULT_ERROR;
		}
	}

        g_object_unref (file);

	if (retval == ARCHIVER_RESULT_OK) {
		ob_archiver_add_processed_files (archiver, 1);
		ob_archiver_push_progress_event (archiver);
	}

	return retval;
}

static GsfOutput *
archiver_pack_subdir (ObArchiver  *archiver,
		      GsfOutput   *dir,
		      const char  *subdirname,
		      GError     **error)
{
	GsfOutput *subdir;

	d(g_printerr ("Packing subdir: %s\n", subdirname));

	subdir  = gsf_outfile_new_child  (GSF_OUTFILE (dir), subdirname, TRUE);
	if (!subdir) {
		g_set_error (error, OB_ERROR, OB_ERROR_BACKUP_GENERIC,
			     "Backup, could not create subdir in archive");
		return NULL;
	}

	return subdir;
}

/* Creates the zip file (optionally encrypted) for the specified category. The
 * created GSF outfiles are returned since they both need to be closed when
 * closing the file. If the filename already exists, the old file is removed.
 */
static ArchiverResult
archiver_zip_create_category_output (ObArchiver  *archiver,
				     ObCategory   category,
				     GsfOutput  **output_zip,
				     GsfOutput  **output_crypt,
				     GError     **error)
{
	ObArchiverZipPriv *priv;
	gchar             *archive_path;
	GError            *gsf_error = NULL;
	GsfOutput         *output;
	GsfOutfile        *outfile_zip;

	priv = OB_ARCHIVER_ZIP (archiver)->priv;

	if (ob_archiver_is_cancelled (archiver)) {
		return ARCHIVER_RESULT_CANCELLED;
	}

	archive_path = ob_backup_info_get_category_archive_path (
		priv->backup_info, category);

	/* We don't care about the return value here, just try to remove any
	 * file that is in the way.
	 */
	gnome_vfs_unlink (archive_path);

	if (ob_archiver_is_cancelled (archiver)) {
		return ARCHIVER_RESULT_CANCELLED;
	}

	output = gsf_output_stdio_new (archive_path, &gsf_error);
	g_free (archive_path);

	if (!output) {
		/* The code is an errno here. */
		if (gsf_error->code == EROFS || gsf_error->code == EPERM ||
		    gsf_error->code == EACCES) {
			/* Read-only filesystem. */
			g_error_free (gsf_error);

			g_set_error (error, OB_ERROR, OB_ERROR_BACKUP_READONLY,
				     "Backup, memory card read-only");
		}
		else if (gsf_error->code == ENOSPC) {
			g_error_free (gsf_error);

			g_set_error (error, OB_ERROR, OB_ERROR_BACKUP_NO_SPACE,
				     "No space");
		} else {
			g_error_free (gsf_error);
			
			g_set_error (error, OB_ERROR, OB_ERROR_BACKUP_GENERIC,
				     "Could not create backup file");
		}

		return ARCHIVER_RESULT_ERROR;
        }

	*output_crypt = gsf_output_crypt_new (output, priv->password, &gsf_error);
        g_object_unref (output);
        if (!*output_crypt) {
                g_error_free (gsf_error);

		g_set_error (error, OB_ERROR, OB_ERROR_BACKUP_GENERIC,
			     "Could not create crypt file");
		
		return ARCHIVER_RESULT_ERROR;
        }

        outfile_zip = gsf_outfile_zip_new (*output_crypt, &gsf_error);
        g_object_unref (*output_crypt);
        if (!outfile_zip) {
                g_error_free (gsf_error);
		
		g_set_error (error, OB_ERROR, OB_ERROR_BACKUP_GENERIC,
			     "Could not create zip file");
		
		return ARCHIVER_RESULT_ERROR;
        }

	*output_zip = GSF_OUTPUT (outfile_zip);

	return ARCHIVER_RESULT_OK;
}

/* Closes the zip file (optionally encrypted). Handles closing the crypt outfile
 * and unreffing the zip file.
 */
static ArchiverResult
archiver_zip_close_category_output (ObArchiver *archiver,
				    GsfOutput  *output_zip,
				    GsfOutput  *output_crypt)
{
	ObArchiverZipPriv *priv;
	gboolean           could_close_zip;
	gboolean           could_close_crypt;

	priv = OB_ARCHIVER_ZIP (archiver)->priv;

	could_close_zip = gsf_output_close (output_zip);

	/* This is already unreffed (the last ref owned by output_zip). */
	could_close_crypt = gsf_output_close (output_crypt);

	g_object_unref (output_zip);

	if (could_close_zip && could_close_crypt) {
		return ARCHIVER_RESULT_OK;
	}

	return ARCHIVER_RESULT_ERROR;
}

/* Pack a category. Each category is saved in its own zip file. If there is an
 * existing file for the category, it is replaced.
 */
static ArchiverResult
archiver_zip_pack_category (ObArchiver  *archiver,
			    ObCategory   category,
			    GError     **error)
{
	ObArchiverZipPriv *priv;
	GsfOutput         *output_zip;
	GsfOutput         *output_crypt;
	GsfOutput         *root;
	GsfOutput         *location_root;
	GList             *uris;
	GList             *include_uris;
	GList             *exclude_uris;
	GSList            *entries = NULL;
	GSList            *l, *tmp_list;
	GnomeVFSFileInfo  *data_file_info;
	ArchiverResult     retval;
	ArchiverResult     close_retval;
	gboolean           could_close_root;
	int                num_files;
	GnomeVFSFileSize   size;

	priv = OB_ARCHIVER_ZIP (archiver)->priv;

	if (ob_archiver_is_cancelled (archiver)) {
		return ARCHIVER_RESULT_CANCELLED;
	}

	/* Just to make the code in the caller cleaner. */
	if (category == 0) {
		return ARCHIVER_RESULT_OK;
	}

	/* Tell the achiver which category we are currently working on
	 * so it can be presented in the progress dialiog 
	 */
	ob_archiver_set_current_category (archiver, category);

	/* Reset the specified category in the backup info. */
	ob_backup_info_reset_categories (priv->backup_info, category);

	/* The list of files to pack come from both ObCategoryFiles (files in
	 * "MyDocs") and the ObBackupLocations (other files added in the
	 * configuration). We get both lists for the specified category and
	 * merge them. For the settings category, GConf data is special-cased.
	 */

	/* Add Locations. */
	include_uris = ob_backup_locations_get_uris (priv->locations, 
						     category, FALSE);
	exclude_uris = ob_backup_locations_get_uris (priv->locations, 
						     category, TRUE);
  	entries = ob_vfs_utils_get_file_entries_filtered (include_uris,   
  							  exclude_uris);  
	g_list_foreach (include_uris, (GFunc) g_free, NULL);
	g_list_free (include_uris);  

	g_list_foreach (exclude_uris, (GFunc) g_free, NULL);
 	g_list_free (exclude_uris); 

	/* Add the files from "MyDocs". */
	tmp_list = ob_category_files_get_file_entries (priv->category_files,
						       category);
	entries = g_slist_concat (entries, tmp_list);

	/* Add special-case GConf data if we have any. */
	if (category == OB_CATEGORY_SETTINGS && priv->gconf_dir) {
#ifdef SHUTDOWN_GCONF
		/* Shutdown gconfd, this should really be done by the
		 * app-killer, but that didn't get implemented so we work around
		 * here.
		 */
		shutdown_gconfd ();
#endif
		
		uris = g_list_prepend (NULL, (gchar*) gnome_vfs_uri_get_path (priv->gconf_dir));
		tmp_list = ob_vfs_utils_get_file_entries_filtered (uris, NULL);
		entries = g_slist_concat (entries, tmp_list);
		g_list_free (uris);
	}

	if (g_slist_length (entries) == 0) {
		ob_file_entry_free_list (entries);
		return ARCHIVER_RESULT_OK;
	}

	retval = archiver_zip_create_category_output (archiver,
						      category,
						      &output_zip,
						      &output_crypt,
						      error);
	if (retval != ARCHIVER_RESULT_OK) {
		ob_file_entry_free_list (entries);
		return retval;
	}

	/* Create a root directory inside the zip file. The data for the
	 * specified category is saved below this root dir. The reason for not
	 * storing the data directly in the root directory is to make it
	 * possible to extend the format if necessary.
	 */
	root = archiver_pack_subdir (archiver, output_zip, "Root", error);
	if (!root) {
		/* We don't care about the return value here, we'll just remove
		 * the file if there is an error.
		 */
		archiver_zip_close_category_output (archiver,
						    output_zip,
						    output_crypt);

		ob_file_entry_free_list (entries);
		return ARCHIVER_RESULT_ERROR;
	}

	retval = ARCHIVER_RESULT_OK;

	/* Record the data amount before packing this category, so that we can
	 * see how much the category adds.
	 */
	num_files = ob_archiver_get_processed_files (archiver);
	size = ob_archiver_get_processed_size (archiver);

	/* Go through all the files and add them to the archive. */
	for (l = entries; l; l = l->next) {
		GnomeVFSURI *data_uri;
		GnomeVFSURI *parent_uri;
		ObFileEntry *entry;
		gchar       *without_slash;

		if (ob_archiver_is_cancelled (archiver)) {
			retval = ARCHIVER_RESULT_CANCELLED;
			break;
		}

		entry = l->data;

		data_uri = entry->uri;
		data_file_info = entry->file_info;

		/* We put the data in the parent directory of the location, so
		 * it ends up relative the root inside the archive, otherwise we
		 * wouldn't know where to restore it. There is always a parent
		 * since all files have at least /.
		 */
		parent_uri = gnome_vfs_uri_get_parent (data_uri);

		/* Remove any leading slash since the path is relative the root
		 * directory. This is to avoid double slashes in the zip file.
		 */
		without_slash = get_path_without_leading_slash_from_uri (parent_uri);

		location_root = archiver_pack_subdir (archiver,
						      root,
						      without_slash,
						      error);

		gnome_vfs_uri_unref (parent_uri);
		g_free (without_slash);

		if (!location_root) {
			g_set_error (error,
				     OB_ERROR, OB_ERROR_BACKUP_GENERIC,
				     "Could not create folder");
			ob_log_warning ("Backup, could not create folder");

			retval = ARCHIVER_RESULT_ERROR;
			break;
		}

		if (data_file_info->type == GNOME_VFS_FILE_TYPE_REGULAR) {
			retval = archiver_pack_file (archiver,
						     location_root,
						     data_uri,
						     data_file_info,
						     error);
		} else {
			/* We should not get directories in the file lists, and
			 * we don't want sockets or symlinks etc. Just skip
			 * those.
			 */
			d(g_printerr ("Skipping non-regular file.\n"));
		}

		if (!gsf_output_close (location_root)) {
			/* Fatal error, but don't set an error if we're
			 * cancelled or already have an error condition.
			 */
			if (retval == ARCHIVER_RESULT_OK) {
				retval = ARCHIVER_RESULT_ERROR;

				g_set_error (
					error,
					OB_ERROR,
					OB_ERROR_BACKUP_GENERIC,
					"Could not close zip dir");
				ob_log_warning ("Backup, could not close zip dir");
			}
		}
		g_object_unref (location_root);

		if (retval != ARCHIVER_RESULT_OK) {
			break;
		}
	}

	ob_file_entry_free_list (entries);

	/* Done, close the root dir. */
	could_close_root = gsf_output_close (root);
	g_object_unref (root);

	switch (retval) {
	case ARCHIVER_RESULT_OK:
		if (!could_close_root) {
			g_set_error (error, OB_ERROR, OB_ERROR_BACKUP_GENERIC,
				     "Could not close archive");
			ob_log_warning ("Backup, could not close archive");
			retval = ARCHIVER_RESULT_ERROR;
		}
		break;

	default:
		/* Just pass it on to the caller. */
		break;
	}

	close_retval = archiver_zip_close_category_output (archiver,
							   output_zip,
							   output_crypt);

	if (close_retval != ARCHIVER_RESULT_OK) {
		retval = close_retval;
	}

	if (retval == ARCHIVER_RESULT_OK) {
		/* Get the data amount added for this category. */
		num_files = ob_archiver_get_processed_files (archiver) - num_files;
		size = ob_archiver_get_processed_size (archiver) - size;

		ob_backup_info_set_data_amount (priv->backup_info,
						category,
						size, num_files);

		priv->written_categories |= category;
	} else {
		gchar *archive_path;

		/* Remove the broken category archive file. */
		archive_path = ob_backup_info_get_category_archive_path (
			priv->backup_info, category);
		gnome_vfs_unlink (archive_path);

		/* Reset the processed files/size since we didn't end up keeping
		 * this category.
		 */
		ob_archiver_set_processed_files (archiver, num_files);
		ob_archiver_set_processed_size (archiver, size);

		/* This category was broken, so make sure it's not saved if it
		 * was in the previous backup.
		 */
		priv->old_categories &= ~category;

		g_free (archive_path);
	}

	return retval;
}

/* Go through all categories and pack the ones that are selected. */
static ArchiverResult
archiver_zip_pack_categories (ObArchiver  *archiver,
			      GError     **error)
{
	ObArchiverZipPriv *priv;
	ArchiverResult     retval;

	priv = OB_ARCHIVER_ZIP (archiver)->priv;

	retval = archiver_zip_pack_category (archiver,
					     priv->categories & OB_CATEGORY_EMAILS,
					     error);
	if (retval != ARCHIVER_RESULT_OK) {
		return retval;
	}

	retval = archiver_zip_pack_category (archiver,
					     priv->categories & OB_CATEGORY_CONTACTS,
					     error);
	if (retval != ARCHIVER_RESULT_OK) {
		return retval;
	}

	retval = archiver_zip_pack_category (archiver,
					     priv->categories & OB_CATEGORY_DOCUMENTS,
					     error);
	if (retval != ARCHIVER_RESULT_OK) {
		return retval;
	}

	retval = archiver_zip_pack_category (archiver,
					     priv->categories & OB_CATEGORY_MEDIA,
					     error);
	if (retval != ARCHIVER_RESULT_OK) {
		return retval;
	}

	retval = archiver_zip_pack_category (archiver,
					     priv->categories & OB_CATEGORY_BOOKMARKS,
					     error);
	if (retval != ARCHIVER_RESULT_OK) {
		return retval;
	}

	retval = archiver_zip_pack_category (archiver,
					     priv->categories & OB_CATEGORY_SETTINGS,
					     error);
	if (retval != ARCHIVER_RESULT_OK) {
		return retval;
	}

	/* The category "other" is only saved when all is checked. */
	if (priv->categories == OB_CATEGORY_ALL) {
		retval = archiver_zip_pack_category (archiver,
						     OB_CATEGORY_OTHER,
						     error);
		if (retval != ARCHIVER_RESULT_OK) {
			return retval;
		}
	}

	return retval;
}

/* Packs a number of categories into separate zip files inside the backup
 * directory on the target location. If a certain category is already existing,
 * it is overwritten.
 */
static ArchiverResult
archiver_zip_pack (ObArchiver *archiver)
{
	ObArchiverZipPriv *priv;
        GError            *error = NULL;
	GnomeVFSResult     result;
	ArchiverResult     retval;
	int                i;

	priv = OB_ARCHIVER_ZIP (archiver)->priv;

	/* Sleep a bit to make sure that apps have acted on the backup start
	 * signal.  We do it in chunks of half a second so that it's still
	 * possible to cancel in a timely fashion, for a total of 3 seconds.
	 */
	for (i = 0; i < 6; i++) {
		g_usleep (0.5 * G_USEC_PER_SEC);
		if (ob_archiver_is_cancelled (archiver)) {
			break;
		}
	}

	/* Make sure that the backup directory exists. */
	result = ob_backup_info_create_dir (priv->backup_info);
	if (result == GNOME_VFS_ERROR_READ_ONLY_FILE_SYSTEM ||
	    result == GNOME_VFS_ERROR_NOT_PERMITTED ||
	    result == GNOME_VFS_ERROR_ACCESS_DENIED) {
		/* Read-only filesystem. */
		error = g_error_new (OB_ERROR, OB_ERROR_BACKUP_READONLY,
				     "Memory card read only");
	}
	else if (result == GNOME_VFS_ERROR_NO_SPACE) {
		error = g_error_new (OB_ERROR, OB_ERROR_BACKUP_NO_SPACE,
				     "No space");
	}
	else if (result != GNOME_VFS_OK) {
		error = g_error_new (OB_ERROR, OB_ERROR_BACKUP_GENERIC,
				     "Could not create backup directory");
	}

	if (error) {
		ob_archiver_push_error_event (archiver, error);
		g_error_free (error);

		return ARCHIVER_RESULT_ERROR;
	}

	/* Go through the categories. For each category do:
	 *
	 * 1. Remove the corresponding zip file if one exists.
	 * 2. Create a new file.
	 * 3. Pack data into it.
	 * 4. Close the file.
	 */

	retval = archiver_zip_pack_categories (archiver, &error);

	if (retval == ARCHIVER_RESULT_ERROR) {
		/* When there is an error, we try to remove the backup since the
		 * error means that we have half-written category files or other
		 * fatal errors.
		 *
		 * We don't care for the return value since we can't do anything
		 * if this fails, and besides, we're already in error mode.
		 */
		ob_backup_info_remove_dir (priv->backup_info);
	} else {
		/* For a successfull or cancelled backup, we try to save the
		 * categories that were written completely, and also the ones
		 * that existed before starting this backup that we didn't
		 * touch. This way updating a backup works when cancelling.
		 */
		ob_backup_info_set_categories (priv->backup_info,
					       priv->written_categories |
					       priv->old_categories);

		if (priv->written_categories | priv->old_categories) {
			ob_backup_info_write_metadata (priv->backup_info);
		} else {
			/* We remove the metadata if there is no data left. This
			 * happens when we only have one category saved, and
			 * cancel it.
			 */
			ob_backup_info_remove_dir (priv->backup_info);
		}
	}

	/* Update the statistics to the real values now that we have them after
	 * a possible cancellation.
	 */
	ob_archiver_set_total_size (
		archiver,
		ob_backup_info_get_size_for_categories (priv->backup_info,
							priv->written_categories |
							priv->old_categories));
	ob_archiver_set_total_files (
		archiver,
		ob_backup_info_get_num_files_for_categories (priv->backup_info,
							     priv->written_categories |
							     priv->old_categories));

	switch (retval) {
	case ARCHIVER_RESULT_OK:
		ob_archiver_push_finished_event (archiver);
		break;

	case ARCHIVER_RESULT_CANCELLED:
		/* The category that was cancelled has been removed at this
		 * stage. It might be a good idea to check if the media is still
		 * mounted and if so remove any .gsf* files, or add an API to
		 * gsf that let's us get the temp filename.
	 	 */
		ob_backend_confirm_cancel (ob_archiver_get_backend (archiver));
		break;

	case ARCHIVER_RESULT_CONFLICT_CANCELLED:
		/* This should never happen when packing, only unpacking. */
		g_assert_not_reached ();
		break;

	case ARCHIVER_RESULT_ERROR:
		/* The category that was cancelled has been removed at this
		 * stage.
		 */
		ob_archiver_push_error_event (archiver, error);
		g_error_free (error);
		break;
	}

	return retval;
}


/*
 * Unpacking
 */

static TempFileEntry *
temp_file_entry_new (GnomeVFSURI *temp_uri,
		     GnomeVFSURI *target_uri)
{
	TempFileEntry *entry;

	entry = g_new0 (TempFileEntry, 1);

	entry->temp_uri = gnome_vfs_uri_ref (temp_uri);
	entry->target_uri = gnome_vfs_uri_ref (target_uri);

	return entry;
}

static void
temp_file_entry_free (TempFileEntry *entry)
{
	gnome_vfs_uri_unref (entry->temp_uri);
	gnome_vfs_uri_unref (entry->target_uri);

	g_free (entry);
}

static void
temp_file_entry_list_free (GSList *list)
{
	g_slist_foreach (list, (GFunc) temp_file_entry_free, NULL);
	g_slist_free (list);
}


/*
 * Conflict handling:
 *
 * When an errors occurs in this function, for example when getting the
 * timestamp for a file, we return OB_CONFLICT_RESPONSE_NO. This means that if
 * there is an error, it will not result in any loss of data.
 */

/* Handles a conflict where a file is in the way of a file. Pushes a conflict
 * event to the backend, which propagates an answer from the user.
 */
static ObConflictResponse
archiver_handle_conflict_file_file (ObArchiver  *archiver,
				    GnomeVFSURI *target_uri,
				    time_t       existing_timestamp,
				    time_t 	 backup_timestamp)
{
	ObConflictResponse response;

	/* If "yes to all" has been previously selected, just replace without
	 * asking.
	 */
	if (ob_archiver_get_replace_all (archiver)) {
		/*g_printerr ("Replacing without asking.\n");*/
		return OB_CONFLICT_RESPONSE_YES;
	}

	/* Push an event to get a response from the user. */
	ob_archiver_push_conflict_event (archiver,
					 OB_CONFLICT_TYPE_FILE_FILE,
					 target_uri,
					 existing_timestamp,
					 backup_timestamp);

	response = ob_backend_wait_for_conflict_response (
		ob_archiver_get_backend (archiver));

        /* remember yes-all */
	if (response == OB_CONFLICT_RESPONSE_YES_ALL)
		ob_archiver_set_replace_all (archiver, TRUE);

	return response;
}

/* Handles a conflict where a dir is in the way of a file. Pushes a conflict
 * event to the backend, which propagates an answer from the user.
 */
static ObConflictResponse
archiver_handle_conflict_dir_file (ObArchiver  *archiver,
				   GnomeVFSURI *target_uri,
				   time_t       existing_timestamp,
				   time_t       backup_timestamp)
{
	ObConflictResponse response;

	/* Push an event to get a response from the user. */
	ob_archiver_push_conflict_event (archiver,
					 OB_CONFLICT_TYPE_DIR_FILE,
					 target_uri,
					 existing_timestamp,
					 backup_timestamp);

	response = ob_backend_wait_for_conflict_response (
		ob_archiver_get_backend (archiver));

	return response;
}

/* Handles a conflict where a file is in the way of a dir. Pushes a conflict
 * event to the backend, which propagates an answer from the user.
 */
static ObConflictResponse
archiver_handle_conflict_file_dir (ObArchiver  *archiver,
				   GsfInput    *input_dir,
				   GnomeVFSURI *uri_file)
{
	ObConflictResponse response;

	/* We can't get a timestamp for a directory in the archive, so we don't
	 * send those in the event here. We also don't use timestamps, since
	 * file/dir conflicts always must be confirmed by the user. We therefore
	 * don't use the "yes to all" flag. */

	/* Push an event to get a response from the user. */
	ob_archiver_push_conflict_event (archiver,
					 OB_CONFLICT_TYPE_FILE_DIR,
					 uri_file,
					 0, 0);

	response = ob_backend_wait_for_conflict_response (
		ob_archiver_get_backend (archiver));

	return response;
}

static gint 
archiver_unpack_find_foreach (const gchar *uri_str,
			      const gchar *to_find)
{
	gboolean found;
	
	found = g_pattern_match_simple (uri_str, to_find);
	d(g_printerr ("  %s entry in excluded list: %s\n", 
		      found ? "Matched" : "Does not match", to_find));

	if (found) {
		return 0;
	}

	return 1;
}

static gboolean
archiver_unpack_check_space (ObArchiver  *archiver,
			     GnomeVFSURI *uri,
			     GsfInput    *input)
{
	GnomeVFSFileSize  free_space;
	GnomeVFSResult    result;
	gsize             num_bytes;
	
	result = gnome_vfs_get_volume_free_space (uri, &free_space);
	if (result != GNOME_VFS_OK) {
		ob_log_warning ("Could't get free space: %s\n",
				gnome_vfs_result_to_string (result));
		return TRUE;
	}

        num_bytes = gsf_input_size (input);
	if (free_space - num_bytes < RESTORE_SPACE_MARGIN) {
		return FALSE;
	}

	return TRUE;
}


/* Unpack a file from an GsfInput into a handle that is already opened.  this
 * function pushes out progress size event, but not errors.  leaves the file in
 * place on errors.
 */
static GnomeVFSResult
archiver_vfs_unpack_file_write (ObArchiver      *archiver,
                                GsfInput        *input,
                                GnomeVFSHandle  *handle,
                                GError         **error)
{
        GnomeVFSFileSize total_bytes_written = 0;
        gsize            num_bytes;

	if (ob_archiver_is_cancelled (archiver)) {
		return GNOME_VFS_ERROR_INTERRUPTED;
	}

        num_bytes = gsf_input_size (input);
        while (num_bytes > 0) {
                gsize           bytes_per_read;
                const guint8   *buf;
                GnomeVFSResult  result;

		bytes_per_read = MIN (num_bytes, BUF_SIZE);
		buf = gsf_input_read (input, bytes_per_read, NULL);

		if (!buf) {
			/* buf should always be != NULL because bytes_per_read > 0. */
			ob_log_error ("Restore IO error from gsf.");
			return GNOME_VFS_ERROR_IO;
		}

                if (ob_archiver_is_cancelled (archiver)) {
                        return GNOME_VFS_ERROR_INTERRUPTED;
		}

		result = ob_vfs_utils_write (handle,
					     buf,
					     bytes_per_read);
                if (result != GNOME_VFS_OK) {
                        return result;
		}
                num_bytes -= bytes_per_read;

                if (ob_archiver_is_cancelled (archiver)) {
                        return GNOME_VFS_ERROR_INTERRUPTED;
		}

                /* Emit progress for each chunk. */
                ob_archiver_add_processed_size (archiver, bytes_per_read);
                total_bytes_written += bytes_per_read;
                if (total_bytes_written % PROGRESS_CHUNK == 0) {
                        ob_archiver_push_progress_event (archiver);
                }

#if 0
		/* Simulate an error. */
                if (total_bytes_written > 0) {
                        g_set_error (error, OB_ERROR, 0, 
				     "Simulated error condition");
                        return GNOME_VFS_ERROR_IO;
                }
#endif
                /* stop thread during restore (simulates device crash) */
		/* kill (getpid(), 19); */
        }

        return GNOME_VFS_OK;
}

static gboolean
archiver_unpack_sudo_restore (ObArchiver  *archiver,
			      GnomeVFSURI *temp_uri,
			      GnomeVFSURI *target_uri)
{
	const char *sudo_cmd;
	char       *cmdline;
	int         exit_status;
	gboolean    script_retval;

	d(g_printerr ("Sudo Restoring\n"));

#ifdef USE_SUDO
	sudo_cmd = "sudo ";
#else
	sudo_cmd = "";
#endif

	cmdline = g_strdup_printf ("%s%s \"%s\" \"%s\"",
				   sudo_cmd,
				   RESTORE_SPECIAL_PROGRAM,
				   gnome_vfs_uri_get_path (temp_uri),
				   gnome_vfs_uri_get_path (target_uri));

	script_retval = g_spawn_command_line_sync (cmdline,
						   NULL, NULL,
						   &exit_status,
						   NULL);

	g_free (cmdline);

	if (!script_retval || exit_status != 0) {
		gnome_vfs_unlink_from_uri (temp_uri);

		ob_log_warning ("Restore, could not restore special file");
		return FALSE;
	}

	return TRUE;
}

static void
archiver_unpack_move_settings (ObArchiver *archiver,
			       GSList     *list)
{
	ObArchiverZipPriv *priv;
	GSList            *l;

	priv = OB_ARCHIVER_ZIP (archiver)->priv;
	
	d(g_printerr ("Moving into place...\n"));

	for (l = list; l; l = l->next) {
		TempFileEntry *entry;

		entry = l->data;

		d(g_printerr ("  %s -> %s\n",
			      gnome_vfs_uri_get_path (entry->temp_uri),
			      gnome_vfs_uri_get_path (entry->target_uri)));
		
		/* Move the temp file into its final target destination. Note:
		 * Some files require a special hack using sudo to be restored,
		 * since only root can touch some directories.
		 */
		if (uri_is_equal_to_path (entry->target_uri, SPECIAL_LOCALE_PATH) ||
		    uri_is_equal_to_path (entry->target_uri, SPECIAL_BTNAME_PATH) ||
		    uri_is_equal_to_path (entry->target_uri, SPECIAL_SOURCES_PATH) ||
		    uri_is_equal_to_path (entry->target_uri, SPECIAL_LOCALTIME_PATH) ||
		    uri_is_under_dir (entry->target_uri, SPECIAL_BLUETOOTH_PATH)) {
			if (archiver_unpack_sudo_restore (archiver,
							  entry->temp_uri,
							  entry->target_uri)) {
				ob_restore_transform_add_file (
					priv->restore_transform,
					OB_CATEGORY_SETTINGS,
					gnome_vfs_uri_get_path (entry->target_uri));
			}
		} else {
			if (gnome_vfs_move_uri (entry->temp_uri, entry->target_uri, TRUE) != GNOME_VFS_OK) {
				/* Clean up. */
				gnome_vfs_unlink_from_uri (entry->temp_uri);
			} else {
				ob_restore_transform_add_file (
					priv->restore_transform,
					OB_CATEGORY_SETTINGS,
					gnome_vfs_uri_get_path (entry->target_uri));
			}
		}
	}

	d(g_printerr ("Finished moving files\n"));
}

/* Special-case settings. We unpack all files into the temporary directory
 * first, and during this we are still cancellable. Then after unpacking all
 * settings, they are moved to their final target, and during this stage we're
 * not cancellable. This is meant to help avoid getting half of the settings
 * restored, which could cause trouble if settings depend on each other.
 *
 * Note that this function somewhat duplicate archiver_unpack_file(), but having
 * it separate makes the code a lot simpler.
 */
static ArchiverResult
archiver_unpack_file_settings (ObArchiver    *archiver,
			       GsfInput      *input,
			       GnomeVFSURI   *target_uri,
			       GSList       **temp_file_list,
			       GError       **error)
{
	ObArchiverZipPriv *priv;
	mode_t             mode;
	time_t             existing_timestamp, backup_timestamp;
	GnomeVFSURI       *parent_uri, *temp_uri = NULL;
	GnomeVFSHandle    *handle = NULL;
	GnomeVFSResult     result;
	const gchar       *tmp;

	priv = OB_ARCHIVER_ZIP (archiver)->priv;

	/* Get file metadata. */
	mode = gsf_infile_zip_get_mode (GSF_INFILE_ZIP (input));
	backup_timestamp = gsf_infile_zip_get_time (GSF_INFILE_ZIP (input));

	/* Guard against broken timestamps. */
	backup_timestamp = MAX (0, backup_timestamp);

	/* for performance reasons, assume file contents didn't change if the
	 * timestamps did not change, and avoid unpacking.
	 */
	result = ob_vfs_utils_uri_get_timestamp (target_uri, &existing_timestamp);
	if (result == GNOME_VFS_OK) {
		time_t diff;

		diff = existing_timestamp - backup_timestamp;
		if (labs (diff) <= 1) { /* zip time format has only 2 seconds accuracy */
			/* Consider the file processed. */
			ob_archiver_add_processed_size (archiver, gsf_input_size (input));
			ob_archiver_add_processed_files (archiver, 1);
			ob_archiver_push_progress_event (archiver);
			return ARCHIVER_RESULT_OK;
		}
	}

	/* Restore file in temporary location. */
	tmp = g_getenv (TMP_ENV);
	if (tmp) {
		parent_uri = gnome_vfs_uri_new (tmp);
	} else {
		parent_uri = gnome_vfs_uri_new (TMP_URI);
	}

	result = ob_vfs_utils_create_temp_file (&handle, parent_uri, mode, &temp_uri);
	gnome_vfs_uri_unref (parent_uri);
	if (result == GNOME_VFS_OK) {
		if (archiver_unpack_check_space (archiver, temp_uri, input)) {
			result = archiver_vfs_unpack_file_write (archiver,
								 input,
								 handle,
								 error);
		} else {
			result = GNOME_VFS_ERROR_NO_SPACE;
		}
		
		if (result != GNOME_VFS_OK) {
			gnome_vfs_close (handle);
		} else { /* Catch all errors */
			result = gnome_vfs_close (handle);
		}

		/* Set timestamp and mode. */
		if (result == GNOME_VFS_OK) {
			GnomeVFSResult tresult;

			tresult = ob_vfs_utils_uri_set_timestamp_and_mode (temp_uri,
									   backup_timestamp,
									   mode);
			if (tresult != GNOME_VFS_OK) {
				/* Non-fatal. */
				ob_log_warning ("Failed to set timestamp and permissions "
						"during restore.");
			}
		}
	}

	if (ob_archiver_is_cancelled (archiver)) {
		gnome_vfs_uri_unref (temp_uri);
		return ARCHIVER_RESULT_CANCELLED;
	}

	/* Add the file to the list of temp files to move into place later. */
	if (result == GNOME_VFS_OK) {
		TempFileEntry *entry;

		entry = temp_file_entry_new (temp_uri, target_uri);
		*temp_file_list = g_slist_prepend (*temp_file_list, entry);

		ob_archiver_add_processed_files (archiver, 1);
		ob_archiver_push_progress_event (archiver);
	}

	gnome_vfs_uri_unref (temp_uri);

	if (result == GNOME_VFS_OK) {
		return ARCHIVER_RESULT_OK;
	}

	/* Handle errors. */
	if (result == GNOME_VFS_ERROR_NO_SPACE) {
		g_set_error (error, OB_ERROR, OB_ERROR_RESTORE_NO_SPACE,
			     "Restore, not enough space");
	} else {
		g_set_error (error, OB_ERROR, OB_ERROR_RESTORE_GENERIC,
			     "Failed to restore setting %s: %s",
			     gnome_vfs_uri_get_path (target_uri),
			     gnome_vfs_result_to_string (result));
	}

	return ARCHIVER_RESULT_ERROR;
}

static ArchiverResult
archiver_unpack_file (ObArchiver    *archiver,
		      GsfInput      *input,
		      GnomeVFSURI   *target_uri,
		      ConflictMode   conflict_mode,
		      ObCategory     category,
		      GError       **error)
{
	ObArchiverZipPriv *priv;
	mode_t             mode;
	time_t             existing_timestamp, backup_timestamp;
	GnomeVFSURI       *parent_uri, *temp_uri;
	GnomeVFSHandle    *handle = NULL;
	GnomeVFSResult     result;
	GnomeVFSFileType   conflict_file_type = GNOME_VFS_FILE_TYPE_UNKNOWN;
	gboolean           resolve_conflicts = FALSE;
	gboolean           conflict_cancelled = FALSE, conflict_ignored = FALSE;
	const char        *tmp;

	priv = OB_ARCHIVER_ZIP (archiver)->priv;

	/* Get file metadata. */
	mode = gsf_infile_zip_get_mode (GSF_INFILE_ZIP (input));
	backup_timestamp = gsf_infile_zip_get_time (GSF_INFILE_ZIP (input));

	/* Guard against broken timestamps. */
	backup_timestamp = MAX (0, backup_timestamp);

	/* For performance reasons, assume file contents didn't change if the
	 * timestamps did not change, and avoid unpacking.
	 */
	result = ob_vfs_utils_uri_get_timestamp (target_uri, &existing_timestamp);

	if (result == GNOME_VFS_OK) {
		time_t diff;

		diff = existing_timestamp - backup_timestamp;
		if (labs (diff) <= 1) { /* zip time format has only 2 seconds accuracy */
			/* Consider the file processed. */
			ob_archiver_add_processed_size (archiver, gsf_input_size (input));
			ob_archiver_add_processed_files (archiver, 1);
			ob_archiver_push_progress_event (archiver);

			return ARCHIVER_RESULT_OK;
		}
	}

	/* Restore file in temporary location. */
	tmp = g_getenv (TMP_ENV);
	if (tmp) {
		parent_uri = gnome_vfs_uri_new (tmp);
	} else {
		parent_uri = gnome_vfs_uri_new (TMP_URI);
	}

	temp_uri = NULL;
	result = ob_vfs_utils_create_temp_file (&handle, parent_uri, mode, &temp_uri);

	gnome_vfs_uri_unref (parent_uri);
	if (result == GNOME_VFS_OK) {
		if (archiver_unpack_check_space (archiver, temp_uri, input)) {
			result = archiver_vfs_unpack_file_write (archiver, input, handle, error);
		} else {
			result = GNOME_VFS_ERROR_NO_SPACE;
		}
		
		if (result != GNOME_VFS_OK) {
			gnome_vfs_close (handle);
		} else { /* Catch all errors */
			result = gnome_vfs_close (handle);
		}

		/* Update timestamp and mode. */
		if (result == GNOME_VFS_OK) {
			GnomeVFSResult tresult;

			tresult = ob_vfs_utils_uri_set_timestamp_and_mode (temp_uri,
									   backup_timestamp,
									   mode);
			if (tresult != GNOME_VFS_OK) {
				/* Non-fatal. */
				ob_log_warning ("Failed to set timestamp and permissions "
						"during restore.");
			}
		}
	}

	/* Move file into target place. */
	if (result == GNOME_VFS_OK) {
		/* Note: We have to special-case bookmarks from older versions
		 * since they are no longer user-writable.
		 */
		if (category == OB_CATEGORY_BOOKMARKS &&
		    uri_is_under_dir (target_uri, SPECIAL_BOOKMARKS_PATH)) {
			ob_log_info ("sudo-restoring bookmarks");
			if (archiver_unpack_sudo_restore (archiver,
							  temp_uri,
							  target_uri)) {
				ob_restore_transform_add_file (
					priv->restore_transform,
					category,
					gnome_vfs_uri_get_path (target_uri));
			}
		} else {
			result = gnome_vfs_move_uri (temp_uri, target_uri, FALSE);
			resolve_conflicts = (result == GNOME_VFS_ERROR_FILE_EXISTS ||
					     result == GNOME_VFS_ERROR_IS_DIRECTORY);
			
			if (result == GNOME_VFS_OK) {
				ob_restore_transform_add_file (
					priv->restore_transform,
					category,
					gnome_vfs_uri_get_path (target_uri));
			}
		}
	}

	/* Find out details about target conflict. */
	if (resolve_conflicts) {
		GnomeVFSResult cresult;

		cresult = ob_vfs_utils_uri_get_file_type (target_uri, &conflict_file_type);
		if (cresult != GNOME_VFS_OK) {
			conflict_file_type = GNOME_VFS_FILE_TYPE_UNKNOWN;
		}
	}

	/* Resolve file/file conflicts. */
	if (conflict_file_type == GNOME_VFS_FILE_TYPE_REGULAR) {
		const char         *path;
		ObConflictResponse  response = OB_CONFLICT_RESPONSE_YES;

		path = gnome_vfs_uri_get_path (target_uri);

 		/* Ask the user if neccessary. */
		if (existing_timestamp > backup_timestamp &&
		    conflict_mode != CONFLICT_MODE_DONT_ASK &&
		    !ob_backup_locations_is_auto_restore (priv->locations, path)) {
			response = archiver_handle_conflict_file_file (archiver,
								       target_uri,
								       existing_timestamp,
								       backup_timestamp);
		}

		/* Handle overwriting. */
		if (response == OB_CONFLICT_RESPONSE_YES ||
		    response == OB_CONFLICT_RESPONSE_YES_ALL) {
		        result = gnome_vfs_move_uri (temp_uri, target_uri, TRUE);

			if (result == GNOME_VFS_OK) {
				ob_restore_transform_add_file (
					priv->restore_transform,
					category,
					gnome_vfs_uri_get_path (target_uri));
			}
		}
		else if (response == OB_CONFLICT_RESPONSE_NO) {
			conflict_ignored = TRUE;
		}

		conflict_cancelled = response == OB_CONFLICT_RESPONSE_CANCEL;
	}

	/* Resolve file/directory conflicts. */
	if (conflict_file_type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
		ObConflictResponse response = OB_CONFLICT_RESPONSE_NO;
                /* Ask the user if neccessary. */
		if (conflict_mode != CONFLICT_MODE_DONT_ASK)
			response = archiver_handle_conflict_dir_file (archiver,
								      target_uri,
								      existing_timestamp,
								      backup_timestamp);

		/* Handle overwriting. */
		if (response == OB_CONFLICT_RESPONSE_YES) {
			GnomeVFSURI *final_target_uri;

			result = ob_vfs_utils_move_to_unique_uri (temp_uri, target_uri,
								  &final_target_uri);
			if (result == GNOME_VFS_OK) {
				ob_restore_transform_add_file (
					priv->restore_transform,
					category,
					gnome_vfs_uri_get_path (final_target_uri));
				gnome_vfs_uri_unref (final_target_uri);
			}
		}
		else if (response == OB_CONFLICT_RESPONSE_NO) {
			conflict_ignored = TRUE;
		}
		conflict_cancelled = response == OB_CONFLICT_RESPONSE_CANCEL;
	}

	/* Signal progress, cleanup. */
	if (!conflict_cancelled) {
		ob_archiver_add_processed_files (archiver, 1);
		ob_archiver_push_progress_event (archiver);
	}

	if (temp_uri) {
		gnome_vfs_unlink_from_uri (temp_uri);
		gnome_vfs_uri_unref (temp_uri);
	}

	/* Handle cancellation (skip error messages). */
	if (conflict_cancelled) {
		return ARCHIVER_RESULT_CONFLICT_CANCELLED;
	}
	else if (ob_archiver_is_cancelled (archiver)) {
		return ARCHIVER_RESULT_CANCELLED;
	}

	/* Report success. */
	if (conflict_ignored || result == GNOME_VFS_OK) {
		return ARCHIVER_RESULT_OK;
	}

	/* Handle errors. */
	if (result == GNOME_VFS_ERROR_NO_SPACE) {
		g_set_error (error, OB_ERROR, OB_ERROR_RESTORE_NO_SPACE,
			     "Restore, not enough space");
	} else {
		g_set_error (error, OB_ERROR, OB_ERROR_RESTORE_GENERIC,
			     "Failed to restore %s: %s",
			     gnome_vfs_uri_get_path (target_uri),
			     gnome_vfs_result_to_string (result));
	}

	return ARCHIVER_RESULT_ERROR;
}

static ArchiverResult
archiver_unpack_dir (ObArchiver    *archiver,
		     GsfInput      *input,
		     GnomeVFSURI   *uri,
		     mode_t         mode,
		     ConflictMode   conflict_mode,
		     GnomeVFSURI  **renamed_uri,
		     gboolean      *stop_traversing,
		     GError       **error)
{
	GnomeVFSResult     result;
	gboolean           rename_dir;
	ObConflictResponse response;

	*renamed_uri = NULL;
	*stop_traversing = FALSE;

	rename_dir = FALSE;

        if (ob_archiver_is_cancelled (archiver)) {
                return ARCHIVER_RESULT_CANCELLED;
	}

	/* First try creating the target directory to see if there is a
	 * conflict.
	 */
	result = gnome_vfs_make_directory_for_uri (uri, mode);
	if (result == GNOME_VFS_OK) {
		/* No conflict, done. */

		/* The umask will mess up the mode if we don't explicitly set
		 * it.
		 */
		result = ob_vfs_utils_uri_set_mode (uri, mode);
		if (result == GNOME_VFS_OK) {
			return ARCHIVER_RESULT_OK;
		} else {
			g_set_error (error, OB_ERROR, OB_ERROR_RESTORE_GENERIC,
				     "Restore, could not set mode for created dir");
			
			return ARCHIVER_RESULT_ERROR;
		}
	}
	else if (result == GNOME_VFS_ERROR_FILE_EXISTS ||
		 result == GNOME_VFS_ERROR_IS_DIRECTORY) {
		/* If the directory already exists, just go on. If it has been
		 * replaced by a file, handle the conflict and if the user wants
		 * to, create a new directory with a unique name.
		 */
		GnomeVFSResult   tmp_result;
		GnomeVFSFileType type;

		tmp_result = ob_vfs_utils_uri_get_file_type (uri, &type);
		if (tmp_result != GNOME_VFS_OK) {
			g_set_error (error, OB_ERROR, OB_ERROR_RESTORE_GENERIC,
				     "Restore, could not get file type for conflict");
			
			return ARCHIVER_RESULT_ERROR;
		}

		if (type == GNOME_VFS_FILE_TYPE_DIRECTORY) {
			/* If there is already a directory in place, just
			 * continue.
			 */
			return ARCHIVER_RESULT_OK;
		} else {
			/* There is a conflict where a file or symlink in the
			 * way of the directory. Note that we don't handle this
			 * for "don't ask" mode, since there is no use in
			 * unpacking such data into a renamed folder.
			 */
			if (conflict_mode == CONFLICT_MODE_DONT_ASK) {
				*stop_traversing = TRUE;
				return ARCHIVER_RESULT_OK;
			}

			response = archiver_handle_conflict_file_dir (archiver,
								      input,
								      uri);

			switch (response) {
			case OB_CONFLICT_RESPONSE_CANCEL:
				return ARCHIVER_RESULT_CONFLICT_CANCELLED;

			case OB_CONFLICT_RESPONSE_YES_ALL:
				/* This is not a valid response for file/dir
				 * conflicts, so it's a bug if we get it.
				 */
				ob_log_warning ("Should not get YES_ALL response to "
						"file/dir conflict.");
				return OB_CONFLICT_RESPONSE_CANCEL;

			case OB_CONFLICT_RESPONSE_YES:
				/* Yes, rename the directory. */
				rename_dir = TRUE;
				break;

			case OB_CONFLICT_RESPONSE_NO:
				/* No, don't unpack the file. */
				rename_dir = FALSE;
				break;

			default:
				g_warning ("Invalid response: %d", response);
				g_assert_not_reached ();
				rename_dir = FALSE;
			}

			if (rename_dir) {
				*renamed_uri = ob_vfs_utils_create_unique_dir (
					uri, mode);

				if (*renamed_uri == NULL) {
					/*g_printerr ("Couldn't create unique dir.\n");*/

					g_set_error (error,
						     OB_ERROR,
						     OB_ERROR_RESTORE_GENERIC,
						     "Restore, could not rename folder");
					
					return ARCHIVER_RESULT_ERROR;
				}
			} else {
				/* Tell the caller that the directory shouldn't
				 * be traversed, but not to interrupt the
				 * operation.
				 */
				*stop_traversing = TRUE;
			}

			return ARCHIVER_RESULT_OK;
		}
	} else {
		/* We couldn't unpack the directory for some other reason. */
		g_set_error (error, OB_ERROR, OB_ERROR_RESTORE_GENERIC,
			     "Could not restore folder %s: %s",
			     gsf_input_name (input),
			     gnome_vfs_result_to_string (result));

		return ARCHIVER_RESULT_ERROR;
	}

	return ARCHIVER_RESULT_OK;
}

static ArchiverResult
archiver_unpack_recurse (ObArchiver    *archiver,
			 ObCategory     category,
			 GsfInput      *archive_parent,
			 GnomeVFSURI   *uri,
			 GList         *excluded_uris,
			 GSList       **temp_file_list,
			 ConflictMode   conflict_mode,
			 GError       **error)
{
	GsfInput       *archive_child;
	GnomeVFSURI    *uri_child;
	const gchar    *uri_child_str;
	gchar          *uri_child_unescaped;
	int             num, i;
	ArchiverResult  retval;
	GnomeVFSURI    *renamed_uri;
	gboolean        stop_traversing;
	gboolean        exclude;
	gint            num_bytes;
	
        if (ob_archiver_is_cancelled (archiver)) {
                return ARCHIVER_RESULT_CANCELLED;
        }

	retval = ARCHIVER_RESULT_OK;

	num = gsf_infile_num_children (GSF_INFILE (archive_parent));
	d(g_printerr ("\nLooking to unpack %d files/directories\n", num));

	for (i = 0; i < num; i++) {
		archive_child = gsf_infile_child_by_index (GSF_INFILE (archive_parent), i);
		uri_child = gnome_vfs_uri_append_path (uri, gsf_input_name (archive_child));
		uri_child_str = gnome_vfs_uri_get_path (uri_child);
		uri_child_unescaped = gnome_vfs_unescape_string (uri_child_str, NULL);
		
		exclude = FALSE;

		if (g_list_find_custom (excluded_uris, uri_child_unescaped,
					(GCompareFunc) archiver_unpack_find_foreach)) {
			exclude = TRUE;
		}

		g_free (uri_child_unescaped);

		if (IS_GSF_DIR (archive_child)) {
			if (!exclude) {
				d(g_printerr ("  Unpacking directory: %s\n", uri_child_str));

				retval = archiver_unpack_dir (archiver,
							      archive_child,
							      uri_child,
							      0777,
							      conflict_mode,
							      &renamed_uri,
							      &stop_traversing,
							      error);
				if (retval != ARCHIVER_RESULT_OK) {
					/* We simply ignore if creating
					 * directories doesn't work. We will get
					 * the real error when an actual file is
					 * tried, and for the sudo restored
					 * bookmarks, the needed directories
					 * will be created by the sudo restore
					 * helper.
					 */
					/*g_object_unref (archive_child);
					gnome_vfs_uri_unref (uri_child);
					break;*/
					retval = ARCHIVER_RESULT_OK;
				}

				if (renamed_uri) {
					/* The directory was renamed to avoid conflict
					 * with an exisiting file. Unpack the children
					 * into the renamed directory, instead of the
					 * original named one.
					 */
					gnome_vfs_uri_unref (uri_child);
					uri_child = renamed_uri;
				}
			} else {
				stop_traversing = FALSE;
			}

			if (!stop_traversing) {
				retval = archiver_unpack_recurse (archiver,
								  category,
								  archive_child,
								  uri_child,
								  excluded_uris,
								  temp_file_list,
								  conflict_mode,
								  error);
				if (retval != ARCHIVER_RESULT_OK) {
					gnome_vfs_uri_unref (uri_child);
					g_object_unref (archive_child);
					break;
				}
			}
		} else {
			if (exclude) {
				num_bytes = gsf_input_size (archive_child);

				ob_archiver_add_processed_size (archiver, num_bytes);
				ob_archiver_add_processed_files (archiver, 1);
				ob_archiver_push_progress_event (archiver);

				d(g_printerr ("  Excluded file: %s\n", uri_child_str));

				continue;
			}
			
 			d(g_printerr ("  Unpacking file: %s\n", uri_child_str));

			/* We special case settings. */
			if (category != OB_CATEGORY_SETTINGS) {
				retval = archiver_unpack_file (archiver,
							       archive_child,
							       uri_child,
							       conflict_mode,
							       category,
							       error);
			} else {
				retval = archiver_unpack_file_settings (archiver,
									archive_child,
									uri_child,
									temp_file_list,
									error);
			}

			if (retval != ARCHIVER_RESULT_OK) {
				gnome_vfs_uri_unref (uri_child);
				g_object_unref (archive_child);
				break;
			}
		}

		gnome_vfs_uri_unref (uri_child);
		g_object_unref (archive_child);
	}

	return retval;
}

/* Opens the zip file (optionally encrypted) for the specified category. */
static ArchiverResult
archiver_zip_create_category_input (ObArchiver  *archiver,
				    ObCategory   category,
				    GsfInput   **input_zip,
				    GError     **error)
{
	ObArchiverZipPriv *priv;
	gchar             *archive_path;
	GError            *gsf_error = NULL;
	GsfInput          *input;
	GsfInfile         *infile_zip;
	GsfInput          *input_crypt;
	ArchiverResult     retval;

	priv = OB_ARCHIVER_ZIP (archiver)->priv;

	if (ob_archiver_is_cancelled (archiver)) {
		return ARCHIVER_RESULT_CANCELLED;
	}

	archive_path = ob_backup_info_get_category_archive_path (priv->backup_info,
								 category);

	/* Open the archive. */
	input = gsf_input_stdio_new (archive_path, &gsf_error);
	g_free (archive_path);
	if (!input) {
		g_error_free (gsf_error);

		g_set_error (error, OB_ERROR, OB_ERROR_RESTORE_GENERIC,
			     "Could not open backup file");
		ob_log_warning ("Could not open backup file");

		return ARCHIVER_RESULT_ERROR;
	}

        if (ob_archiver_is_cancelled (archiver)) {
                g_object_unref (input);
		return ARCHIVER_RESULT_CANCELLED;
	}

	input_crypt = gsf_input_crypt_new (input, priv->password, &gsf_error);
        g_object_unref (input);
        if (!input_crypt) {
		char *tmp;

		tmp = g_strdup_printf ("gsf: (%d) %s\n",
				       gsf_error->code,
				       gsf_error->message);
		ob_log_error (tmp);
		g_free (tmp);
		
		g_error_free (gsf_error);

		/* We should not be called with a password if the backup is not
		 * protected, so therefore we can use this information to decide
		 * if the password was wrong, or if there is some other
		 * error.
		 */
		if (priv->password) {
			g_set_error (error, OB_ERROR, OB_ERROR_RESTORE_WRONG_PASSWORD,
				     "Wrong password");
		} else {
			g_set_error (error, OB_ERROR, OB_ERROR_RESTORE_GENERIC,
				     "Could not open crypt file");
		}

		retval = ARCHIVER_RESULT_ERROR;
		goto done;
        }

	gsf_error = NULL;
	infile_zip = gsf_infile_zip_new (input_crypt, &gsf_error);
	g_object_unref (input_crypt);
	if (gsf_error) {
		g_set_error (error, OB_ERROR, OB_ERROR_RESTORE_GENERIC,
			     "Failed to open zip file: %s", gsf_error->message);
		g_error_free (gsf_error);
		ob_log_error ((*error)->message);
		retval = ARCHIVER_RESULT_ERROR;
		goto done;
	}

	*input_zip = GSF_INPUT (infile_zip);
	retval =  ARCHIVER_RESULT_OK;

 done:

	return retval;
}

static ArchiverResult
archiver_zip_unpack_category (ObArchiver  *archiver,
			      ObCategory   category,
			      GError     **error)
{
	ObArchiverZipPriv *priv;
	GsfInput          *input_zip;
	GsfInput          *root;
	ArchiverResult     retval;
	GnomeVFSURI       *unpack_root_uri;
	GList             *excluded_uris;
	GSList            *temp_file_list = NULL;
	
	priv = OB_ARCHIVER_ZIP (archiver)->priv;

	if (ob_archiver_is_cancelled (archiver)) {
		return ARCHIVER_RESULT_CANCELLED;
	}

	/* Just to make the code in the caller cleaner. */
	if (category == 0) {
		return ARCHIVER_RESULT_OK;
	}

	/* Don't try to unpack files that aren't there. */
	if (!(ob_backup_info_get_categories (priv->backup_info) & category)) {
		return ARCHIVER_RESULT_OK;
	}

	/* Tell the achiver which category we are currently working on
	 * so it can be presented in the progress dialiog 
	 */
	ob_archiver_set_current_category (archiver, category);

	d(g_printerr ("Unpacking %s\n",
		      ob_category_to_non_translated_string (category)));

	retval = archiver_zip_create_category_input (archiver,
						     category,
						     &input_zip,
						     error);
	if (retval != ARCHIVER_RESULT_OK) {
		return retval;
	}

	root = gsf_infile_child_by_name (GSF_INFILE (input_zip), "Root");
	if (!root) {
		/* If we can't find the dir, just skip this category. It
		 * shouldn't happen but if it does, we can at least handle it
		 * gracefully.
		 */
		g_object_unref (input_zip);
		return ARCHIVER_RESULT_OK;
	}

	/* We assume that the restore base directory exists. For real usage, it
	 * is the root (/). For testing unpacking to another base directory,
	 * change this URI.
	 */
	unpack_root_uri = gnome_vfs_uri_new ("file:///");

	excluded_uris = ob_backup_locations_get_uris (priv->locations, 
						      category, TRUE);
	
	retval = archiver_unpack_recurse (archiver,
					  category,
					  root,
					  unpack_root_uri,
					  excluded_uris,
					  &temp_file_list,
					  CONFLICT_MODE_ASK,
					  error);

	g_list_foreach (excluded_uris, (GFunc) g_free, NULL);
	g_list_free (excluded_uris);

	gnome_vfs_uri_unref (unpack_root_uri);

	if (retval == ARCHIVER_RESULT_OK && category == OB_CATEGORY_SETTINGS) {
		archiver_unpack_move_settings (archiver, temp_file_list);
	}
	temp_file_entry_list_free (temp_file_list);

	g_object_unref (root);
	g_object_unref (input_zip);

	return retval;
}

static ArchiverResult
archiver_zip_unpack_categories (ObArchiver  *archiver,
				GError     **error)
{
	ObArchiverZipPriv *priv;
	ArchiverResult     retval;

	priv = OB_ARCHIVER_ZIP (archiver)->priv;

	retval = ARCHIVER_RESULT_OK;

	retval = archiver_zip_unpack_category (archiver,
					       priv->categories & OB_CATEGORY_EMAILS,
					       error);
	if (retval != ARCHIVER_RESULT_OK) {
		return retval;
	}

	retval = archiver_zip_unpack_category (archiver,
					       priv->categories & OB_CATEGORY_CONTACTS,
					       error);
	if (retval != ARCHIVER_RESULT_OK) {
		return retval;
	}

	retval = archiver_zip_unpack_category (archiver,
					       priv->categories & OB_CATEGORY_DOCUMENTS,
					       error);
	if (retval != ARCHIVER_RESULT_OK) {
		return retval;
	}

	retval = archiver_zip_unpack_category (archiver,
					       priv->categories & OB_CATEGORY_MEDIA,
					       error);
	if (retval != ARCHIVER_RESULT_OK) {
		return retval;
	}

	retval = archiver_zip_unpack_category (archiver,
					       priv->categories & OB_CATEGORY_BOOKMARKS,
					       error);
	if (retval != ARCHIVER_RESULT_OK) {
		return retval;
	}

	retval = archiver_zip_unpack_category (archiver,
					       priv->categories & OB_CATEGORY_SETTINGS,
					       error);
	if (retval != ARCHIVER_RESULT_OK) {
		return retval;
	}

	/* The category "other" is only restored when "all" is checked. */
	if (priv->categories == OB_CATEGORY_ALL) {
		retval = archiver_zip_unpack_category (archiver,
						       priv->categories & OB_CATEGORY_OTHER,
						       error);

		if (retval != ARCHIVER_RESULT_OK) {
			return retval;
		}
	}

	ob_archiver_push_progress_finalizing_event (archiver);

	/* Run any transform scripts if necessary. */
	ob_restore_transform_execute (priv->restore_transform);
	
	return retval;
}

/* Unpacks a number of categories from separate zip files inside the backup
 * directory to the original location.
 */
static ArchiverResult
archiver_zip_unpack (ObArchiver *archiver)
{
	ObArchiverZipPriv *priv;
	GError            *error = NULL;
	ArchiverResult     retval;
	int                i;

	priv = OB_ARCHIVER_ZIP (archiver)->priv;

	/* Hack: Sleep a bit just to be on the safe side if something is not
	 * immediately shut down by the app killer script. We do it in the
	 * thread to not lock up the GUI. We do it in chunks of half a second so
	 * that it's still possible to cancel in a timely fashion.
	 */
	for (i = 0; i < 5; i++) {
		g_usleep (0.5 * G_USEC_PER_SEC);
		if (ob_archiver_is_cancelled (archiver)) {
			break;
		}
	}
	
	/* Setup progress info. */
	ob_archiver_set_total_size (
		archiver,
		ob_backup_info_get_size_for_categories (priv->backup_info,
							priv->categories));

	ob_archiver_set_total_files (
		archiver,
		ob_backup_info_get_num_files_for_categories (priv->backup_info,
							     priv->categories));

	retval = archiver_zip_unpack_categories (archiver, &error);

	switch (retval) {
	case ARCHIVER_RESULT_OK:
		ob_archiver_push_finished_event (archiver);
		break;
	case ARCHIVER_RESULT_CANCELLED:
		ob_backend_confirm_cancel (ob_archiver_get_backend (archiver));
		break;
	case ARCHIVER_RESULT_CONFLICT_CANCELLED:
		ob_archiver_push_cancelled_event (archiver);
		break;
	case ARCHIVER_RESULT_ERROR:
		ob_archiver_push_error_event (archiver, error);
		g_error_free (error);
		break;
	}

	return retval;
}

static gpointer
archiver_zip_pack_thread_func (ObArchiver *archiver)
{
	ObArchiverZipPriv *priv;
	int                fd;

	d(g_printerr ("ArchiverZip pack thread starting.\n"));

	priv = OB_ARCHIVER_ZIP (archiver)->priv;

	g_return_val_if_fail (!priv->is_restore, NULL);

	fd = archiver_zip_lock_card (OB_ARCHIVER_ZIP (archiver));
	archiver_zip_pack (archiver);
	if (fd != -1) {
		close (fd);
	}
	
	d(g_printerr ("Thread done.\n"));

	return NULL;
}

static gpointer
archiver_zip_unpack_thread_func (ObArchiver *archiver)
{
	ObArchiverZipPriv *priv;
	int                fd;

	d(g_printerr ("ArchiverZip unpack thread starting.\n"));

	priv = OB_ARCHIVER_ZIP (archiver)->priv;

	g_return_val_if_fail (priv->is_restore, NULL);

	fd = archiver_zip_lock_card (OB_ARCHIVER_ZIP (archiver));
	archiver_zip_unpack (archiver);
	if (fd != -1) {
		close (fd);
	}

	d(g_printerr ("Thread done.\n"));

	return NULL;
}


