/*
 * This file is part of osso-backup
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Andrey Kochanov <andrey.kochanov@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <config.h>
#include <string.h>
#include <gsf/gsf-output-impl.h>
#include <gsf/gsf-impl-utils.h>
#include <gsf/gsf-utils.h>
#include <openssl/aes.h>

#include "gsf-output-crypt.h"
#include "ob-utils.h"

#define d(x) 

static GObjectClass *parent_class;

struct _GsfOutputCrypt {
	GsfOutput  output;

	GsfOutput *sink; /* encrypted data */

	guchar    *xor;

	guchar    *left_data;
	size_t     left_num_bytes;

	guchar    *hash;
	AES_KEY    aes;
};

typedef struct {
	GsfOutputClass output_class;
} GsfOutputCryptClass;


static gboolean
init_crypt (GsfOutputCrypt *crypt, GError **err)
{
	/* Just pass throgh when no password. */
	if (!crypt->hash) {
		return TRUE;
	}

	/* Initialize the AES data. */
	AES_set_encrypt_key (crypt->hash, 128, &crypt->aes);
	
	return TRUE;
}

/**
 * gsf_output_crypt_new :
 * @sink : The underlying data source.
 * @password: Password or NULL if no password was used to protect the file.
 * @err	   : optionally NULL.
 *
 * Adds a reference to @sink.
 *
 * Returns a new file or NULL.
 **/
GsfOutput *
gsf_output_crypt_new (GsfOutput *sink, const char *password, GError **err)
{
	GsfOutputCrypt *crypt;

	g_return_val_if_fail (GSF_IS_OUTPUT (sink), NULL);

	crypt = g_object_new (GSF_OUTPUT_CRYPT_TYPE, NULL);
	crypt->sink = g_object_ref (sink);

	if (password) {
		crypt->hash = ob_utils_get_128_bits_hash (password);
	}

	if (!init_crypt (crypt, err)) {
		g_object_unref (crypt);
		return NULL;
	}

	/* Write out the IV when the stream is protected. */
	if (password && !gsf_output_write (crypt->sink, 16, crypt->xor)) {
		g_object_unref (crypt);
		return NULL;
	}

	return GSF_OUTPUT (crypt);
}

static void
gsf_output_crypt_finalize (GObject *obj)
{
	GsfOutputCrypt *crypt = (GsfOutputCrypt *)obj;

	if (crypt->sink != NULL) {
		g_object_unref (crypt->sink);
		crypt->sink = NULL;
	}

	g_free (crypt->hash);
	g_free (crypt->xor);
	g_free (crypt->left_data);

	parent_class->finalize (obj);
}

static void
encrypt_data (AES_KEY *aes,
	      guchar  *xor,
	      guchar  *data,
	      gsize    num_bytes)
{
	gsize offset;

	offset = 0;
	while (num_bytes > 0) {
		ob_utils_xor_16_bytes (data + offset, xor);

		AES_encrypt (data + offset,
			     data + offset,
			     aes);

		memcpy (xor, data + offset, 16);

		offset += 16;
		num_bytes -= 16;
	}
}

static gboolean
gsf_output_crypt_write (GsfOutput    *output,
			size_t        num_bytes,
			guint8 const *data)
{
	GsfOutputCrypt *crypt = GSF_OUTPUT_CRYPT (output);
	int             pad;
	size_t          real_num_bytes;
	guchar         *full_data;
 	guchar          xor[16];
	
	g_return_val_if_fail (data != NULL, FALSE);

	/* Pass through when not protected. */
	if (!crypt->hash) {
		return gsf_output_write (crypt->sink, num_bytes, data);
	}
		
	d(g_print ("crypt_write: crypt %p\n", crypt));
	
	d(g_print ("----\n"));
	d(g_print ("write, %d bytes at %d\n", num_bytes, (int) output->cur_offset));

	/* The encryption algorithm works at 16 bytes at a time. This means that
	 * we can only write when we have at least 16 bytes, otherwise we keep
	 * the data and wait for more. If we get more than 16 bytes and it's not
	 * a multiple of 16, we keep the last bit for the next round.
	 */

	real_num_bytes = crypt->left_num_bytes + num_bytes;

	if (real_num_bytes < 16) {
		guchar *tmp;

		/* Not enough yet, save for the next round. */
		d(g_print ("not enough yet... %d\n", real_num_bytes));
		
		tmp = g_malloc (real_num_bytes);
		memcpy (tmp, crypt->left_data, crypt->left_num_bytes);
		memcpy (tmp + crypt->left_num_bytes, data, num_bytes);

		g_free (crypt->left_data);
		crypt->left_data = tmp;
		crypt->left_num_bytes = real_num_bytes;

		return TRUE;
	}
	
	pad = real_num_bytes % 16;

	real_num_bytes -= pad;
	
	d(g_print ("writing %d old bytes + new: %d\n", crypt->left_num_bytes, real_num_bytes));

	/* Put any left over data from the last round together with the new one. */
	full_data = g_malloc (real_num_bytes);
	if (crypt->left_num_bytes) {
		memcpy (full_data, crypt->left_data, crypt->left_num_bytes);
	}

	memcpy (full_data + crypt->left_num_bytes, data, real_num_bytes - crypt->left_num_bytes);

	/* Get the previous chunk for xor value. */
	memcpy (xor, crypt->xor, 16);
	
	encrypt_data (&crypt->aes,
		      xor,
		      full_data,
		      real_num_bytes);

	/* Write the result. */
	if (!gsf_output_write (crypt->sink, real_num_bytes, full_data)) {
		g_free (full_data);
		return FALSE;
	}
	
	g_free (full_data);

	/* Keep the last xor for the next round. */
	g_free (crypt->xor);
	crypt->xor = g_memdup (xor, 16);

	/* And keep the remainder of the data for the next round. */
	g_free (crypt->left_data);
	if (pad > 0) {
		crypt->left_data = g_memdup (data + num_bytes - pad, pad);
	} else {
		crypt->left_data = NULL;
	}
	crypt->left_num_bytes = pad;

	d(g_print ("%d bytes left\n", pad));

	return TRUE;
}

static gboolean
gsf_output_crypt_flush (GsfOutput *output)
{
	GsfOutputCrypt *crypt = GSF_OUTPUT_CRYPT (output);
	size_t          num_bytes;
	guint8 const   *data;
	guchar         *full_data;
 	guchar          xor[16];
	char            pad_char;
	int             padding;
	
	data = crypt->left_data;
	num_bytes = crypt->left_num_bytes;
	
	full_data = g_malloc (16);

	memcpy (full_data, crypt->left_data, num_bytes);

	/* Add padding. */
	padding = 16 - num_bytes;
	d(g_print ("flushing, padding with %d\n", padding));

	if (padding >= 1 && padding <= 9) {
		pad_char = '0' + padding;
	}
	else if (padding >= 10 && padding <= 16) {
		pad_char = 'A' + padding - 10;
	} else {
		g_warning ("Wrong padding\n");
		return FALSE;
	}
	
	memset (full_data + num_bytes, pad_char, padding);

	/* Get the previous chunk for xor value. */
	memcpy (xor, crypt->xor, 16);
	
	encrypt_data (&crypt->aes,
		      xor,
		      full_data,
		      16);

	/* Write the result. */
	if (!gsf_output_write (crypt->sink, 16, full_data)) {
		g_free (full_data);
		return FALSE;
	}

	g_free (full_data);
	
	return TRUE;
}

static gboolean
gsf_output_crypt_seek (GsfOutput *output,
		       gsf_off_t  offset,
		       GSeekType  whence)
{
	GsfOutputCrypt *crypt = GSF_OUTPUT_CRYPT (output);

	/* Just pass through when no password. */
	if (!crypt->hash) {
		return gsf_output_seek (crypt->sink, offset, whence);
	}
	
	return FALSE;
}

static gboolean
gsf_output_crypt_close (GsfOutput *output)
{
	GsfOutputCrypt *crypt = GSF_OUTPUT_CRYPT (output);

	if (crypt->hash) {
		return gsf_output_crypt_flush (output);
	}
	
	return TRUE;
}

static void
gsf_output_crypt_init (GObject *obj)
{
	GsfOutputCrypt *crypt = GSF_OUTPUT_CRYPT (obj);

	crypt->sink = NULL;
	crypt->xor = ob_utils_create_random_16_bytes ();
}

static void
gsf_output_crypt_class_init (GObjectClass *gobject_class)
{
	GsfOutputClass *output_class = GSF_OUTPUT_CLASS (gobject_class);

	gobject_class->finalize = gsf_output_crypt_finalize;
	output_class->Write	= gsf_output_crypt_write;
	output_class->Seek	= gsf_output_crypt_seek;
	output_class->Close	= gsf_output_crypt_close;

	parent_class = g_type_class_peek_parent (gobject_class);
}

GSF_CLASS (GsfOutputCrypt, gsf_output_crypt,
	   gsf_output_crypt_class_init, gsf_output_crypt_init, GSF_OUTPUT_TYPE)



