/*
 * Maemo Games Startup
 * Copyright (c) 2006 INdT.
 * @author Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 * @author Cidorvan Leite <cidorvan.leite@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#include <libosso.h>
#include "startup_ui.h"
#include "startup_i18n.h"

#include <startup_config.h>

#include <glib.h>
#include <gtk/gtk.h>

#include <stdlib.h>
#include <string.h>

#include <hildon-widgets/hildon-note.h>
#include <hildon-widgets/gtk-infoprint.h>

#include <gconf/gconf.h>
#include <gconf/gconf-client.h>

/* Layoyt */
#define GAME_IMAGE_HEIGHT       179
#define GAME_IMAGE_WIDTH         -1
#define LABEL_HEIGHT             30
#define LABEL_WIDTH              -1
#define CTRL_BUTTON_HEIGHT       60
#define CTRL_BUTTON_WIDTH       160
#define PLUGIN_HEIGHT           100
#define PLUGIN_WIDTH             -1

/* menu_action identifiers for plugin*/
#define ME_GAME_OPEN     20
#define ME_GAME_SAVE     21
#define ME_GAME_SAVE_AS  22

/* menu_action identifiers */
#define MA_GAME_PLAY 		  1
#define MA_GAME_RESTART 	  2
#define MA_GAME_CLOSE 		  3

GtkWidget *play_item;
GtkWidget *restart_item;
GtkWidget *open_item;
GtkWidget *save_item       = NULL;
GtkWidget *save_as_item    = NULL;
//GtkWidget *notebook 	   = NULL;
gchar     *game_title      = NULL;
gboolean  paused           = FALSE;
//GtkWidget *join_play_button, *join_refresh_button, *join_back_button;
//GtkWidget *join_play_menu, *join_refresh_menu, *join_back_menu;


static GtkWidget *startup_ui_create_common_view       (StartupUI    *ui);
static void startup_ui_create_menu              (StartupUI    *ui);

// callbacks
static void     startup_ui_top_cb               (StartupApp   *app, 
                                                 gpointer      data);
static void     startup_ui_state_change_cb      (StartupApp   *app, 
                                                 guint         oldstate,
                                                 guint         newstate,
                                                 gpointer      data);
static void     startup_ui_menu_action_cb       (gpointer      data, 
                                                 guint         action, 
                                                 GtkWidget    *menu_item);
static void     startup_ui_game_quit_cb         (GtkWidget    *widget,
                                                 gpointer      data);
static void     startup_ui_game_play_cb         (GtkWidget    *widget, 
                                                 gpointer      data);
static void     startup_ui_game_restart_cb      (GtkWidget    *widget, 
                                                 gpointer      data);
static gboolean ui_show_restart_game_dialog     (StartupUI *ui);

StartupUI *
startup_ui_new (StartupApp *app)
{ 
  StartupUI *ui;
  const gchar *plugin_path;
  const gchar *game_gettext_package;
  const gchar *msgid;
  g_return_val_if_fail (app != NULL, NULL);

  ui = g_new0 (StartupUI, 1);
  g_assert (ui);

  ui->app = app;
  startup_app_set_top_cb (app, startup_ui_top_cb, (gpointer) ui);
  startup_app_set_state_change_cb (app, startup_ui_state_change_cb, (gpointer) ui);
  plugin_path = startup_config_read_entry (startup_app_get_config (ui->app), "PluginPath", NULL);
  if (plugin_path){
    ui->plugin_single = startup_plugin_load (plugin_path);
    ui->plugin_single->gs->ui = ui;
    ui->plugin_single->gs->startup_ui_menu_action_cb = startup_ui_menu_action_cb;
    ui->plugin_single->gs->startup_ui_state_change_cb = startup_ui_state_change_cb;
  } else 
    ui->plugin_single = NULL;

  ui->hildon_app = HILDON_APP (hildon_app_new ());
  g_assert (ui->hildon_app);

  ui->hildon_appview = HILDON_APPVIEW (hildon_appview_new (NULL));
  g_assert (ui->hildon_appview);
  
  /* Connect destroy signal */
  g_signal_connect (G_OBJECT (ui->hildon_appview), "destroy",
		          G_CALLBACK (startup_ui_game_quit_cb), (gpointer) ui);
      
  gtk_container_add (GTK_CONTAINER (ui->hildon_appview), startup_ui_create_common_view (ui));

  gtk_widget_show_all(GTK_WIDGET(hildon_appview_get_menu (ui->hildon_appview)));

  game_gettext_package = startup_config_read_entry (startup_app_get_config (ui->app), "GettextPackage", NULL);
  if (ui->plugin_single && game_gettext_package)
    ui->plugin_single->gettext_package = game_gettext_package;

  msgid = startup_config_read_entry (startup_app_get_config (ui->app), "TitleId", NULL);
  
  if (game_gettext_package && msgid) {
    game_title = dgettext (game_gettext_package, msgid);
  }
  
  if (!game_title || game_title == msgid) {
    game_title = startup_config_read_entry (startup_app_get_config (ui->app), "Title", NULL);
  }
  g_assert (game_title);

  hildon_app_set_title (ui->hildon_app, game_title);
  hildon_app_set_appview (ui->hildon_app, ui->hildon_appview);
  gtk_widget_show_all (GTK_WIDGET (ui->hildon_app));

  return ui;
}

void
startup_ui_destroy (StartupUI *ui)
{
  g_return_if_fail (ui != NULL);
}

static GtkWidget *startup_ui_create_common_view (StartupUI *ui)
{
  GtkWidget *alignment;
  GtkWidget *main_vbox;
  GtkWidget *image, *back;
  GtkWidget *controls_vbox;
  GtkWidget *controls_hbox;
  GtkWidget *game_widget;  

  /* Create a vbox for all our stuff inside main view */
  main_vbox = gtk_vbox_new (FALSE, 0);
  g_assert (main_vbox);

  image = gtk_image_new ();
  g_assert (image);
  gtk_image_set_from_file (GTK_IMAGE (image), startup_config_read_entry (startup_app_get_config (ui->app), "Image", NULL));
  gtk_widget_set_size_request (image, GAME_IMAGE_WIDTH, GAME_IMAGE_HEIGHT);
  gtk_container_add (GTK_CONTAINER (main_vbox), image);

  /* Create a vbox for common controls */
  controls_vbox = gtk_vbox_new (FALSE, 0);
  g_assert (controls_vbox);

  gtk_container_add (GTK_CONTAINER (main_vbox), controls_vbox);

  /* Status label */
  ui->controls_state_label = gtk_label_new (_("game_ia_startup_not_started"));
  g_assert (ui->controls_state_label);
    
  gtk_widget_set_size_request (ui->controls_state_label, LABEL_WIDTH, LABEL_HEIGHT);
  gtk_container_add (GTK_CONTAINER (controls_vbox), ui->controls_state_label);

  /* Create a hbox for common controls */
  controls_hbox = gtk_hbox_new (FALSE, 10);
  g_assert (controls_hbox);

  gtk_container_add (GTK_CONTAINER (controls_vbox), controls_hbox);

  /* Play button */
  ui->play_button = gtk_button_new_with_label (_("game_bv_startup_play"));
  g_assert (ui->play_button);
  gtk_widget_set_size_request (ui->play_button, CTRL_BUTTON_WIDTH, CTRL_BUTTON_HEIGHT);

  g_signal_connect (G_OBJECT (ui->play_button), "clicked", G_CALLBACK (startup_ui_game_play_cb), (gpointer) ui);

  alignment = gtk_alignment_new (1.0, 0.5, 0, 0);
  gtk_container_add (GTK_CONTAINER (alignment), ui->play_button);
  
  gtk_box_pack_start (GTK_BOX (controls_hbox), alignment, TRUE, TRUE, 0);

  /* Restart button */
  ui->restart_button = gtk_button_new_with_label (_("game_bv_startup_restart"));
  g_assert (ui->restart_button);
  gtk_widget_set_size_request (ui->restart_button, CTRL_BUTTON_WIDTH, CTRL_BUTTON_HEIGHT);

  g_signal_connect (G_OBJECT (ui->restart_button), "clicked", G_CALLBACK (startup_ui_game_restart_cb), (gpointer) ui);

  alignment = gtk_alignment_new (0, 0.5, 0, 0);
  gtk_container_add (GTK_CONTAINER (alignment), ui->restart_button);
  
  gtk_widget_set_sensitive (ui->restart_button, FALSE);

  gtk_box_pack_start (GTK_BOX (controls_hbox), alignment, TRUE, TRUE, 0);
  
  if (ui->plugin_single) 
    game_widget = ui->plugin_single->info->load ();
  else
    game_widget = gtk_vbox_new (FALSE, 0);
  g_assert (game_widget);
  
  gtk_widget_set_size_request (game_widget, PLUGIN_WIDTH, PLUGIN_HEIGHT);
  gtk_container_add (GTK_CONTAINER (main_vbox), game_widget);

  startup_ui_create_menu(ui);

  return main_vbox;
}

static void 
startup_ui_create_menu (StartupUI *ui)
{
  GtkMenu   *main_menu_bar;
  GtkWidget *close_item;
  GtkWidget *game_menu;
  GtkWidget *game_item;
  GtkWidget **plugin_menu; 
  gint nmenu_items = 0, i;
 
  // Get application view's menu bar 
  main_menu_bar = hildon_appview_get_menu (ui->hildon_appview);
  game_item = gtk_menu_item_new_with_label (_("game_me_chess_main_menu_game"));
  gtk_menu_bar_append (GTK_MENU (main_menu_bar), game_item);

  game_menu = gtk_menu_new (); 
  gtk_menu_item_set_submenu (GTK_MENU_ITEM (game_item), game_menu);

  play_item = gtk_menu_item_new_with_label (_("game_me_chess_menu_game_play"));
  gtk_menu_append (GTK_MENU (game_menu), play_item);
  g_signal_connect (G_OBJECT (play_item), "activate",
                            G_CALLBACK (startup_ui_game_play_cb),
                            (gpointer) ui);
  
  restart_item = gtk_menu_item_new_with_label (_("game_me_chess_menu_game_restart"));
  gtk_menu_append (GTK_MENU (game_menu), restart_item);
  g_signal_connect (G_OBJECT (restart_item), "activate",
                            G_CALLBACK (startup_ui_game_restart_cb),
                            (gpointer) ui);

  if (ui->plugin_single && ui->plugin_single->info->load_menu) {
    if(ui->plugin_single->menu_open_save && ui->plugin_single->info->plugin_cb) {
      gtk_menu_append (GTK_MENU (game_menu), gtk_menu_item_new());  
      save_item = gtk_menu_item_new_with_label (_("game_me_chess_menu_game_save"));
      gtk_menu_append (GTK_MENU (game_menu), save_item);
      g_signal_connect (G_OBJECT (save_item), "activate",
                        G_CALLBACK (ui->plugin_single->info->plugin_cb),
                        (gpointer) ME_GAME_SAVE);    

      save_as_item = gtk_menu_item_new_with_label (_("game_me_chess_menu_game_save_as"));
      gtk_menu_append (GTK_MENU (game_menu), save_as_item);
      g_signal_connect (G_OBJECT (save_as_item), "activate",
                        G_CALLBACK (ui->plugin_single->info->plugin_cb),
                        (gpointer) ME_GAME_SAVE_AS);    

      open_item = gtk_menu_item_new_with_label (_("game_me_chess_menu_game_open"));
      gtk_menu_append (GTK_MENU (game_menu), open_item);
      g_signal_connect (G_OBJECT (open_item), "activate",
                        G_CALLBACK (ui->plugin_single->info->plugin_cb),
                        (gpointer) ME_GAME_OPEN);    

      if (startup_app_get_game_state (ui->app) == GAME_UNLOADED ||
          startup_app_get_game_state (ui->app) == GAME_CLOSED) {
        gtk_widget_set_sensitive(save_item, FALSE);
        gtk_widget_set_sensitive(save_as_item, FALSE);
      }
    }

    plugin_menu = ui->plugin_single->info->load_menu (&nmenu_items);
    // Conditional creation, by game type.
    for(i=0; i<nmenu_items; i++){
       if (GTK_IS_MENU_ITEM(plugin_menu[i])){
         gtk_menu_bar_append (GTK_MENU (main_menu_bar), plugin_menu[i]);
       }
    }
    if (ui->plugin_single->info->update_menu)
      ui->plugin_single->info->update_menu();
  }
  close_item = gtk_menu_item_new_with_label (_("game_me_main_menu_close"));
  gtk_menu_bar_append (GTK_MENU (main_menu_bar), close_item);
  g_signal_connect (G_OBJECT (close_item), "activate",
                            G_CALLBACK (startup_ui_game_quit_cb),
                            (gpointer) ui);
  
  if (startup_app_get_game_state (ui->app) == GAME_UNLOADED ||
      startup_app_get_game_state (ui->app) == GAME_CLOSED)
    gtk_widget_set_sensitive(restart_item, FALSE);
}

static void 
startup_ui_top_cb (StartupApp *app, 
                   gpointer    data)
{
  StartupUI *ui = STARTUP_UI (data);
  gtk_window_present (GTK_WINDOW (ui->hildon_app));
}

static void 
startup_ui_state_change_cb (StartupApp *app, 
                            guint       oldstate,
                            guint       newstate,
                            gpointer    data)
{
  StartupUI *ui = STARTUP_UI (data);

  switch (newstate) {
    case GAME_UNLOADED:
      gtk_label_set_text (GTK_LABEL (ui->controls_state_label),
        _("game_ia_startup_not_started"));

      gtk_button_set_label (GTK_BUTTON (ui->play_button), 
        _("game_bv_startup_play"));

      gtk_widget_set_sensitive (ui->restart_button, FALSE);

      gtk_widget_set_sensitive(restart_item, FALSE);
      gtk_widget_set_sensitive (ui->restart_button, FALSE);
      if (save_item && save_as_item){
        gtk_widget_set_sensitive(save_item, FALSE);
        gtk_widget_set_sensitive(save_as_item, FALSE);
      }
      break;

    case GAME_PAUSED:
      gtk_label_set_text (GTK_LABEL (ui->controls_state_label),
        _("game_ia_startup_paused"));

      gtk_button_set_label (GTK_BUTTON (ui->play_button), 
        _("game_bv_startup_continue"));
        
      gtk_widget_set_sensitive (ui->restart_button, TRUE);

      gtk_widget_set_sensitive(restart_item, TRUE);
      if (save_item && save_as_item){
        gtk_widget_set_sensitive(save_item, TRUE);
        gtk_widget_set_sensitive(save_as_item, TRUE);
      }
      if (ui->plugin_single && ui->plugin_single->info->update_menu)
        ui->plugin_single->info->update_menu ();
      paused = TRUE;
      break;
      
    case GAME_CLOSED:
      gtk_label_set_text (GTK_LABEL (ui->controls_state_label),
        _("game_ia_startup_finished"));

      gtk_button_set_label (GTK_BUTTON (ui->play_button), 
        _("game_bv_startup_play"));

      gtk_widget_set_sensitive (ui->restart_button, FALSE);

      gtk_widget_set_sensitive(restart_item, FALSE);

      if (save_item && save_as_item){
        gtk_widget_set_sensitive(save_item, FALSE);
        gtk_widget_set_sensitive(save_as_item, FALSE);
      }

      if (ui->plugin_single && ui->plugin_single->info->update_menu)
        ui->plugin_single->info->update_menu ();
      break;

    default:
      break;
  }
}

// callbacks
static void 
startup_ui_menu_action_cb (gpointer   data, 
                           guint      action, 
                           GtkWidget *menu_item)
{
  StartupUI *ui = STARTUP_UI (data);
  
  switch (action) {
    case MA_GAME_PLAY:
      startup_ui_game_play_cb (menu_item, ui);
      break;
      
    case MA_GAME_RESTART:
      startup_ui_game_restart_cb (menu_item, ui);
      break;
      
    case MA_GAME_CLOSE:
      startup_ui_game_quit_cb (menu_item, ui);
      break;
  }
}

static void 
startup_ui_game_play_cb (GtkWidget *widget, 
                         gpointer   data)
{
  StartupUI *ui = STARTUP_UI (data);
  if (ui->plugin_single && ui->plugin_single->info->write_config)
    ui->plugin_single->info->write_config ("", "");
    guint state = startup_app_get_game_state (ui->app);
  if (state == GAME_PAUSED || paused){
    paused = FALSE;
    startup_app_send_game_msg (ui->app, GAME_CONTINUE);
  }
  else {
    startup_app_send_game_msg (ui->app, GAME_RESTART);
  }
}

static void 
startup_ui_game_restart_cb (GtkWidget *widget, 
                            gpointer   data)
{
  StartupUI *ui = STARTUP_UI (data);
  if (paused){
    if(ui_show_restart_game_dialog(ui)){
      if (ui->plugin_single && ui->plugin_single->info->write_config)
        ui->plugin_single->info->write_config ("", "");
      
      hildon_app_set_title (ui->hildon_app, game_title);
      startup_app_send_game_msg (ui->app, GAME_RESTART);
    }
  }
}

static void 
startup_ui_game_quit_cb (GtkWidget *widget,
                         gpointer   data)
{
  StartupUI *ui = STARTUP_UI (data);
  guint state;

  if (ui->plugin_single && ui->plugin_single->info->write_config)
    ui->plugin_single->info->write_config ("", "");

    state = startup_app_get_game_state (ui->app);
    if (state == GAME_RUNNING){
      startup_app_send_game_msg (ui->app, GAME_CLOSE);
    }
  
  gtk_widget_hide (GTK_WIDGET (ui->hildon_app));
  gtk_main_iteration ();
  gtk_main_quit (); 
}

gboolean ui_show_restart_game_dialog(StartupUI *ui)
{
  gboolean answer = FALSE;
  HildonNote *note = HILDON_NOTE(hildon_note_new_confirmation_add_buttons(
        GTK_WINDOW(ui->hildon_app),
        _("game_nc_restart_game"),
        _("game_bd_restart_game_ok"), 1,
        _("game_bd_restart_game_cancel"), 0,
        NULL));
  answer = gtk_dialog_run(GTK_DIALOG(note));
  gtk_widget_destroy(GTK_WIDGET(note));
  return answer;
}
