/*
 * This file is part of lessertunjo
 *
 * Copyright (C) 2005 Nokia Corporation.
 *
 * Contact: Kuisma Salonen <kuisma.salonen@nokia.com>
 * Author: Kuisma Salonen <kuisma.salonen@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; version 2.1 of the License.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


/**
 * @file sappd_init.c
 * shadow application library, daemon side library,
 * (de)initialization part
 * <p>
 */

/**********************************************************/

#include <time.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <sys/un.h>
#include <unistd.h>
#include <sys/socket.h>

#include <glib.h>
#include <libosso.h>
#include <gconf/gconf-client.h>

#include "shadowappd.h"

int sappd_error = 0;
int sappd_crashed = 0;
sapp_d_data *main_cont = NULL;

char statedatafile[128] = { 0 } ;

static int cont_active = 0;


static void fall_out_on_signal(int signum)
{
  if(!cont_active) {
    return;
  }


  send(main_cont->connection_sd, "bye\n", 4, 0);

  usleep(160000);

  kill(main_cont->child_pid, SIGKILL);

  sappd_deinitialize(main_cont, SAPPD_DEINIT_REPORT | SAPPD_DEINIT_CRASHED);

  exit(-1);
}

static void ctx_flush(sapp_d_data *ctx)
{
  if(ctx) {
    close(ctx->gcq_sd);
    if(ctx->alt_addr) {
      remove(ctx->alt_addr->sun_path);
      free(ctx->alt_addr);
      ctx->alt_addr = NULL; }
    close(ctx->listener_sd);
    if(ctx->our_addr) {
      remove(ctx->our_addr->sun_path);
      free(ctx->our_addr);
      ctx->our_addr = NULL; }
    if(ctx->osso_context) {
      osso_deinitialize(ctx->osso_context);
      ctx->osso_context = NULL; }
    if(ctx->executable) {
      free(ctx->executable); }
    if(ctx->versionstring) {
      free(ctx->versionstring); }
    if(ctx->appname) {
      free(ctx->appname); }
    free(ctx);
  }
}

sapp_d_data *sappd_initialize(char *filename, GMainContext *gctx, GMainLoop *loopie)
{
  sapp_d_data *retval = NULL;
  FILE *pFile;
  char *filebuffer;

  /* load structure from a file */

  retval = (sapp_d_data *)malloc(sizeof(sapp_d_data));
  if(retval == NULL) {
    sappd_error = SAPP_ERROR_NOMEMORY;
#ifdef SAPPD_DEBUG
    _sd_debugprint("no memory to allocate main context");
#endif
    return NULL;
  }

  memset(retval, 0, sizeof(sapp_d_data));

  pFile = fopen(filename, "r");
  if(pFile) {
    filebuffer = (char *)malloc(256);
    filebuffer[255] = 0;
    fgets(filebuffer, 255, pFile);
    if(strstr(filebuffer, " ") || strstr(filebuffer, "/") ||
       strstr(filebuffer, ".")) {
      free(filebuffer);
      free(retval);
      sappd_error = SAPP_ERROR_FILEFORMAT;
      return NULL;
    }
    retval->appname = (char *)malloc(strlen(filebuffer));
    memcpy(retval->appname, filebuffer, strlen(filebuffer) - 1);
    retval->appname[strlen(filebuffer) - 1] = 0;
    memset(filebuffer, 0, 256);

    fgets(filebuffer, 255, pFile);
    if(strlen(filebuffer) == 0) {
      free(filebuffer);
      free(retval->appname);
      free(retval);
      sappd_error = SAPP_ERROR_FILEFORMAT;
      return NULL;
    }
    retval->executable = (char *)malloc(strlen(filebuffer));
    memcpy(retval->executable, filebuffer, strlen(filebuffer) - 1);
    retval->executable[strlen(filebuffer) - 1] = 0;
    memset(filebuffer, 0, 256);

    fgets(filebuffer, 255, pFile);
    if(strstr(filebuffer, " ") || !strstr(filebuffer, ".") ||
       (strlen(filebuffer) == 0)) {
      retval->service = NULL;
    } else {
      retval->service = (char *)malloc(strlen(filebuffer));
      memcpy(retval->service, filebuffer, strlen(filebuffer) - 1);
      retval->service[strlen(filebuffer) - 1] = 0;
    }

    fgets(filebuffer, 255, pFile);
    if(strstr(filebuffer, " ") || (filebuffer[0] != '/') ||
       (strlen(filebuffer) == 0)) {
      retval->path = NULL;
    } else {
      retval->path = (char *)malloc(strlen(filebuffer));
      memcpy(retval->path, filebuffer, strlen(filebuffer) - 1);
      retval->path[strlen(filebuffer) - 1] = 0;
    }

    fgets(filebuffer, 255, pFile);
    if(strstr(filebuffer, " ") || !strstr(filebuffer, ".") ||
       (strlen(filebuffer) == 0)) {
      retval->iface = NULL;
    } else {
      retval->iface = (char *)malloc(strlen(filebuffer));
      memcpy(retval->iface, filebuffer, strlen(filebuffer) - 1);
      retval->iface[strlen(filebuffer) - 1] = 0;
    }
    fgets(filebuffer, 255, pFile);
    if(strstr(filebuffer, " ") || strstr(filebuffer, "/") ||
       (strlen(filebuffer) == 0) || (filebuffer[0] == '\n')) {
      retval->versionstring = (char *)malloc(8);
      memcpy(retval->versionstring, "1.3.3.7", 7);
      retval->versionstring[7] = 0;
    } else {
      retval->versionstring = (char *)malloc(strlen(filebuffer));
      memcpy(retval->versionstring, filebuffer, strlen(filebuffer) - 1);
      retval->versionstring[strlen(filebuffer) - 1] = 0;
    }

    fclose(pFile);
  } else {
    free(retval);
    sappd_error = SAPP_ERROR_FILEOPEN;
    return NULL;
  }

#ifdef SAPPD_DEBUG
  sprintf(filebuffer, "app name: %s  version: %s  service: %s  path: %s  iface: %s  executable: %s",
          retval->appname, retval->versionstring, retval->service,
          retval->path, retval->iface, retval->executable);
  _sd_debugprint(filebuffer);
#endif

  free(filebuffer);

  /* init libosso */

  retval->osso_context = osso_initialize(retval->appname,
                                         retval->versionstring,
                                         FALSE, gctx);
  if(retval->osso_context == NULL) {
    free(retval->executable);
    free(retval->versionstring);
    free(retval->appname);
    free(retval);
    sappd_error = SAPP_ERROR_OSSOINIT;
#ifdef SAPPD_DEBUG
    _sd_debugprint("cannot gain osso context");
#endif
    return NULL;
  }

#ifdef SAPPD_DEBUG
  _sd_debugprint("libosso initialized");
#endif

  if((osso_application_set_top_cb(retval->osso_context,
                                  sappd_osso_callback_top,
                                  (gpointer)retval) < 0) ||
     (osso_rpc_set_cb_f(retval->osso_context,
                        retval->service,
                        retval->path,
                        retval->iface,
                        sappd_osso_callback_rpc,
                        (gpointer)retval) < 0)) {
    osso_deinitialize(retval->osso_context);
    free(retval->executable);
    free(retval->versionstring);
    free(retval->appname);
    free(retval);
    sappd_error = SAPP_ERROR_OSSOINIT;
    return NULL;
  }

  sprintf(statedatafile, "/tmp/.gamewrapper/statedata/%s.w", retval->appname);

  /* init socket to child */

  retval->listener_sd = socket(PF_UNIX, SOCK_STREAM, 0);
  if(retval->listener_sd == -1) {
    osso_deinitialize(retval->osso_context);
    free(retval->executable);
    free(retval->versionstring);
    free(retval->appname);
    free(retval);
    sappd_error = SAPP_ERROR_SOCKET;
#ifdef SAPPD_DEBUG
    _sd_debugprint("cannot gain socket");
#endif
    return NULL;
  }

  retval->our_addr = (struct sockaddr_un *)malloc(sizeof(struct sockaddr_un));
  if(retval->our_addr == NULL) {
    close(retval->listener_sd);
    osso_deinitialize(retval->osso_context);
    free(retval->executable);
    free(retval->versionstring);
    free(retval->appname);
    free(retval);
    sappd_error = SAPP_ERROR_NOMEMORY;
    return NULL;
  }
  retval->our_addr->sun_family = AF_UNIX;
  sprintf(retval->our_addr->sun_path, "/tmp/.gamewrapper/sockets/s%sv%s",
          retval->appname, retval->versionstring);
  retval->socket_lenght = sizeof(struct sockaddr_un);
  remove(retval->our_addr->sun_path);
  if(bind(retval->listener_sd, (struct sockaddr *)retval->our_addr,
          retval->socket_lenght) == -1) {
#ifdef SAPPD_DEBUG
    _sd_debugprint("cannot bind socket to address:");
    _sd_debugprint(retval->our_addr->sun_path);
#endif
    close(retval->listener_sd);
    osso_deinitialize(retval->osso_context);
    free(retval->our_addr);
    free(retval->executable);
    free(retval->versionstring);
    free(retval->appname);
    free(retval);
    sappd_error = SAPP_ERROR_SOCKET;
    return NULL;
  }
  if(listen(retval->listener_sd, 4) == -1) {
    close(retval->listener_sd);
    remove(retval->our_addr->sun_path);
    osso_deinitialize(retval->osso_context);
    free(retval->our_addr);
    free(retval->executable);
    free(retval->versionstring);
    free(retval->appname);
    free(retval);
    sappd_error = SAPP_ERROR_SOCKET;
#ifdef SAPPD_DEBUG
    _sd_debugprint("cannot listen to socket");
#endif
    return NULL;
  }

#ifdef SAPPD_DEBUG
  _sd_debugprint("starting to initialize gconf socket interface...");
#endif

  /* init socket for gconf */

  retval->gcq_sd = socket(PF_UNIX, SOCK_STREAM, 0);
  if(retval->gcq_sd == -1) {
    ctx_flush(retval);
    sappd_error = SAPP_ERROR_SOCKET;
#ifdef SAPPD_DEBUG
    _sd_debugprint("cannot gain socket for gconf interface");
#endif
    return NULL;
  }

  retval->alt_addr = (struct sockaddr_un *)malloc(sizeof(struct sockaddr_un));
  if(retval->alt_addr == NULL) {
    ctx_flush(retval);
    sappd_error = SAPP_ERROR_NOMEMORY;
    return NULL;
  }
  retval->alt_addr->sun_family = AF_UNIX;
  sprintf(retval->alt_addr->sun_path, "/tmp/.gamewrapper/sockets/s%scfg",
          retval->appname);
  retval->socket_lenght = sizeof(struct sockaddr_un);
  remove(retval->alt_addr->sun_path);
  if(bind(retval->gcq_sd, (struct sockaddr *)retval->alt_addr,
          retval->socket_lenght) == -1) {
#ifdef SAPPD_DEBUG
    _sd_debugprint("cannot bind socket to address:");
    _sd_debugprint(retval->alt_addr->sun_path);
#endif
    ctx_flush(retval);
    sappd_error = SAPP_ERROR_SOCKET;
    return NULL;
  }
  if(listen(retval->gcq_sd, 4) == -1) {
    ctx_flush(retval);
    sappd_error = SAPP_ERROR_SOCKET;
#ifdef SAPPD_DEBUG
    _sd_debugprint("cannot listen to socket");
#endif
    return NULL;
  }

#ifdef SAPPD_DEBUG
  _sd_debugprint("initializing watcher for the socket...");
#endif

  retval->confchan = g_io_channel_unix_new(retval->gcq_sd);
  g_io_add_watch(retval->confchan, G_IO_IN | G_IO_HUP | G_IO_ERR,
                 sappd_confsock_callback, (gpointer)retval);
  g_io_channel_unref(retval->confchan);

  /* gconf */

#ifdef SAPPD_DEBUG
  _sd_debugprint("initializing gconf itself...");
#endif

  retval->gconf_client = gconf_client_get_default();

  /* generate valid state here */
  retval->connstate = SAPPD_CONNSTATE_NONE;

  pFile = fopen(statedatafile, "rb");
  if(pFile) {
    fread(&(retval->state), 4, 1, pFile);
    fclose(pFile);
  } else {
    retval->state = SAPPD_STATE_INACTIVE;
  }

#ifdef SAPPD_DEBUG
  _sd_debugprint("we are up");
#endif

  retval->realbuflen = 0;
  retval->mainloop = loopie;

  return retval;
}

int sappd_init_sigchld(sapp_d_data *context)
{
  if(context == NULL) {
    return SAPP_ERROR_INVALIDPARAMS;
  }

  main_cont = context;
  cont_active = 1;
  signal(SIGCHLD, sigchld_handler);
  signal(SIGHUP, fall_out_on_signal);
  signal(SIGINT, fall_out_on_signal);
  signal(SIGQUIT, fall_out_on_signal);
  signal(SIGILL, fall_out_on_signal);
  signal(SIGABRT, fall_out_on_signal);
  signal(SIGSEGV, fall_out_on_signal);
  signal(SIGPIPE, fall_out_on_signal);
  signal(SIGALRM, fall_out_on_signal);
  signal(SIGTERM, fall_out_on_signal);
  signal(SIGUSR1, SIG_IGN);
  signal(SIGUSR2, SIG_IGN);
  signal(SIGBUS, fall_out_on_signal);
  signal(SIGPOLL, SIG_IGN);
  signal(SIGPROF, SIG_IGN);
  signal(SIGSYS, fall_out_on_signal);
  signal(SIGTRAP, fall_out_on_signal);
  signal(SIGURG, SIG_IGN);
/*signal(SIGVTALARM, SIG_IGN);*/
  signal(SIGXCPU, fall_out_on_signal);
  signal(SIGXFSZ, fall_out_on_signal);
/*signal(SIGEMT, SIG_IGN);*/
  signal(SIGSTKFLT, fall_out_on_signal);
  signal(SIGIO, SIG_IGN);
  signal(SIGPWR, fall_out_on_signal);
/*signal(SIGLOST, fall_out_on_signal);*/

  return SAPP_ERROR_NONE;
}

int sappd_deinitialize(sapp_d_data *context, int flags)
{
  FILE *pFile;

  /* if called at exit, we should enable this particular
     flag which remembers current state                  */


  if(!context) {
    return SAPP_ERROR_INVALIDPARAMS;
  }


  cont_active = 0;


  if(flags & SAPPD_DEINIT_STATESAVE) {
    pFile = fopen(statedatafile, "wb");
    if(pFile) {
      fwrite(&(context->state), 4, 1, pFile);
      fclose(pFile);
    }
  } else {
    remove(statedatafile);
  }

  if(flags & SAPPD_DEINIT_REPORT) {
    if(flags & SAPPD_DEINIT_CRASHED) {
      sappd_rpc(context, "game_crashed");
    } else if(context->state == SAPPD_STATE_INACTIVE) {
      sappd_rpc(context, "game_close");
    } else {
      sappd_rpc(context, "game_pause");
    }
  }

  ctx_flush(context);

  return SAPP_ERROR_NONE;
}


int _sd_debugprint(char *text)
{
  FILE *pFile;
  int currenttime = (int)time(NULL);

  pFile = fopen(SAPP_DEBUG_FILENAME, "a");
  if(pFile) {
    fprintf(pFile, "(day %d, %d:%d:%d) %s\n", (currenttime/86400),
            (((currenttime/3600)%24) + 3), ((currenttime/60)%60),
            (currenttime%60),text);
    fclose(pFile);
  } else {
    printf("%s\n", text);
  }

  return 0;
}
