
/*
 * drivers/media/video/omap/sensor_tcm825x.c
 *
 * TCM825X Sensor driver for OMAP camera sensor interface
 *
 * Author: David Cohen (david.cohen@indt.org.br)
 *
 * This file is licensed under the terms of the GNU General Public License 
 * version 2. This program is licensed "as is" without any warranty of any 
 * kind, whether express or implied.
 * 
 * This driver was based on ov9640 sensor driver from MontaVista
 */

#include <linux/errno.h>
#include <linux/i2c.h>
#include <linux/videodev.h>
#include <linux/delay.h>
#include <media/video-buf.h>

#include <asm/mach-types.h>

#include <asm/arch/board.h>
#include <asm/arch/gpio.h>

#include "sensor_if.h"
#include "tcm825x.h"

struct tcm825x_sensor {
	const struct omap_camera_sensor_config * sensor_config;
	struct omap_camera_sensor * sensor_if;
	struct i2c_client i2c_client;
	struct i2c_driver *i2c_driver;
};

static struct tcm825x_sensor tcm825x;

static struct i2c_driver tcm825x_i2c_driver = {
	.driver = {
		.name           = "TCM825x I2C driver",
	},
        .id		= I2C_DRIVERID_MISC,           /* Experimental ID */
};

/* list of image formats supported by TCM825X sensor */
const static struct v4l2_fmtdesc tcm825x_formats[] = {
	{
		.description = "YUYV (YUV 4:2:2), packed",
		.pixelformat = V4L2_PIX_FMT_UYVY,
	}, {
		/* Note:  V4L2 defines RGB565 as:
		 *
		 *      Byte 0                    Byte 1
		 *      g2 g1 g0 r4 r3 r2 r1 r0   b4 b3 b2 b1 b0 g5 g4 g3
		 *
		 * We interpret RGB565 as:
		 *
		 *      Byte 0                    Byte 1
		 *      g2 g1 g0 b4 b3 b2 b1 b0   r4 r3 r2 r1 r0 g5 g4 g3
		 */
		.description = "RGB565, le",
		.pixelformat = V4L2_PIX_FMT_RGB565,
	},
};

#define TCM825X_NUM_CAPTURE_FORMATS ARRAY_SIZE(tcm825x_formats)
#define NUM_OVERLAY_FORMATS 2

/* register initialization tables for TCM825X */

#define TCM825X_REG_TERM 0xFF	/* terminating list entry for reg */
#define TCM825X_VAL_TERM 0xFF	/* terminating list entry for val */

/* common TCM825X register initialization for all image sizes, pixel
 * formats, and frame rates
 */
const static struct tcm825x_reg tcm825x_common[] = {
    {0x00, 0x00}, {0x30, 0x01}, {0x8c, 0x02}, {0x00, 0x03},
    {0x0f, 0x04}, {0x02, 0x05}, {0x0d, 0x06}, {0xc0, 0x07},
    {0x38, 0x08}, {0x50, 0x09}, {0x80, 0x0a}, {0x40, 0x0b},
    {0x40, 0x0c}, {0x00, 0x0d}, {0x04, 0x0e}, {0x04, 0x0f},
    {0x22, 0x10}, {0x96, 0x11}, {0xf0, 0x12}, {0x08, 0x13},
    {0x08, 0x14}, {0x30, 0x15}, {0x30, 0x16}, {0x01, 0x17},
    {0x40, 0x18}, {0x87, 0x19}, {0x2b, 0x1a}, {0x84, 0x1b},
    {0x52, 0x1c}, {0x44, 0x1d}, {0x68, 0x1e}, {0x00, 0x1f},
    {0x00, 0x20}, {0x01, 0x21}, {0x27, 0x22}, {0x40, 0x23},
    {0x27, 0x24}, {0x5f, 0x25}, {0x00, 0x26}, {0x16, 0x27},
    {0x23, 0x28}, {0x03, 0x29}, {0xaa, 0x2a}, {0xc0, 0x2b},
    {0x10, 0x2c}, {0x4c, 0x2d}, {0x00, 0x2e}, {0x00, 0x2f},
    {0x00, 0x30}, {0x00, 0x31}, {0x00, 0x32}, {0x00, 0x33},
    {0x00, 0x34}, {0x00, 0x34}, {0x00, 0x35}, {0x00, 0x36},
    {0x00, 0x37}, {0x00, 0x38}, {0x8c, 0x39}, {0xc8, 0x3A},
    {0x80, 0x3b}, {0x00, 0x3c}, {0x17, 0x3d}, {0x85, 0x3e},
    {0x9c, 0x3f}, {0xa0, 0x40}, {0x00, 0x41}, {0x00, 0x42},
    {0x00, 0x43}, {0x08, 0x44}, {0x12, 0x45}, {0x00, 0x46},
    {0x20, 0x47}, {0x30, 0x48}, {0x18, 0x49}, {0x20, 0x4a},
    {0x4d, 0x4b}, {0x0c, 0x4c}, {0xe0, 0x4d}, {0x20, 0x4e},
    {0x89, 0x4f}, {0x21, 0x50}, {0x80, 0x51}, {0x02, 0x52},
    {0x00, 0x53}, {0x30, 0x54}, {0x90, 0x55}, {0x40, 0x56},
    {0x06, 0x57}, {0x0f, 0x58}, {0x23, 0x59}, {0x08, 0x5A},
    {0x04, 0x5b}, {0x08, 0x5c}, {0x08, 0x5d}, {0x08, 0x5e},
    {0x08, 0x5f},
    {TCM825X_VAL_TERM, TCM825X_REG_TERM}
};

/* TCM825X register configuration for all combinations of pixel format and 
 * image size
 */
const static struct tcm825x_reg subqcif = { 0x20, TCM825X_PICSIZ };
const static struct tcm825x_reg qcif =    { 0x18, TCM825X_PICSIZ };
const static struct tcm825x_reg cif =     { 0x14, TCM825X_PICSIZ };
const static struct tcm825x_reg qqvga =   { 0x0c, TCM825X_PICSIZ };
const static struct tcm825x_reg qvga =    { 0x04, TCM825X_PICSIZ };
const static struct tcm825x_reg vga =     { 0x00, TCM825X_PICSIZ };

const static struct tcm825x_reg yuv422 = { 0x00, TCM825X_PICFMT };
const static struct tcm825x_reg rgb565 = { 0x02, TCM825X_PICFMT };

/* Our own specific controls */
#define V4L2_CID_ALC                      V4L2_CID_PRIVATE_BASE
#define V4L2_CID_H_EDGE_EN                V4L2_CID_PRIVATE_BASE + 1
#define V4L2_CID_V_EDGE_EN                V4L2_CID_PRIVATE_BASE + 2
#define V4L2_CID_LENS                     V4L2_CID_PRIVATE_BASE + 3
#define V4L2_CID_MAX_EXPOSURE_TIME        V4L2_CID_PRIVATE_BASE + 4
#define V4L2_CID_LAST_PRIV                V4L2_CID_MAX_EXPOSURE_TIME

/*  Video controls  */
static struct vcontrol {
	struct v4l2_queryctrl qc;
	u16 reg;
	u16 start_bit;
} video_control[] = {
	{
		{
			.id = V4L2_CID_GAIN,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Gain",
			.minimum = 0,
			.maximum = 63,
			.step = 1,
		},
		.reg = TCM825X_AG,
		.start_bit = 0,
	}, 
	{
		{
			.id = V4L2_CID_RED_BALANCE,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Red Balance",
			.minimum = 0,
			.maximum = 255,
			.step = 1,
		},
		.reg = TCM825X_MRG,
		.start_bit = 0,
	}, 
	{ 
		{
			.id = V4L2_CID_BLUE_BALANCE,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Blue Balance",
			.minimum = 0,
			.maximum = 255,
			.step = 1,
		},
		.reg = TCM825X_MBG,
		.start_bit = 0,
	},
	{ 
		{
			.id = V4L2_CID_AUTO_WHITE_BALANCE,
			.type = V4L2_CTRL_TYPE_BOOLEAN,
			.name = "Auto White Balance",
			.minimum = 0,
			.maximum = 1,
			.step = 0,
		},
		.reg = TCM825X_AWBSW,
		.start_bit = 7,
	},
	{
		{
			.id = V4L2_CID_EXPOSURE,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Exposure Time",
			.minimum = 0,
			.maximum = 0x1fff,
			.step = 1,
		},
		.reg = TCM825X_ESRSPD_U,
		.start_bit = 0,
	},

	{
		{
			.id = V4L2_CID_HFLIP,
			.type = V4L2_CTRL_TYPE_BOOLEAN,
			.name = "Mirror Image",
			.minimum = 0,
			.maximum = 1,
			.step = 0,
		},
		.reg = TCM825X_H_INV,
		.start_bit = 6,
	},
	{
		{
			.id = V4L2_CID_VFLIP,
			.type = V4L2_CTRL_TYPE_BOOLEAN,
			.name = "Vertical Flip",
			.minimum = 0,
			.maximum = 1,
			.step = 0,
		},
		.reg = TCM825X_V_INV,
		.start_bit = 7,
	},
	/* Private controls */
	{
		{
			.id = V4L2_CID_ALC,
			.type = V4L2_CTRL_TYPE_BOOLEAN,
			.name = "Auto Luminance Control",
			.minimum = 0,
			.maximum = 1,
			.step = 0,
		},
		.reg = TCM825X_ALCSW,
		.start_bit = 7,
	},
	{
		{
			.id = V4L2_CID_H_EDGE_EN,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Horizontal Edge Enhancement",
			.minimum = 0,
			.maximum = 0xff,
			.step = 1,
		},
		.reg = TCM825X_HDTG,
		.start_bit = 0,
	},
	{
		{
			.id = V4L2_CID_V_EDGE_EN,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Vertical Edge Enhancement",
			.minimum = 0,
			.maximum = 0xff,
			.step = 1,
		},
		.reg = TCM825X_VDTG,
		.start_bit = 0,
	},
	{
		{
			.id = V4L2_CID_LENS,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Lens Shading Compensation",
			.minimum = 0,
			.maximum = 0x3f,
			.step = 1,
		},
		.reg = TCM825X_LENS,
		.start_bit = 0,
	},
	{
		{
			.id = V4L2_CID_MAX_EXPOSURE_TIME,
			.type = V4L2_CTRL_TYPE_INTEGER,
			.name = "Maximum Exposure Time",
			.minimum = 0,
			.maximum = 0x3,
			.step = 1,
		},
		.reg = TCM825X_ESRLIM,
		.start_bit = 5,
	},
};


const static struct tcm825x_reg *tcm825x_siz_reg[NUM_IMAGE_SIZES] =
    { &subqcif, &qqvga, &qcif, &qvga, &cif, &vga };

const static struct tcm825x_reg *tcm825x_fmt_reg[NUM_PIXEL_FORMATS] = { &yuv422, &rgb565 };

/* 
 * Read a value from a register in an TCM825X sensor device.  The value is 
 * returned in 'val'.
 * Returns zero if successful, or non-zero otherwise.
 */
static int tcm825x_read_reg(struct i2c_client *client, int reg)
{
	int err;
	struct i2c_msg msg[2];
	u8 reg_buf, data_buf = 0;

	if (!client->adapter)
		return -ENODEV;

	msg[0].addr = client->addr;
	msg[0].flags = 0;
	msg[0].len = 1;
	msg[0].buf = &reg_buf;
	msg[1].addr = client->addr;
	msg[1].flags = I2C_M_RD;
	msg[1].len = 1;
	msg[1].buf = &data_buf;

	reg_buf = reg;

	err = i2c_transfer(client->adapter, msg, 2);
	if (err < 0)
		return err;
	return data_buf;
}

/* Write a value to a register in an TCM825X sensor device.
 * Returns zero if successful, or non-zero otherwise.
 */
static int tcm825x_write_reg(struct i2c_client *client, u8 reg, u8 val)
{
	int err;
	struct i2c_msg msg[1];
	unsigned char data[2];

	if (!client->adapter)
		return -ENODEV;

	msg->addr = client->addr;
	msg->flags = 0;
	msg->len = 2;
	msg->buf = data;
	data[0] = reg;
	data[1] = val;
	err = i2c_transfer(client->adapter, msg, 1);
	if (err >= 0)
		return 0;
	return err;
}

static int __tcm825x_write_reg_mask(struct i2c_client *client, u8 reg, u8 val, u8 mask)
{
	int rc;

	/* need to do read - modify - write */
	rc = tcm825x_read_reg(client, reg);
	if (rc < 0)
		return rc;
	
	rc &= (~mask);	/* Clear the masked bits */
	val &= mask;	/* Enforce mask on value */
	val |= rc;

	/* write the new value to the register */
	if ((rc = tcm825x_write_reg(client, reg, val)))
		return rc;

	return 0;
}

#define tcm825x_write_reg_mask(client, regmask, val) \
        __tcm825x_write_reg_mask(client, TCM825X_ADDR((regmask)), val, \
				 TCM825X_MASK((regmask)))


/* Initialize a list of TCM825X registers.
 * The list of registers is terminated by the pair of values 
 * { TCM825X_REG_TERM, TCM825X_VAL_TERM }.
 * Returns zero if successful, or non-zero otherwise.
 */
static int tcm825x_write_default_regs(struct i2c_client *client, const struct tcm825x_reg reglist[])
{
	int err;
	const struct tcm825x_reg *next = reglist;

	printk(KERN_DEBUG "%s()\n", __FUNCTION__);

	while (!((next->reg == TCM825X_REG_TERM)
		 && (next->val == TCM825X_VAL_TERM))) {
		err = tcm825x_write_reg(client, next->reg, next->val);
		udelay(100);
		if (err) {
			printk(KERN_ERR "%s(): Register writing failed\n", __FUNCTION__);
			return err;
		}
		next++;
	}

	return 0;
}

/* Matches the control ID and returns the vcontrol pointer */
static struct vcontrol * find_vctrl(int id)
{
	int i;

	if (id < V4L2_CID_BASE)
		return NULL;

	for (i = 0; i < ARRAY_SIZE(video_control); i++)
		if (video_control[i].qc.id == id)
			return &video_control[i];

	return NULL;
}

/* Configure the TCM825X for a specified image size, pixel format, and frame 
 * period.  xclk is the frequency (in Hz) of the xclk input to the TCM825X.  
 * fper is the frame period (in seconds) expressed as a fraction.
 * Returns zero if successful, or non-zero otherwise.
 * The actual frame period is returned in fper.
 */
static int tcm825x_configure(struct i2c_client *client,
			     enum image_size isize, enum pixel_format pfmt,
			     unsigned long xclk, struct v4l2_fract *fper)
{
	int err;
	u8 val;

	/* common register initialization */
	err = tcm825x_write_default_regs(client, tcm825x_common);
	if (err)
		return err;

	/* configure image size */
	val = tcm825x_siz_reg[isize]->val;
	printk(KERN_DEBUG "%s(): configuring Image Size %d\n", __FUNCTION__, isize);
	err = tcm825x_write_reg_mask(client, tcm825x_siz_reg[isize]->reg, val);
	if (err)
		return err;

	val = tcm825x_fmt_reg[pfmt]->val;
	/* configure pixel format */
	printk(KERN_DEBUG "%s(): configuring Pixel Format %d\n", __FUNCTION__, pfmt);
	err = tcm825x_write_reg_mask(client, tcm825x_fmt_reg[pfmt]->reg, val);
	if (err)
		return err;

	return 0;
}

static int tcm825x_detect(struct tcm825x_sensor *sensor)
{
	int r;

	r = tcm825x_read_reg(&sensor->i2c_client, 0x01);
	if (r < 0)
		return r;
	if (r == 0) {
		dev_err(&sensor->i2c_client.dev, "device not detected\n");
		return -EIO;
	}
	return 0;
}


/* This function registers an I2C client via i2c_attach_client() for an TCM825X 
 * sensor device.  If 'probe' is non-zero, then the I2C client is only 
 * registered if the device can be detected.  If 'probe' is zero, then no 
 * device detection is attempted and the I2C client is always registered.
 * Returns zero if an I2C client is successfully registered, or non-zero 
 * otherwise.
 */
static int tcm825x_i2c_attach_client(struct i2c_adapter *adap, int addr, int probe)
{
	struct tcm825x_sensor *sensor = &tcm825x;
	struct i2c_client *client = &sensor->i2c_client;
	int err;

	if (client->adapter)
		return -EBUSY;	/* our client is already attached */

	client->addr = addr;
	client->flags = 0;
	client->driver = sensor->i2c_driver;
	client->adapter = adap;
	strlcpy(client->name, "TCM825x I2C driver", sizeof(client->name));

	err = i2c_attach_client(client);
	if (err) {
		client->adapter = NULL;
		return err;
	}

	if (probe) {
		err = tcm825x_detect(sensor);
		if (err < 0) {
			i2c_detach_client(client);
			client->adapter = NULL;
			return err;
		}
	}
	return 0;
}

/* This function is called by i2c_del_adapter() and i2c_del_driver() 
 * if the adapter or driver with which this I2C client is associated is 
 * removed.  This function unregisters the client via i2c_detach_client().
 * Returns zero if the client is successfully detached, or non-zero 
 * otherwise.
 */
static int tcm825x_i2c_detach_client(struct i2c_client *client)
{
	int err;

	if (!client->adapter)
		return -ENODEV;	/* our client isn't attached */

	err = i2c_detach_client(client);
	client->adapter = NULL;

	return err;
}

/* This function will be called for each registered I2C bus adapter when our 
 * I2C driver is registered via i2c_add_driver().  It will also be called 
 * whenever a new I2C adapter is registered after our I2C driver is registered.
 * This function probes the specified I2C bus adapter to determine if an 
 * TCM825X sensor device is present.  If a device is detected, an I2C client 
 * is registered for it via tcm825x_i2c_attach_client().  Note that we can't use 
 * the standard i2c_probe() function to look for the sensor because the OMAP 
 * I2C controller doesn't support probing.
 * Returns zero if an TCM825X device is detected and an I2C client successfully 
 * registered for it, or non-zero otherwise.
 */
static int tcm825x_i2c_probe_adapter(struct i2c_adapter *adap)
{
	return tcm825x_i2c_attach_client(adap, TCM825X_I2C_ADDR, 1);
}

/* Find the best match for a requested image capture size.  The best match 
 * is chosen as the nearest match that has the same number or fewer pixels 
 * as the requested size, or the smallest image size if the requested size 
 * has fewer pixels than the smallest image.
 */
static enum image_size tcm825x_find_size(unsigned int width, unsigned int height)
{
	enum image_size isize;
	unsigned long pixels = width * height;

	for (isize = subQCIF; isize < VGA; isize++) {
		if (tcm825x_sizes[isize + 1].height * tcm825x_sizes[isize + 1].width > pixels) {
			printk(KERN_DEBUG "%s(): size %d\n", __FUNCTION__, isize);
			return isize;
		}
	}
	
	printk(KERN_DEBUG "%s(): format default VGA\n", __FUNCTION__);
	return VGA;
}

/* following are sensor interface functions implemented by 
 * TCM825X sensor driver.
 */
static int tcm825xsensor_query_control(struct v4l2_queryctrl *qc, void *priv)
{
	struct vcontrol * control;

	control = find_vctrl(qc->id);

	if (control == NULL)
		return -EINVAL;

	*qc = control->qc;

	return 0;
}

static int tcm825xsensor_get_control(struct v4l2_control *vc, void *priv)
{
	struct tcm825x_sensor *sensor = (struct tcm825x_sensor *)priv;
	struct i2c_client *client = &sensor->i2c_client;
	int val, r;
	struct vcontrol *lvc;

	/* exposure time is special, spread accross 2 registers */
	if (vc->id == V4L2_CID_EXPOSURE) {
		int val_lower, val_upper;

		val_upper = tcm825x_read_reg(client,
					     TCM825X_ADDR(TCM825X_ESRSPD_U));
		if (val_upper < 0)
			return val_upper;
		val_lower = tcm825x_read_reg(client,
					     TCM825X_ADDR(TCM825X_ESRSPD_L));
		if (val_lower < 0)
			return val_lower;

		vc->value = ((val_upper & 0x1f) << 8) | (val_lower);
		return 0;
	}

	lvc = find_vctrl(vc->id);
	if (lvc == NULL)
		return -EINVAL;

	r = tcm825x_read_reg(client, TCM825X_ADDR(lvc->reg));
	if (r < 0)
		return r;
	val = r & TCM825X_MASK(lvc->reg);
	val >>= lvc->start_bit;
	
	if (val < 0)
		return val;

	vc->value = val;
	return 0;
}

static int tcm825xsensor_set_control(struct v4l2_control *vc, void *priv)
{
	struct tcm825x_sensor *sensor = (struct tcm825x_sensor *)priv;
	struct i2c_client *client = &sensor->i2c_client;
	struct vcontrol *lvc;
	int val = vc->value;

	/* exposure time is special, spread accross 2 registers */
	if (vc->id == V4L2_CID_EXPOSURE) {
		int val_lower, val_upper;
		val_lower = val & TCM825X_MASK(TCM825X_ESRSPD_L);
		val_upper = (val >> 8) & TCM825X_MASK(TCM825X_ESRSPD_U);

		if (tcm825x_write_reg_mask(client,
					   TCM825X_ESRSPD_U, val_upper))
			return -EIO;

		if (tcm825x_write_reg_mask(client,
					   TCM825X_ESRSPD_L, val_lower))
			return -EIO;

		return 0;
	}

	lvc = find_vctrl(vc->id);
	if (lvc == NULL)
		return -EINVAL;

	val = val << lvc->start_bit;
	if (tcm825x_write_reg_mask(client, lvc->reg, val))
		return -EIO;
	
	return 0;
}

/* Implement the VIDIOC_ENUM_FMT ioctl for the CAPTURE buffer type.
 */
static int tcm825xsensor_enum_pixformat(struct v4l2_fmtdesc *fmt, void *priv)
{
	int index = fmt->index;
	enum v4l2_buf_type type = fmt->type;

	memset(fmt, 0, sizeof(*fmt));
	fmt->index = index;
	fmt->type = type;

	switch (fmt->type) {
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
		if (index >= TCM825X_NUM_CAPTURE_FORMATS)
			return -EINVAL;
		break;

	case V4L2_BUF_TYPE_VIDEO_OVERLAY:
		if (index >= NUM_OVERLAY_FORMATS)
			return -EINVAL;
		break;

	default:
		return -EINVAL;
	}

	fmt->flags = tcm825x_formats[index].flags;
	strlcpy(fmt->description, tcm825x_formats[index].description, sizeof(fmt->description));
	fmt->pixelformat = tcm825x_formats[index].pixelformat;

	return 0;
}

/* Implement the VIDIOC_TRY_FMT ioctl for the CAPTURE buffer type.  This 
 * ioctl is used to negotiate the image capture size and pixel format 
 * without actually making it take effect.
 */
static int tcm825xsensor_try_format(struct v4l2_pix_format *pix, void *priv)
{
	enum image_size isize;
	int ifmt;

	isize = tcm825x_find_size(pix->width, pix->height);
	printk(KERN_DEBUG "%s(): isize = %d num_capture = %d\n", 
	       __FUNCTION__, isize, TCM825X_NUM_CAPTURE_FORMATS);
	
	pix->width = tcm825x_sizes[isize].width;
	pix->height = tcm825x_sizes[isize].height;
	
	for (ifmt = 0; ifmt < TCM825X_NUM_CAPTURE_FORMATS; ifmt++)
		if (pix->pixelformat == tcm825x_formats[ifmt].pixelformat)
			break;
	
	if (ifmt == TCM825X_NUM_CAPTURE_FORMATS)
		ifmt = 0;	/* Default = YUV 4:2:2 */
	
	pix->pixelformat = tcm825x_formats[ifmt].pixelformat;
	pix->field = V4L2_FIELD_NONE;
	pix->bytesperline = pix->width * 2;
	pix->sizeimage = pix->bytesperline * pix->height;
	pix->priv = 0;
	printk(KERN_DEBUG "%s(): format = 0x%08x\n", __FUNCTION__,pix->pixelformat);
	
	switch (pix->pixelformat) {
	case V4L2_PIX_FMT_UYVY:
	default:
		pix->colorspace = V4L2_COLORSPACE_JPEG;
		break;
	case V4L2_PIX_FMT_RGB565:
		pix->colorspace = V4L2_COLORSPACE_SRGB;
		break;
	}
	
	return 0;
}

/* Given the image capture format in pix, the nominal frame period in 
 * timeperframe, calculate the required xclk frequency 
 * The nominal xclk input frequency of the TCM825X is 24MHz, maximum 
 * frequency is 48MHz, and minimum frequency is 10MHz.
 */
static unsigned long tcm825xsensor_calc_xclk(struct v4l2_pix_format *pix,
					     struct v4l2_fract *timeperframe,
					     void *priv)
{
	unsigned long tgt_xclk;	/* target xclk */
	unsigned long tgt_fpm;	/* target frames per minute */
	enum image_size isize;

	/* We use arbitrary rules to select the xclk frequency.  If the 
	 * capture size is VGA and the frame rate is greater than 900 
	 * frames per minute, or if the capture size is SXGA and the 
	 * frame rate is greater than 450 frames per minutes, then the 
	 * xclk frequency will be set to 48MHz.  Otherwise, the xclk 
	 * frequency will be set to 24MHz.  If the mclk frequency is such that 
	 * the target xclk frequency is not achievable, then xclk will be set 
	 * as close as to the target as possible.
	 */
	if ((timeperframe->numerator == 0)
	    || (timeperframe->denominator == 0)) {
		/* supply a default nominal_timeperframe of 15 fps */
		timeperframe->numerator = 1;
		timeperframe->denominator = 15;
	}
	
	tgt_fpm = (timeperframe->denominator * 60)
	    / timeperframe->numerator;
	tgt_xclk = 24000000;
	isize = tcm825x_find_size(pix->width, pix->height);
	
	switch (isize) {
	case VGA:
		if (tgt_fpm > 900)
			tgt_xclk = 24000000;	//48000000;
		break;
	default:
		break;
	}
	
	return tgt_xclk;
}

/* Given a capture format in pix, the frame period in timeperframe, and
 * the xclk frequency, set the capture format of the TCM825X sensor.
 * The actual frame period will be returned in timeperframe.
 */
static int tcm825xsensor_configure(struct v4l2_pix_format *pix,
				   unsigned long xclk,
				   struct v4l2_fract *timeperframe,
				   void *priv)
{
	struct tcm825x_sensor *sensor = (struct tcm825x_sensor *)priv;
	enum pixel_format pfmt = RGB565;

	switch (pix->pixelformat) {
	case V4L2_PIX_FMT_RGB565:
		pfmt = RGB565;
		break;
	case V4L2_PIX_FMT_UYVY:
		pfmt = YUV422;
		break;
	default:
		pfmt = RGB565;
		break;
	}

	return tcm825x_configure(&sensor->i2c_client,
				 tcm825x_find_size(pix->width, pix->height),
				 pfmt, xclk, timeperframe);
}

static int tcm825xsensor_power_on(void *data)
{
	return tcm825x.sensor_config->power_on(NULL);
}

static int tcm825xsensor_power_off(void *data)
{
	return tcm825x.sensor_config->power_off(NULL);
}

/* Prepare for the driver to exit.
 * Balances tcm825xsensor_init().
 * This function must de-initialize the sensor and its associated data 
 * structures.
 */
static int tcm825xsensor_cleanup(void *priv)
{
	struct tcm825x_sensor *sensor = &tcm825x;

	i2c_del_driver(sensor->i2c_driver);

	return 0;
}


/* Initialize the TCM825X sensor.
 * This routine allocates and initializes the data structure for the sensor, 
 * powers up the sensor, registers the I2C driver, and sets a default image 
 * capture format in pix.  The capture format is not actually programmed 
 * into the TCM825X sensor by this routine.
 * This function must return a non-NULL value to indicate that 
 * initialization is successful.
 */
static void *tcm825xsensor_init(struct v4l2_pix_format *pix)
{
	struct tcm825x_sensor *sensor = &tcm825x;

	sensor->i2c_driver->attach_adapter = tcm825x_i2c_probe_adapter;
	sensor->i2c_driver->detach_client = tcm825x_i2c_detach_client;

	if (i2c_add_driver(sensor->i2c_driver)) {
		printk(KERN_ERR "%s(): Failed to register TCM825x I2C client\n", __FUNCTION__);
		goto err_add;
	}

	if (!sensor->i2c_client.adapter) {
		printk(KERN_WARNING "%s(): Failed to detect TCM825x sensor chip\n", __FUNCTION__);
		goto err_detect;
	}

	/* Make the default capture format QVGA RGB565 */
	pix->width = tcm825x_sizes[QVGA].width;
	pix->height = tcm825x_sizes[QVGA].height;
	pix->pixelformat = V4L2_PIX_FMT_RGB565;
	tcm825xsensor_try_format(pix, NULL);

	return (void *)sensor;

err_detect:
	i2c_del_driver(sensor->i2c_driver);
err_add:
	return NULL;
}

struct omap_camera_sensor tcm825x_sensor_if = {
      .version = 0x01,
      .name = "TCM825x",
      .init = tcm825xsensor_init,
      .cleanup = tcm825xsensor_cleanup,
      .enum_pixformat = tcm825xsensor_enum_pixformat,
      .try_format = tcm825xsensor_try_format,
      .calc_xclk = tcm825xsensor_calc_xclk,
      .configure = tcm825xsensor_configure,
      .query_control = tcm825xsensor_query_control,
      .get_control = tcm825xsensor_get_control,
      .set_control = tcm825xsensor_set_control,
};

#ifdef CONFIG_ARCH_OMAP24XX
extern int omap24xxcam_register_sensor(struct omap_camera_sensor *sensor);
#endif
int __init sensor_tcm825x_init(void)
{
	int ret = 0;
	struct tcm825x_sensor *sensor = &tcm825x;

	memset(sensor, 0, sizeof(*sensor));

	sensor->sensor_config = omap_get_config(OMAP_TAG_CAMERA_SENSOR,
						struct omap_camera_sensor_config);
	if (sensor->sensor_config == NULL ||
	    sensor->sensor_config->power_on == NULL ||
	    sensor->sensor_config->power_off == NULL)
		return -ENODEV;

	tcm825x_sensor_if.power_on = tcm825xsensor_power_on;
	tcm825x_sensor_if.power_off = tcm825xsensor_power_off;

	sensor->i2c_driver = &tcm825x_i2c_driver;
	sensor->sensor_if = &tcm825x_sensor_if;

#ifdef CONFIG_ARCH_OMAP24XX
	ret = omap24xxcam_register_sensor(sensor->sensor_if);
	if (ret < 0)
		return ret;
#endif

	return 0;
}

void sensor_tcm825x_cleanup(void)
{
}

late_initcall(sensor_tcm825x_init);
module_exit(sensor_tcm825x_cleanup);
