/*
 * drivers/hwmon/tmp105.c
 *
 * TMP105 temperature sensor chip driver
 *
 * Copyright (C) 2006 Nokia Corporation
 * Mikko Ylinen <mikko.k.ylinen@nokia.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/module.h>
#include <linux/err.h>
#include <linux/i2c.h>
#include <linux/interrupt.h>
#include <linux/device.h>
#include <linux/delay.h>
#include <linux/hwmon.h>
#include <linux/sysfs.h>

#include <asm/mach-types.h>
#include <asm/mach/irq.h>

#include <asm/arch/gpio.h>
#include <asm/arch/board.h>

#define DRIVER_NAME			"tmp105"

#define TMP105_I2C_ADDRESS		0x48

#define TMP105_TEMPERATURE_MAX		2000
#define TMP105_TEMPERATURE_CRITICAL	1280
#define TMP105_TEMPERATURE_CRIT_HYST	1200

#define TMP105_REG_TEMPERATURE		0 /* RO */
#define TMP105_REG_CONFIGURATION	1 /* RW */
#define TMP105_REG_TLOW			2 /* RW */
#define TMP105_REG_THIGH		3 /* RW */

#define TMP105_BIT_SD			(1 << 0) /* shutdown mode */
#define TMP105_BIT_TM			(1 << 1) /* thermostat mode */
#define TMP105_BIT_POL			(1 << 2) /* polarity */
#define TMP105_BIT_F0			(1 << 3) /* fault queue bit 1 */
#define TMP105_BIT_F1			(1 << 4) /* fault queue bit 2 */
#define TMP105_BIT_R0			(1 << 5) /* ADC resolution bit 1 */
#define TMP105_BIT_R1			(1 << 6) /* ADC resolution bit 2 */
#define TMP105_BIT_OS			(1 << 7) /* one shot */

#define TMP105_INIT_BITS		0x36	 /* R0 | F1 | TM | POL */

#define temp2reg(x) ((u16)((((x) & 0xf) << 8) | (((x) & 0xff0) >> 4)))
#define reg2temp(x) ((signed short)(((x) >> 8) | (((x) & 0xff) << 8)) >> 4)

static unsigned short normal_i2c[] = { TMP105_I2C_ADDRESS, I2C_CLIENT_END };

I2C_CLIENT_INSMOD;

struct tmp105_data {
	struct i2c_client	client;
	struct class_device	*class_dev;
	struct work_struct	work;
	unsigned short		irq_pin;
};

static int tmp105_detach_client(struct i2c_client *client);
static int tmp105_attach_adapter(struct i2c_adapter *bus);

static inline int tmp105_write_reg8(struct i2c_client *client, int reg,
				    u8 val)
{
	return i2c_smbus_write_byte_data(client, reg, val);
}

static inline int tmp105_write_reg(struct i2c_client *client, int reg,
				   u16 val)
{
	return i2c_smbus_write_word_data(client, reg, val);
}

static int tmp105_read_reg(struct i2c_client *client, int reg)
{
	int val = i2c_smbus_read_word_data(client, reg);
	if (val < 0)
		dev_err(&client->dev, "%s failed\n", __FUNCTION__);

	return val;
}

static int tmp105_read_temperature(struct i2c_client *client, int *temp)
{
	int val;

	val = tmp105_read_reg(client, TMP105_REG_TEMPERATURE);
	if (val < 0)
		return val;

	*temp = reg2temp(val);

	return 0;
}

static int tmp105_is_hot(struct i2c_client *client)
{
	int temp, ret;

	ret = tmp105_read_temperature(client, &temp);
	if (ret < 0)
		return ret;

	ret = tmp105_read_reg(client, TMP105_REG_THIGH);
	if (ret < 0)
		return ret;

	return temp >= reg2temp(ret);
}

static ssize_t tmp105_temp1_crit_show(struct device *dev,
				      struct device_attribute *attr,
				      char *buf)
{
	struct i2c_client *client = to_i2c_client(dev);
	int val;

	val = tmp105_read_reg(client, TMP105_REG_THIGH);
	if (val < 0)
		return -EAGAIN;

	return sprintf(buf, "%d\n", (reg2temp(val) * 1000) / 16);
}

static ssize_t tmp105_temp1_crit_store(struct device *dev,
				       struct device_attribute *attr,
				       const char *buf, size_t count)
{
	struct i2c_client *client = to_i2c_client(dev);
	int high, ret;

	ret = sscanf(buf, "%d", &high);
	if (ret != 1)
		return -EINVAL;

	high = (high * 16) / 1000;

	ret = tmp105_read_reg(client, TMP105_REG_TLOW);
	if (ret < 0)
		return -EAGAIN;

	if ((high <= reg2temp(ret)) || (high > TMP105_TEMPERATURE_MAX))
		return -EINVAL;

	ret = tmp105_write_reg(client, TMP105_REG_THIGH, temp2reg(high));
	if (ret < 0)
		return -EAGAIN;

	return count;
}

static DEVICE_ATTR(temp1_crit, S_IRUGO | S_IWUSR, tmp105_temp1_crit_show,
						  tmp105_temp1_crit_store);

static ssize_t tmp105_temp1_crit_hyst_show(struct device *dev,
					   struct device_attribute *attr,
					   char *buf)
{
	struct i2c_client *client = to_i2c_client(dev);
	int val;

	val = tmp105_read_reg(client, TMP105_REG_TLOW);
	if (val < 0)
		return -EAGAIN;

	return sprintf(buf, "%d\n", (reg2temp(val)/ 16) * 1000);
}

static ssize_t tmp105_temp1_crit_hyst_store(struct device *dev,
					    struct device_attribute *attr,
					    const char *buf, size_t count)
{
	struct i2c_client *client = to_i2c_client(dev);
	int low, ret;

	ret = sscanf(buf, "%d", &low);
	if (ret != 1)
		return -EINVAL;

	low = (low * 16) / 1000;

	ret = tmp105_read_reg(client, TMP105_REG_THIGH);
	if (ret < 0)
		return -EAGAIN;

	if (low >= reg2temp(ret))
		return -EINVAL;

	ret = tmp105_write_reg(client, TMP105_REG_TLOW, temp2reg(low));
	if (ret < 0)
		return -EAGAIN;

	return count;
}

static DEVICE_ATTR(temp1_crit_hyst, S_IRUGO | S_IWUSR,
		   tmp105_temp1_crit_hyst_show, tmp105_temp1_crit_hyst_store);

static ssize_t tmp105_temp1_input_show(struct device *dev,
				       struct device_attribute *attr,
				       char *buf)
{
	struct i2c_client *client = to_i2c_client(dev);
	int temp, ret;

	ret = tmp105_read_temperature(client, &temp);
	if (ret < 0)
		return -EAGAIN;

	return sprintf(buf, "%d\n", (temp * 1000) / 16);
}

DEVICE_ATTR(temp1_input, S_IRUGO, tmp105_temp1_input_show, NULL);

static ssize_t tmp105_temp1_crit_alarm_show(struct device *dev,
				  struct device_attribute *attr,
				  char *buf)
{
	struct i2c_client *client = to_i2c_client(dev);
	int ret;

	ret = tmp105_is_hot(client);
	if (ret < 0)
		return -EAGAIN;

	return sprintf(buf, "%d\n", ret);
}

DEVICE_ATTR(temp1_crit_alarm, S_IRUGO, tmp105_temp1_crit_alarm_show, NULL);

static inline int tmp105_create_sysfs_files(struct i2c_client *client)
{
	struct device *dev = &client->dev;
	int err;

	err = device_create_file(dev, &dev_attr_temp1_crit);
	if (unlikely(err != 0))
		return err;

	err = device_create_file(dev, &dev_attr_temp1_crit_hyst);
	if (unlikely(err != 0))
		goto fail1;

	err = device_create_file(dev, &dev_attr_temp1_input);
	if (unlikely(err != 0))
		goto fail2;

	err = device_create_file(dev, &dev_attr_temp1_crit_alarm);
	if (unlikely(err != 0))
		goto fail3;

	return 0;

fail3:
	device_remove_file(dev, &dev_attr_temp1_input);
fail2:
	device_remove_file(dev, &dev_attr_temp1_crit_hyst);
fail1:
	device_remove_file(dev, &dev_attr_temp1_crit);

	return err;
}

static void tmp105_work(void *data)
{
	struct tmp105_data *t = data;
	struct i2c_client *client = &t->client;

	/* unconditionally notify userland about alarm */
	sysfs_notify(&client->dev.kobj, NULL, "temp1_crit_alarm");

	return;
}

static irqreturn_t tmp105_irq_handler(int irq, void *chip,
				      struct pt_regs *regs)
{
	struct tmp105_data *data = chip;

	(void)schedule_work(&data->work);

	return IRQ_HANDLED;
}

#if CONFIG_ARCH_OMAP

static int tmp105_powered;

static int tmp105_dev_power(int power)
{
	const struct omap_tmp105_config *info;
	int err;

	info = omap_get_config(OMAP_TAG_TMP105, struct omap_tmp105_config);
	if (info == NULL || info->set_power == NULL || tmp105_powered == power)
		return 0;
	err = info->set_power(power);
	if (err < 0)
		return err;
	tmp105_powered = power;
	return 0;
}

static int tmp105_dev_init(void *data)
{
	struct tmp105_data *chip = data;
	const struct omap_tmp105_config *info;
	int err;

	info = omap_get_config(OMAP_TAG_TMP105, struct omap_tmp105_config);
	if (info == NULL)
		return 0;

	chip->irq_pin = info->tmp105_irq_pin;

	err = omap_request_gpio(chip->irq_pin);
	if (err < 0) {
		printk(KERN_ERR "couldn't get IRQ pin\n");
		return err;
	}

	/* tmp105 gives an interrupt if temperature oversteps defined
	 * range. Reserve a GPIO line and register a handler for tmp105
	 * interrupts. */
	omap_set_gpio_direction(chip->irq_pin, 1);

	err = request_irq(OMAP_GPIO_IRQ(chip->irq_pin), tmp105_irq_handler,
			  IRQF_TRIGGER_RISING, DRIVER_NAME, chip);
	if (err < 0) {
		printk(KERN_ERR "unable to register IRQ handler\n");
		return err;
	}

	return 0;
}

static void tmp105_dev_exit(void *data)
{
	struct tmp105_data *chip = data;
	if (chip->irq_pin) {
		free_irq(OMAP_GPIO_IRQ(chip->irq_pin), chip);
		omap_free_gpio(chip->irq_pin);
	}
}

#else

static int tmp105_dev_init(void *data)
{
	return 0;
}

static void tmp105_dev_exit(void *data)
{
}

static int tmp105_dev_power(int power)
{
	return 0;
}

#endif

static struct i2c_driver tmp105_i2c_driver = {
	.id		= I2C_DRIVERID_MISC,
	.attach_adapter	= tmp105_attach_adapter,
	.detach_client	= tmp105_detach_client,
	.driver = {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
	}
};

static int tmp105_probe(struct i2c_adapter *adapter, int address, int kind)
{
	struct i2c_client *client;
	struct tmp105_data *data;
	int err;

	pr_info(DRIVER_NAME ": found at address 0x%02x\n", address);

	data = kzalloc(sizeof(struct tmp105_data), GFP_KERNEL);
	if (!data)
		return -ENOMEM;

	client = &data->client;

	strncpy(client->name, DRIVER_NAME, sizeof(client->name));
	i2c_set_clientdata(client, data);

	client->addr		= address;
	client->adapter		= adapter;
	client->driver		= &tmp105_i2c_driver;
	client->flags		= 0;

	if ((err = tmp105_dev_init(data)) < 0)
		goto fail1;

	if ((err = i2c_attach_client(client)) < 0) {
		printk(KERN_ERR DRIVER_NAME ": unable to attach I2C client\n");
		goto fail2;
	}

	/* set proper configuration */
	err = tmp105_write_reg8(client, TMP105_REG_CONFIGURATION,
				TMP105_INIT_BITS);
	if (err < 0) {
		printk(KERN_ERR DRIVER_NAME ": failed to configure chip\n");
		goto fail3;
	}

	/* Register sysfs hooks */
	data->class_dev = hwmon_device_register(&client->dev);
	if (IS_ERR(data->class_dev))
		goto fail3;

	/* set up sysfs attributes we provide */
	err = tmp105_create_sysfs_files(client);
	if (err < 0)
		goto fail4;

	INIT_WORK(&data->work, tmp105_work, data);

	msleep(30); /* the first conversion after power up is slooow... */

	err = tmp105_write_reg(client, TMP105_REG_THIGH,
			       temp2reg(TMP105_TEMPERATURE_CRITICAL));
	if (err < 0)
		goto fail4;

	err = tmp105_write_reg(client, TMP105_REG_TLOW,
			       temp2reg(TMP105_TEMPERATURE_CRIT_HYST));
	if (err < 0)
		goto fail4;

	return 0;

fail4:
	hwmon_device_unregister(data->class_dev);
fail3:
	i2c_detach_client(client);
fail2:
	tmp105_dev_exit(data);
fail1:
	kfree(data);

	return err;
}

static int tmp105_detach_client(struct i2c_client *client)
{
	struct tmp105_data *data = i2c_get_clientdata(client);
	struct device *dev = &client->dev;
	int err;

	hwmon_device_unregister(data->class_dev);

	device_remove_file(dev, &dev_attr_temp1_crit);
	device_remove_file(dev, &dev_attr_temp1_crit_hyst);
	device_remove_file(dev, &dev_attr_temp1_input);
	device_remove_file(dev, &dev_attr_temp1_crit_alarm);

	if ((err = i2c_detach_client(client)) < 0) {
		dev_err(&client->dev, "client deregistration failed\n");
		return err;
	}

	tmp105_dev_power(0);
	tmp105_dev_exit(data);

	kfree(data);

	return 0;
}

static int tmp105_attach_adapter(struct i2c_adapter *bus)
{
	int err;

	if (!i2c_check_functionality(bus, I2C_FUNC_SMBUS_WRITE_BYTE_DATA |
				     I2C_FUNC_SMBUS_READ_WORD_DATA |
					  I2C_FUNC_SMBUS_WRITE_WORD_DATA)) {
		printk(KERN_ERR DRIVER_NAME ": missing I2C functionalities\n");
		return -EINVAL;
	}

	err = tmp105_dev_power(1);
	if (err < 0)
		return err;

	return i2c_probe(bus, &addr_data, &tmp105_probe);
}

static int __init tmp105_init(void)
{
	int res;

	pr_info(DRIVER_NAME ": initializing\n");

	if ((res = i2c_add_driver(&tmp105_i2c_driver)) < 0) {
		printk(KERN_ERR DRIVER_NAME ": I2C registration failed\n");
		return res;
	}

	return 0;
}

static void __exit tmp105_exit(void)
{
	if (i2c_del_driver(&tmp105_i2c_driver) < 0)
		printk(KERN_ERR DRIVER_NAME ": driver remove failed\n");
}

MODULE_AUTHOR("Mikko K. Ylinen <mikko.k.ylinen@nokia.com");
MODULE_DESCRIPTION("I2C interface driver for a TMP105 temperature sensor.");
MODULE_LICENSE("GPL");

module_init(tmp105_init);
module_exit(tmp105_exit);
