/*
 * This file is part of libgst0.10-dsp
 *
 * Copyright (C) 2006 Nokia Corporation. All rights reserved.
 *
 * Contact: Stefan Kost <stefan.kost@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <unistd.h>
#include <string.h>
#include <gst/gst.h>

#include "gstdspaacsink.h"

#define AUDIO_SEND_CHUNK_SIZE 4000

#define ADIF_MAX_SIZE 40 /* Should be enough */
#define ADTS_MAX_SIZE 10 /* Should be enough */

static int aac_sample_rates[] = { SAMPLE_RATE_96KHZ,
                                  SAMPLE_RATE_88_2KHZ,
                                  SAMPLE_RATE_64KHZ,
                                  SAMPLE_RATE_48KHZ,
                                  SAMPLE_RATE_44_1KHZ,
                                  SAMPLE_RATE_32KHZ,
                                  SAMPLE_RATE_24KHZ,
                                  SAMPLE_RATE_22_05KHZ,
                                  SAMPLE_RATE_16KHZ,
                                  SAMPLE_RATE_12KHZ,
                                  SAMPLE_RATE_11_025KHZ,
                                  SAMPLE_RATE_8KHZ };

GstElementDetails gst_dspaacsink_details = GST_ELEMENT_DETAILS ("DSP AAC Sink",
    "Sink/Audio",
    "AAC audio sink",
    "Makoto Sugano <makoto.sugano@nokia.com>");

static gchar *devname = "/dev/dsptask/aacdec";

static GstStaticPadTemplate dspaacsink_sink_template =
GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ( "audio/mpeg, "
        "mpegversion = (int) { 2, 4 }, "
        "framed = (bool) false" ) );

static guint gst_dspaacsink_signals[DSPAAC_SIGNAL_LAST] = { 0 };

// Function prototypes
static void gst_dspaacsink_class_init (GstDSPAACSinkClass * klass);
static void gst_dspaacsink_base_init (gpointer g_class);
static void gst_dspaacsink_init (GstDSPAACSink *dspmp3sink, GstDSPAACSinkClass *g_class);
static gboolean gst_dspaacsink_initialize (GstDSPAACSink *dsp);
static GstClockTime gst_dspaacsink_get_time (GstClock * clock, gpointer sink);

// For GObject
static void gst_dspaacsink_set_property (GObject * object, guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_dspaacsink_get_property (GObject * object, guint prop_id, GValue * value, GParamSpec * pspec);

// For GstElement
static const GstQueryType *gst_dspaacsink_get_query_types (GstElement * element);
static gboolean gst_dspaacsink_query (GstElement * element, GstQuery *query);
static GstClock *gst_dspaacsink_provide_clock (GstElement * elem);
static GstStateChangeReturn gst_dspaacsink_change_state (GstElement * element, GstStateChange transition);
// static gboolean gst_dspaacsink_event (GstElement * element, GstEvent * event);

// For GstBaseSink
static GstFlowReturn gst_dspaacsink_render (GstBaseSink *sink, GstBuffer * buffer);
static gboolean gst_dspaacsink_setcaps (GstBaseSink *sink, GstCaps *caps);
static GstFlowReturn gst_dspaacsink_preroll (GstBaseSink *sink, GstBuffer * buffer);
static gboolean gst_dspaacsink_event2 (GstBaseSink *sink, GstEvent *event);


/**
 *
 *
 */

void _do_init()
{
  DBG_PRINT("AAC DO_INIT\n");
}

GST_BOILERPLATE_FULL (GstDSPAACSink, gst_dspaacsink, GstBaseSink,
                      GST_TYPE_BASE_SINK, _do_init);


/**
 * gst_dspaacsink_dispose:
 * @object: GObject pointer to element to be deleted
 *
 * Deletes AAC sink element instance. Called automatically by
 * GLib framework when element needs to be disposed.
 */

static void
gst_dspaacsink_dispose (GObject * object)
{
  GstDSPAACSink *dsp = (GstDSPAACSink *) object;

  if(dsp->audio->clock) {
    gst_object_unref(dsp->audio->clock);
    dsp->audio->clock = NULL;
  }

  if(dsp->audio) {
    gst_dspaudio_destroy(dsp->audio);
    dsp->audio = NULL;
  }
  g_object_unref(dsp->adapter);

  G_OBJECT_CLASS (parent_class)->dispose (object);
}


/**
 * gst_dspaacsink_base_init:
 * @g_class: AAC sink class
 *
 * Does the basic initialization of AAC sink element class. This is
 * called by GStreamer framework.
 */

static void
gst_dspaacsink_base_init (gpointer g_class)
{
  GstElementClass *gstelement_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_add_pad_template (gstelement_class,
      gst_static_pad_template_get (&dspaacsink_sink_template));
  gst_element_class_set_details (gstelement_class, &gst_dspaacsink_details);

}

/**
 * gst_dspaacsink_class_init:
 * @klass: AAC Sink class object
 *
 * Initializes AAC sink element class. This is called by GStreamer
 * framework.
 */

static void
gst_dspaacsink_class_init (GstDSPAACSinkClass * klass)
{
  GObjectClass *gobject_class;
  GstElementClass *gstelement_class;
  GstBaseSinkClass *gstbase_sink_class;

  gobject_class = (GObjectClass *) klass;
  gstelement_class = (GstElementClass *) klass;
  gstbase_sink_class = (GstBaseSinkClass *) klass;

  gobject_class->set_property = gst_dspaacsink_set_property;
  gobject_class->get_property = gst_dspaacsink_get_property;
  gobject_class->dispose = gst_dspaacsink_dispose;

  gstelement_class->provide_clock = GST_DEBUG_FUNCPTR (gst_dspaacsink_provide_clock);
  gstelement_class->change_state = gst_dspaacsink_change_state;
  gstelement_class->get_query_types = gst_dspaacsink_get_query_types;
//   gstelement_class->send_event = gst_dspaacsink_event;
  gstelement_class->query = gst_dspaacsink_query;

   // Use dspaudio utility to install default properties
  gst_dspaudio_install_properties(G_OBJECT_CLASS (klass));

  gst_dspaacsink_signals[DSPAAC_SIGNAL_STREAMID] =
      g_signal_new ("stream_id", G_TYPE_FROM_CLASS (klass), G_SIGNAL_RUN_FIRST,
                    G_STRUCT_OFFSET (GstDSPAACSinkClass, stream_id), NULL, NULL,
                    g_cclosure_marshal_VOID__INT, G_TYPE_NONE, 1, G_TYPE_INT);

  gstbase_sink_class->render = GST_DEBUG_FUNCPTR (gst_dspaacsink_render);
  gstbase_sink_class->set_caps = GST_DEBUG_FUNCPTR (gst_dspaacsink_setcaps);
  gstbase_sink_class->event = GST_DEBUG_FUNCPTR (gst_dspaacsink_event2);
  gstbase_sink_class->preroll = GST_DEBUG_FUNCPTR(gst_dspaacsink_preroll);
}


/**
 * gst_dspaacsink_init:
 * @dsp: DSP AAC sink object
 *
 * Initializes the given AAC sink element instance and allocates required
 * resources. This also creates an audio stream object. This is
 * called by GStreamer framework.
 */

static void
gst_dspaacsink_init (GstDSPAACSink * dspaacsink, GstDSPAACSinkClass *g_class)
{
  dspaacsink->audio = gst_dspaudio_new();
  dspaacsink->adapter = gst_adapter_new();
  dspaacsink->audio->clock = gst_audio_clock_new ("clock",
                                                  (GstAudioClockGetTimeFunc) gst_dspaacsink_get_time,
                                                  dspaacsink);
}


/**
 * gst_dspaacsink_handle_sink_event:
 * @dsp: DSP AAC sink object
 *
 * Handles an incoming event by reading the incoming data
 * stream.
 *
 * Returns: TRUE if event was not EOS, otherwise FALSE
 */

static gboolean
  gst_dspaacsink_send_remaining (GstDSPAACSink *dsp)
{
  if(dsp->audio->discont_sent == FALSE) {
    DSPWaitStatus status;
    short int tmp[2];
    const guint8 *data;

    guint avail = gst_adapter_available(dsp->adapter);
    DBG_PRINT("AAC remaining: %d bytes\n", avail);

    // It does not matter if avail == 0
    tmp[0] = DSP_CMD_EOF;
    tmp[1] = avail/2;

    // FIXME: What if avail > mmap_buffer_size?

    // Send remaining bytes to DSP and wait for EOF signal
    do {
      g_mutex_lock(dsp->audio->dsp_mutex);
      status = gst_dspaudio_wait_buffer(dsp->audio);
      g_mutex_unlock(dsp->audio->dsp_mutex);
    }
    while (status == DSP_WAIT_RETRY);

    if(status == DSP_WAIT_OK) {
      g_mutex_lock(dsp->audio->dsp_mutex);
      DBG_PRINT("Got EOS data write\n");
      if(avail) {
        data = gst_adapter_peek(dsp->adapter, avail);
        memcpy(dsp->audio->codec.mmap_buffer, data, avail);
      }
      write(dsp->audio->codec.fd, tmp, 2*sizeof(short int));
      DBG_PRINT("Wrote CMD_EOF (%d bytes)\n", avail);
      gst_dspaudio_wait_eos(dsp->audio);
      DBG_PRINT("GOT EOF REPLY\n");
      dsp->audio->rw_pending = 0;
      dsp->audio->mode = DSP_MODE_EOS;
      g_mutex_unlock(dsp->audio->dsp_mutex);
    }
    gst_adapter_flush(dsp->adapter, avail);
  }
  else {
    DBG_PRINT("DISCONT SENT TO DSP -> NOT SENDING EOF\n");
  }

  return TRUE;
}


/**
 * gst_dspaacsink_setcaps:
 * @sink: GstBaseSink
 * @caps: GStCaps offered
 *
 * Returns: TRUE if caps are accepted
 */

static gboolean
gst_dspaacsink_setcaps(GstBaseSink *sink, GstCaps *caps)
{
  DBG_PRINT("gst_dspaacsink_setcaps\n");
  GstDSPAACSink *dsp = GST_DSPAACSINK(sink);
  GstStructure *structure = gst_caps_get_structure (caps, 0);
  const gchar *mimetype = gst_structure_get_name (structure);
  gint mpegversion;

  if(strncmp(mimetype, "audio/mpeg", 10)) {
    return FALSE;
  }

  if(!gst_structure_has_field(structure, "mpegversion" )) {
    return FALSE;
  }

  gst_structure_get_int (structure, "mpegversion", &mpegversion);

  if(mpegversion != 2 && mpegversion != 4) {
    return FALSE;
  }

  // 3GP demuxer offers us this information
  if(gst_structure_has_field(structure, "rate_index" ) &&
     gst_structure_has_field(structure, "channels" ) &&
     gst_structure_has_field(structure, "object_type" ))
  {
    gst_structure_get_int (structure, "rate_index", &(dsp->sample_rate));
    gst_structure_get_int (structure, "channels", &(dsp->channels));
    gst_structure_get_int (structure, "object_type", &(dsp->object_type));
    dsp->got_stream_info = TRUE;
  }

  if(dsp->sample_rate < 3) {
    GST_ELEMENT_ERROR (dsp, STREAM, CODEC_NOT_FOUND, (NULL),
                       ("illegal samplerate"));
    return FALSE;
  }

  g_mutex_lock(dsp->audio->dsp_mutex);
  gboolean ret = gst_dspaacsink_initialize(dsp);
  g_mutex_unlock(dsp->audio->dsp_mutex);

  if(!ret) {
    GST_ELEMENT_ERROR (dsp, RESOURCE, SETTINGS, (NULL),
                       ("gst_dspaacsink_initialize"));
    return FALSE;
  }

  return TRUE;
}


/**
 * gst_dspaacsink_get_stream_details:
 * @dsp: DSP AAC sink object
 *
 * Parses the incoming AAC data stream and extracts
 * stream information from it. This information is then
 * stored into DSP sink object itself.
 *
 * Returns: TRUE if the info was found, otherwise FALSE
 */


static gboolean
gst_dspaacsink_get_stream_details(GstDSPAACSink *dsp)
{
  gboolean retval = FALSE;
  gboolean found = FALSE;
  const guint8 *buffer;
  guint i = 0;

  guint64 avail = gst_adapter_available(dsp->adapter);

  dsp->header_type = DSPAAC_HEADER_UNKNOWN;
  buffer = gst_adapter_peek(dsp->adapter, avail);

  for(i=0; i < avail - ADIF_MAX_SIZE; i++) {

    if (memcmp (&(buffer[i]), "ADIF", 4) == 0) {
      int skip_size = 0;
      int bitstream;
      int sf_idx;

  //     DBG_PRINT("ADIF format detected!\n");
      dsp->header_type = DSPAAC_HEADER_ADIF;

      // Skip the "ADIF" bytes
      const guint8 *adif = &(buffer[i+4]);

      /* copyright string */
      if(adif[0] & 0x80)
        skip_size += 9; /* skip 9 bytes */

      bitstream = adif[0 + skip_size] & 0x10;
      dsp->bitrate = ((unsigned int)(adif[0 + skip_size] & 0x0F)<<19) |
                     ((unsigned int)adif[1 + skip_size]<<11) |
                     ((unsigned int)adif[2 + skip_size]<<3) |
                     ((unsigned int)adif[3 + skip_size] & 0xE0);

      if (bitstream == 0) {
        dsp->object_type = ((adif[6 + skip_size]&0x01)<<1)|((adif[7 + skip_size]&0x80)>>7);
        sf_idx = (adif[7 + skip_size]&0x78)>>3;
      }
      else {
        dsp->object_type = (adif[4 + skip_size] & 0x18)>>3;
        sf_idx = ((adif[4 + skip_size] & 0x07)<<1)|((adif[5 + skip_size] & 0x80)>>7);
      }
      dsp->sample_rate = aac_sample_rates[sf_idx];

      // FIXME: Can we assume this?
      dsp->channels = 2;
  //    dsp->object_type = 1;

      DBG_PRINT("ADIF: br=%d, samplerate=%d, objtype=%d\n",
                dsp->bitrate, dsp->sample_rate, dsp->object_type);

      found = TRUE;
      break;
    }

    else if ((buffer[i] == 0xFF) && ((buffer[i+1] & 0xF6) == 0xF0)) {
      gint sf_idx;
      gint len;

      DBG_PRINT("ADTS ID: %d (position %d)\n", (buffer[i+1] & 0x08)>>3, i);
      len = ((buffer[i + 3] & 0x03) << 11) |
          (buffer[i + 4] << 3) | ((buffer[i + 5] & 0xe0) >> 5);

      DBG_PRINT("ADTS FRAME LEN: %d bytes\n", len);
      if(i + len + 1 > avail) {
        break;
      }

      if ((buffer[i+len] == 0xFF) && ((buffer[i+len+1] & 0xF6) == 0xF0)) {
        DBG_PRINT("ANOTHER ADTS FRAME FOUND\n");
        dsp->header_type = DSPAAC_HEADER_ADTS;
        dsp->object_type = (buffer[i+2] & 0xC0)>>6;
        sf_idx = (buffer[i+2] & 0x3C)>>2;
        dsp->sample_rate = aac_sample_rates[sf_idx];
        dsp->channels = ((buffer[i+2] & 0x01)<<2) | ((buffer[i+3] & 0xC0)>>6);
        dsp->bitrate = ((buffer[i+5] & 0x1F)<<6) | ((buffer[i+6] & 0xFC)>>2);
        DBG_PRINT("ADTS: samplerate %d, channels %d, bitrate %d, objtype %d\n",
                  dsp->sample_rate, dsp->channels, dsp->bitrate, dsp->object_type );

        gst_adapter_flush(dsp->adapter, i);
        found = TRUE;
        break;
      }
    }
  }

  if(found) {
    if(dsp->object_type == 1 || dsp->object_type == 3) {
      dsp->sync = TRUE;
      retval = TRUE;
    }
  }
/*  else {
    DBG_PRINT("RAW AAC format. Using following parameters:\n");
    DBG_PRINT("Samplerate: %d\nChannels: %d\nObjType: %d\n",
              dsp->sample_rate, dsp->channels, dsp->object_type);

    dsp->header_type = DSPAAC_HEADER_NONE;
    dsp->bitrate = 128000;
    retval = TRUE;
  }*/
  return retval;
}


/**
 *
 *
 */

static gboolean
gst_dspaacsink_find_sync(GstDSPAACSink *dsp)
{
  const guint8 *data = NULL;

  if(dsp->header_type != DSPAAC_HEADER_ADTS) {
    DBG_PRINT("CANNOT FIND SYNC - NOT ADTS STREAM\n");
    return FALSE;
  }

  int counter = 0;
  while( TRUE ) {

    if(gst_adapter_available(dsp->adapter) < 2) {
      DBG_PRINT("CANNOT FIND SYNC WITHIN %d bytes\n", counter);
      return FALSE;
    }
    data = gst_adapter_peek(dsp->adapter, 2);

    // Check for ADTS header
    if ((data[0] == 0xFF) && ((data[1] & 0xF6) == 0xF0)) {
      DBG_PRINT("ADTS SYNC FOUND\n");
      break;
    }
    gst_adapter_flush(dsp->adapter, 1);
    counter++;
  }

  return TRUE;
}


/**
 * gst_dspaacsink_send_discont:
 * @dsp: DSP AAC sink object
 *
 * Send a DISCONT command to DSP
 */

static void
gst_dspaacsink_send_discont(GstDSPAACSink *dsp)
{
  if(gst_dspaudio_discont(dsp->audio)) {
    if(gst_dspaudio_read_cmd(dsp->audio, 0) != DSP_WAIT_OK) {
      GST_ELEMENT_ERROR (dsp, RESOURCE, SEEK, (NULL), ("discont"));
    }
    DBG_PRINT("GOT REPLY\n");
  }
}


/**
 * gst_dspaacsink_render:
 * @pad:
 * @buffer:
 *
 */

#define AAC_CHUNK_SIZE 500
#define AAC_SYNC_SIZE 100000
#define AAC_DETECT_BLOCK_SIZE 4096

static GstFlowReturn
gst_dspaacsink_render(GstBaseSink *sink, GstBuffer * buffer)
{
  GstFlowReturn ret = GST_FLOW_OK;
  DSPWaitStatus status = DSP_WAIT_ERROR;
  GstDSPAACSink *dsp = GST_DSPAACSINK (sink);
  const guint8 *data;
  guint avail;

//   DBG_PRINT("AAC RENDER: %d bytes\n", GST_BUFFER_SIZE(buffer));

  gst_buffer_ref(buffer);
  gst_adapter_push (dsp->adapter, buffer);

  if(dsp->header_type == DSPAAC_HEADER_NOT_PARSED) {

    g_mutex_lock(dsp->audio->dsp_mutex);

    if(!dsp->got_stream_info) {
      // Don't even try to detect the stream before enough data is
      // available
      if(gst_adapter_available(dsp->adapter) < AAC_DETECT_BLOCK_SIZE) {
        g_mutex_unlock(dsp->audio->dsp_mutex);
        return GST_FLOW_OK;
      }

      if(!gst_dspaacsink_get_stream_details(dsp)) {
        g_mutex_unlock(dsp->audio->dsp_mutex);
        GST_ELEMENT_ERROR (sink, STREAM, CODEC_NOT_FOUND, (NULL),
                          ("gst_dspaacsink_get_stream_details"));
        DBG_PRINT("RENDER RETURNING GST_FLOW_ERROR 1\n");
        return GST_FLOW_ERROR;
      }
      if(!gst_dspaacsink_initialize(dsp)) {
        g_mutex_unlock(dsp->audio->dsp_mutex);
        GST_ELEMENT_ERROR (dsp, RESOURCE, SETTINGS, (NULL),
                           ("gst_dspaacsink_initialize"));
        return GST_FLOW_ERROR;
      }
    }
    else {
      DBG_PRINT("NOT NEED TO EXAMINE STREAM\n");
      dsp->header_type = DSPAAC_HEADER_NONE;
    }

  // Settings might have changed between PAUSE->PLAY transition
    gst_dspaudio_update_dsp_settings(dsp->audio);
    gst_dspaudio_play(dsp->audio);
    g_mutex_unlock(dsp->audio->dsp_mutex);
  }

  avail = gst_adapter_available(dsp->adapter);

  while(avail > AAC_CHUNK_SIZE) {

    if(!dsp->streamid_signal_sent) {
      g_signal_emit (G_OBJECT (dsp), gst_dspaacsink_signals[DSPAAC_SIGNAL_STREAMID],
                    0, dsp->audio->codec.stream_id);
      dsp->streamid_signal_sent = TRUE;
    }

    // Syncing issues -----------------------------------------------
    if(!dsp->sync && dsp->header_type == DSPAAC_HEADER_ADTS) {
      DBG_PRINT("RENDER SYNCING...\n");

      if(avail < AAC_SYNC_SIZE) {
        return GST_FLOW_OK;
      }

      if(!gst_dspaacsink_find_sync(dsp)) {
        DBG_PRINT("RENDER RETURNING GST_FLOW_ERROR 2\n");
        GST_ELEMENT_ERROR (sink, STREAM, FAILED, (NULL),
                           ("gst_dspaacsink_find_sync"));
        return GST_FLOW_ERROR;
      }

      dsp->sync = TRUE;
      DBG_PRINT("RENDER SYNCING OK\n");
      return GST_FLOW_OK;
    }

    g_mutex_lock(dsp->audio->dsp_mutex);

    // Wait for a permission to write data --------------------------
    status = gst_dspaudio_wait_buffer(dsp->audio);

    if(status == DSP_WAIT_RETRY) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
//       DBG_PRINT("AAC RENDER DSP_WAIT_RETRY\n");
      continue;
    }
    if(status == DSP_WAIT_INTERRUPT) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
      DBG_PRINT("AAC RENDER DSP_WAIT_INTERRUPT\n");
      break;
    }
    else if(status == DSP_WAIT_ERROR) {
      g_mutex_unlock(dsp->audio->dsp_mutex);
      GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE, (NULL),
                         ("loop1: error cmd: %d status: %d",
                          dsp->audio->error_cmd,
                          dsp->audio->error_status));
      ret = GST_FLOW_ERROR;
      break;
    }

//     dsp->audio->rw_pending = 1;

  // DSP_ERROR_SYNC and DSP_ERROR_STREAM errors can happen without
  // return value "DSP_WAIT_ERROR" in wait_buffer()
    if(dsp->audio->error_status) {
      if(dsp->header_type != DSPAAC_HEADER_ADTS) {
        GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE, (NULL),
                           ("loop2: error cmd: %d status: %d",
                            dsp->audio->error_cmd,
                            dsp->audio->error_status));
        ret = GST_FLOW_ERROR;
      }

      DBG_PRINT("RENDER() DETECTED ERROR SITUATION\n");
      dsp->audio->error_status = 0;
      dsp->audio->discont_sent = TRUE; // DSP has just been reset, no need for discont
      dsp->sync = FALSE;
      g_mutex_unlock(dsp->audio->dsp_mutex);
      break; // Cannot continue, not enough data to find sync
    }


    data = gst_adapter_peek(dsp->adapter, AAC_CHUNK_SIZE);
//     DBG_PRINT("AAC: Got %d bytes\n", AAC_CHUNK_SIZE);
    avail -= AAC_CHUNK_SIZE;

    dsp->outbuf.data_size = AAC_CHUNK_SIZE / sizeof(short int);
    memcpy(dsp->audio->codec.mmap_buffer, data, AAC_CHUNK_SIZE);
    gst_adapter_flush(dsp->adapter, AAC_CHUNK_SIZE);

    if(write(dsp->audio->codec.fd, &(dsp->outbuf), sizeof(AAC_DATA_WRITE)) < 0) {
      dsp->audio->mode = DSP_MODE_ERROR;
      GST_ELEMENT_ERROR (dsp, RESOURCE, WRITE, (NULL), ("write = -1"));
      g_mutex_unlock(dsp->audio->dsp_mutex);
      ret = GST_FLOW_ERROR;
      break;
    }

//     DBG_PRINT("AAC: Wrote %d bytes to DSP\n", dsp->outbuf.data_size * sizeof(short int));
    dsp->bytes_written += AAC_CHUNK_SIZE;
    dsp->audio->rw_pending = 0;
    dsp->audio->discont_sent = FALSE;
    dsp->sync = TRUE;
    g_mutex_unlock(dsp->audio->dsp_mutex);
  }

  return ret;
}


/**
 * gst_dspaacsink_initialize:
 * @dsp: DSP AAC sink object
 *
 * Initializes the underlying audio stream object and sets up the
 * AAC audio parameters in DSP.
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */

static gboolean
gst_dspaacsink_initialize(GstDSPAACSink *dsp)
{
  AUDIO_PARAMS_DATA init_data;
  DSP_CMD_STATUS cmd_status;
  short int stat_buf[3];

  init_data.dsp_cmd = DSP_CMD_SET_PARAMS;
  init_data.sample_rate = dsp->sample_rate;
  init_data.number_channels = dsp->channels;
  init_data.stream_priority = dsp->audio->priority;
  init_data.ds_stream_ID = 0;

  // Read the junk out...
  gst_dspaudio_flush(&dsp->audio->codec);

  if(!gst_dspaudio_setparams(dsp->audio, (char *)&init_data, sizeof(AUDIO_PARAMS_DATA))) {
    dsp->audio->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  stat_buf[0] = MP4AACDEC_SET_STATUS;
  stat_buf[1] = dsp->sample_rate;

  // 0 - Main Profile
  // 1 - LC Profile
  // 2 - SSR Profile
  // 3 - LTP profile
  stat_buf[2] = dsp->object_type;

  if(write(dsp->audio->codec.fd, stat_buf, 3 * sizeof(short int)) < 0) {
    dsp->audio->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  int done = read(dsp->audio->codec.fd, &cmd_status, sizeof(DSP_CMD_STATUS));
  if(done == -1 || cmd_status.status != DSP_OK) {
    dsp->audio->mode = DSP_MODE_ERROR;
    return FALSE;
  }

  DBG_PRINT("DSP AAC configured\n");
  return TRUE;
}


/**
 * gst_dspaacsink_preroll:
 * @sink:
 * @buffer:
 *
 * This method is automatically called by GStreamer framework, when
 * a scheduler decides to give execution to this element.
 * The purpose is to check the media stream.
 */

static GstFlowReturn
gst_dspaacsink_preroll (GstBaseSink *sink, GstBuffer * buffer)
{
  GstDSPAACSink *dsp = GST_DSPAACSINK (sink);

  DBG_PRINT("gst_dspaacsink_preroll\n");

  // We want this to run only once (to initialize the codec)
  if( dsp->audio->mode == DSP_MODE_UNINITIALIZED ) {
    DBG_PRINT("gst_dspaacsink_preroll try to init DSP\n");

    gboolean ret = FALSE;

    if(dsp->got_stream_info == FALSE && buffer) {
      gst_buffer_ref(buffer);
      gst_adapter_push (dsp->adapter, buffer);
      ret = gst_dspaacsink_get_stream_details(dsp);
      gst_adapter_clear (dsp->adapter);
    }

    if(dsp->got_stream_info || ret) {
      DBG_PRINT("AAC INITIALIZED IN PREROLL\n");
      g_mutex_lock(dsp->audio->dsp_mutex);
      if(!gst_dspaacsink_initialize(dsp)) {
        g_mutex_unlock(dsp->audio->dsp_mutex);
        GST_ELEMENT_ERROR (dsp, RESOURCE, SETTINGS, (NULL),
                           ("gst_dspaacsink_initialize"));
        return GST_FLOW_ERROR;
      }
    }
  }
  return GST_FLOW_OK;
}


/**
 * gst_dspaacsink_set_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Stores the property value into GObject
 */

static void
gst_dspaacsink_set_property(GObject * object,
                            guint prop_id,
                            const GValue * value,
                            GParamSpec * pspec)
{
  GstDSPAACSink *dsp = GST_DSPAACSINK (object);

  if(!gst_dspaudio_set_property(dsp->audio, prop_id, value)) {
    g_warning("trying to set illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}


/**
 * gst_dspaacsink_get_property:
 * @object: GObject where the property should be retrieved
 * @prop_id: Unique identifier of the desired property
 * @value: Pointer to a variable where the value should be put
 * @pspec: Parameter specification
 *
 * Retrieves the desired parameter value from object
 */

static void
gst_dspaacsink_get_property(GObject * object,
                            guint prop_id,
                            GValue * value,
                            GParamSpec * pspec)
{
  GstDSPAACSink *dsp = GST_DSPAACSINK (object);

  if(!gst_dspaudio_get_property(dsp->audio, prop_id, value)) {
    g_warning("trying to get illegal property");
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
  }
}

/**
 * gst_dspaacsink_get_query_types:
 * @element: GstElement that received this query
 *
 * Returns: a list of supported query types
 */

static const GstQueryType *
gst_dspaacsink_get_query_types(GstElement * element)
{
  static const GstQueryType gst_dspaacsink_query_types[] = {
    GST_QUERY_POSITION,
    GST_QUERY_DURATION,
    0
  };

  return gst_dspaacsink_query_types;
}


/**
 * gst_dspaacsink_query:
 * @element: GstElement that received this query
 * @type: Type of the query, e.g. TOTAL, POSITION
 * @format: The format of the response, e.g. BYTES, TIME
 * @value: Pointer where the return value should be stored
 *
 * Returns: TRUE, iff the query was successful.
 */

static gboolean
gst_dspaacsink_query (GstElement * element, GstQuery *query)
{
  GstBaseSink *bsink = GST_BASE_SINK(element);
  GstQueryType type = GST_QUERY_TYPE (query);
  gboolean res = FALSE;
  gint64 value;

  if(type == GST_QUERY_POSITION) {
    // We can only hope that neighbor element (demux?) knows this
    if(gst_dspaudio_query_peer_position(bsink->sinkpad, &value)) {
      gst_query_set_position (query, GST_FORMAT_TIME, value);
      res = TRUE;
    }
  }
  else if(type == GST_QUERY_DURATION) {
    // We can only hope that neighbor element (demux?) knows this
    if(gst_dspaudio_query_peer_duration(bsink->sinkpad, &value)) {
      gst_query_set_duration (query, GST_FORMAT_TIME, value);
      res = TRUE;
    }
  }

/*  else if(   type == GST_QUERY_POSITION &&
     *format == GST_FORMAT_PERCENT) {

    guint64 currpos = gst_bytestream_tell(dsp->bs);
    guint64 total_bytes = gst_bytestream_length(dsp->bs);

    if(total_bytes != 0 && currpos != -1) {
      *value = 100 * currpos / total_bytes;
      res = TRUE;
    }
  }*/
  return res;
}


/**
 * gst_dspaacsink_event:
 * @element: GstElement that received this event
 * @event: The GstEvent structure itself
 *
 * Returns: TRUE, iff the event was successful
 *
 * This handles events that are directly sent to this element
 */

// static gboolean
// gst_dspaacsink_event(GstElement * element, GstEvent * event)
// {
//   GstDSPAACSink *dsp = GST_DSPAACSINK (element);
//   gboolean res = FALSE;
//
//   GstEventType type = event
//                     ? GST_EVENT_TYPE (event)
//                     : GST_EVENT_UNKNOWN;
//
//   GST_DEBUG ("dsp: event %p %d", event, type);
//
//   //   FIXME: Is it possible to seek AAC stream?
//   if(type == GST_EVENT_SEEK) {
//       DBG_PRINT("AAC SEEK");
//   }
//   gst_event_unref (event);
//   return res;
// }


/**
 *
 * This method handles events coming from basesink
 */

static gboolean gst_dspaacsink_event2 (GstBaseSink *sink, GstEvent * event)
{
  DBG_PRINT("^^^^^^^^^^^^^^^^^^^GOT EVENT FROM BASESINK: %d\n", GST_EVENT_TYPE (event));
  GstDSPAACSink *dsp = GST_DSPAACSINK (sink);

  GstEventType type = event ? GST_EVENT_TYPE (event) : GST_EVENT_UNKNOWN;
  if(type == GST_EVENT_EOS) {
    DBG_PRINT("EOS EVENT\n");
    dsp->audio->codec.stream_id = 0;
    g_signal_emit (G_OBJECT (sink), gst_dspaacsink_signals[DSPAAC_SIGNAL_STREAMID], 0, 0);
    if(!dsp->sync) {
      DBG_PRINT("EOS WITHOUT SYNC\n");
      GST_ELEMENT_ERROR (sink, STREAM, FAILED, (NULL),
                         ("gst_dspaacsink_find_sync"));
    }
    else {
      gst_dspaacsink_send_remaining(dsp);
    }
    dsp->audio->mode = DSP_MODE_EOS;
  }
  else if(type == GST_EVENT_NEWSEGMENT)
  {
    DBG_PRINT("NEWSEGMENT EVENT\n");
    gst_adapter_clear(dsp->adapter);
  }
  else if (type == GST_EVENT_FLUSH_START ) {
    DBG_PRINT("AAC FLUSH_START\n");
    g_mutex_lock(dsp->audio->dsp_mutex);
    gst_dspaacsink_send_discont(dsp);
    g_mutex_unlock(dsp->audio->dsp_mutex);
  }

  return TRUE;
}



/**
 * gst_dspaacsink_change_state:
 * @element: The GstElement whose state to be changed
 *
 * Changes the state of given element. The desired state information
 * is contained by the element itself, and this method is just
 * a callback, so the plugin knows when the state is changing.
 * This method is called by GStreamer framework
 *
 * Returns: TRUE if operation was successful, otherwise FALSE
 */

static GstStateChangeReturn
gst_dspaacsink_change_state(GstElement * element, GstStateChange transition)
{
  GstDSPAACSink *dspaacsink = GST_DSPAACSINK (element);
  GstStateChangeReturn ret;

  gst_dspaudio_clock_change_state(dspaacsink->audio, transition);

  switch (transition) {

    case GST_STATE_CHANGE_NULL_TO_READY:
      DBG_PRINT("AAC NULL TO READY\n");
      dspaacsink->bytes_written = 0;
      dspaacsink->sync = TRUE;
      dspaacsink->outbuf.dsp_cmd = DSP_CMD_DATA_WRITE;
      dspaacsink->header_type = DSPAAC_HEADER_NOT_PARSED;
      dspaacsink->got_stream_info = FALSE;
      dspaacsink->streamid_signal_sent = FALSE;

      // Set default (guessed) stream values
      dspaacsink->sample_rate = SAMPLE_RATE_44_1KHZ;
      dspaacsink->channels = 2;
      dspaacsink->object_type = 1;

      if (gst_dspaudio_open(dspaacsink->audio, devname) == FALSE) {
        GST_ELEMENT_ERROR (element, RESOURCE,
                           OPEN_READ_WRITE, (NULL),
                           ("gst_dspaudio_open"));
        return GST_STATE_CHANGE_FAILURE;
      }
      break;

    case GST_STATE_CHANGE_READY_TO_PAUSED:
      DBG_PRINT("AAC READY TO PAUSED\n");
      gst_adapter_clear(dspaacsink->adapter);
      break;

    case GST_STATE_CHANGE_PAUSED_TO_PLAYING:
      g_mutex_lock(dspaacsink->audio->dsp_mutex);
      DBG_PRINT("AAC PAUSED TO PLAYING. MODE = %d\n", dspaacsink->audio->mode);

      if(dspaacsink->audio->mode == DSP_MODE_PAUSED) {
        // Flush node
        gst_dspaudio_flush(&dspaacsink->audio->codec);
        gst_dspaudio_play(dspaacsink->audio);
        dspaacsink->audio->rw_pending = 0;
      }
      DBG_PRINT("AAC PAUSED TO PLAYING OK\n");
      g_mutex_unlock(dspaacsink->audio->dsp_mutex);
      break;

    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      DBG_PRINT("AAC PLAYING TO PAUSED V1\n");
      gst_dspaudio_interrupt_render(dspaacsink->audio);
      break;

    default:
      break;
  }

  ret = GST_ELEMENT_CLASS (parent_class)->change_state (element, transition);

  switch (transition) {

    case GST_STATE_CHANGE_PLAYING_TO_PAUSED:
      gst_dspaudio_remove_interrupt(dspaacsink->audio);
      g_mutex_lock(dspaacsink->audio->dsp_mutex);
      DBG_PRINT("AAC PLAYING TO PAUSED V2\n");

      // FIXME: Send PAUSE only if we are NOT IN ERROR STATE
      if(gst_dspaudio_pause(dspaacsink->audio)) {
        gst_dspaudio_read_cmd(dspaacsink->audio, 0);
        dspaacsink->audio->rw_pending = 0;
      }
      DBG_PRINT("AAC PLAYING TO PAUSED V2 OK\n");
      g_mutex_unlock(dspaacsink->audio->dsp_mutex);
      break;

    case GST_STATE_CHANGE_PAUSED_TO_READY:
      DBG_PRINT("AAC PAUSED TO READY\n");
      g_mutex_lock(dspaacsink->audio->dsp_mutex);
      if(gst_dspaudio_stop(dspaacsink->audio)) {
        gst_dspaudio_read_cmd(dspaacsink->audio, 0);
      }
      g_mutex_unlock(dspaacsink->audio->dsp_mutex);
      gst_adapter_clear(dspaacsink->adapter);
      break;

    case GST_STATE_CHANGE_READY_TO_NULL:
      DBG_PRINT("AAC READY To NULL\n");
      gst_dspaudio_close(dspaacsink->audio);
      gst_dspaudio_reset(dspaacsink->audio);
      break;

    default:
      break;
  }

  DBG_PRINT("AAC STATE CHANGE SUCCESSFUL\n");
  return ret;
}


/**
 *
 *
 */

static GstClock *
gst_dspaacsink_provide_clock (GstElement * elem)
{
  DBG_PRINT("gst_dspaacsink_provide_clock\n");
  GstDSPAACSink *dsp = GST_DSPAACSINK (elem);
  return GST_CLOCK (gst_object_ref (GST_OBJECT (dsp->audio->clock)));
}


/**
 *
 *
 */

static GstClockTime
gst_dspaacsink_get_time (GstClock * clock, gpointer sink)
{
  GstDSPAACSink *dsp = GST_DSPAACSINK (sink);
  return gst_dspaudio_get_dsp_clock(dsp->audio);
}


/**
 * plugin_init:
 * @plugin: GstPlugin to be initialized
 *
 * This initializes the plugin when it is loaded by GStreamer
 *
 * Returns: Boolean value, TRUE if initialization was successful
 */

static gboolean
plugin_init (GstPlugin * plugin)
{
  return gst_element_register (plugin, "dspaacsink", GST_RANK_PRIMARY,
      GST_TYPE_DSPAACSINK);
}


/**
 * This is the "boilerplate" of the plugin
 *
 */

GST_PLUGIN_DEFINE (GST_VERSION_MAJOR,
                   GST_VERSION_MINOR,
                   "dspaac",
                   "DSP AAC audio sink",
                   plugin_init,
                   VERSION,
                   "LGPL",
                   "dspaudio",
                   "Nokia Corporation");
