/*
 * rtpstream.c - Source for RTP plugin stream implementation
 *
 * Farsight RTP/AVP/SAVP/AVPF Module
 * Copyright (C) 2005,2006 Collabora Ltd.
 * Copyright (C) 2005,2006 Nokia Corporation
 *   @author Rob Taylor <rob.taylor@collabora.co.uk>
 *   @author Philippe Kalaf <philippe.kalaf@collabora.co.uk>
 * Copyright (C) 2005 INdT 
 *   @author Andre Moreira Magalhaes <andre.magalhaes@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "rtp.h"
#include "rtpstream.h"
#include "rtpsession.h"
#include "rtpgstcodecs.h"
#include "farsight-transport.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "helpers/farsight-interfaces.h"
#include <arpa/inet.h>

#ifdef HAVE_SOFIASIP
#include <sofia-sip/su.h>
#include <sofia-sip/su_localinfo.h>
#endif

#ifdef HAVE_GNET
#include <gnet.h>
#endif

#ifdef HAVE_CLINKC
#include "helpers/upnpigd.h"
#endif

#include <gst/gst.h>

#define DEFAULT_AUDIO_PORT 7078
#define DEFAULT_VIDEO_PORT 9078

#define DEFAULT_STUN_IP "stun.l.google.com"
#define DEFAULT_STUN_PORT 19302

#define DEFAULT_CONN_TIMEOUT 45

enum
{
  PROP_0,
  PROP_STUN_IP,
  PROP_STUN_PORT,
  PROP_TURN_IP,
  PROP_TURN_PORT,
  PROP_CONN_TIMEOUT,
  PROP_CODEC_PREF
};

FarsightCodecPreference codec_pref[] = {
  { "AMR", 8000 },
  { "GSM", 8000 },
  { "speex", 8000 },
  { "speex", 16000 },
  { "iLBC", 8000 },
  { "PCMA", 8000 },
  { "PCMU", 8000 }
};

struct _FarsightRTPStreamPrivate
{
  gboolean disposed;

  GList *local_codecs;
  GList *remote_codecs;
  GHashTable *pt_caps_table;

  /* send/recv pipeline */
  GstElement *main_pipeline; /*this is an optional user provided GstPipeline*/
  GstElement *pipeline;
  GstElement *rtpbin;
  GstElement *send_codec_bin;

  GstElement *src;
  GstCaps *src_filter;

  GstElement *rtpdemux;
  GstElement *sink;
  GstCaps *sink_filter;

  gboolean sending;

  guint bus_watch;

  gint recv_codec_id;
  gint send_codec_id;
  
  gboolean start_me;
  gboolean build_send_pipeline;

  GHashTable *depayloader_chains_info;

  gboolean prepared; /*set if prepare has been called*/
#ifdef HAVE_JINGLEP2P
  socketclient_t socket_client;
#endif

  gchar *stun_ip;
  guint stun_port;
  gchar *turn_ip;
  guint turn_port;

  guint conn_timeout;
  gint timeout_src;

  GList *local_candidates;
  GList *remote_candidates;

  gchar *active_native_candidate;
  gchar *active_remote_candidate;

  guint16 base_local_port;
  guint16 local_port;
  gint local_socket; /* socket used for symetric RTP */

  gboolean use_upnp;
};

#define FARSIGHT_RTP_STREAM_GET_PRIVATE(o)  \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), FARSIGHT_TYPE_RTP_STREAM, \
                                FarsightRTPStreamPrivate))

static void farsight_rtp_stream_class_init (FarsightRTPStreamClass *klass);
static void farsight_rtp_stream_init (FarsightRTPStream *rtp_stream);
static GObject * farsight_rtp_stream_constructor (GType type,
    guint n_construct_properties,
    GObjectConstructParam *construct_properties);

static void farsight_rtp_stream_dispose (GObject *object);
static void farsight_rtp_stream_finalize (GObject *object);

static void farsight_rtp_stream_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec);
static void farsight_rtp_stream_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec);

static gboolean farsight_rtp_stream_bus_watch_cb (GstBus *bus, GstMessage
    *message, gpointer user_data);
static void farsight_rtp_stream_new_payload_type (GstElement *element, 
                                                  gint pt, GstPad *pad, 
                                                  gpointer user_data);
static void farsight_rtp_stream_payload_type_change (GstElement *element, 
                                                     gint pt,
                                                     gpointer user_data);
static void farsight_rtp_stream_create_new_pt_recv_pipeline (FarsightStream *stream, 
                                                             gint id);
static GstElement *farsight_rtp_stream_get_pipeline (FarsightStream *stream);
static gboolean farsight_rtp_stream_set_pipeline (FarsightStream *stream,
    GstElement *pipeline);
static void farsight_rtp_stream_set_active_codec (FarsightStream *stream, 
                                                  gint id);
static gint farsight_rtp_stream_get_active_codec (FarsightStream *stream);
static G_CONST_RETURN GList *farsight_rtp_stream_get_local_codecs (FarsightStream *stream);
static void farsight_rtp_stream_set_remote_codecs (FarsightStream *stream, 
                                                   const GList *codecs);
static GList *farsight_rtp_stream_get_codec_intersection (FarsightStream *stream);
static void farsight_rtp_stream_set_codec_preference_list (FarsightStream *stream,
    FarsightCodecPreference codec_pref[]);
static gboolean farsight_rtp_stream_set_source (FarsightStream *stream, 
                                                GstElement *source);
static gboolean farsight_rtp_stream_set_source_filter (FarsightStream *stream,
    GstCaps *filter);
static GstElement *farsight_rtp_stream_get_source (FarsightStream *stream);
static gboolean farsight_rtp_stream_set_sink (FarsightStream *stream, 
                                              GstElement *sink);
static gboolean farsight_rtp_stream_set_sink_filter (FarsightStream *stream,
    GstCaps *filter);
static GstElement *farsight_rtp_stream_get_sink (FarsightStream *stream);

static gboolean farsight_rtp_stream_build_base_pipeline (FarsightRTPStream *self);
static gboolean farsight_rtp_stream_build_send_pipeline (FarsightRTPStream *selft);
static gboolean farsight_rtp_stream_start (FarsightStream *stream);
static void farsight_rtp_stream_stop (FarsightStream *stream);
static gboolean farsight_rtp_stream_set_sending (FarsightStream *stream,
    gboolean sending);
static gboolean farsight_rtp_stream_choose_codec (FarsightRTPStream *self, 
                                                  CodecInternal **codec_internal);
static gboolean farsight_rtp_stream_candidate_exists (FarsightStream *stream, 
                                                      const GList *candidate_list, 
                                                      const GList *candidate);

#ifdef HAVE_JINGLEP2P
static void farsight_rtp_stream_native_candidate_ready (gpointer stream, 
                                                        const GList *candidate);
static void farsight_rtp_stream_socket_state_changed (gpointer stream, 
                                                      gint state);
static void farsight_rtp_stream_network_error (gpointer stream);
#endif

static void farsight_rtp_stream_upnp_send_request(FarsightRTPStream *self,
    FarsightTransportInfo *derived_trans);
static void farsight_rtp_stream_upnp_close_ports (FarsightRTPStream *self);

static void farsight_rtp_stream_prepare_transports (FarsightStream *self);
static gboolean farsight_rtp_stream_add_remote_candidate_to_rtpbin (FarsightRTPStream *self, 
                                                                    const gchar *remote_candidate_id);
static gboolean farsight_rtp_stream_set_active_candidate_pair (FarsightStream *stream,
                                                               const gchar *native_candidate_id, 
                                                               const gchar *remote_candidate_id);
static GList *farsight_rtp_stream_get_native_candidate (FarsightStream *stream, 
                                                        const gchar *candidate_id);
static G_CONST_RETURN GList *farsight_rtp_stream_get_native_candidate_list (FarsightStream *stream);
static void farsight_rtp_stream_set_remote_candidate_list (FarsightStream *stream, 
                                                           const GList *remote_candidates);
static void farsight_rtp_stream_add_remote_candidate (FarsightStream *self,
                                                      const GList *remote_candidate);

static gboolean farsight_rtp_stream_connection_timed_out (gpointer data);

FarsightStreamClass *rtp_stream_parent_class = NULL;

static void
blocked_cb (GstPad *pad, gboolean blocked,
    gpointer user_data);

GType
farsight_rtp_stream_get_type (void)
{
  static GType type = 0;

  if (type == 0) {
    static const GTypeInfo info = {
      sizeof (FarsightRTPStreamClass),
      NULL,
      NULL,
      (GClassInitFunc) farsight_rtp_stream_class_init,
      NULL,
      NULL,
      sizeof (FarsightRTPStream),
      0,
      (GInstanceInitFunc) farsight_rtp_stream_init
    };

    type = g_type_register_static (FARSIGHT_TYPE_STREAM,
        "FarsightRTPStream", &info, 0);
  }

  return type;
}

static void
farsight_rtp_stream_class_init (FarsightRTPStreamClass *klass)
{
  GObjectClass *gobject_class;
  FarsightStreamClass *farsight_stream_class;

  gobject_class = (GObjectClass *) klass;
  farsight_stream_class = (FarsightStreamClass *) klass;
  rtp_stream_parent_class = g_type_class_peek_parent (klass);

  gobject_class->constructor = farsight_rtp_stream_constructor;

  gobject_class->set_property = farsight_rtp_stream_set_property;
  gobject_class->get_property = farsight_rtp_stream_get_property;

  g_object_class_install_property (gobject_class, PROP_STUN_IP,
      g_param_spec_string ("stun_ip", "STUN server ip",
        "The IP address of the STUN server to use",
        NULL, G_PARAM_WRITABLE));

  g_object_class_install_property (gobject_class, PROP_STUN_PORT,
      g_param_spec_uint ("stun_port", "STUN server port",
        "The port to the STUN server", 0, G_MAXUINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_STUN_IP,
      g_param_spec_string ("turn_ip", "TURN server ip",
        "The IP address of the TURN server to use",
        NULL, G_PARAM_WRITABLE));

  g_object_class_install_property (gobject_class, PROP_TURN_PORT,
      g_param_spec_uint ("turn_port", "TURN server port",
        "The port to the TURN server", 0, G_MAXUINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_CONN_TIMEOUT,
      g_param_spec_uint ("conn_timeout", "Connection timeout",
        "Number of secs before connection timeout", 0, G_MAXUINT, 0,
        G_PARAM_READWRITE));

  g_object_class_install_property (G_OBJECT_CLASS (klass), PROP_CODEC_PREF,
      g_param_spec_pointer ("codec_pref", "Codec preference list",
          "A pointer to an array of FarsightCodecPreference",
          G_PARAM_READWRITE));

  gobject_class->dispose = farsight_rtp_stream_dispose;
  gobject_class->finalize = farsight_rtp_stream_finalize;

  farsight_stream_class->prepare_transports =
      farsight_rtp_stream_prepare_transports;
  farsight_stream_class->get_native_candidate_list =
      farsight_rtp_stream_get_native_candidate_list;
  farsight_stream_class->get_native_candidate =
      farsight_rtp_stream_get_native_candidate;
  farsight_stream_class->set_remote_candidate_list =
      farsight_rtp_stream_set_remote_candidate_list;
  farsight_stream_class->add_remote_candidate =
      farsight_rtp_stream_add_remote_candidate;
  //farsight_stream_class->remove_remote_candidate =
      //farsight_rtp_stream_remove_remote_candidate;
  farsight_stream_class->set_active_candidate_pair =
      farsight_rtp_stream_set_active_candidate_pair;
  farsight_stream_class->get_local_codecs =
      farsight_rtp_stream_get_local_codecs;
  farsight_stream_class->set_remote_codecs =
      farsight_rtp_stream_set_remote_codecs;
  farsight_stream_class->get_codec_intersection =
    farsight_rtp_stream_get_codec_intersection;
  farsight_stream_class->set_codec_preference_list =
    farsight_rtp_stream_set_codec_preference_list;
  farsight_stream_class->set_active_codec =
      farsight_rtp_stream_set_active_codec;
  farsight_stream_class->get_active_codec =
      farsight_rtp_stream_get_active_codec;
  farsight_stream_class->set_sink = farsight_rtp_stream_set_sink;
  farsight_stream_class->set_sink_filter = farsight_rtp_stream_set_sink_filter;
  farsight_stream_class->get_sink = farsight_rtp_stream_get_sink;
  farsight_stream_class->set_source = farsight_rtp_stream_set_source;
  farsight_stream_class->set_source_filter =
    farsight_rtp_stream_set_source_filter;
  farsight_stream_class->get_source = farsight_rtp_stream_get_source;
  farsight_stream_class->get_pipeline = farsight_rtp_stream_get_pipeline;
  farsight_stream_class->set_pipeline = farsight_rtp_stream_set_pipeline;
  farsight_stream_class->start = farsight_rtp_stream_start;
  farsight_stream_class->stop = farsight_rtp_stream_stop;
  farsight_stream_class->set_sending = farsight_rtp_stream_set_sending;

  g_type_class_add_private (klass, sizeof (FarsightRTPStreamPrivate));
}

static void
farsight_rtp_stream_init (FarsightRTPStream *self)
{
  self->priv = FARSIGHT_RTP_STREAM_GET_PRIVATE (self);
  self->priv->remote_codecs = NULL;

  self->priv->pipeline = NULL;
  self->priv->rtpbin = NULL;
  self->priv->send_codec_bin = NULL;

  self->priv->src = NULL;
  self->priv->src_filter = NULL;

  self->priv->rtpdemux = NULL;
  self->priv->sink = NULL;
  self->priv->sink_filter = NULL;

  if (farsight_stream_get_direction (FARSIGHT_STREAM (self)) ==
      FARSIGHT_STREAM_DIRECTION_RECEIVEONLY)
  {
    self->priv->sending = FALSE;
  }
  else
  {
    self->priv->sending = TRUE;
  }

  self->priv->prepared = FALSE;

  self->priv->bus_watch = 0;

#ifdef HAVE_JINGLEP2P
  self->priv->socket_client = NULL;
#endif

  self->priv->stun_ip = g_strdup (DEFAULT_STUN_IP);
  self->priv->stun_port = DEFAULT_STUN_PORT;
  self->priv->turn_ip = NULL;
  self->priv->turn_port = 0;

  self->priv->local_candidates = NULL;
  self->priv->remote_candidates = NULL;

  /* this tells _start to auto pick a codec */
  self->priv->send_codec_id = -1;

  self->priv->start_me = FALSE;
  self->priv->build_send_pipeline = FALSE;

  self->priv->timeout_src = -1;
  self->priv->conn_timeout = DEFAULT_CONN_TIMEOUT;

#ifdef HAVE_CLINKC
  if (!getenv ("FARSIGHT_DISABLE_UPNP"))
  {
    self->priv->use_upnp = TRUE;
    g_message ("Setting uPnP to true");
    if (!upnp_cp_init())
    {
      g_warning ("Error setting up uPnP");
      self->priv->use_upnp = FALSE;
    }
  }
#endif

}

static GObject *
farsight_rtp_stream_constructor (GType type,
    guint n_construct_properties,
    GObjectConstructParam *construct_properties)
{
  GObject *obj;

  {
    /* Invoke parent constructor. */
    FarsightRTPStreamClass *klass;
    GObjectClass *parent_class;  
    klass = FARSIGHT_RTP_STREAM_CLASS (g_type_class_peek (FARSIGHT_TYPE_RTP_STREAM));
    parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
    obj = parent_class->constructor (type,
                                     n_construct_properties,
                                     construct_properties);
  }

  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (obj);

  guint media_type;

  g_object_get (G_OBJECT (self), "media-type", &media_type, NULL);
  switch (media_type) {
    case FARSIGHT_MEDIA_TYPE_AUDIO:
      self->priv->base_local_port = 7078;
      break;
    case FARSIGHT_MEDIA_TYPE_VIDEO:
      self->priv->base_local_port = 9098;
      break;
    default:
      self->priv->base_local_port = 0;
  }

  g_message ("Media type is %d", media_type);

  /* get locally available codecs */
  const GList *load_codecs_result = load_codecs (media_type);
  if (!load_codecs_result)
  {
    g_error ("Error running load_codecs()");
    return obj;
  }

  self->priv->local_codecs = g_list_copy
    ((GList *)load_codecs_result);

  if (media_type == FARSIGHT_MEDIA_TYPE_AUDIO)
  {
    sort_codecs (&self->priv->local_codecs,
            codec_pref,
            sizeof (codec_pref) / sizeof (FarsightCodecPreference));
  }

  return obj;
}

static void
farsight_rtp_stream_dispose (GObject *object)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (object);
  guint media_type;

  if (self->priv->disposed) {
    /* If dispose did already run, return. */
    return;
  }

  self->priv->disposed = TRUE;

  farsight_rtp_stream_stop (FARSIGHT_STREAM (self));

  g_object_get (G_OBJECT (self), "media-type", &media_type, NULL);

  unload_codecs (media_type);

  /* chain up to parent */
  G_OBJECT_CLASS (rtp_stream_parent_class)->dispose (object);
}

static void
farsight_rtp_stream_finalize (GObject *object)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (object);

  g_return_if_fail (self != NULL);
  g_return_if_fail (FARSIGHT_IS_RTP_STREAM (self));

  if (self->priv->local_candidates)
    farsight_transport_list_destroy (self->priv->local_candidates);

  if (self->priv->remote_candidates)
    farsight_transport_list_destroy (self->priv->remote_candidates);

  if (self->priv->local_codecs)
    g_list_free (self->priv->local_codecs);

  if (self->priv->remote_codecs)
    farsight_codec_list_destroy (self->priv->remote_codecs);

  if (self->priv->depayloader_chains_info)
  {
    g_hash_table_destroy (self->priv->depayloader_chains_info);
  }

  if (self->priv->active_native_candidate) {
    g_free (self->priv->active_native_candidate);
  }

  if (self->priv->active_remote_candidate) {
    g_free (self->priv->active_remote_candidate);
  }

  if (self->priv->stun_ip) {
    g_free (self->priv->stun_ip);
  }

  if (self->priv->turn_ip) {
    g_free (self->priv->turn_ip);
  }

  if (self->priv->pt_caps_table) {
    g_hash_table_destroy (self->priv->pt_caps_table);
  }

  if (self->priv->sink) {
    gst_object_unref (self->priv->sink);
  }

  if (self->priv->sink_filter) {
    gst_caps_unref (self->priv->sink_filter);
  }

  if (self->priv->src) {
    gst_object_unref (self->priv->src);
  }

  if (self->priv->src_filter) {
    gst_caps_unref (self->priv->src_filter);
  }

  /* TODO free GSTelements */

  G_OBJECT_CLASS (rtp_stream_parent_class)->finalize (object);
}

static void farsight_rtp_stream_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (object);

  switch (prop_id) {
    case PROP_STUN_IP:
      self->priv->stun_ip = g_value_dup_string (value);
      break;
    case PROP_STUN_PORT:
      self->priv->stun_port = g_value_get_uint (value);
      break;
    case PROP_TURN_IP:
      self->priv->turn_ip = g_value_dup_string (value);
      break;
    case PROP_TURN_PORT:
      self->priv->turn_port = g_value_get_uint (value);
      break;
    case PROP_CONN_TIMEOUT:
      self->priv->conn_timeout = g_value_get_uint (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void farsight_rtp_stream_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (object);

  switch (prop_id) {
    case PROP_STUN_IP:
      g_value_set_string (value, self->priv->stun_ip);
      break;
    case PROP_STUN_PORT:
      g_value_set_uint (value, self->priv->stun_port);
      break;
    case PROP_TURN_IP:
      g_value_set_string (value, self->priv->turn_ip);
      break;
    case PROP_TURN_PORT:
      g_value_set_uint (value, self->priv->turn_port);
      break;
    case PROP_CONN_TIMEOUT:
      g_value_set_uint (value, self->priv->conn_timeout);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static gboolean
farsight_rtp_stream_bus_watch_cb (GstBus *bus, GstMessage *message, 
                                  gpointer user_data)
{
  FarsightStream *stream = FARSIGHT_STREAM (user_data);

  switch (GST_MESSAGE_TYPE (message)) {
    case GST_MESSAGE_EOS:
      g_debug ("%s (%d): end of stream on stream pipeline",
          __FUNCTION__, __LINE__);
      farsight_stream_signal_error (stream, 
                                    FARSIGHT_STREAM_ERROR_EOS, NULL);
      farsight_rtp_stream_stop (stream);
      break;
    case GST_MESSAGE_ERROR:
      {
      gchar *debug;
      GError *err;

      gst_message_parse_error (message, &err, &debug);
      g_free (debug);

      g_warning ("%s (%d): error on stream pipeline. "
                 "Error code=%d message=%s",
                 __FUNCTION__, __LINE__, err->code, err->message);
      g_error_free (err);

      if (err->domain == GST_RESOURCE_ERROR)
      {
        farsight_stream_signal_error (stream,
            FARSIGHT_STREAM_ERROR_RESOURCE, err->message);
      }
      else
      {
        farsight_stream_signal_error (stream,
            FARSIGHT_STREAM_ERROR_UNKNOWN, err->message);
      }
      farsight_rtp_stream_stop (stream);
      break;
      }
    default:
      break;
  }

  return TRUE;
}

static void
farsight_rtp_stream_new_payload_type (GstElement *element, gint pt, 
                                      GstPad *pad, gpointer user_data)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (user_data);
  /*CodecInternal *codec_internal;*/
  DepayloaderChainInfo *chain_info;

  /*
  codec_internal = 
    lookup_codec_by_pt (
        farsight_stream_get_media_type (FARSIGHT_STREAM(stream)), pt);
  if (!codec_internal) {
    g_warning ("%s (%d): g_hash_table_lookup on %d return null", __FUNCTION__,
        __LINE__, pt);
    farsight_rtp_stream_stop (FARSIGHT_STREAM(stream));
    farsight_stream_signal_error (FARSIGHT_STREAM(stream), 
                                  FARSIGHT_STREAM_ERROR_UNKNOWN, NULL);
    return;
  }
  */

  if (!self->priv->depayloader_chains_info) {
    self->priv->depayloader_chains_info =
        g_hash_table_new (g_direct_hash, g_direct_equal);
  }
  chain_info = g_new0 (DepayloaderChainInfo, 1);
  chain_info->pad = pad;
  g_hash_table_insert (self->priv->depayloader_chains_info,
                       (gpointer) pt, (gpointer) chain_info);
}

static void
farsight_rtp_stream_payload_type_change (GstElement *element, gint pt,
                                         gpointer user_data)
{
  FarsightStream *self = FARSIGHT_STREAM (user_data);
  farsight_rtp_stream_create_new_pt_recv_pipeline(self, pt);
}

static gboolean
error_on_create_recv_pipeline (gpointer data)
{
  FarsightStream *stream = FARSIGHT_STREAM (data);

  farsight_rtp_stream_stop (stream);
  farsight_stream_signal_error (stream, FARSIGHT_STREAM_ERROR_PIPELINE_SETUP,
      "Error creating new recv pipeline");

  return FALSE;
}

static void
farsight_rtp_stream_create_new_pt_recv_pipeline (FarsightStream *stream, gint id)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  DepayloaderChainInfo *chain_info = NULL;
  gchar *name = NULL;
  GstElement *codec_bin = NULL;
  CodecInternal *codec_internal = NULL;

  /* WARNING This function gets called in the recv pipeline thread */

  codec_internal = lookup_codec_by_pt (farsight_stream_get_media_type (stream),
      id);
  if (!codec_internal) {
    g_warning ("Payload type %d not supported", id);
    goto error;
  }

  g_debug ("%s (%d): active PT change to %d", __FUNCTION__, __LINE__, id);

  chain_info = g_hash_table_lookup (self->priv->depayloader_chains_info,
          (gpointer) ((gint) id));

  /* Let's see if a recv codec bin for this codec already exists */
  name = g_strdup_printf ("recv%d", id);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->priv->pipeline), name);
  g_free (name);

  if (!codec_bin) {
    g_message ("%s (%d): setting up new pipeline for id=%d",
            __FUNCTION__, __LINE__, id);

    codec_bin = create_codec_bin (farsight_stream_get_media_type (stream),
        id, DIR_RECV);
    if (!codec_bin)
    {
      g_warning ("Couldn't create elements, check your gstreamer installation");
      goto error;
    }
    gst_bin_add (GST_BIN (self->priv->pipeline), codec_bin);

    self->priv->recv_codec_id = id;

    /* we emit now so we can set anything we need on the elements before they
     * are playing */
    farsight_stream_signal_codec_changed (stream, id);

    gst_element_set_state (codec_bin, GST_STATE_PAUSED);

    gchar *padname = gst_pad_get_name (chain_info->pad);
    gst_element_link_pads (self->priv->rtpdemux, padname, GST_ELEMENT(codec_bin),
            "sink");

    gst_element_set_state (codec_bin, GST_STATE_PLAYING);

    g_free (padname);

  } else {
    g_debug ("%s (%d): pipeline for id %d already configured, using it",
        __FUNCTION__, __LINE__, id);
    gst_object_unref (GST_OBJECT (codec_bin));
  }

  if (!codec_internal->has_sink)
  {
    if (self->priv->sink)
    {
      GstPad *sinkpad = gst_element_get_pad (self->priv->sink, "sink");
      if (!gst_pad_is_linked (sinkpad))
      {
        g_debug ("%s (%d): linking audio sink to codec bin",
            __FUNCTION__, __LINE__);

        if ((self->priv->main_pipeline && 
              gst_element_get_parent (self->priv->sink) == NULL)
            || !self->priv->main_pipeline)
        {
          gst_bin_add (GST_BIN (self->priv->pipeline), self->priv->sink);
        }
      }
      else
      {
        GstBin *old_codec_bin;

        g_debug ("%s (%d): relinking audio sink to new codec bin with pt %d",
            __FUNCTION__, __LINE__, id);

        name = g_strdup_printf ("recv%d", self->priv->recv_codec_id);
        old_codec_bin = GST_BIN (gst_bin_get_by_name (GST_BIN (self->priv->pipeline), name));
        g_free (name);

        /* TODO must block the src pad on the codec bin before unlinking or
         * pause it (look at part-block.txt) */

        gst_element_unlink (GST_ELEMENT (old_codec_bin), self->priv->sink);

        gst_object_unref (GST_OBJECT (old_codec_bin));
      }
      gst_element_set_state (self->priv->sink, GST_STATE_READY);
      gst_element_link_filtered (GST_ELEMENT (codec_bin), self->priv->sink, self->priv->sink_filter);
      gst_element_set_state (self->priv->sink, GST_STATE_PLAYING);

      gst_object_unref (GST_OBJECT (sinkpad));
    }
    else 
    {
      g_warning("received stream while sink unset, blocking recv pipeline");
      GstPad *codec_bin_src_pad =
        gst_element_get_pad (codec_bin, "src");

      gst_pad_set_blocked_async (codec_bin_src_pad, TRUE,
          blocked_cb, NULL);
    }
  }

  /* restart self pipeline */
  //gst_xml_write_file (GST_ELEMENT (self->priv->pipeline), stdout);
  g_debug ("%s: setting state to playing", G_STRFUNC);
  return;

error:
  g_warning ("%s: PT change failed", G_STRFUNC);
  g_idle_add (error_on_create_recv_pipeline, stream);
}

static GstElement*
farsight_rtp_stream_get_pipeline (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;

  if (self->priv->main_pipeline)
  {
    return self->priv->main_pipeline;
  }
  else
  {
    return self->priv->pipeline;
  }
}

static gboolean
farsight_rtp_stream_set_pipeline (FarsightStream *stream,
    GstElement *pipeline)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;

  if (self->priv->main_pipeline)
  {
    g_warning ("Pipeline already set! Ignoring");
    return FALSE;
  }

  self->priv->main_pipeline = pipeline;
  return TRUE;
}

static void
farsight_rtp_stream_set_active_codec (FarsightStream *stream, gint id)
{
  /* TODO This dosen't work yet, need to make sure the audiosrc or whatever else is
   * used sends a new_segment after a reconnection, this is done by
   * overwriting the link/unlink functions
   * in the current case without the new segment the timestamp stops
   * incrementing 
   * Also make sure that if there is a user defined pipeline, we change the
   * ghost pad target to the new send_codec_bin */

  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  gchar *name;
  GstElement *codec_bin = NULL;
  GstElement *old_codec_bin = NULL;
  GstElement *new_codec_bin = NULL;
  CodecInternal *codec_internal = NULL;

  g_debug ("%s (%d): called to change codec from %d to %d", __FUNCTION__,
          __LINE__, farsight_rtp_stream_get_active_codec(stream), id);

  g_debug ("%s: this does not work yet, returning", __FUNCTION__);
  return;

  if (id == farsight_rtp_stream_get_active_codec(stream))
    return;

  /* Let's check if the given codec id is valid */
  codec_internal = lookup_codec_by_pt 
    (farsight_stream_get_media_type (stream), id);
  if (!codec_internal)
  {
    g_message ("%s (%d): invalid codec id %d", 
              __FUNCTION__, __LINE__, id);
    return;
  }

  /* Let's make sure codec is in the remote_codec list */
  /* FIXME This will not work if the remote codecs are not set. But we usually
   * call set_active_codec before setting the remote codecs. This allows the
   * send pipeline creation code not to auto-chose a codec. We might just have
   * to trust the user to set an available active codec */
#if 0
  if (!self->priv->remote_codecs)
  {
    g_warning ("You need to set the remote codecs before changing the active"
        " codec");
    return;
  }

  GList *codec_list = self->priv->remote_codecs;
  FarsightCodec *cur_codec;
  do {
    cur_codec = codec_list->data;
    if (codec_internal->codec->id < 96)
    {
      if (codec_internal->codec->id == cur_codec->id)
      {
        break;
      }
    }
    else
    {
      if (g_ascii_strcasecmp (codec_internal->codec->encoding_name,
            cur_codec->encoding_name) == 0)
      {
        break;
      }
    }

    codec_list = codec_list->next;
  } while (codec_list);

  if (!codec_list)
  {
    g_warning ("Called farsight_stream_set_active_codec() with codec (%d %s)"
        " not available in remote codec list! Ignoring", 
        id, codec_internal->codec->encoding_name);
    return;
  }
#endif

  if (!self->priv->send_codec_bin)
    goto set_var;

  g_debug ("%s (%d): changing active send PT to %d", __FUNCTION__,
          __LINE__, id);

  /* Let's see if a send codec bin for this codec already exists
   * in shouldn't really exist... */
  name = g_strdup_printf ("send%d", id);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->priv->pipeline), name);
  g_free (name);
  if (codec_bin) {
    gst_object_unref (GST_OBJECT (codec_bin));
    g_warning ("Send codec already exists for codec %d, this shouldn't happen",
        id);
    goto error;
  }

  /* Let's unlink the existing codec bin and delete it */
  name = g_strdup_printf ("send%d", self->priv->send_codec_id);
  old_codec_bin = gst_bin_get_by_name (GST_BIN(self->priv->pipeline), name);
  g_debug ("removing send codec bin %p on pipeline %p", codec_bin, self->priv->pipeline);
  g_free (name);
  if (!old_codec_bin)
  {
    g_warning ("Couldn't find current send codec for codec %d!",
        self->priv->send_codec_id);
    goto error;
  }

  /* TODO must block the src pad before unlinking
   * (look at part-block.txt) */
  gst_element_unlink (old_codec_bin, self->priv->rtpbin);
  if (self->priv->src)
  {
    gst_element_unlink (self->priv->src, old_codec_bin);
  }

  gst_bin_remove (GST_BIN (self->priv->pipeline), old_codec_bin);

  gst_object_unref (GST_OBJECT (old_codec_bin));

  self->priv->send_codec_bin = NULL;

  /* Let's create a new send codec bin for this codec */
  new_codec_bin = create_codec_bin (farsight_stream_get_media_type (stream), id,
      DIR_SEND); 
  if (!new_codec_bin)
  {
    g_warning ("Couldn't create elements, check your gstreamer installation");
    goto error;
  }

  g_debug ("adding send codec bin %p on pipeline %p", new_codec_bin, self->priv->pipeline);
  gst_bin_add (GST_BIN (self->priv->pipeline), new_codec_bin);
  /* now we link the src and rtpbin to it */
  gst_element_link (new_codec_bin, self->priv->rtpbin);
  if (self->priv->src)
  {
    gst_element_link_filtered (self->priv->src, new_codec_bin, self->priv->src_filter);
  }

  self->priv->send_codec_bin = new_codec_bin;

set_var:
  self->priv->send_codec_id = id;

  return;

error:
  farsight_rtp_stream_stop (stream);
  farsight_stream_signal_error (stream, FARSIGHT_STREAM_ERROR_PIPELINE_SETUP,
      "Error while changing send codec");
}

static gint
farsight_rtp_stream_get_active_codec (FarsightStream *stream)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  return self->priv->send_codec_id; 
}

static void
unblocked_cb (GstPad *pad, gboolean blocked,
    gpointer user_data)
{
  g_debug ("Pad unblocked successfully");
  gst_object_unref (GST_OBJECT (pad));
}

static void
blocked_cb (GstPad *pad, gboolean blocked,
    gpointer user_data)
{
  g_debug ("Pad blocked successfully");
  gst_object_unref (GST_OBJECT (pad));
}

/* this function will be called when a block has succeeded, it will unlink the
 * pads and insert the new element if given in user_data.
 * Also works if the new element is a sink it implements part-block.txt on
 * dynamically switching element in PLAYING state.
 * It expects element4 to be in the correct bin
 * 
  .----------.      .----------.      .----------.
  | element1 |      | element2 |      | element3 |
 ...        src -> sink       src -> sink       ...
  '----------'      '----------'      '----------'
                    .----------.
                    | element4 |
                   sink       src
                    '----------'
*/
static void
unlink_and_replace (GstPad *element1_src_pad, gboolean blocked,
    gpointer user_data)
{
  g_debug ("Blocked pad successfully, unlinking and replacing downstream");
  GstPad *element2_sink_pad = NULL;
  element2_sink_pad = gst_pad_get_peer (element1_src_pad);

  if (element2_sink_pad == NULL)
  {
    return;
  }

  GstElement *element4 = (GstElement *)user_data;
  GstElement *element2 = gst_pad_get_parent_element (element2_sink_pad);

  if (element2 == NULL)
  {
    gst_object_unref (GST_OBJECT (element2_sink_pad));
    return;
  }

  gst_pad_unlink (element1_src_pad, element2_sink_pad);

  if (element4)
  {
    GstElement *element1 = NULL;
    GstPad *element2_src_pad = NULL;

    element1 = gst_pad_get_parent_element (element1_src_pad);
    element2_src_pad = gst_element_get_pad (element2, "src");
    if (element2_src_pad)
    {
      GstPad *element3_sink_pad = NULL;
      GstElement *element3 = NULL;
      element3_sink_pad = gst_pad_get_peer (element2_src_pad);
      element3 = gst_pad_get_parent_element (element3_sink_pad);
      gst_pad_unlink (element2_src_pad, element3_sink_pad);
      gst_element_link (element4, element3);
      gst_object_unref (GST_OBJECT (element3_sink_pad));
      gst_object_unref (GST_OBJECT (element3));
      gst_object_unref (GST_OBJECT (element2_src_pad));
    }
    gst_element_link (element1, element4);
    //gst_element_sync_state_with_parent (element4);
    gst_element_set_state (element4, GST_STATE_PLAYING);
    gst_object_unref (GST_OBJECT (element1));
    gst_pad_set_blocked_async (element1_src_pad, FALSE, unblocked_cb, NULL);
    gst_object_unref (GST_OBJECT (element4));
  }

  gst_object_unref (GST_OBJECT (element2_sink_pad));
  gst_object_unref (GST_OBJECT (element2));
}

struct _replace_sink_data
{
  FarsightRTPStream *self;
  GstElement *sink;
};

typedef struct _replace_sink_data replace_sink_data;

static void
unlink_and_replace_sink (GstPad *element1_src_pad, gboolean blocked,
    gpointer user_data)
{
  replace_sink_data *data = (replace_sink_data *)user_data;
  FarsightRTPStream *self = data->self;

  unlink_and_replace (element1_src_pad, blocked, data->sink);

  /* remove old sink and update references */
  if (gst_element_set_state (self->priv->sink, GST_STATE_NULL) ==
      GST_STATE_CHANGE_ASYNC)
  {
    g_debug ("waiting for state change");
    gst_element_get_state (self->priv->sink, NULL, NULL, GST_CLOCK_TIME_NONE);
    g_debug ("done");
  }
  if (!self->priv->main_pipeline)
  {
    gst_bin_remove (GST_BIN (self->priv->pipeline), self->priv->sink);
  }
  /* unref previous sink */
  gst_object_unref (self->priv->sink);
  /* set new sink */
  self->priv->sink = data->sink;
  g_free (data);
}

static void
farsight_rtp_stream_unlink_source (FarsightRTPStream *self)
{
  /* we want to safely unlink the source here */
  if (!self->priv->src)
  {
    return;
  }
  /* FIXME Decide how to properly unlink the source. If the source is a tee
   * we don't need to block it, we just unlink. But if the source is something
   * else, and we just unlink it, it might return a GST_FLOW_ERROR and stop the
   * whole pipeline. Problem with blocking the source is that blocking is done
   * async, and by the time we block+unlink the whole farsight stream might have
   * been unreffed. Doing a sync block might hang forever if the source stops
   * emitting buffers... For now I just unlink if there is a user defined
   * pipeline assuming it will be a tee. If there is no user defined pipeline
   * then there is no need to unlink the source since it will be NULLED along
   * the whole pipeline */
#if 0
  GstPad *codec_bin_sink_pad = gst_element_get_pad (self->priv->send_codec_bin, "sink");
  if (codec_bin_sink_pad == NULL)
  {
    g_warning ("Could not find sink pad on send codec bin! This should not"
        "happen! Source will not be unlinked properly");
    return;
  }
  /* this assumes the send codec bin is connected to the src */
  GstPad *src_src_pad = gst_pad_get_peer (codec_bin_sink_pad);
  gst_object_ref (GST_OBJECT (src_src_pad));
  gst_object_unref (GST_OBJECT (codec_bin_sink_pad));
  if (!src_src_pad)
  {
    return;
  }
  if (!gst_pad_set_blocked_async (src_src_pad, TRUE, unlink_and_replace, NULL))
  {
    g_warning ("Trying to block an already blocked pad!");
  }
  else
  {
    g_debug ("Waiting for source src pad to block");
  }
#endif
  if (self->priv->main_pipeline)
  {
    GstPad *pad = gst_element_get_pad (self->priv->pipeline, "sink");
    GstPad *tee_src = gst_pad_get_peer (pad);
    if (!tee_src)
    {
      g_debug ("Nothing linked to pipeline sink");
      return;
    }
    g_debug ("unlinking source");
    if (self->priv->sending)
    {
      g_debug ("waiting for pad to block");
      gst_pad_set_blocked (tee_src, TRUE);
      g_debug ("pad blocked, unlinking");
    }
    gst_element_unlink (self->priv->src, self->priv->pipeline);
    if (self->priv->sending)
    {
      g_debug ("unblocking source");
      gst_pad_set_blocked (tee_src, FALSE);
      g_debug ("done unblocking");
    }
    /*
    const gchar *longname = gst_element_factory_get_longname
      (gst_element_get_factory (self->priv->src));
    if (g_ascii_strcasecmp (longname, "Tee pipe fitting") == 0)
    {
      g_debug ("RELEASING REQUEST PAD");
      gst_element_release_request_pad (self->priv->src, tee_src);
    }
    */
    gst_object_unref (pad);
    gst_object_unref (tee_src);
  }
  /* gst_object_unref (GST_OBJECT (codec_bin_sink_pad)); */
}

static G_CONST_RETURN GList *
farsight_rtp_stream_get_local_codecs (FarsightStream *stream)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  return self->priv->local_codecs;
}

static void
farsight_rtp_stream_set_remote_codecs (FarsightStream *stream, 
                                       const GList *codecs)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  self->priv->remote_codecs = farsight_codec_list_copy (codecs);

  /* we sort the remote codecs according to our priorities */
  /* TODO What is the correct behaviour?
   * it seems wrong to ignore the remote preferences, should they be included in
   * the reply? Should they be only kept in memory for the stream we send while
   * our reply has out preference? 
   * GTalk's behaviour is to save our preference, send us what we prefer, but
   * in the intersection reply, he overwrites our preference with his. The
   * result is that we can get 2 different codecs being sent/recved. This won't
   * happen if we sort as done below. */
  sort_codecs (&self->priv->remote_codecs,
      codec_pref,
      sizeof (codec_pref) / sizeof (FarsightCodecPreference));

  GList *codec_list = self->priv->remote_codecs;
  do {
    FarsightCodec *codec = codec_list->data;
    g_debug ("remote_codec %s %d", codec->encoding_name, codec->clock_rate);
    codec_list = codec_list->next;
  } while (codec_list);

  update_local_dynamic_codecs (farsight_stream_get_media_type (stream),
      self->priv->remote_codecs);

  if (self->priv->pt_caps_table)
  {
    g_hash_table_destroy (self->priv->pt_caps_table);
  }
  self->priv->pt_caps_table =
      create_pt_caps_hashtable (self->priv->local_codecs);

  if (self->priv->rtpbin)
  {
    g_object_set (G_OBJECT(self->priv->rtpbin), "pt-map", self->priv->pt_caps_table, NULL);
  }
  else
  {
    farsight_stream_signal_error (FARSIGHT_STREAM(self),
        FARSIGHT_STREAM_ERROR_UNKNOWN, 
        "You need to run farsight_stream_prepare_transports() before setting"
        "the remote candidates");
    return;
  }

  /* We can create the send pipeline now */
  farsight_rtp_stream_build_send_pipeline (self);
}

static GList *
farsight_rtp_stream_get_codec_intersection (FarsightStream *stream)
{
  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  g_return_val_if_fail (self->priv->remote_codecs != NULL, NULL);

  GList *codecs = NULL;
  GList *el = self->priv->remote_codecs;

  for (el = self->priv->remote_codecs; el; el = el->next)
  {
    CodecInternal *ci;

    if ((ci = match_codec (farsight_stream_get_media_type (stream), el->data)))
    {
      codecs = g_list_prepend (codecs, farsight_codec_copy (ci->codec));
    }
  }

  codecs = g_list_reverse (codecs);

  return codecs;
}

static void
farsight_rtp_stream_set_codec_preference_list (FarsightStream *stream, 
                                       FarsightCodecPreference codec_pref_list[])
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  sort_codecs (&self->priv->local_codecs,
      codec_pref_list,
      sizeof (codec_pref_list) / sizeof (FarsightCodecPreference));
}

static gboolean
farsight_rtp_stream_set_source (FarsightStream *stream, GstElement *source)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  g_debug ("%s (%d): setting src", __FUNCTION__, __LINE__);
  if (self->priv->src)
  {
    if (self->priv->send_codec_bin)
    {
      g_warning ("Send pipeline already created,"
          "will attempt to replace while pipeline is running");
      /* TODO do I need to pause the pipeline? pause the src? just remove it and
       * replace a new one? */
    }
    gst_object_unref(self->priv->src);
  }
  self->priv->src = source;
  gst_object_ref(source);

  if (self->priv->build_send_pipeline)
  {
    farsight_rtp_stream_build_send_pipeline (self);
  }
  return TRUE;
}

static gboolean
farsight_rtp_stream_set_source_filter (FarsightStream *stream, GstCaps *filter)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  g_debug ("%s (%d): setting source filter", __FUNCTION__, __LINE__);
  if (self->priv->src_filter)
    gst_caps_unref (self->priv->src_filter);
  self->priv->src_filter = filter;
  gst_caps_ref(filter);
  return TRUE;
}

static GstElement *
farsight_rtp_stream_get_source (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  if (self->priv->src)
    return self->priv->src;

  g_return_val_if_fail (self->priv->pipeline != NULL, NULL);

  GstElement *codec_bin = NULL;
  GstIterator *iter;
  gchar *name;

  /* see if we have a src inside the pipeline */
  /* Let's see if a recv codec bin for this codec already exists */
  name = g_strdup_printf ("send%d", self->priv->send_codec_id);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->priv->pipeline), name);
  g_free (name);

  if (codec_bin) {
    gboolean done = FALSE;
    iter = gst_bin_iterate_elements (GST_BIN(codec_bin));

    while (!done) {
      gpointer data;
      switch (gst_iterator_next (iter, &data)) {
        case GST_ITERATOR_OK:
          {
            GstElement *child;
            gboolean is_src;
            child = GST_ELEMENT_CAST (data);

            GST_OBJECT_LOCK (child);
            if (!GST_OBJECT_FLAG_IS_SET (child, GST_ELEMENT_IS_SINK) &&
                !child->numsinkpads) {
              is_src = TRUE;
            }
            else
            {
              is_src = FALSE;
            }
            GST_OBJECT_UNLOCK (child);

            gst_object_unref (child);

            if (is_src)
            {
              gst_iterator_free (iter);
              return child;
            }
            break;
          }
        case GST_ITERATOR_RESYNC:
          gst_iterator_resync (iter);
          break;
        case GST_ITERATOR_DONE:
          done = TRUE;
          break;
        case GST_ITERATOR_ERROR:
          g_assert_not_reached ();
          break;
      }
    }
    gst_iterator_free (iter);
    gst_object_unref (GST_OBJECT (codec_bin));
  }
  return NULL;
}

static gboolean
farsight_rtp_stream_set_sink (FarsightStream *stream, GstElement *sink)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  g_debug ("%s (%d): setting sink", __FUNCTION__, __LINE__);

  GstElement *codec_bin = NULL;
  if (self->priv->pipeline)
  {
    gchar *name;
    name = g_strdup_printf ("recv%d", self->priv->recv_codec_id);
    codec_bin = gst_bin_get_by_name (GST_BIN(self->priv->pipeline), name);
    g_free (name);
  }

  if (codec_bin)
  {
    GstPad *codec_bin_src_pad =
      gst_element_get_pad (codec_bin, "src");

    /* no new sink */
    if (sink == NULL)
    {
      /* in this case we just want to block the recv bin and unlink the sink */
      replace_sink_data *user_data = g_new0 (replace_sink_data, 1);
      user_data->self = self;
      user_data->sink = NULL;

      gst_pad_set_blocked_async (codec_bin_src_pad, TRUE, unlink_and_replace_sink,
          NULL);
      self->priv->sink = NULL;
    }
    /* new sink provided */
    else
    {
      if (!self->priv->main_pipeline)
      {
        gst_bin_add (GST_BIN (self->priv->pipeline), sink);
      }
      /* in this case we just want to unblock the recv bin and connect the new
       * sink */
      if (self->priv->sink == NULL)
      {
        gst_element_set_state (sink, GST_STATE_READY);
        gst_element_link (codec_bin, sink);
        gst_element_set_state (sink, GST_STATE_PLAYING);
        gst_pad_set_blocked_async (codec_bin_src_pad, FALSE, unblocked_cb,
          NULL);
        gst_object_ref (sink);
        self->priv->sink = sink;
      }
      else
      {
        /* we want to block, remove old sink, add new sink, link and unblock */
        g_debug ("sink already exists, replacing");
        replace_sink_data *user_data = g_new0 (replace_sink_data, 1);
        user_data->self = self;
        user_data->sink = sink;
        gst_object_ref (sink);
        if (!gst_pad_set_blocked_async (codec_bin_src_pad, TRUE,
              unlink_and_replace_sink, user_data))
        {
          g_debug ("Recv pipeline already blocked, connecting new sink");
          gst_element_set_state (sink, GST_STATE_READY);
          gst_element_link (codec_bin, sink);
          gst_element_set_state (sink, GST_STATE_PLAYING);
          gst_pad_set_blocked_async (codec_bin_src_pad, FALSE, unblocked_cb,
              NULL);
          self->priv->sink = sink;
        }
      }
    }
    gst_object_unref (GST_OBJECT (codec_bin));
  }
  else
  {
    self->priv->sink = sink;
    if (sink)
    {
      gst_object_ref(sink);
    }
  }

  return TRUE;
}

static gboolean
farsight_rtp_stream_set_sink_filter (FarsightStream *stream, GstCaps *filter)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  g_debug ("%s (%d): setting sink filter", __FUNCTION__, __LINE__);
  if (self->priv->sink_filter)
    gst_caps_unref (self->priv->sink_filter);
  self->priv->sink_filter = filter;
  gst_caps_ref(filter);
  return TRUE;
}

static GstElement *
farsight_rtp_stream_get_sink (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream*) stream;
  if (self->priv->sink)
    return self->priv->sink;

  /* see if we have a sink inside the pipeline */
  /* we need a pipeline in this case */
  if (!self->priv->pipeline)
  {
    return NULL;
  }

  GstElement *codec_bin = NULL;
  GstIterator *iter;
  gchar *name;

  /* Let's see if a recv codec bin for this codec already exists */
  name = g_strdup_printf ("recv%d", self->priv->recv_codec_id);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->priv->pipeline), name);
  g_free (name);

  if (codec_bin) {
    gboolean done = FALSE;
    iter = gst_bin_iterate_elements (GST_BIN(codec_bin));

    while (!done) {
      gpointer data;

      switch (gst_iterator_next (iter, &data)) {
        case GST_ITERATOR_OK:
          {
            GstElement *child;
            gboolean is_sink;
            child = GST_ELEMENT_CAST (data);

            GST_OBJECT_LOCK (child);
            is_sink = GST_OBJECT_FLAG_IS_SET (child, GST_ELEMENT_IS_SINK);
            GST_OBJECT_UNLOCK (child);

            gst_object_unref (child);

            if (is_sink)
            {
              gst_iterator_free (iter);
              return child;
            }
            break;
          }
        case GST_ITERATOR_RESYNC:
          gst_iterator_resync (iter);
          break;
        case GST_ITERATOR_DONE:
          done = TRUE;
          break;
        case GST_ITERATOR_ERROR:
          g_assert_not_reached ();
          break;
      }
    }
    gst_iterator_free (iter);
    gst_object_unref (GST_OBJECT (codec_bin));
  }

  return NULL;
}

/* this build the core of the pipeline, meaning RTPBin and rtpdemux. The
 * send/recv parts are added later */
static gboolean
farsight_rtp_stream_build_base_pipeline (FarsightRTPStream *self)
{
  GstElement *rtpbin = NULL;
  GstBus *pipe_bus;

  g_return_val_if_fail (self != NULL, FALSE);

  g_debug ("%s (%d): creating core RTP pipeline", __FUNCTION__,
      __LINE__);
  /* 
   * build base pipeline 
   */
  if (self->priv->pipeline == NULL) {
    if (self->priv->main_pipeline)
    {
      /* Create a bin and a gst bus for it and set it on the bin */
      self->priv->pipeline = gst_bin_new (NULL);
      if (!self->priv->pipeline)
      {
        goto error;
      }
      /* let's make our bin independant of the parent state changes */
      gst_element_set_locked_state (self->priv->pipeline, TRUE);
      gst_bin_add (GST_BIN (self->priv->main_pipeline), self->priv->pipeline);
      pipe_bus = gst_bus_new ();
      gst_element_set_bus (GST_ELEMENT_CAST (self->priv->pipeline), pipe_bus);

      /* let's create a ghostpad for our sink */
      GstPad *ghostpad;
      ghostpad = gst_ghost_pad_new_no_target ("sink", GST_PAD_SINK);
      gst_element_add_pad (self->priv->pipeline, ghostpad);
    }
    else
    {
      /* Create a pipeline and listen on it's bus */
      self->priv->pipeline = gst_pipeline_new ("pipeline");
      if (!self->priv->pipeline)
      {
        goto error;
      }
      pipe_bus = gst_pipeline_get_bus (GST_PIPELINE (self->priv->pipeline));
    }
    self->priv->bus_watch = gst_bus_add_watch (pipe_bus, farsight_rtp_stream_bus_watch_cb, self);
    gst_object_unref (pipe_bus);

    /* create rtpbin element */
    rtpbin = gst_element_factory_make ("rtpbin", NULL);
    if (!rtpbin)
    {
      g_warning ("Couldn't create rtpbin, check your gstreamer install");
      goto error;
    }
    self->priv->rtpbin = rtpbin;
    gst_bin_add (GST_BIN (self->priv->pipeline), rtpbin);

    g_object_set (G_OBJECT (rtpbin),
        "rtp-sockfd", self->priv->local_socket,
        "rtcp-support", FALSE,
        "localport", self->priv->local_port, NULL);

#ifdef HAVE_JINGLEP2P 
    /* create jingle specific icesrc element that get data from
     * libjingle */
    g_object_set (G_OBJECT(rtpbin), "bypass-udp", TRUE, NULL);
    GstElement *icesrc = NULL;

    icesrc = gst_element_factory_make ("icesrc", NULL);
    //icesink = gst_element_factory_make ("fakesink", NULL);

    gst_bin_add (GST_BIN (self->priv->pipeline), icesrc);
    g_debug("added icesrc %p to pipeline %p with sockclient %p", icesrc,
        self->priv->pipeline, self->priv->socket_client);

    g_object_set (G_OBJECT(icesrc), "socketclient", self->priv->socket_client,
        NULL);

    /* TODO HACK ALERT HACK ALERT */
    /* just adding a boggus destination for now */
    /* so that jrtplib releases the packets to someone */
    g_object_set (G_OBJECT (rtpbin), "destinations", "64.34.23.11:5000", NULL);
#endif

    /* link rtpbin to rtpdemux to be able to discover what codec 
     * should be used.
     * We don't actually connect the depayloaders decoders yet */
    self->priv->rtpdemux = gst_element_factory_make ("rtpdemux", NULL);
    if (!self->priv->rtpdemux)
    {
      g_warning ("Couldn't create rtpdemux, check your gstreamer install");
      goto error;
    }
    g_signal_connect (G_OBJECT (self->priv->rtpdemux), "new-payload-type",
        G_CALLBACK (farsight_rtp_stream_new_payload_type), self);
    g_signal_connect (G_OBJECT (self->priv->rtpdemux), "payload-type-change",
        G_CALLBACK (farsight_rtp_stream_payload_type_change), self);

    gst_bin_add (GST_BIN (self->priv->pipeline),
        self->priv->rtpdemux);
    if (!gst_element_link_pads (rtpbin, "src%d", self->priv->rtpdemux,
        "sink"))
    {
      g_warning ("Could not link rtpbin:src to rtpdemux:sink");
      goto error;
    }

#ifdef HAVE_JINGLEP2P
    if (!gst_element_link_pads (icesrc, "src", rtpbin, "rtpsink"))
    {
      g_warning ("Could not link icesrc:src to rtpbin:rtpsink");
      goto error;
    }
#endif
  }

  return TRUE;

error:
  g_warning ("%s (%d): error setting up core RTP pipeline",
      __FUNCTION__, __LINE__);
  if (self->priv->pipeline) {
    gst_object_unref (GST_OBJECT (self->priv->pipeline));
    self->priv->pipeline = NULL;
  }
  farsight_stream_signal_error (FARSIGHT_STREAM(self),
          FARSIGHT_STREAM_ERROR_PIPELINE_SETUP,
          "Error setting up core RTP pipeline");
  return FALSE;
}

static gboolean
farsight_rtp_stream_build_send_pipeline (FarsightRTPStream *self)
{
  CodecInternal *codec_internal = NULL;
  GstElement *codec_bin;

  g_return_val_if_fail (self != NULL, FALSE);
  /* let's create the base pipeline if not already done so (prepare_transports
   * not called) */
  if (!self->priv->rtpbin || !self->priv->pipeline)
  {
    farsight_rtp_stream_build_base_pipeline (self);
  }

  g_return_val_if_fail (self->priv->rtpbin != NULL, FALSE);
  g_return_val_if_fail (self->priv->pipeline != NULL, FALSE);

  if (self->priv->send_codec_bin)
  {
    g_warning ("Send pipeline already created, will not recreate");
    return TRUE;
  }

  /* Let us automaticaly pick a codec if not set by the user */
  if (self->priv->send_codec_id == -1)
  {
    if (!farsight_rtp_stream_choose_codec (self, &codec_internal))
    {
      return FALSE;
    }

    self->priv->send_codec_id = codec_internal->codec->id;
  }
  else
  {
    /* let's get the CodecInternal for the selected PT */
    codec_internal = lookup_codec_by_pt (
        farsight_stream_get_media_type (FARSIGHT_STREAM (self)), 
        self->priv->send_codec_id);
    if (!codec_internal) {
      g_warning ("Codec %d not supported", self->priv->send_codec_id);
      goto error;
    }
  }

  g_message ("%s (%d): creating send pipeline with codec %d", __FUNCTION__,
      __LINE__, self->priv->send_codec_id);

  if (!self->priv->src && !codec_internal->has_src)
  {
    g_message ("No source has been set yet, send pipeline build for later");
    self->priv->build_send_pipeline = TRUE;
    return FALSE;
  }

  /* add active remote candidate as destination if set */
  if (self->priv->active_remote_candidate)
  {
    farsight_rtp_stream_add_remote_candidate_to_rtpbin(self,
        self->priv->active_remote_candidate);
  }

  /*
   * sending part of pipeline
   */

  if (self->priv->src)
  {
    /* let's add the source to our bin if there is no user provided pipeline or
     * if the given source is not added to the user pipeline */
    if ((self->priv->main_pipeline && gst_element_get_parent (self->priv->src) == NULL)
        || !self->priv->main_pipeline)
      gst_bin_add (GST_BIN (self->priv->pipeline),
          self->priv->src);
  }

  /* build send part based on send_pipeline_factory */
  /* put all these elements in a bin */
  codec_bin = create_codec_bin (farsight_stream_get_media_type
      (FARSIGHT_STREAM(self)), self->priv->send_codec_id, DIR_SEND);
  if (!codec_bin)
  {
    g_warning ("Couldn't create elements for codec %d", self->priv->send_codec_id);
    goto error;
  }

  gst_bin_add (GST_BIN (self->priv->pipeline), codec_bin);

  if (self->priv->src)
  {
    if ((GstElement *)gst_element_get_parent (self->priv->src) == self->priv->pipeline)
    {
      /* connect src to codec_bin */
      gchar *tmp_caps = gst_caps_to_string (self->priv->src_filter);
      g_debug ("linking src %p to codec bin %p with caps %s", self->priv->src, codec_bin,
          tmp_caps);
      g_free (tmp_caps);
      if (!gst_element_link_filtered (self->priv->src, codec_bin, self->priv->src_filter))
      {
        g_warning ("Could not link src to codec bin");
        goto error;
      }
    }
    else
    {
      /* codec_bin and source have different parents, let's use our ghostpad */
      GstPad *ghostpad = gst_element_get_pad (self->priv->pipeline, "sink");
      GstPad *codec_bin_sink_pad = gst_element_get_pad (codec_bin, "sink");
      gst_ghost_pad_set_target ((GstGhostPad *)ghostpad, codec_bin_sink_pad);
      gst_element_link_filtered (self->priv->src, self->priv->pipeline,
          self->priv->src_filter);
      gst_object_unref (ghostpad);
      gst_object_unref (codec_bin_sink_pad);
    }
  }

  /* connect rtpbin to codec_bin */
  if (!gst_element_link_pads (codec_bin, "src", self->priv->rtpbin, "sink%d"))
  {
    g_warning ("Could not link codec_bin:src to rtpbin:sink");
    goto error;
  }

#ifdef HAVE_JINGLEP2P
  /* create jingle specific icesink element that give data to
   * libjingle */
  GstElement *icesink = NULL;
  icesink = gst_element_factory_make ("icesink", NULL);

  gst_bin_add (GST_BIN (self->priv->pipeline), icesink);

  g_object_set (G_OBJECT(icesink), "socketclient", self->priv->socket_client,
      NULL);

  if (!gst_element_link_pads (self->priv->rtpbin, "rtpsrc", icesink, "sink"))
  {
    g_warning ("Could not link rtpbin:rtpsrc to icesink:sink");
    goto error;
  }
#endif

  self->priv->send_codec_bin = codec_bin;

  self->priv->build_send_pipeline = FALSE;

  /* if user requested to block the pad, let's do it */
  if (self->priv->sending == FALSE)
  {
    self->priv->sending = TRUE;
    farsight_rtp_stream_set_sending (FARSIGHT_STREAM (self), FALSE);
  }

  if (self->priv->start_me)
  {
    farsight_rtp_stream_start (FARSIGHT_STREAM(self));
  }

return TRUE;

error:
  g_warning ("%s (%d): error setting up send codec pipeline",
      __FUNCTION__, __LINE__);
  return FALSE;
}

static gboolean
farsight_rtp_stream_start (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;

  g_return_val_if_fail (self != NULL, FALSE);
  if (self->priv->pipeline == NULL ||
      self->priv->rtpbin == NULL ||
      self->priv->send_codec_bin == NULL ||
      farsight_stream_get_state (stream) != FARSIGHT_STREAM_STATE_CONNECTED)
  {
    self->priv->start_me = TRUE;
    return FALSE;
  }

  self->priv->start_me = FALSE;

  gst_element_set_state (self->priv->pipeline, GST_STATE_PLAYING);

  return TRUE;
}


/**
 * Stop a #FarsightRTPStream instance.
 *
 * @param stream #FarsightRTPStream instance
 */
static void
farsight_rtp_stream_stop (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;

  g_return_if_fail (stream != NULL);

  if (self->priv->timeout_src != -1)
  {
      g_source_remove (self->priv->timeout_src);
      self->priv->timeout_src = -1;
  }

  if (self->priv->pipeline)
  {
    g_debug ("%s (%d): stopping media pipeline", __FUNCTION__, __LINE__);

    /* let's unlink from the source first */
    farsight_rtp_stream_unlink_source (self);

    /* let's unlink the sink now */
    if (self->priv->main_pipeline)
    {
      gchar *name = g_strdup_printf ("recv%d", self->priv->recv_codec_id);
      GstElement *codec_bin = gst_bin_get_by_name (GST_BIN(self->priv->pipeline), name);
      g_free (name);

      if (codec_bin && self->priv->sink)
      {
        g_debug ("unlinking sink");
        gst_element_unlink (codec_bin, self->priv->sink);
      }
      if (codec_bin)
      {
        gst_object_unref (GST_OBJECT (codec_bin));
      }
    }

    GstStateChangeReturn state_set_return;
    g_debug ("%s: Setting state to NULL", __FUNCTION__);
    state_set_return = gst_element_set_state (self->priv->pipeline, GST_STATE_NULL);
    g_debug ("%s: DONE Setting state to NULL returned %d", __FUNCTION__, state_set_return);

    if (state_set_return == GST_STATE_CHANGE_ASYNC)
    {
      GstStateChangeReturn state;
      g_debug ("%s: Getting state", __FUNCTION__);
      state = gst_element_get_state (self->priv->pipeline, NULL, NULL, 5 * GST_SECOND);
      g_debug ("%s: DONE Getting state", __FUNCTION__);
      switch(state)
      {
        case GST_STATE_CHANGE_FAILURE:
          g_warning("Unable to set pipeline to NULL! This could break the "
              "teardown");
          break;
        case GST_STATE_CHANGE_ASYNC:
          g_warning ("State change not finished, after 5 seconds. This could "
              "break the teardown");
          break;
        default:
          break;
      }
    }
    else if (state_set_return == GST_STATE_CHANGE_FAILURE)
    {
      g_warning ("State change unsuccessfull. This could break the teardown");
    }
    else if (state_set_return == GST_STATE_CHANGE_SUCCESS)
    {
      g_debug ("Changed pipeline state to NULL succesfully");
    }

    if (self->priv->bus_watch)
      g_source_remove (self->priv->bus_watch);

    if (!self->priv->main_pipeline)
    {
      gst_object_unref (GST_OBJECT (self->priv->pipeline));
    }
    else
    {
      gst_bin_remove (GST_BIN (self->priv->main_pipeline), self->priv->pipeline);
    }
    self->priv->pipeline = NULL;
    self->priv->src = NULL;
    self->priv->sink = NULL;

    if (self->priv->depayloader_chains_info) {
      g_hash_table_destroy (self->priv->depayloader_chains_info);
      self->priv->depayloader_chains_info = NULL;
    }
  }
  else
  {
    if (self->priv->src)
    {
      gst_object_unref (GST_OBJECT (self->priv->src));
      self->priv->src = NULL;
    }
    if (self->priv->sink)
    {
      gst_object_unref (GST_OBJECT (self->priv->sink));
      self->priv->sink = NULL;
    }
  }

#ifdef HAVE_JINGLEP2P
  if (self->priv->socket_client)
  {
    g_debug ("%s (%d): destroying socket client", __FUNCTION__, __LINE__);
    socketclient_destroy (self->priv->socket_client);
    self->priv->socket_client = NULL;
  }
#endif

  if (self->priv->use_upnp)
  {
    farsight_rtp_stream_upnp_close_ports (self);
#ifdef HAVE_CLINKC
    upnp_cp_shutdown();
#endif
  }

  farsight_stream_signal_state_changed (stream,
      FARSIGHT_STREAM_STATE_DISCONNECTED,
      farsight_stream_get_direction (stream));
}

static gboolean
farsight_rtp_stream_set_sending (FarsightStream *stream, gboolean sending)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;

  GstElement *codec_bin = NULL;
  GstPad *codec_bin_src_pad  = NULL;
  gchar *name = NULL;

  if (self->priv->sending == sending)
  {
    return TRUE;
  }

  self->priv->sending = sending;

  if (self->priv->pipeline == NULL)
  {
    g_warning ("No pipeline present, will set sending later");
    return FALSE;
  }

  name = g_strdup_printf ("send%d", self->priv->send_codec_id);
  codec_bin = gst_bin_get_by_name (GST_BIN(self->priv->pipeline), name);
  g_free (name);

  if (codec_bin)
  {
    codec_bin_src_pad = gst_element_get_pad (codec_bin, "src");
    if (!codec_bin_src_pad)
    {
      g_warning ("send codec has no source pad! This shouldn't happen");
      return FALSE;
    }
    gst_object_unref (GST_OBJECT (codec_bin));
  }
  else
  {
    g_message ("send codec bin not created yet, will set sending later");
    return TRUE;
  }

  if (sending)
  {
    g_debug ("Setting sending to %d", sending);
    gst_pad_set_blocked_async (codec_bin_src_pad, FALSE, unblocked_cb, NULL);
   /* g_object_set (G_OBJECT (stream), "direction",
        FARSIGHT_STREAM_DIRECTION_BOTH, NULL);*/
  }
  else
  {
    g_debug ("Setting sending on %d", sending);
    gst_pad_set_blocked_async (codec_bin_src_pad, TRUE, blocked_cb, NULL);
    /*g_object_set (G_OBJECT (stream), "direction",
        FARSIGHT_STREAM_DIRECTION_RECEIVEONLY, NULL);*/
  }

  return TRUE;
}

gboolean 
farsight_rtp_stream_choose_codec (FarsightRTPStream *self, 
                                  CodecInternal **codec_internal)
{
  GList *codec_list = self->priv->remote_codecs;
  do {
    if ((*codec_internal = match_codec (farsight_stream_get_media_type
            (FARSIGHT_STREAM (self)), codec_list->data)))
    {
      return TRUE;
    }
    codec_list = codec_list->next;
  } while (codec_list);
  return FALSE;
}

static void
farsight_rtp_stream_upnp_send_request(FarsightRTPStream *self,
    FarsightTransportInfo *derived_trans)
{
#ifdef HAVE_CLINKC
  FarsightTransportInfo *trans = NULL;
  const GList *lp;
  GList *candidate_list = self->priv->local_candidates;


  g_debug ("Looking for local ip");

  for (lp = candidate_list; lp; lp = g_list_next (lp)) {
    trans = (FarsightTransportInfo *) lp->data;
    if (trans->type == FARSIGHT_CANDIDATE_TYPE_LOCAL)
    {
      if (trans->ip)
      {
        /* let's ignore loopback */
        if (g_ascii_strcasecmp(trans->ip, "127.0.0.1") != 0)
        {
          g_debug ("Found local_ip %s", trans->ip);
          /* open ports */
          upnp_open_port_all_igds (trans->ip, derived_trans->port, derived_trans->port,
              (derived_trans->proto == FARSIGHT_NETWORK_PROTOCOL_UDP)?
              IPPROTO_UDP:IPPROTO_TCP);
        }
      }
    }
  }
#endif
}

static void
farsight_rtp_stream_upnp_close_ports (FarsightRTPStream *self)
{
#ifdef HAVE_CLINKC
  FarsightTransportInfo *trans = NULL;
  const GList *lp;
  GList *candidate_list = self->priv->local_candidates;

  for (lp = candidate_list; lp; lp = g_list_next (lp)) {
    trans = (FarsightTransportInfo *) lp->data;
    if (trans->type == FARSIGHT_CANDIDATE_TYPE_DERIVED)
    {
      g_debug ("Found derived ip %s", trans->ip);
      /* close ports */
      upnp_close_port_all_igds (trans->port, 
          (trans->proto == FARSIGHT_NETWORK_PROTOCOL_UDP)?
            IPPROTO_UDP:IPPROTO_TCP);
    }
  }
#endif
}

static gboolean 
farsight_rtp_stream_candidate_exists (FarsightStream *stream, 
                                      const GList *candidate_list, 
                                      const GList *candidate)
{
  FarsightTransportInfo *trans = NULL;
  FarsightTransportInfo *trans2 = NULL;
  const GList *lp = NULL;
  const GList *lp2 = NULL;
  gint i = 0;

  if (candidate_list == NULL || candidate == NULL)
    return FALSE;

  /* we check ip and port */
  for (lp = candidate; lp; lp = g_list_next (lp)) {
    trans = (FarsightTransportInfo *) lp->data;

    for (lp2 = candidate_list; lp2; lp2 = g_list_next (lp2)) {
      trans2 = (FarsightTransportInfo *) lp2->data;
      if (farsight_transport_are_equal (trans, trans2))
        i++;
    }
  }

  if (i == g_list_length ((GList *)candidate))
    return TRUE;
  else if (i == 0)
    return FALSE;
  else
  {
    g_error("Candidate only partially exists (some components do), \
              this should not happen!");
    return FALSE;
  }
}

#ifdef HAVE_JINGLEP2P

static void 
farsight_rtp_stream_native_candidate_ready (gpointer stream, 
                                            const GList *candidate)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;
  FarsightTransportInfo *trans = NULL;
  GList *candidate_copy = NULL;

  g_debug ("Called farsight_rtp_stream_native_candidate_ready");

  if (farsight_rtp_stream_candidate_exists (stream, self->priv->local_candidates,
              candidate))
  {
    g_message ("Native candidate already in list, not adding");
    return;
  }
  else
  {
    g_debug ("Native candidates found, adding to list");
    candidate_copy = farsight_transport_list_copy(candidate);
    self->priv->local_candidates = g_list_concat(self->priv->local_candidates, 
        candidate_copy);
    trans = (FarsightTransportInfo *) candidate_copy->data;
    farsight_stream_signal_new_native_candidate (stream,
            (gchar *)trans->candidate_id);

    if (self->priv->use_upnp)
    {
      /* TODO this assumes the local IP is found before the derived IP */
      /* if not it won't send any request since it needs the local ip */
      if (trans->type == FARSIGHT_CANDIDATE_TYPE_DERIVED)
      {
        farsight_rtp_stream_upnp_send_request (self, trans);
      }
    }
  }

  // farsight_stream_signal_native_candidates_prepared (stream);
}

static void 
farsight_rtp_stream_socket_state_changed (gpointer stream, gint state)
{
  g_message ("socket state changed to %d", state);

  FarsightRTPStream *self = FARSIGHT_RTP_STREAM (stream);
  if (state == 1)     /* writable */
    {
      if (self->priv->timeout_src)
      {
        g_source_remove (self->priv->timeout_src);
        self->priv->timeout_src = -1;
      }

      /* FIXME: dirty hack */
      farsight_stream_signal_new_active_candidate_pair
        (FARSIGHT_STREAM (stream), "foo", "bar");
      farsight_stream_signal_state_changed (stream,
          FARSIGHT_STREAM_STATE_CONNECTED,
          farsight_stream_get_direction (stream));

      if (self->priv->start_me)
      {
        farsight_rtp_stream_start (stream);
      }
    }
  else if (state == 0)
    {
      if (self->priv->timeout_src)
      {
        g_source_remove (self->priv->timeout_src);
        self->priv->timeout_src = -1;
      }
      /* Let us set a timer for a timeout on reestablishing a connection */
      self->priv->timeout_src = g_timeout_add (self->priv->conn_timeout * 1000,
          farsight_rtp_stream_connection_timed_out,
          self);

      /* farsight_rtp_stream_stop (FARSIGHT_STREAM (stream)); */
      /* What we really want here is some sort of _pause/_resume functionality
       * in farsight */
      farsight_stream_signal_state_changed (stream,
          FARSIGHT_STREAM_STATE_CONNECTING,
          farsight_stream_get_direction (stream));
    }
}

static void 
farsight_rtp_stream_network_error (gpointer stream)
{
  g_warning ("error from jinglep2p socketmanager.");
  farsight_rtp_stream_stop (stream);
  farsight_stream_signal_error (stream,
          FARSIGHT_STREAM_ERROR_NETWORK, 
          "Network error from jinglep2p socketmanager");
}
#endif

static gboolean
farsight_rtp_stream_connection_timed_out (gpointer data)
{
  FarsightRTPStream *self = (FarsightRTPStream *) data;
 
  self->priv->timeout_src = -1;

  /* let's check if we are connected yet, throw an error if not */
  if (farsight_stream_get_state (FARSIGHT_STREAM(self)) ==
      FARSIGHT_STREAM_STATE_CONNECTING)
  {
    g_warning ("Could not establish a connection");
    farsight_stream_signal_error (FARSIGHT_STREAM(self),
        FARSIGHT_STREAM_ERROR_TIMEOUT, "Could not establish a connection");
    farsight_rtp_stream_stop (FARSIGHT_STREAM(self));
  }

  return FALSE;
}

/* This is to signal you to prepare the list of transport condidates,
 * e.g. start off any network probing, like STUN
 * */
static void 
farsight_rtp_stream_prepare_transports (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;

#ifdef HAVE_JINGLEP2P
  if (self->priv->prepared)
    return;

  g_message ("Init and running jinglep2p to prepare native candiates, using stun %s : %d and turn %s : %d", 
      self->priv->stun_ip, self->priv->stun_port, self->priv->turn_ip, self->priv->turn_port);
  self->priv->socket_client = socketclient_init(self->priv->stun_ip,
          self->priv->stun_port, self->priv->turn_ip, self->priv->turn_port);

  connect_signal_candidates_ready (self->priv->socket_client,
          farsight_rtp_stream_native_candidate_ready, stream);
  connect_signal_socket_state_change (self->priv->socket_client,
          farsight_rtp_stream_socket_state_changed, stream);
  connect_signal_network_error (self->priv->socket_client,
          farsight_rtp_stream_network_error, stream);

  socketclient_create_socket (self->priv->socket_client, "rtp");

  socketclient_start_processing_candidates (self->priv->socket_client);

  farsight_stream_signal_state_changed (stream,
      FARSIGHT_STREAM_STATE_CONNECTING, 
      farsight_stream_get_direction (stream));
#else
#ifdef HAVE_SOFIASIP
  FarsightTransportInfo *trans;

  su_sockaddr_t *su;
  su_localinfo_t *interfaces, *li;
  su_getlocalinfo (NULL, &interfaces);
  char buf[256];

  if (self->priv->prepared)
    return;

  /* TODO : replace by farsight net socket api from kai */
  struct sockaddr_in address;
  gint sock;
  guint16 port = self->priv->base_local_port;
  gint bindresult = -1;
  while (bindresult != 0)
  {
    if ((sock = socket (AF_INET,SOCK_DGRAM,0)) > 0)
        g_debug ("The socket was created");
    else
        g_debug ("Error creating socket");
    address.sin_family = AF_INET;
    address.sin_addr.s_addr = INADDR_ANY;
    address.sin_port = htons (port);
    bindresult = bind (sock, (struct sockaddr *) &address, sizeof (address));
    if (bindresult != 0)
    {
      g_debug ("could not bind port %d", port);
      port++;
    }
  }
  g_debug ("bound to port %d", port);
  self->priv->local_socket = sock;
  self->priv->local_port = port;
  /* \ TODO */

  gint i = 1;
  for (li = interfaces; li; li=li->li_next) {
    if (li->li_family == AF_INET) {
      su = li->li_addr;
      trans = g_new0 (FarsightTransportInfo,1);
      trans->candidate_id = g_strdup_printf ("L%d", i);
      trans->component = 1;
      trans->ip = g_strdup (inet_ntop (su->su_family, SU_ADDR(su), 
                                       buf, sizeof(buf)));
      trans->port = self->priv->local_port;
      trans->proto = FARSIGHT_NETWORK_PROTOCOL_UDP;
      trans->proto_subtype = g_strdup ("RTP");
      trans->proto_profile = g_strdup ("AVP");
      trans->preference = 1.0;
      trans->type = FARSIGHT_CANDIDATE_TYPE_LOCAL;
      self->priv->local_candidates = g_list_prepend (self->priv->local_candidates, trans);
      i++;
      farsight_stream_signal_new_native_candidate (stream, (gchar *) trans->candidate_id);
    }
  }
  self->priv->local_candidates = g_list_reverse (self->priv->local_candidates);
  self->priv->prepared = TRUE;

  farsight_stream_signal_native_candidates_prepared (stream); 
#else
  g_debug ("Getting local interfaces");
  FarsightTransportInfo *trans;
  GList *ips = NULL;

  ips = farsight_get_local_ips(FALSE);

  if (self->priv->prepared)
    return;

  /* TODO : replace by farsight net socket api from kai */
  struct sockaddr_in address;
  gint sock;
  guint16 port = self->priv->base_local_port;
  gint bindresult = -1;
  while (bindresult != 0)
  {
    if ((sock = socket (AF_INET,SOCK_DGRAM,0)) > 0)
        g_debug ("The socket was created");
    else
        g_debug ("Error creating socket");
    address.sin_family = AF_INET;
    address.sin_addr.s_addr = INADDR_ANY;
    address.sin_port = htons (port);
    bindresult = bind (sock, (struct sockaddr *) &address, sizeof (address));
    if (bindresult != 0)
    {
      g_debug ("could not bind port %d", port);
      port++;
    }
  }
  g_debug ("bound to port %d", port);
  self->priv->local_socket = sock;
  self->priv->local_port = port;
  /* \ TODO */

  gint i = 1;
  GList *current;
  for (current = ips; current; current = g_list_next(current)) {
    trans = g_new0 (FarsightTransportInfo,1);
    trans->candidate_id = g_strdup_printf ("L%d", i);
    trans->component = 1;
    trans->ip = g_strdup (current->data);
    trans->port = self->priv->local_port;
    trans->proto = FARSIGHT_NETWORK_PROTOCOL_UDP;
    trans->proto_subtype = g_strdup ("RTP");
    trans->proto_profile = g_strdup ("AVP");
    trans->preference = 1.0;
    trans->type = FARSIGHT_CANDIDATE_TYPE_LOCAL;
    self->priv->local_candidates = g_list_prepend (self->priv->local_candidates, trans);
    i++;
    farsight_stream_signal_new_native_candidate (stream, (gchar *) trans->candidate_id);
  }
  self->priv->local_candidates = g_list_reverse (self->priv->local_candidates);
  self->priv->prepared = TRUE;

  farsight_stream_signal_native_candidates_prepared (stream); 

  /* free list of ips */
  g_list_foreach (ips, (GFunc) g_free, NULL);
  g_list_free (ips);
#endif
#endif

  /* Let us set a timer for a timeout on establishing a connection */
  self->priv->timeout_src = g_timeout_add (self->priv->conn_timeout * 1000,
      farsight_rtp_stream_connection_timed_out,
      self);

  /* We can already create the core pipeline now */
  farsight_rtp_stream_build_base_pipeline (self);
}

static gboolean 
farsight_rtp_stream_add_remote_candidate_to_rtpbin (FarsightRTPStream *self, 
                                                    const gchar *remote_candidate_id)
{
  GList *remote_candidate;
  gchar *addr = NULL;
  FarsightTransportInfo *trans = NULL;
  const GList *lp;

  if (self->priv->rtpbin)
  {
    remote_candidate = farsight_transport_get_list_for_candidate_id
        (self->priv->remote_candidates, remote_candidate_id);
    if (remote_candidate == NULL)
      return FALSE;

    /* Find remote candidate that is of subtype RTP */
    for (lp = remote_candidate; lp; lp = g_list_next (lp)) {
      trans = (FarsightTransportInfo *) lp->data;
      if (g_ascii_strcasecmp(trans->proto_subtype, "RTP") == 0)
      {
        break;
      }
    }
    if (trans == NULL)
      return FALSE;
    /* Set the ip:port of that transport to rtpbin */
    /* TODO there should be a way to set alternative RTCP ip:port that are
     * not simply ip:port+1 */
    addr = g_strdup_printf ("%s:%d", trans->ip, trans->port);
    g_object_set (G_OBJECT (self->priv->rtpbin),
            "destinations", addr, NULL);

    g_free(addr);
    return TRUE;
  }
  return FALSE;
}

static gboolean 
farsight_rtp_stream_set_active_candidate_pair (FarsightStream *stream,
                                               const gchar *native_candidate_id, 
                                               const gchar *remote_candidate_id)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;

  self->priv->active_native_candidate = g_strdup (native_candidate_id);
  self->priv->active_remote_candidate = g_strdup (remote_candidate_id);

  /* this will not work if the pipeline has not yet been created */
  if (!farsight_rtp_stream_add_remote_candidate_to_rtpbin (self,
              remote_candidate_id))
    return FALSE;

  return TRUE;
}

static GList *
farsight_rtp_stream_get_native_candidate (FarsightStream *stream, 
                                          const gchar *candidate_id)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;
  GList *candidate = NULL;

  candidate = farsight_transport_get_list_for_candidate_id
          (self->priv->local_candidates, candidate_id);

  return candidate;
}

static G_CONST_RETURN GList *
farsight_rtp_stream_get_native_candidate_list (FarsightStream *stream)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;
  return self->priv->local_candidates;
}

static void 
farsight_rtp_stream_set_remote_candidate_list (FarsightStream *stream, 
                                               const GList *remote_candidates)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;

  self->priv->remote_candidates = farsight_transport_list_copy(remote_candidates);
}

static void 
farsight_rtp_stream_add_remote_candidate (FarsightStream *stream,
                                          const GList *remote_candidate)
{
  FarsightRTPStream *self = (FarsightRTPStream *) stream;
  FarsightTransportInfo *info = (FarsightTransportInfo *)remote_candidate->data;
  g_debug ("%s (%d): adding remote candidate %s %d",
          __FUNCTION__, __LINE__, info->ip, info->port);
  GList *rc_copy = farsight_transport_list_copy (remote_candidate);

  if (self->priv->remote_candidates == NULL)
  {
    self->priv->remote_candidates = rc_copy;
  }
  else
  {
    if (farsight_rtp_stream_candidate_exists (stream,
                self->priv->remote_candidates, rc_copy))
    {
      g_message ("Remote candidate already in list, not adding");
      return;
    }
    else
    {
      self->priv->remote_candidates = g_list_concat(self->priv->remote_candidates, 
                rc_copy);
      g_debug ("%s (%d): Added remote candidate",
                __FUNCTION__, __LINE__);
    }
  }

/* if using jingle, we need to add the candidates to the socket */
#ifdef HAVE_JINGLEP2P
  if (self->priv->socket_client)
    socketclient_add_remote_candidates(self->priv->socket_client,
              rc_copy);
#endif
}
