/* Certificate Management library
 * 
 * Copyright (C) 2005 Nokia. All rights reserved.
 * Author: Ed Bartosh <Eduard.Bartosh@nokia.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/** cst_crypt.c **/
#include "cst_t.h"
#include <openssl/bio.h>
#include <openssl/evp.h>
#include <openssl/rand.h>
#include <string.h>

//#define ALGORITHM EVP_rc5_32_12_16_cbc()
#define ALGORITHM DEFAULT_ALGORITHM
#define MD EVP_sha1()

int crypt_reader_eof(CRYPT_FILE * cf)
{
#ifdef SECURE
    return BIO_eof(cf->cipher);
#else
    return feof(cf->fp);
#endif
}

CRYPT_FILE *crypt_reader_open(const char *filename, char *password)
{
    g_assert(filename);
    g_assert(password && (password[0] != 0));
    TRACE("*");

    CRYPT_FILE *cf = g_malloc(sizeof(CRYPT_FILE));
#ifdef SECURE
    cf->file = BIO_new_file(filename, "rb");
    if (!cf->file)
    {
        CST_error(CST_ERROR_NOT_FOUND);
        g_free(cf);
        return NULL;
    }

    /* Make salt and write it to cf->file */ 
    unsigned char salt[PKCS5_SALT_LEN];
    cst_bio_read(cf->file, salt, sizeof(salt));
#ifdef NEED_TRACE
    printf("READ SALT: %s\n", buffer_to_hex(salt, sizeof(salt)));
#endif
    /* Make key based on password and salt*/
    unsigned char key[EVP_MAX_KEY_LENGTH];
    unsigned char iv[EVP_MAX_IV_LENGTH];
    EVP_BytesToKey(ALGORITHM, MD, salt, (unsigned char *)password,
            strlen(password), 1, key, iv);
    
    
    cf->buffer = BIO_new(BIO_f_buffer());
    cf->cipher = BIO_new(BIO_f_cipher());

    TRACE_S(password);
    BIO_set_cipher(cf->cipher, ALGORITHM, key, iv, 0);

    BIO_push(cf->cipher, cf->buffer);
    BIO_push(cf->buffer, cf->file);
#else
    cf->fp = fopen(filename, "rb");
    if (!cf->fp)
    {
        g_free(cf);
        return NULL;
    }
#endif
    return cf;
}

CRYPT_FILE *crypt_writer_open(const char *filename, char *password)
{
    g_assert(filename);
    g_assert(password && (password[0] != 0));

    TRACE("*");

    g_assert(filename);
    CRYPT_FILE *cf = g_malloc(sizeof(CRYPT_FILE));
#ifdef SECURE
    cf->file = BIO_new_file(filename, "wb");
    if (!cf->file)
    {
        CST_error(CST_ERROR_CREATE_FILE);
        g_free(cf);
        return NULL;
    }

    /* Make salt and write it to cf->file */ 
    unsigned char salt[PKCS5_SALT_LEN];
    if (RAND_pseudo_bytes(salt, sizeof(salt)) < 0)
    {
        CST_error(CST_ERROR_CREATE_FILE);
        g_free(cf);
        return NULL; 
    }
    cst_bio_write(cf->file, salt, sizeof(salt));
    BIO_flush(cf->file);
#ifdef NEED_TRACE
    printf("WRITE SALT: %s\n", buffer_to_hex(salt, sizeof(salt)));
#endif
    
    /* Make key based on password and salt*/
    unsigned char key[EVP_MAX_KEY_LENGTH];
    unsigned char iv[EVP_MAX_IV_LENGTH];
    EVP_BytesToKey(ALGORITHM, MD, salt, (unsigned char *)password,
            strlen(password), 1, key, iv);

    /* Create BIO need for encrypt */
    cf->buffer = BIO_new(BIO_f_buffer());
    cf->cipher = BIO_new(BIO_f_cipher());
    
    BIO_set_cipher(cf->cipher, ALGORITHM, key, iv, 1);

    BIO_push(cf->cipher, cf->buffer);
    BIO_push(cf->buffer, cf->file);
#else
    cf->fp = fopen(filename, "wb");
    if (!cf->fp)
    {
        g_free(cf);
        return NULL;
    }
#endif
    return cf;
}

void crypt_reader_close(CRYPT_FILE * cf)
{
    TRACE("*");
#ifdef SECURE
    BIO_free_all(cf->cipher);
#else
    fclose(cf->fp);
#endif
    g_free(cf);
}

void crypt_writer_close(CRYPT_FILE * cf)
{
    g_assert(cf);
    TRACE("*");
#ifdef SECURE
    BIO_flush(cf->cipher);
    BIO_free_all(cf->cipher);
#else
    fflush(cf->fp);
    fclose(cf->fp);
#endif
    g_free(cf);
}

int cst_bio_read(BIO * bio, void *buffer, int len)
{
    g_assert(bio);
    g_assert(buffer);
    
    int total, read;
    for (total = 0; total < len; total += read)
    {
        if ((read =
             BIO_read(bio, buffer + total, len - total)) <= 0)
        {
            if (BIO_should_retry(bio))
            {
                read = 0;
                continue;
            }
            break;
        }
    }
    return total;
}


int crypt_reader_get(CRYPT_FILE * cf, void *buffer, int len)
{
    g_assert(cf);
    g_assert(buffer);
#ifdef SECURE
    return cst_bio_read(cf->cipher, buffer, len);
#else
    g_assert(cf->fp);
    return fread(buffer, 1, len, cf->fp);
#endif
}

int cst_bio_write(BIO * bio, void *buffer, int len)
{
    g_assert(bio);

    if (buffer)
    {
        int total, written;

        for (total = 0; total < len; total += written)
        {
            if ((written =
                 BIO_write(bio, buffer + total, len - total)) <= 0)
            {
                if (BIO_should_retry(bio))
                {
                    written = 0;
                    continue;
                }
                break;
            }
        }
        return total;
    }
    return 0;
}


int crypt_writer_put(CRYPT_FILE * cf, void *buffer, int len)
{
    g_assert(cf);

    if (buffer)
    {
#ifdef SECURE
        return cst_bio_write(cf->cipher, buffer, len);
#else
        g_assert(cf->fp);
        return fwrite(buffer, 1, len, cf->fp);
#endif
    }
    return 0;
}
