/**
   X Colorkey eXtension

   Authors:
   Tapani Paelli <tapani.palli@nokia.com>

   Stores a colorkey value (int) in a given GC and uses it to
   create optimizing tables for pixmaps. When pixmaps are drawn,
   these tables can be used to speed up drawing.

   ideas :
     * implement dimming here
     * draw half of scanlines to increase fullscreen game speed

   todo
     * some other type optimization data for a 'complex' pixmap
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <X11/extensions/xckproto.h>
#include "xck.h"
#include "fb.h"

/* no new events or errors introduced here */
#define event_amount  0
#define error_amount  0

#define XCK_DEBUG

static unsigned char XCK_request_code;

/* some function declarations */
static DISPATCH_PROC (ProcXCKDispatch);     /* request handler, called from servers dispatch loop */
static DISPATCH_PROC (ProcXCKQueryVersion); /* version query function */
static DISPATCH_PROC (ProcXCKSetColorkey);  /* sets a colorkey value into given GC */
static DISPATCH_PROC (ProcXCKScanPixmap);   /* create optimizing tables if possible */
static DISPATCH_PROC (ProcXCKBlit);         /* blit drawable->drawable */
static DISPATCH_PROC (ProcXCKSetState);     /* set colorkey state in a GC (TRUE or FALSE) */

void ProcXCKCloseDown (ExtensionEntry*);


/**
   Initializes PixmapPrivateIndex for use

   @par current screen
   @return boolean success value
*/
BOOL init_screen (ScreenPtr pScreen)
{
  static int XCKGeneration = -1;
  xckScrPrivPtr pScrPriv;

  if (XCKGeneration != serverGeneration)
  {
    xckScrPrivateIndex = AllocateScreenPrivateIndex ();
    if (xckScrPrivateIndex == -1)
      return FALSE;

    xckGCPrivateIndex = AllocateGCPrivateIndex();
    if (xckGCPrivateIndex == -1)
      return FALSE;

    xckPixPrivateIndex = AllocatePixmapPrivateIndex();
    if (xckPixPrivateIndex == -1)
      return FALSE;

    /* allocate indexes only once */
    XCKGeneration = serverGeneration;
  }
  
  if (pScreen->devPrivates[xckScrPrivateIndex].ptr)
    return TRUE; /* indexes already exist */

  if(!AllocatePixmapPrivate(pScreen, xckPixPrivateIndex, sizeof(XCK_data_rec)))
    return FALSE;

  if (!AllocateGCPrivate (pScreen, xckGCPrivateIndex, sizeof (xckGCPrivRec)))
    return FALSE;

  /* allocate memory for xck screen structure */
  pScrPriv = (xckScrPrivPtr) xalloc (sizeof (xckScrPrivRec));
  if (!pScrPriv)
    return FALSE;

  /* wrap needed functions */
  wrap (pScrPriv, pScreen, CreatePixmap, xckCreatePixmap);
  wrap (pScrPriv, pScreen, DestroyPixmap, xckDestroyPixmap);
  wrap (pScrPriv, pScreen, CreateGC, xckCreateGC);

  pScreen->devPrivates[xckScrPrivateIndex].ptr = (pointer) pScrPriv;
  return TRUE;
}


/**
	Initializes extension-entry structure
	and allocates memory for needed structures.

	@return void
*/
void InitColorkeyExtension (void)
{
    ExtensionEntry *entry;
    int s;
    /**
	Here I use 'StandardMinorOpcode'. It means that request
	structs must have minorOpcode as their second variable.
    */
    entry = AddExtension (XCK_NAME,event_amount,error_amount,
                          ProcXCKDispatch, ProcXCKDispatch,
                          ProcXCKCloseDown,StandardMinorOpcode);
    if (!entry)
    {
        FatalError ("XCK :: AddExtension failed");
        return;
    }

    /* initialize every screen       */
   for (s = 0; s < screenInfo.numScreens; s++)
   {
     if(!init_screen (screenInfo.screens[s]))
     {
       FatalError ("XCK :: init_screen failed");
       return;
     }
   }
    XCK_request_code = (unsigned char) entry->base;
}


/**
	Dispatch function. works as a request handler.

	usable macros : server/include/dix.h

	@par client	client pointer
	@return int	function pointer
*/
int ProcXCKDispatch (ClientPtr client)
{
  /* declare local variable 'stuff' */
  /* that points to an xReq struct  */
  REQUEST (xReq);

  switch (stuff->data)
  {
  case X_XCKQueryVersion :
    return ProcXCKQueryVersion (client);
  case X_XCKSetColorkey :
    return ProcXCKSetColorkey (client);
  case X_XCKScanPixmap :
    return ProcXCKScanPixmap (client);
  case X_XCKBlit :
    return ProcXCKBlit (client);
  case X_XCKSetState :
    return ProcXCKSetState (client);
  default :
    return BadRequest;
  }
}


/**
   Sets colorkey state in a GC.

   @par client	client pointer
 */
static int ProcXCKSetState (ClientPtr client)
{
  REQUEST (xXCKSetStateReq);
  REQUEST_SIZE_MATCH (xXCKSetStateReq);

  GCPtr gc;
  gc = (GCPtr) LookupIDByType (stuff->gc_id, RT_GC);
  if (!gc)
  {
    client->errorValue = stuff -> gc_id;
    return BadMatch;
  }

  xckGCPrivPtr pGCPriv = xckGetGCPriv (gc);
  pGCPriv -> colorkey_active = stuff->state;
  client->lastGC = gc;

  return (client->noClientException);
}


/**
   Scan a pixmap to create optimizing tables.

   @par client	client pointer
*/
static int ProcXCKScanPixmap (ClientPtr client)
{
  REQUEST (xXCKScanPixmapReq);
  REQUEST_SIZE_MATCH (xXCKScanPixmapReq);

  PixmapPtr pix;
  DrawablePtr drawable;

  pix = (PixmapPtr) LookupIDByType (stuff->pixmap , RT_PIXMAP);
  if (!pix)
  {
    fprintf(stderr,"xck: not a valid pixmap\n");
    client->errorValue = stuff -> pixmap;
    return BadMatch;
  }

  drawable = (DrawablePtr) LookupIDByClass (stuff->pixmap , RC_DRAWABLE);
  if (!drawable)
  {
    fprintf(stderr,"xck: bad drawable\n");
    client->errorValue = stuff -> pixmap;
    return BadDrawable;
  }

  /* there HAS to be a lastGC before this */
  if (client->lastGC == NULL)
    return (client->noClientException);
  xckGCPrivPtr pGCPriv = xckGetGCPriv (client->lastGC);

  /* xckSetColorkey has to be run before */
  if (pGCPriv->colorkey >= 0)
  {
    xckScanPixmap (drawable, pGCPriv->colorkey);
  }
  else
  {
    fprintf(stderr,"xck: scan failure, no colorkey defined\n");
    return BadRequest;
  }

  return (client->noClientException);
}


/**
   Reply to a version query.
   'WriteToClient' is used to send the reply.

   @par client	client pointer
 */
static int ProcXCKQueryVersion (ClientPtr client)
{
  REQUEST (xXCKQueryVersionReq);
  ck_version_reply reply;
  int temp;
 
  /* is the request size correct? */
  REQUEST_SIZE_MATCH (xXCKQueryVersionReq);

  /* build the reply */
  reply.type = X_Reply;
  reply.sequenceNumber = client -> sequence;
  reply.length = 0;
  reply.majorVersion = XCK_VERSION_MAJOR;
  reply.minorVersion = XCK_VERSION_MINOR;

  if (client->swapped)
  {
      swaps (&reply.sequenceNumber, temp);
      swaps (&reply.majorVersion, temp);
      swaps (&reply.minorVersion, temp);
  }

  (void) WriteToClient (client, SIZEOF(ck_version_reply), (char*)&reply);
  return (client->noClientException);
}


/**
   Sets a new colorkey value to GC devprivates.

   @par client	client pointer
*/
static int ProcXCKSetColorkey (ClientPtr client)
{
  REQUEST (xXCKSetColorkeyReq);
  REQUEST_SIZE_MATCH (xXCKSetColorkeyReq);

  GCPtr gc;
  gc = (GCPtr) LookupIDByType (stuff->gc_id, RT_GC);
  if (!gc)
  {
    client->errorValue = stuff -> gc_id;
    return BadMatch;
  }

  xckGCPrivPtr pGCPriv = xckGetGCPriv(gc);
  pGCPriv -> colorkey = stuff->pixel;
  pGCPriv -> colorkey_active = TRUE;
  client->lastGC = gc;

  return (client->noClientException);
}


/**
   Blitting with a colorkey.

   @par client	client pointer
*/
static int ProcXCKBlit (ClientPtr client)
{
  DrawablePtr pSrc;
  DrawablePtr pDst;
  RegionPtr pRgn;
  PixmapPtr srcpix;

  REQUEST (xXCKBlitReq);
  REQUEST_SIZE_MATCH (xXCKBlitReq);

  /* source has to be a pixmap */
  srcpix = (PixmapPtr) LookupIDByType (stuff->src , RT_PIXMAP);
  if (!srcpix)
    {
      client->errorValue = stuff -> src;
      return BadMatch;
    }

  pSrc = (DrawablePtr) LookupIDByClass (stuff->src , RC_DRAWABLE);
  if (!pSrc)
    {
      client->errorValue = stuff -> src;
      return BadDrawable;
    }
  pDst = (DrawablePtr) LookupIDByClass (stuff->dst , RC_DRAWABLE);
  if (!pDst)
    {
      client->errorValue = stuff -> dst;
      return BadDrawable;
    }

  XCK_data p = xckGetPixPriv(srcpix);
  xckGCPrivPtr pGCPriv = xckGetGCPriv(client->lastGC);

  /* initialize a new colorkey */
  pGCPriv->colorkey = stuff->pixel;

  if (p->type == XCK_UNKNOWN)
  {
    xckScanPixmap (pSrc, pGCPriv->colorkey);
  }

  /* no transparency found */
  if (p->type == XCK_NORMAL)
    ;

  /* optimized, offsets and scanline positions */
  else if (p->type == XCK_OPTIMIZE)
    colorkey_blit (pSrc,pDst,0,0);

  /* pixel per pixel */
  else  if (p->type == XCK_COMPLEX)
  {
    pRgn = fbDoCopy (pSrc, pDst,
		     client->lastGC,
		     0, 0,
		     pSrc -> width, pSrc -> height,
		     0, 0,
		     (fbCopyProc) xckCopyNtoN,
		     0, 0);
  }
  return (client->noClientException);
}


/**
     Closes extension, deallocates memory

     @return void
 */
void ProcXCKCloseDown (ExtensionEntry *entry)
{
  /* free allocated memory here */
}
