/*
    Copyright (C) 2003 Otso Kiveks, <otso.kivekas@movial.fi>

    This file is part of Outo.

    Outo is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published 
    by the Free Software Foundation; either version 2.1 of the License, or
    (at your option) any later version.

    Outo is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with Outo; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <stdlib.h>
/* #include <signal.h> */
#include <unistd.h>
/* #include <errno.h> */
#include <string.h>
#include <dlfcn.h>
#include <getopt.h>
#include <time.h>
#include <sys/types.h>
#include <sys/wait.h>

#include <outo.h>
#include "report.h"


/*------------------ GLOBALS ---------------------*/
/*
  hey, this is C. in C we use globals when needed.
  (okay, some of these could and should be avoided)
*/

/* the default debug file */
char* debugfile = "/tmp/.outo";

stat c;
stat *curstat = &c;


/* for dlopen */
void *handle;
char *error;

/* int assertsigs = ASSERT_SIGNALS; not in use yet */


/*------------------- TESTING ---------------------*/

/** inits the test library by calling in it function init_tests()
 * if sush function exists. called before every test case.
 */
void init_test()
{  
  void* (*initfunc)();

  initfunc = dlsym(handle, "init_test");
  if ((error = dlerror()) == NULL) 
    initfunc();
}

/** closes the test library by calling in it function post_tests()
 * if sush function exists. called after every test case.
 */
void post_test()
{
  void* (*postfunc)();

  postfunc = dlsym(handle, "post_test");
  if ((error = dlerror()) == NULL) 
    postfunc();
}


/** 
 *  calls the actual test case, run under fork
 *  @return nothing, but exits with 0 on succes, or 1 on fail
 *  or fails on some signal or exits with 2 if test didn't fail on
 * assert when it should have 
 */
static void do_test(testcase* tcase)
{
  int result=0;

  /* set stderr & stdout */
  stderr = freopen(debugfile, "a", stderr); 
  stdout = freopen(debugfile, "a", stdout); 

  printf("\n\n   STARTING TEST %i\n", curstat->cur);
  fflush(stdout);

  init_test();
  result = tcase->test();
  post_test();

  /* check result based on EXPECTED */
  if(tcase->ex_result == EXPECT_OK){
    PRINTT(curstat->cur, tcase, result==1); /* this goes to the debug file */

    if(result==1)
      exit(0);
    else exit(1);
  }
  else {
    PRINTT_(curstat->cur, tcase, 0); /* this goes to the debug file */
    printf(" didn't throw assert\n");
    exit(2);
  }
}



/** 
 * forks and the sets stdout & stderr to a file and calls do_test under 
 * the fork. reports output after the forked one have died.
 */
static void test(testcase* tcase)
{
  int pid=0, statloc;

  pid = fork();
  if( pid < 0){
    fprintf(stderr, "forking failed\n");
    exit(1);
  }
  else if(pid == 0){
    do_test(tcase);
  }
  else{ 
    DPRINT("starting wait...\n");
    wait(&statloc);
    DPRINT("finished wait...\n");

    report(tcase, statloc, curstat);    
  }
}





/*----------------- MAIN AND HELPERS -----------------*/

/** 
 * checks if this test case is a duplicate of some earlier test case
 * @param cur current test (the one to test)
 * @param tcases the first testcase in the array
 * @return 0 if this ain't duplicate, number of duplicated test if this is
 */
static int duplicate(testcase *cur, testcase *tcases)
{
  /* obs. curtest == index of current test + 1 */
  testcase *t = tcases;
  int i;
  for(i=0; i<curstat->cur-1; i++,t++){
    if(t->test == cur->test)
      return i+1;
  }
  return 0;
}


#define PRINT_USAGE {fprintf(stderr, \
 "\nusage:\n%s\n%s\n%s\n %s\n \n%s\n%s \n%s\n%s \n%s\n%s \n%s\n%s \n\n",\
 "outo [options] libname [libname ...]", \
 "      where library must be a dynamic library (libnmame.so)", \
 "      in LD_LIBRARY_PATH or path to the library", \
 "options: ", \
 " -d debugfile \n --debug debugfile", \
 "      sets the debug to go to specified file", \
 " -f\n --force", \
 "      forces writing over debug file, if file exists", \
 " -v\n --version", \
 "      prints version number and exits", \
 " -h\n --help", \
 "      prints this message and exits" \
 ); exit(1);}

/*
%s\n%s\n%s\n%s\n%s\n%s \n",\
"Possible options are:", \
" -o int          sets the explanation field width in output", \
"                 without -d, -f means deleting the default debug file", \
"                 /tmp/.movialtester, which would otherwice be appended" \
*/


int main(int argc, char **argv){
  char *testlib=0, *dpath=0;
  char **tmp=0;
  testcase* (*getfunc)();
  testcase *tcases; /* all the tests in current lib*/
  int duplicates=0, totaldup=0;
  stat totals = {0,0,0,0};

  int force=0;
  FILE *f; /* filehandle for testing openings ect */
  time_t t;

  /* get params */
  if(argc<2)
    PRINT_USAGE;

  while (1) {
    int this_option_optind = optind ? optind : 1;
    int option_index = 0;
    int c;
    static struct option long_options[] = {
      {"debug", 1, 0, 'd'},
      {"force", 0, 0, 'f'},
      {"version", 0, 0, 'v'},
      {"help", 0, 0, 'h'},
      {0}
    };
    c = getopt_long (argc, argv, "d:f",
                     long_options, &option_index);
    if (c == -1)
      break; /* ends the while */
   
    switch (c) {
      /*
    case 0:
      printf ("option %s", long_options[option_index].name);
      if (optarg)
        printf (" with arg %s", optarg);
      printf ("\n");
      break;
      */
    case 'd':
      dpath=optarg;
      break;
    case 'f':
      force=1;
      break;
    case 'v':
      /* change to get number from configure */
      printf("Outo version 0.1.1\n\n"); 
      exit(0);
    case 'h':
    case '?': /* some better output here */
      PRINT_USAGE;
      break;
    default:
      /* this do be bad and needs better commenst althoug should never happen*/
      fprintf(stderr, "?? getopt returned character code 0%o ??\n", c);
      exit(1);
    }
  }

  /*check at least one testslib name is given*/
  if(optind >= argc){
    fprintf(stderr, "please give at least one testlib name\n");
    PRINT_USAGE;
  }
  /* set debug file */
  if(dpath != 0){
    if( (f = fopen(dpath, "r")) != NULL && !force){
      fprintf(stderr, "\ndebug file '%s' exists already\n", dpath);
      fprintf(stderr, "to override file, use -f or --force\n");
      exit(-1);
    }
    else if( (f = fopen(dpath, "w")) != NULL){
      /* printf("file '%s' opened for writing\n", dpath); */
      fprintf(f, "Outo debug file\n");
      time(&t);
      fprintf(f, "%s\n", ctime(&t));      
      debugfile = dpath;
      fclose(f);
    }
    else{
      fprintf(stderr, "\ndebug file '%s' could not be opened\n", dpath);
      exit(-1);
    }
  }
  else if( (f = fopen(debugfile, "w")) != NULL){
    /*printf("file %s opened for writing\n", dpath); */
    fprintf(f, "Outo debug file\n");
    time(&t);
    fprintf(f, "%s\n", ctime(&t));  
    fclose(f);
  }
  else{
    fprintf(stderr, "\ndebug file '%s' could not be opened\n", debugfile);
    exit(-1);
  }

  /* here starts the real work */
  begin_report();

  /* iterate trough testlibs */
  while(optind < argc){
    testcase *te; /* just a helper for iteration */
    testlib = argv[optind];
    
    curstat->tests=0;
    curstat->pass=0;
    curstat->fail=0;
    curstat->cur=1;
    duplicates=0;


    /* get the handle */
    handle = dlopen (testlib, RTLD_NOW);
    if (handle == NULL) {
      fprintf (stderr, 
               "coudn't open the test library '%s': %s\n", 
               testlib, dlerror());
      break;
    }
    
    getfunc = dlsym(handle, "get_tests");
    if ((error = dlerror()) != NULL)  {
      fprintf (stderr, 
               "coudn't open the get-tests() function in library '%s': %s\n", 
               testlib, error);
      break;
    }


    /* call the tests, check duplicates */
    
    tmp = dlsym(handle, "outo_name");
    if ((error = dlerror()) == NULL && tmp != NULL) 
      /* better testing would be nice */
      testlib = *tmp;

    begin_lib_report(testlib);
    tcases = getfunc();
    te=tcases;
    while(te->test != 0){
      int res;
      if( res=duplicate(te, tcases) ){
        printf("%3i: is a duplicate of test %i\n", curstat->cur, res);
        duplicates++;
      }
      else
        test(te);
      te++;
      curstat->cur++;
    }
    curstat->tests = curstat->cur - duplicates -1;
    end_lib_report(curstat, testlib);
    dlclose(handle);
    optind++;

    totals.tests += curstat->tests;
    totals.pass += curstat->pass;
    totals.fail += curstat->fail;
    totals.cur++; /* using this for counting libraries */

    totaldup += duplicates;
  }   
  /* finish */
  if(totals.cur > 1) /* only if more than one suite*/
    end_report(&totals);
  
  if(totals.fail==0 && totaldup==0)
    return 0;
  else return totals.fail+totaldup;
  
}
