/**
 * @file libgalago/galago-presence.c Galago Presence API
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago/galago-presence.h>
#include <libgalago/galago-context.h>
#include <libgalago/galago-core.h>
#include <libgalago/galago-marshal.h>
#include <libgalago/galago-private.h>
#include <stdio.h>
#include <string.h>

struct _GalagoPresencePrivate
{
	GalagoAccount *account;

	gboolean idle;
	time_t idle_start_time;

	GList *statuses;

	GalagoStatus *active_status;
};

enum
{
	PROP_0,
	PROP_ACCOUNT
};

enum
{
	IDLE_CHANGED,
	STATUS_ADDED,
	STATUS_REMOVED,
	STATUS_UPDATED,
	CHANGED,
	LAST_SIGNAL
};

static void _galago_dbus_account_destroy_presence(GalagoAccount *account,
												  GalagoPresence *presence);
static void _galago_presence_remove_status(GalagoPresence *presence,
										   GalagoStatus *status);

static unsigned int status_type_scores[] =
{
	0,     /* unset         */
	-500,  /* offline       */
	100,   /* available     */
	-50,   /* hidden        */
	-100,  /* away          */
	-200,  /* extended away */
};

/**************************************************************************
 * Object/Class support
 **************************************************************************/
static void galago_presence_destroy(GalagoObject *object);
static void galago_presence_dbus_message_append(DBusMessageIter *iter,
											 const GalagoObject *object);
static void *galago_presence_dbus_message_get(DBusMessageIter *iter);
static gchar *galago_presence_dbus_get_signature(void);
static void galago_presence_set_property(GObject *object, guint prop_id,
										 const GValue *value,
										 GParamSpec *pspec);
static void galago_presence_get_property(GObject *object, guint prop_id,
										 GValue *value, GParamSpec *pspec);

static GalagoObjectClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = {0};

G_DEFINE_TYPE(GalagoPresence, galago_presence, GALAGO_TYPE_OBJECT);

static void
galago_presence_class_init(GalagoPresenceClass *klass)
{
	GalagoObjectClass *object_class  = GALAGO_OBJECT_CLASS(klass);
	GObjectClass      *gobject_class = G_OBJECT_CLASS(klass);

	parent_class = g_type_class_peek_parent(klass);

	object_class->destroy = galago_presence_destroy;
	object_class->dbus_interface = GALAGO_DBUS_PRESENCE_INTERFACE;

	object_class->dbus_message_append = galago_presence_dbus_message_append;
	object_class->dbus_message_get    = galago_presence_dbus_message_get;
	object_class->dbus_get_signature  = galago_presence_dbus_get_signature;

	gobject_class->set_property = galago_presence_set_property;
	gobject_class->get_property = galago_presence_get_property;

	signals[IDLE_CHANGED] =
		g_signal_new("idle_changed",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoPresenceClass, idle_changed),
					 NULL, NULL,
					 galago_marshal_VOID__BOOLEAN_ULONG,
					 G_TYPE_NONE, 2,
					 G_TYPE_BOOLEAN,
					 G_TYPE_ULONG);

	signals[STATUS_ADDED] =
		g_signal_new("status_added",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoPresenceClass, status_added),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__POINTER,
					 G_TYPE_NONE, 1,
					 G_TYPE_POINTER);

	signals[STATUS_REMOVED] =
		g_signal_new("status_removed",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoPresenceClass, status_removed),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__POINTER,
					 G_TYPE_NONE, 1,
					 G_TYPE_POINTER);

	signals[STATUS_UPDATED] =
		g_signal_new("status_updated",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoPresenceClass, status_updated),
					 NULL, NULL,
					 galago_marshal_VOID__POINTER_STRING,
					 G_TYPE_NONE, 2,
					 G_TYPE_POINTER,
					 G_TYPE_STRING);

	signals[CHANGED] =
		g_signal_new("changed",
					 G_TYPE_FROM_CLASS(klass),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoPresenceClass, changed),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__VOID,
					 G_TYPE_NONE, 0);

	g_object_class_install_property(gobject_class, PROP_ACCOUNT,
		g_param_spec_object("account", "Account",
							"The account that owns this presence",
							GALAGO_TYPE_ACCOUNT,
							G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY |
							G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
}

static void
galago_presence_init(GalagoPresence *presence)
{
	presence->priv = g_new0(GalagoPresencePrivate, 1);
}

static void
galago_presence_destroy(GalagoObject *object)
{
	GalagoPresence *presence = GALAGO_PRESENCE(object);

	if (presence->priv != NULL)
	{
		GalagoAccount *account;

		account = galago_presence_get_account(presence);

		if (account != NULL && GALAGO_OBJECT_IS_LOCAL(presence) &&
			galago_account_get_presence(account, FALSE) == presence)
		{
			_galago_account_set_presence(account, NULL);
			_galago_dbus_account_destroy_presence(account, presence);
		}

		if (presence->priv->statuses != NULL)
		{
			g_list_foreach(presence->priv->statuses,
						   (GFunc)galago_object_destroy, NULL);
			g_list_free(presence->priv->statuses);
		}

		g_free(presence->priv);
		presence->priv = NULL;
	}

	if (GALAGO_OBJECT_CLASS(parent_class)->destroy != NULL)
		GALAGO_OBJECT_CLASS(parent_class)->destroy(object);
}

static void
galago_presence_dbus_message_append(DBusMessageIter *iter,
								   const GalagoObject *object)
{
	GalagoPresence *presence = (GalagoPresence *)object;
	gboolean is_idle;
	dbus_uint32_t idle_time;

	galago_dbus_message_iter_append_string_or_nil(iter,
		galago_object_get_dbus_path(object));

	galago_dbus_message_iter_append_object(iter,
		GALAGO_OBJECT(galago_presence_get_account(presence)));

	is_idle = galago_presence_is_idle(presence);
	dbus_message_iter_append_basic(iter, DBUS_TYPE_BOOLEAN, &is_idle);

	idle_time = galago_presence_get_idle_time(presence);
	dbus_message_iter_append_basic(iter, DBUS_TYPE_UINT32, &idle_time);

	galago_dbus_message_iter_append_object_list(iter, GALAGO_TYPE_STATUS,
		galago_presence_get_statuses(presence));
}

static void *
galago_presence_dbus_message_get(DBusMessageIter *iter)
{
	GalagoPresence *presence;
	GalagoAccount *account;
	GList *l, *statuses;
	const char *obj_path;
	gboolean idle;
	time_t idle_start_time = 0;

	obj_path = galago_dbus_message_iter_get_string_or_nil(iter);
	dbus_message_iter_next(iter);

	account = galago_dbus_message_iter_get_object(iter, GALAGO_TYPE_ACCOUNT);
	dbus_message_iter_next(iter);

	dbus_message_iter_get_basic(iter, &idle);
	dbus_message_iter_next(iter);

	dbus_message_iter_get_basic(iter, &idle_start_time);
	dbus_message_iter_next(iter);

	presence = _galago_presence_new(account, obj_path);
	galago_presence_set_idle(presence, idle, idle_start_time);

	statuses = galago_dbus_message_iter_get_object_list(iter,
														GALAGO_TYPE_STATUS);

	for (l = statuses; l != NULL; l = l->next)
		galago_presence_add_status(presence, GALAGO_STATUS(l->data));

	return presence;
}

static gchar *
galago_presence_dbus_get_signature(void)
{
	return g_strconcat(
		DBUS_TYPE_STRING_AS_STRING,  // object path
		galago_object_type_get_dbus_signature(GALAGO_TYPE_ACCOUNT), // account
		DBUS_TYPE_BOOLEAN_AS_STRING, // idle flag
		DBUS_TYPE_UINT32_AS_STRING,  // idle time
		DBUS_TYPE_ARRAY_AS_STRING,   // array of statuses
		galago_object_type_get_dbus_signature(GALAGO_TYPE_STATUS),
		NULL);
}

static void
galago_presence_set_property(GObject *object, guint prop_id,
							 const GValue *value, GParamSpec *pspec)
{
	GalagoPresence *presence = GALAGO_PRESENCE(object);

	switch (prop_id)
	{
		case PROP_ACCOUNT:
			presence->priv->account =
				GALAGO_ACCOUNT(g_value_get_object(value));
			g_object_notify(G_OBJECT(presence), "account");
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void
galago_presence_get_property(GObject *object, guint prop_id,
							 GValue *value, GParamSpec *pspec)
{
	GalagoPresence *presence = GALAGO_PRESENCE(object);

	switch (prop_id)
	{
		case PROP_ACCOUNT:
			g_value_set_object(value, galago_presence_get_account(presence));
			break;

		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

GalagoPresence *
_galago_presence_new(GalagoAccount *account, const char *obj_path)
{
	GalagoPresence *presence;

	g_return_val_if_fail(galago_is_initted(),        NULL);
	g_return_val_if_fail(account != NULL,            NULL);
	g_return_val_if_fail(GALAGO_IS_ACCOUNT(account), NULL);
	g_return_val_if_fail(obj_path == NULL || *obj_path != '\0', NULL);

	presence = galago_account_get_presence(account, FALSE);

	if (presence == NULL)
	{
		galago_context_push(galago_object_get_context(GALAGO_OBJECT(account)));
		presence = g_object_new(GALAGO_TYPE_PRESENCE,
			"origin",  galago_object_get_origin(GALAGO_OBJECT(account)),
			"account", account,
			NULL);
		galago_context_pop();

		if (obj_path == NULL)
			_galago_dbus_account_create_presence(account, presence);
		else
			galago_object_set_dbus_path(GALAGO_OBJECT(presence), obj_path);

		_galago_account_set_presence(account, presence);
	}

	return presence;
}


/**************************************************************************
 * GalagoPresence API
 **************************************************************************/
void
galago_presence_set_idle(GalagoPresence *presence, gboolean idle,
						 time_t idle_start_time)
{
	GalagoPerson *person;
	GalagoAccount *account;

	g_return_if_fail(presence != NULL);
	g_return_if_fail(GALAGO_IS_PRESENCE(presence));
	g_assert(!idle || idle_start_time > 0);

	if (presence->priv->idle == idle &&
		presence->priv->idle_start_time == idle_start_time)
	{
		return;
	}

	presence->priv->idle = idle;
	presence->priv->idle_start_time = idle_start_time;

	account = galago_presence_get_account(presence);
	person  = galago_account_get_person(account);

	if (GALAGO_OBJECT_IS_LOCAL(person) && galago_is_connected() &&
		galago_is_feed())
	{
		galago_dbus_send_message(GALAGO_OBJECT(presence), "SetIdle",
			galago_value_new(GALAGO_VALUE_TYPE_BOOLEAN, &idle, NULL),
			galago_value_new(GALAGO_VALUE_TYPE_ULONG, &idle_start_time, NULL),
			NULL);
	}

	g_signal_emit(presence, signals[IDLE_CHANGED], 0, idle, idle_start_time);
	g_signal_emit(presence, signals[CHANGED], 0);
}

void
galago_presence_set_statuses(GalagoPresence *presence, GList *statuses)
{
	GList *l;
	GalagoAccount *account;

	g_return_if_fail(presence != NULL);
	g_return_if_fail(GALAGO_IS_PRESENCE(presence));

	if (presence->priv->statuses == statuses)
		return;

	if (presence->priv->statuses != NULL)
	{
		for (l = presence->priv->statuses; l != NULL; l = l->next)
		{
			GalagoStatus *status = (GalagoStatus *)l->data;

			g_signal_emit(presence, signals[STATUS_REMOVED], 0, status);

			galago_object_destroy(GALAGO_OBJECT(status));
		}

		g_list_free(presence->priv->statuses);

		presence->priv->active_status = NULL;
	}

	presence->priv->statuses = statuses;

	account = galago_presence_get_account(presence);

	if (GALAGO_OBJECT_IS_LOCAL(account) &&
		galago_is_connected() && galago_is_feed())
	{
		galago_dbus_send_message(GALAGO_OBJECT(presence), "SetStatuses",
			galago_value_new_list(GALAGO_VALUE_TYPE_OBJECT,
								  galago_presence_get_statuses(presence),
								  (void *)GALAGO_TYPE_STATUS),
			NULL);
	}

	for (l = statuses; l != NULL; l = l->next)
		g_signal_emit(presence, signals[STATUS_ADDED], 0, l->data);

	g_signal_emit(presence, signals[CHANGED], 0);
}

void
galago_presence_add_status(GalagoPresence *presence, GalagoStatus *status)
{
	GalagoAccount *account;
	GalagoPerson *person;
	const char *status_id;

	g_return_if_fail(presence != NULL);
	g_return_if_fail(status   != NULL);
	g_return_if_fail(GALAGO_IS_PRESENCE(presence));
	g_return_if_fail(GALAGO_IS_STATUS(status));

	status_id = galago_status_get_id(status);

	account = galago_presence_get_account(presence);
	person  = galago_account_get_person(account);

	if (galago_presence_get_status(presence, status_id) != NULL)
	{
		GalagoAccount *account = galago_presence_get_account(presence);

		g_warning("A status with ID %s has already been added "
				  "to the presence for account %s",
				  status_id, galago_account_get_username(account));

		galago_object_destroy(GALAGO_OBJECT(status));

		return;
	}

	if (GALAGO_OBJECT_IS_LOCAL(person) && galago_is_connected() &&
		galago_is_feed())
	{
		char *obj_path = galago_dbus_send_message_with_reply(
			GALAGO_OBJECT(presence), "AddStatus",
			galago_value_new(GALAGO_VALUE_TYPE_STRING, NULL, NULL),
			galago_value_new_object(GALAGO_TYPE_STATUS, G_OBJECT(status)),
			NULL);

		g_assert(obj_path != NULL);

		galago_object_set_dbus_path(GALAGO_OBJECT(status), obj_path);
		g_free(obj_path);
	}

	if (galago_status_is_exclusive(status))
	{
		GalagoStatus *active_status;

		active_status = galago_presence_get_active_status(presence);

		if (active_status != NULL)
			_galago_presence_remove_status(presence, active_status);

		presence->priv->active_status = status;
	}

	presence->priv->statuses = g_list_append(presence->priv->statuses, status);

	galago_status_set_presence(status, presence);

	g_signal_emit(presence, signals[STATUS_ADDED], 0, status);
	g_signal_emit(presence, signals[CHANGED], 0);
}

static void
_galago_presence_remove_status(GalagoPresence *presence, GalagoStatus *status)
{
	GalagoAccount *account;

	g_return_if_fail(presence != NULL);
	g_return_if_fail(status   != NULL);
	g_return_if_fail(GALAGO_IS_PRESENCE(presence));
	g_return_if_fail(GALAGO_IS_STATUS(status));

	presence->priv->statuses = g_list_remove(presence->priv->statuses, status);

	account = galago_presence_get_account(presence);

	if (GALAGO_OBJECT_IS_LOCAL(account) &&
		!galago_status_is_exclusive(status) && galago_is_connected() &&
		galago_is_feed())
	{
		const char *id = galago_status_get_id(status);

		galago_dbus_send_message(GALAGO_OBJECT(presence), "RemoveStatus",
			galago_value_new(GALAGO_VALUE_TYPE_STRING, &id, NULL),
			NULL);
	}

	g_signal_emit(presence, signals[STATUS_REMOVED], 0, status);
	g_signal_emit(presence, signals[CHANGED], 0);

	galago_object_destroy(GALAGO_OBJECT(status));
}

void
galago_presence_remove_status(GalagoPresence *presence, const char *status_id)
{
	GalagoStatus *status;

	g_return_if_fail(presence  != NULL);
	g_return_if_fail(status_id != NULL && *status_id != '\0');
	g_return_if_fail(GALAGO_IS_PRESENCE(presence));

	status = galago_presence_get_status(presence, status_id);

	if (status == NULL)
	{
		GalagoAccount *account = galago_presence_get_account(presence);

		g_warning("Attempted to remove an unknown status %s from "
				  "the presence for account %s",
				  status_id, galago_account_get_username(account));

		return;
	}

	g_return_if_fail(!galago_status_is_exclusive(status));

	_galago_presence_remove_status(presence, status);
}

void
galago_presence_clear_statuses(GalagoPresence *presence)
{
	g_return_if_fail(presence != NULL);
	g_return_if_fail(GALAGO_IS_PRESENCE(presence));

	galago_presence_set_statuses(presence, NULL);
}

GalagoAccount *
galago_presence_get_account(const GalagoPresence *presence)
{
	g_return_val_if_fail(presence != NULL,             NULL);
	g_return_val_if_fail(GALAGO_IS_PRESENCE(presence), NULL);

	return presence->priv->account;
}

gboolean
galago_presence_is_idle(const GalagoPresence *presence)
{
	g_return_val_if_fail(presence != NULL,             FALSE);
	g_return_val_if_fail(GALAGO_IS_PRESENCE(presence), FALSE);

	return presence->priv->idle;
}

time_t
galago_presence_get_idle_time(const GalagoPresence *presence)
{
	g_return_val_if_fail(presence != NULL,             0);
	g_return_val_if_fail(GALAGO_IS_PRESENCE(presence), 0);

	return time(NULL) - presence->priv->idle_start_time;
}

time_t
galago_presence_get_idle_start_time(const GalagoPresence *presence)
{
	g_return_val_if_fail(presence != NULL,             0);
	g_return_val_if_fail(GALAGO_IS_PRESENCE(presence), 0);

	return presence->priv->idle_start_time;
}

gboolean
galago_presence_is_discarded(const GalagoPresence *presence)
{
	g_return_val_if_fail(presence != NULL,             TRUE);
	g_return_val_if_fail(GALAGO_IS_PRESENCE(presence), TRUE);

	return (galago_presence_get_statuses(presence) == NULL);
}

gboolean
galago_presence_is_available(const GalagoPresence *presence)
{
	GalagoStatus *status;

	g_return_val_if_fail(presence != NULL,             FALSE);
	g_return_val_if_fail(GALAGO_IS_PRESENCE(presence), FALSE);

	status = galago_presence_get_active_status(presence);

	return ((status != NULL && galago_status_is_available(status)) &&
			!galago_presence_is_idle(presence));
}

GList *
galago_presence_get_statuses(const GalagoPresence *presence)
{
	g_return_val_if_fail(presence != NULL,             NULL);
	g_return_val_if_fail(GALAGO_IS_PRESENCE(presence), NULL);

	return presence->priv->statuses;
}

GalagoStatus *
galago_presence_get_active_status(const GalagoPresence *presence)
{
	g_return_val_if_fail(presence != NULL,             NULL);
	g_return_val_if_fail(GALAGO_IS_PRESENCE(presence), NULL);

	return presence->priv->active_status;
}

gboolean
galago_presence_is_status_exclusive(const GalagoPresence *presence,
									const char *status_id)
{
	GalagoStatus *status;

	g_return_val_if_fail(presence  != NULL,                       FALSE);
	g_return_val_if_fail(status_id != NULL && *status_id != '\0', FALSE);
	g_return_val_if_fail(GALAGO_IS_PRESENCE(presence),            FALSE);

	status = galago_presence_get_status(presence, status_id);

	if (status == NULL)
		return FALSE;

	return galago_status_is_exclusive(status);
}

GalagoStatus *
galago_presence_get_status(const GalagoPresence *presence,
						   const char *status_id)
{
	GList *l;

	g_return_val_if_fail(presence  != NULL,            NULL);
	g_return_val_if_fail(status_id != NULL,            NULL);
	g_return_val_if_fail(GALAGO_IS_PRESENCE(presence), NULL);

	for (l = galago_presence_get_statuses(presence);
		 l != NULL;
		 l = l->next)
	{
		GalagoStatus *status = (GalagoStatus *)l->data;

		if (!strcmp(galago_status_get_id(status), status_id))
			return status;
	}

	return NULL;
}

gboolean
galago_presence_has_status(const GalagoPresence *presence,
						   const char *status_id)
{
	g_return_val_if_fail(presence  != NULL,            FALSE);
	g_return_val_if_fail(status_id != NULL,            FALSE);
	g_return_val_if_fail(GALAGO_IS_PRESENCE(presence), FALSE);

	return (galago_presence_get_status(presence, status_id) != NULL);
}

gboolean
galago_presence_has_status_type(const GalagoPresence *presence,
								GalagoStatusType type)
{
	GList *l;

	g_return_val_if_fail(presence != NULL,             FALSE);
	g_return_val_if_fail(GALAGO_IS_PRESENCE(presence), FALSE);
	g_return_val_if_fail(type != GALAGO_STATUS_UNSET,  FALSE);

	for (l = galago_presence_get_statuses(presence);
		 l != NULL;
		 l = l->next)
	{
		GalagoStatus *status = (GalagoStatus *)l->data;

		if (galago_status_get_primitive(status) == type)
			return TRUE;
	}

	return FALSE;
}

int
galago_presence_compare(const GalagoPresence *presence1,
						const GalagoPresence *presence2)
{
	int score1 = 0, score2 = 0;
	GList *l;

	g_return_val_if_fail(presence1 == NULL || GALAGO_IS_PRESENCE(presence1),
						 1);
	g_return_val_if_fail(presence2 == NULL || GALAGO_IS_PRESENCE(presence2),
						 -1);

	if (presence1 == presence2)
		return 0;
	else if (presence1 == NULL)
		return 1;
	else if (presence2 == NULL)
		return -1;

	/* Compute the score of the first set of statuses. */
	for (l = galago_presence_get_statuses(presence1); l != NULL; l = l->next)
	{
		GalagoStatus *status = (GalagoStatus *)l->data;
		score1 += status_type_scores[galago_status_get_primitive(status)];
	}

	/* Compute the score of the second set of statuses. */
	for (l = galago_presence_get_statuses(presence2); l != NULL; l = l->next)
	{
		GalagoStatus *status = (GalagoStatus *)l->data;
		score2 += status_type_scores[galago_status_get_primitive(status)];
	}

	if (score1 > score2)
		return 1;
	else if (score1 < score2)
		return -1;
	else
	{
		gboolean idle1 = galago_presence_is_idle(presence1);
		gboolean idle2 = galago_presence_is_idle(presence2);

		if (!idle1 && !idle2)
			return 0;
		else if (idle1 && !idle2)
			return 1;
		else if (!idle1 && idle2)
			return -1;
		else
		{
			time_t idle_time_1;
			time_t idle_time_2;

			idle_time_1 = galago_presence_get_idle_time(presence1);
			idle_time_2 = galago_presence_get_idle_time(presence2);

			if (idle_time_1 < idle_time_2)
				return 1;
			else if (idle_time_1 > idle_time_2)
				return -1;
			else
				return 0;
		}
	}
}

void
_galago_dbus_account_create_presence(GalagoAccount *account,
									 GalagoPresence *presence)
{
	char *obj_path;

	if (!galago_is_connected() || !galago_is_feed())
		return;

	obj_path = galago_dbus_send_message_with_reply(
		GALAGO_OBJECT(account), "CreatePresence",
		galago_value_new(GALAGO_VALUE_TYPE_STRING, NULL, NULL),
		NULL);

	g_assert(obj_path != NULL);

	galago_object_set_dbus_path(GALAGO_OBJECT(presence), obj_path);
	g_free(obj_path);
}

static void
_galago_dbus_account_destroy_presence(GalagoAccount *account,
									  GalagoPresence *presence)
{
	if (!galago_is_connected() || !galago_is_feed())
		return;

	galago_dbus_send_message(GALAGO_OBJECT(account), "DestroyPresence", NULL);
}
