/**
 * @file libgalago/galago-account.h Galago Account API
 * @ingroup Presence
 *
 * @Copyright (C) 2004-2006 Christian Hammond
 *
 * #GalagoAccount models a single account a #GalagoPerson has with a
 * provider (a #GalagoService).
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#ifndef _GALAGO_ACCOUNT_H_
#define _GALAGO_ACCOUNT_H_

typedef struct _GalagoAccount        GalagoAccount;
typedef struct _GalagoAccountClass   GalagoAccountClass;
typedef struct _GalagoAccountPrivate GalagoAccountPrivate;

#include <libgalago/galago-image.h>
#include <libgalago/galago-object.h>
#include <libgalago/galago-person.h>
#include <libgalago/galago-presence.h>
#include <libgalago/galago-service.h>

/**
 * An account on a service.
 */
struct _GalagoAccount
{
	GalagoObject parent_object;

	GalagoAccountPrivate *priv;

#ifndef _DOXYGEN_
	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
#endif
};

/**
 * An account's class.
 */
struct _GalagoAccountClass
{
	GalagoObjectClass parent_class;

	/* Signals */
	void (*connection_state_changed)(GalagoAccount *account);
	void (*display_name_changed)(GalagoAccount *account);
	void (*presence_created)(GalagoAccount *account, GalagoPresence *presence);
	void (*presence_deleted)(GalagoAccount *account);
	void (*avatar_set)(GalagoAccount *account, GalagoImage *image);
	void (*contact_added)(GalagoAccount *account, GalagoAccount *contact);
	void (*contact_removed)(GalagoAccount *account, GalagoAccount *contact);

#ifndef _DOXYGEN_
	void (*reserved_1)(void);
	void (*reserved_2)(void);
	void (*reserved_3)(void);
	void (*reserved_4)(void);
#endif
};

#define GALAGO_TYPE_ACCOUNT (galago_account_get_type())

/** Cast @c obj to a GalagoAccount. */
#define GALAGO_ACCOUNT(obj) \
		(G_TYPE_CHECK_INSTANCE_CAST((obj), GALAGO_TYPE_ACCOUNT, \
									GalagoAccount))

/** Cast @c klass to a GalagoAccount. */
#define GALAGO_ACCOUNT_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_CAST((klass), GALAGO_TYPE_ACCOUNT, \
								 GalagoAccountClass))

/** Check if @c obj is a GalagoAccount. */
#define GALAGO_IS_ACCOUNT(obj) \
		(G_TYPE_CHECK_INSTANCE_TYPE((obj), GALAGO_TYPE_ACCOUNT))

/** Check if @c klass is a GalagoAccountClass. */
#define GALAGO_IS_ACCOUNT_CLASS(klass) \
		(G_TYPE_CHECK_CLASS_TYPE((klass), GALAGO_TYPE_ACCOUNT))

/** Get the class for the GalagoAccount @c obj. */
#define GALAGO_ACCOUNT_GET_CLASS(obj) \
		(G_TYPE_INSTANCE_GET_CLASS((obj), GALAGO_TYPE_ACCOUNT, \
								   GalagoAccountClass))

#define GALAGO_DBUS_ACCOUNT_INTERFACE "org.freedesktop.Galago.Account"

G_BEGIN_DECLS

/**************************************************************************/
/** @name Account API                                                     */
/**************************************************************************/
/*@{*/

/**
 * Returns GalagoAccount's GType.
 *
 * @return The GType.
 */
GType galago_account_get_type(void);

/**
 * Sets whether or not the account is connected.
 *
 * @param account   The account.
 * @param connected The connected state.
 */
void galago_account_set_connected(GalagoAccount *account,
								  gboolean connected);

/**
 * Returns an account's service.
 *
 * @param account The account.
 *
 * @return The account's service.
 */
GalagoService *galago_account_get_service(const GalagoAccount *account);

/**
 * Returns the person that owns an account.
 *
 * @param account The account.
 *
 * @return The person that owns the account.
 */
GalagoPerson *galago_account_get_person(const GalagoAccount *account);

/**
 * Returns an account's username.
 *
 * @param account The account.
 *
 * @return The account's username.
 */
const char *galago_account_get_username(const GalagoAccount *account);

/**
 * Returns whether or not an account is connected.
 *
 * @param account The account.
 *
 * @return The account's connected state.
 */
gboolean galago_account_is_connected(const GalagoAccount *account);

/**
 * Sets the account's displayed name.
 *
 * This is the alias that the account may be shown as on another client.
 * It's purely optional.
 *
 * @param account      The account.
 * @param display_name The display name.
 */
void galago_account_set_display_name(GalagoAccount *account,
									 const char *display_name);

/**
 * Returns the account's displayed name.
 *
 * If the displayed name is not explicitly set, this will return the
 * screen name.
 *
 * @param account The account.
 *
 * @return The displayed name.
 */
const char *galago_account_get_display_name(const GalagoAccount *account);

/**
 * Returns whether or not a custom displayed name is set.
 *
 * @param account The account.
 *
 * @return TRUE if a custom displayed name is set, or FALSE.
 */
gboolean galago_account_is_display_name_set(const GalagoAccount *account);

/**
 * Adds a contact to an account.
 *
 * @param account The account.
 * @param contact The contact's account to add.
 */
void galago_account_add_contact(GalagoAccount *account,
								GalagoAccount *contact);

/**
 * Removes a contact from an account.
 *
 * @param account The account.
 * @param contact The contact's account to remove.
 */
void galago_account_remove_contact(GalagoAccount *account,
								   GalagoAccount *contact);

/**
 * Returns the contact with the specified username in an account.
 *
 * @param account  The account.
 * @param username The username.
 * @param query    TRUE if a remote query should be done if there is no
 *                 local contact found, or FALSE.
 *
 * @return The contact's account, if found, or NULL.
 */
GalagoAccount *galago_account_get_contact(const GalagoAccount *account,
										  const char *username,
										  gboolean query);

/**
 * Returns a list of accounts of users seen from this account.
 *
 * This may emit a contact-added signal for every object that returns. If
 * your code connects to this signal and calls galago_account_get_contacts()
 * as a result, you will want to add a lock so that you don't end up with
 * unwanted side-effects.
 *
 * @param account The account.
 * @param query   TRUE if a remote query should be done if there is no
 *                local contact found, or FALSE.
 *
 * @return A list of accounts of other users, or NULL.
 */
GList *galago_account_get_contacts(const GalagoAccount *account,
								   gboolean query);

/**
 * Creates an account's presence.
 *
 * The account should be a local account.
 *
 * If the account already has a presence, this will return the existing one.
 *
 * @param account The account.
 *
 * @return The presence.
 */
GalagoPresence *galago_account_create_presence(GalagoAccount *account);

/**
 * Returns the account's presence.
 *
 * @param account The account.
 * @param query   TRUE if a remote query should be done if there is no
 *                local presence found, or FALSE.
 *
 * @return The presence, if found, or NULL.
 */
GalagoPresence *galago_account_get_presence(const GalagoAccount *account,
											gboolean query);

/**
 * Sets the account's avatar.
 *
 * The account should be a local account.
 *
 * @param account The account.
 * @param avatar  The avatar to set.
 */
void galago_account_set_avatar(GalagoAccount *account, GalagoImage *avatar);

/**
 * Returns the account's avatar.
 *
 * @param account The account.
 * @param query   TRUE if a remote query should be done if there is no
 *                local avatar found, or FALSE.
 *
 * @return The avatar, if found, or NULL.
 */
GalagoImage *galago_account_get_avatar(const GalagoAccount *account,
									   gboolean query);

/*@}*/

G_END_DECLS

#endif /* _GALAGO_ACCOUNT_H_ */
