#include <linux/module.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <asm/arch/gpio.h>
#include <../drivers/cbus/tahvo.h>

#define MODULE_NAME "omap_usbtest"
#define OMAP_USBTEST_TIMEOUT_MS	100

#define OMAP_USBTEST_PASSED	0
#define OMAP_USBTEST_FAILED	-1
#define OMAP_USBTEST_INIT_ERROR	-2

#define PU_PD_SEL_NA		0	/* No pu_pd reg available */
#define PULL_DWN_CTRL_NA	0	/* No pull-down control needed */

#define MUX_REG(reg, mode_offset, mode) .mux_reg = reg, \
					.mask_offset = mode_offset, \
					.mask = mode,

#define PULL_REG(reg, bit, status)	.pull_reg = PULL_DWN_CTRL_##reg, \
					.pull_bit = bit, \
					.pull_val = status,

#define PU_PD_REG(reg, status)		.pu_pd_reg = PU_PD_SEL_##reg, \
					.pu_pd_val = status,

#define MUX_CFG(desc, mux_reg, mode_offset, mode,	\
		pull_reg, pull_bit, pull_status,	\
		pu_pd_reg, pu_pd_status)		\
{							\
	.name =	 desc,					\
	MUX_REG(mux_reg, mode_offset, mode)		\
	PULL_REG(pull_reg, pull_bit, pull_status)	\
	PU_PD_REG(pu_pd_reg, pu_pd_status)		\
},

typedef struct {
	char *name;

	const unsigned int mux_reg;
	const unsigned char mask_offset;
	const unsigned char mask;

	const unsigned int pull_reg;
	const unsigned char pull_val;
	const unsigned char pull_bit;

	const unsigned int pu_pd_reg;
	const unsigned char pu_pd_val;
} usbtest_reg_cfg_set;

enum usbtest_gpios {
	GPIO_USB_PUEN = 8,
	GPIO_USB0_TXEN = 25,
	GPIO_USB0_VM = 18,
	GPIO_USB0_VP = 26
};

typedef enum {
	USBTEST_P9_GPIO8,
	USBTEST_W9_GPIO25,
	USBTEST_R9_GPIO18,
	USBTEST_AA9_GPIO26,
	USBTEST_P9_USB_PUEN,
	USBTEST_W9_USB0_TXEN,
	USBTEST_R9_USB0_VM,
	USBTEST_AA9_USB0_VP,
} usbtest_reg_cfg_t;

#define USB_TRANSCEIVER_CTRL  (0xfffe1000 + 0x0064)


static usbtest_reg_cfg_set usbtest_reg_cfg_table[] = {
/*
 *	 description		mux        mode mux  pull pull pull pu_pd  pu
 *				reg      offset mode reg  bit  ena  reg
 */
MUX_CFG("P9_GPIO8",	USB_TRANSCEIVER_CTRL, 4,  6,  NA,   0,   0,  NA,    0)
MUX_CFG("W9_GPIO25",	FUNC_MUX_CTRL_B,      9,  7,   2,  19,   0,   2,    0)
MUX_CFG("R9_GPIO18",	FUNC_MUX_CTRL_C,     18,  7,   3,   0,   0,   3,    0)
MUX_CFG("AA9_GPIO26",	FUNC_MUX_CTRL_B,      6,  7,   2,  18,   0,   2,    0)
MUX_CFG("P9_USB_PUEN",  USB_TRANSCEIVER_CTRL, 4,  6,  NA,   0,   0,  NA,    0)
MUX_CFG("W9_USB0_TXEN",	FUNC_MUX_CTRL_B,      9,  5,   2,  19,   0,   2,    0)
MUX_CFG("R9_USB0_VM",   FUNC_MUX_CTRL_C,     18,  5,   3,   0,   0,   3,    0)
MUX_CFG("AA9_USB0_VP",	FUNC_MUX_CTRL_B,      6,  5,   2,  18,   0,   2,    0)
};

/*
 * Sets the Omap MUX and PULL_DWN registers based on the table
 */
static int omap_usbtest_cfg_reg(const usbtest_reg_cfg_t reg_cfg)
{
	usbtest_reg_cfg_set *cfg;
	unsigned int reg_orig = 0, reg = 0, pu_pd_orig = 0, pu_pd = 0,
		pull_orig = 0, pull = 0;
	unsigned int mask;

	if (reg_cfg > ARRAY_SIZE(usbtest_reg_cfg_table)) {
		printk(KERN_ERR "MUX: reg_cfg %d\n", reg_cfg);
		return OMAP_USBTEST_INIT_ERROR;
	}

	cfg = &usbtest_reg_cfg_table[reg_cfg];

	/* Check the mux register in question */
	if (cfg->mux_reg) {
		unsigned	tmp1, tmp2;

		reg_orig = omap_readl(cfg->mux_reg);

		/* The mux registers always seem to be 3 bits long */
		mask = (0x7 << cfg->mask_offset);
		tmp1 = reg_orig & mask;
		reg = reg_orig & ~mask;

		tmp2 = (cfg->mask << cfg->mask_offset);
		reg |= tmp2;

		omap_writel(reg, cfg->mux_reg);
	}

	/* Check for pull up or pull down selection on 1610 */
	if (cfg->pu_pd_reg && cfg->pull_val) {
		pu_pd_orig = omap_readl(cfg->pu_pd_reg);
		mask = 1 << cfg->pull_bit;

		if (cfg->pu_pd_val) {
			/* Use pull up */
			pu_pd = pu_pd_orig | mask;
		} else {
			/* Use pull down */
			pu_pd = pu_pd_orig & ~mask;
		}
		omap_writel(pu_pd, cfg->pu_pd_reg);
	}

	/* Check for an associated pull down register */
	if (cfg->pull_reg) {
		pull_orig = omap_readl(cfg->pull_reg);
		mask = 1 << cfg->pull_bit;

		if (cfg->pull_val) {
			/* Low bit = pull enabled */
			pull = pull_orig & ~mask;
		} else {
			/* High bit = pull disabled */
			pull = pull_orig | mask;
		}

		omap_writel(pull, cfg->pull_reg);
	}

	return 0;
}

static int omap_usbtest_gpio_init(void)
{
	if (omap_request_gpio(GPIO_USB_PUEN))
		return OMAP_USBTEST_INIT_ERROR;

	if (omap_request_gpio(GPIO_USB0_TXEN))
		goto free1;

	if (omap_request_gpio(GPIO_USB0_VM))
		goto free2;

	if (omap_request_gpio(GPIO_USB0_VP))
		goto free3;

	omap_set_gpio_direction(GPIO_USB_PUEN, 0);
	omap_set_gpio_direction(GPIO_USB0_TXEN, 0);
	omap_set_gpio_direction(GPIO_USB0_VM, 1);
	omap_set_gpio_direction(GPIO_USB0_VP, 1);

	/* deassert output pins by default */
	omap_set_gpio_dataout(GPIO_USB_PUEN, 0);
	omap_set_gpio_dataout(GPIO_USB0_TXEN, 0);

	omap_usbtest_cfg_reg(USBTEST_P9_GPIO8);
	omap_usbtest_cfg_reg(USBTEST_W9_GPIO25);
	omap_usbtest_cfg_reg(USBTEST_R9_GPIO18);
	omap_usbtest_cfg_reg(USBTEST_AA9_GPIO26);

	return 0;

free1:
	omap_free_gpio(GPIO_USB0_VM);

free2:
	omap_free_gpio(GPIO_USB0_TXEN);

free3:
	omap_free_gpio(GPIO_USB_PUEN);
	return OMAP_USBTEST_INIT_ERROR;
}

static void omap_usbtest_gpio_cleanup(void)
{
	omap_usbtest_cfg_reg(USBTEST_P9_USB_PUEN);
	omap_usbtest_cfg_reg(USBTEST_W9_USB0_TXEN);
	omap_usbtest_cfg_reg(USBTEST_R9_USB0_VM);
	omap_usbtest_cfg_reg(USBTEST_AA9_USB0_VP);

	omap_free_gpio(GPIO_USB0_VP);
	omap_free_gpio(GPIO_USB0_VM);
	omap_free_gpio(GPIO_USB0_TXEN);
	omap_free_gpio(GPIO_USB_PUEN);
}

static int omap_usbtest_wait_for_response(int level)
{
	unsigned long end = jiffies + msecs_to_jiffies(OMAP_USBTEST_TIMEOUT_MS);
	int usb0_vm;
	int usb0_vp;

	while (1) {
		usb0_vm = omap_get_gpio_datain(GPIO_USB0_VM);
		usb0_vp = omap_get_gpio_datain(GPIO_USB0_VP);

		if (usb0_vm == level && usb0_vp == level)
			return 0;
		if (time_after(jiffies, end))
			break;
	}
	printk(KERN_ERR MODULE_NAME ":level %d not set.USB0_VM %d USB0_VP %d\n",
		level, usb0_vm, usb0_vp);
	return OMAP_USBTEST_FAILED;
}

static int omap_usbtest_perform_test(void)
{
	int r = OMAP_USBTEST_PASSED;
	u16 w;

	if ((r = omap_usbtest_gpio_init()))
		goto out;
	w = tahvo_read_reg(TAHVO_REG_USBR);
	/* REGOUT, Slave_Control, SUSPEND */
	tahvo_write_reg(TAHVO_REG_USBR, (1 << 1) | (1 << 5) | (1 << 8));
	omap_set_gpio_dataout(GPIO_USB_PUEN, 1);
	omap_set_gpio_dataout(GPIO_USB0_TXEN, 1);
	if ((r = omap_usbtest_wait_for_response(1)))
		goto out;
	omap_set_gpio_dataout(GPIO_USB_PUEN, 0);
	if ((r = omap_usbtest_wait_for_response(0)))
		goto out;
out:
	tahvo_write_reg(TAHVO_REG_USBR, w);
	omap_usbtest_gpio_cleanup();
	return r;
}

static ssize_t omap_usbtest_show_result(struct device *dev,
					struct device_attribute *attr,
					char *buf)
{
	char *s;

	switch (omap_usbtest_perform_test()) {
	case OMAP_USBTEST_PASSED:
		s = "passed";
		break;
	case OMAP_USBTEST_FAILED:
		s = "failed";
		break;
	case OMAP_USBTEST_INIT_ERROR:
		s = "init error";
		break;
	default:
		s = "unknown error";
		break;
	}
	return snprintf(buf, PAGE_SIZE, "%s\n", s);
}

static DEVICE_ATTR(run_test, 0440, omap_usbtest_show_result, NULL);

static int omap_usbtest_probe(struct device *dev)
{
	return device_create_file(dev, &dev_attr_run_test);
}

static int omap_usbtest_remove(struct device *dev)
{
	device_remove_file(dev, &dev_attr_run_test);
	return 0;
}

static void omap_usbtest_release_dev(struct device *dev)
{
}

static struct platform_device omap_usbtest_device = {
	.name	= MODULE_NAME,
	.id	= -1,
	.dev = {
		.release = omap_usbtest_release_dev,
	},
	.num_resources = 0,
};

static struct device_driver omap_usbtest_driver = {
	.name = MODULE_NAME,
	.bus = &platform_bus_type,
	.probe = omap_usbtest_probe,
	.remove = omap_usbtest_remove,
};

static int __init omap_usbtest_init(void)
{
	if (platform_device_register(&omap_usbtest_device)) {
		printk(KERN_ERR MODULE_NAME ":device registration failed\n");
		return -ENODEV;
	}

	if (driver_register(&omap_usbtest_driver)) {
		printk(KERN_ERR MODULE_NAME ":driver registration failed\n");
		return -ENODEV;
	}

	return 0;
}

static void __exit omap_usbtest_cleanup(void)
{
	driver_unregister(&omap_usbtest_driver);
	platform_device_unregister(&omap_usbtest_device);
}

module_init(omap_usbtest_init);
module_exit(omap_usbtest_cleanup);

MODULE_LICENSE("GPL");
